/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/
 
#pragma ident "@(#)test_key.c	1.13 95/11/17"
 
#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <netdb.h>
#include <stropts.h>
#include <sys/param.h>
#include <sys/debug.h>
#include <sys/errno.h>
#include <sys/time.h>
#ifdef SYSV
#include <sys/systeminfo.h>
#include <sys/byteorder.h>
#endif
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>

#include <skip_proto.h>
#include <skip_types.h>
#include <skip_keymgrmsgs.h>

extern int		errno;
char			keymgr[] = "/dev/skip_key";
int			debug = 0;

#define	MGTCSIZE	2048
#define	MGTDSIZE	2048
#define	MAX_MESSAGES	5

char			*messages[] = {
			"invalid",
			"SKIP_ENCRYPTED_PKT_KEY_REQ",
			"SKIP_ENCRYPTED_PKT_KEY_RSP",
			"SKIP_DECRYPTED_PKT_KEY_REQ",
			"SKIP_DECRYPTED_PKT_KEY_RSP",
			"SKIP_DH_CERTIFICATE_REQ",
			"SKIP_DH_CERTIFICATE_RSP"
			};

static void
usage(char *who)
{
	fprintf(stderr, "usage: %s [-d]\n", who);
	fprintf(stderr, "-d for debug mode, key = zeroes\n");
	exit(1);
}

static void
print_time()
{
	struct tm		*tm;
	char			buf[128];
	long			clock;

	clock = time((time_t *) 0);
	tm = localtime(&clock);
	strftime(buf, sizeof (buf), "%c", tm);
	printf("%s: ", buf);
}

/*
 * interface to kernel management software
 */
open_device(char *dev)
{
	int	fd;
	printf("Opening %s\n", dev);
	fd = open(dev, O_RDWR);
	if (fd < 0) {
		perror("open");
		exit(1);
	}
	return (fd);
}

/* skip_keymgr_getmsg()
 *
 * Retrieve a message from the key manager
 *
 * Returns: -1 on error, message size on success
 */
static int
skip_keymgr_getmsg(int fd, void *buf, int buflen)
{
#ifdef SYSV
	struct strbuf		ctlbuf;
	int			rc, flags;

	ctlbuf.maxlen = buflen;
	ctlbuf.buf = buf;
	flags = 0;

	rc = getmsg(fd, &ctlbuf, NULL, &flags);

	return (rc < 0 ? rc : ctlbuf.len);
#else
	return(read(fd, (char *) buf, buflen));
#endif
}

/* skip_keymgr_putmsg()
 *
 * Send a message to the key manager
 *
 * Returns: -1 on error, message size on success
 */
static int
skip_keymgr_putmsg(int fd, void *buf, int buflen)
{
#ifdef SYSV
	struct strbuf		ctlbuf;
	int			rc, flags;

	ctlbuf.len = buflen;
	ctlbuf.buf = buf;
	flags = 0;

	rc = putmsg(fd, &ctlbuf, NULL, flags);

	return (rc < 0 ? rc : ctlbuf.len);
#else
	return(write(fd, (char *) buf, buflen));
#endif
}

static void
set_nodeid(int fd)
{
	struct Set_local_keyid	set;
	int			flags;
	char			hostname[MAXHOSTNAMELEN];
	struct hostent		*hp;
	char			*p_buf;
	short			idx;

	/*
	 * set the nodeid to be the system's primary IP address
	 */
#ifdef SYSV
	(void) sysinfo(SI_HOSTNAME, hostname, MAXHOSTNAMELEN);
#else
	(void) gethostname(hostname, MAXHOSTNAMELEN);
#endif
	hp = gethostbyname(hostname);

	if (hp == NULL) {
		fprintf(stderr, "Cannot get host information for %s\n",
			hostname);
		perror("gethostbyname");
		exit(1);
	}

	/*
	 *  Set the local Mster Key ID for NSID IPV4 to local IP Address
	 */
	set.msgtype	= SKIP_SET_LOCAL_KEYID;
	set.nsid	= SKIP_NSID_IPV4;
	set.mkeyid.len	= 4;
	memcpy(set.mkeyid.buf, hp->h_addr_list[0], set.mkeyid.len);

	printf("Setting Local Master Key Id 0x%08x for NSID %d...",
			ntohl(*(unsigned long *)set.mkeyid.buf), set.nsid); 

	print_time();

	fflush(stdout);

	if (skip_keymgr_putmsg(fd, &set, sizeof(set)) < 0) {
		if (errno == EINTR) {
			return;
		}
		if ((errno == EAGAIN) || (errno == EWOULDBLOCK)) {
			return;
		}
		perror("skip_keymhr_putmsg");
	}
	printf("Done\n");

	/*
	 *  Set the local Master Key ID for NSID 8 to 4 times local IP Address
	 */
	set.msgtype	= SKIP_SET_LOCAL_KEYID;
	set.nsid	= SKIP_NSID_MD5_DH_PUB;
	set.mkeyid.len	= 4;
	p_buf		= set.mkeyid.buf;

	for (idx = 0; idx < 4; idx++) {
		memcpy(p_buf, hp->h_addr_list[0], set.mkeyid.len);
		p_buf += set.mkeyid.len;
	}
	set.mkeyid.len	*= 4;

	printf("Setting Local Master Key Id for NSID %d...", set.nsid); 

	print_time();

	fflush(stdout);

	if (skip_keymgr_putmsg(fd, &set, sizeof(set)) < 0) {
		if (errno == EINTR) {
			return;
		}
		if ((errno == EAGAIN) || (errno == EWOULDBLOCK)) {
			return;
		}
		perror("skip_keymhr_putmsg");
	}
	printf("Done\n");
}

static int
print_params(skip_param_t *params)
{
	unsigned long nodeid;

	nodeid = ntohl(*((unsigned long *)params->r_mkeyid.buf));

	printf("\tversion = %01x\n", params->version);

	printf("\tip @ = %08x nodeid = %08x counter_n = %08x ttl = %d\n",
					ntohl(params->ip_addr.s_addr),
					nodeid,
					params->counter_n,
					params->ttl);

	printf("\ts_nsid = %01x r_nsid = %01x, r_mkeyid = %d\n",
					params->s_nsid,
					params->r_nsid,
					params->r_mkeyid.len);
					

	printf("\tkp_alg = %d kij_alg = %d mac_alg = %d comp_alg = %d\n",
					params->kp_alg,
					params->kij_alg,
					params->mac_alg,
					params->comp_alg);
}

static void
print_message(union skip_messages *dp)
{
	int		i;
	unsigned char	*c;

	if (dp->msgtype > MAX_MESSAGES) {
		fprintf(stderr, "Bad message id (%d) received\n", dp->msgtype);
		return;
	}

	printf("%s:\n", messages[dp->msgtype]);

	switch (dp->msgtype) {

	case SKIP_ENCRYPTED_PKT_KEY_REQ:

		print_params(&(dp->enc_req.params));
		break;

	case SKIP_ENCRYPTED_PKT_KEY_RSP:

		print_params(&dp->enc_rsp.params);
		printf("\tekp = ");
		c = (unsigned char *) dp + dp->enc_rsp.ekp_offset;
		for (i = 0; i < dp->enc_rsp.ekp_keylen; i++) {
			printf("%02x ", *c++);
		}
		putchar('\n');
		printf("\tkp = ");
		c = (unsigned char *) dp + dp->enc_rsp.kp_offset;
		for (i = 0; i < dp->enc_rsp.kp_keylen; i++) {
			printf("%02x ", *c++);
		}
		putchar('\n');
		break;

	case SKIP_DECRYPTED_PKT_KEY_REQ:

		print_params(&dp->dec_req.params);
		printf("\tekp = ");
		c = (unsigned char *) dp + dp->dec_req.ekp_offset;
		for (i = 0; i < dp->dec_req.ekp_keylen; i++) {
			printf("%02x ", *c++);
		}
		putchar('\n');
		break;

	case SKIP_DECRYPTED_PKT_KEY_RSP:

		print_params(&dp->dec_rsp.params);
		printf("\tekp = ");
		c = (unsigned char *) dp + dp->dec_rsp.ekp_offset;
		for (i = 0; i < dp->dec_rsp.ekp_keylen; i++) {
			printf("%02x ", *c++);
		}
		putchar('\n');
		printf("\tkp = ");
		c = (unsigned char *) dp + dp->dec_rsp.kp_offset;
		for (i = 0; i < dp->dec_rsp.kp_keylen; i++) {
			printf("%02x ", *c++);
		}
		putchar('\n');
		break;

	default:
		fprintf(stderr, "Unsupported message (%d) in print_message\n",
				dp->msgtype);
	}
}

static void
write_key(char *addr, unsigned int len)
{
	struct timeval	tv;

	if (debug) {
		memset(addr, 0, len);
		return;
	}
	if (len > sizeof tv) {
		memcpy(addr, (char *) write_key, len);
	} else {
		(void) gettimeofday(&tv, NULL);
		memcpy(addr, (char *) &tv, len);
	}
}

static void
make_response(union skip_messages *dp, union skip_messages *response, int *size)
{
	skip_param_t	*params;

	memset((char *) response, 0, sizeof (*response));

	if (dp->msgtype > MAX_MESSAGES) {
		fprintf(stderr, "Bad message id (%d) received\n", dp->msgtype);
		return;
	}

	/*
	 * XXX - Stupid debuging values...
	 */
	dp->enc_req.params.counter_n	= 0x44997753;
	dp->enc_req.params.ttl		= 0x11223344;
	dp->dec_req.params.counter_n	= 0x44997753;
	dp->dec_req.params.ttl		= 0x11223344;

	switch (dp->msgtype) {

	case SKIP_ENCRYPTED_PKT_KEY_REQ:
		params 				= &response->enc_rsp.params;
		response->enc_rsp.msgtype	= SKIP_ENCRYPTED_PKT_KEY_RSP;
		response->enc_rsp.params	= dp->enc_req.params;
		response->enc_rsp.kp_keylen	= 8;
		response->enc_rsp.ekp_keylen	= 8;
		response->enc_rsp.iv_len	= 8;
		response->enc_rsp.kp_offset	= sizeof(response->enc_rsp);
		response->enc_rsp.ekp_offset	= sizeof(response->enc_rsp);
		response->enc_rsp.iv_offset	= sizeof(response->enc_rsp);
		write_key((char *) response + sizeof(response->enc_rsp), 8);
		*size 				= sizeof (response->enc_rsp)
									+ 8;
		/*
		 * ESP/AH mode only ... 
		 */
		if (params->version == SKIP_V2 ) {

			if (params->kp_alg) {
				response->enc_rsp.E_kp_keylen = 8;
				response->enc_rsp.E_kp_offset = 
						sizeof(response->enc_rsp);
			} else  {
				response->enc_rsp.E_kp_keylen	= 0;
			}

			if (params->mac_alg) {
				response->enc_rsp.A_kp_keylen = 8;
				response->enc_rsp.A_kp_offset = 
						sizeof(response->enc_rsp);
			} else  {
				response->enc_rsp.A_kp_keylen	= 0;
			}
		}
		break;

	case SKIP_DECRYPTED_PKT_KEY_REQ:
		params 				= &response->dec_rsp.params;
		response->dec_rsp.msgtype	= SKIP_DECRYPTED_PKT_KEY_RSP;
		response->dec_rsp.params	= dp->dec_req.params;
		response->dec_rsp.kp_keylen	= dp->dec_req.ekp_keylen;
		response->dec_rsp.ekp_keylen	= dp->dec_req.ekp_keylen;
		response->dec_rsp.kp_offset     = sizeof(response->dec_rsp);
		response->dec_rsp.ekp_offset	= sizeof (response->dec_rsp);
		memcpy((char *) response  +  sizeof (response->dec_rsp),
			(char *) dp + dp->dec_req.ekp_offset,
			dp->dec_req.ekp_keylen);
		*size 				= sizeof (response->dec_rsp) +
						  dp->dec_req.ekp_keylen;
		/*
		 * ESP/AH mode only ... 
		 */
		if (params->version == SKIP_V2 ) {

			if (params->kp_alg) {
				response->dec_rsp.E_kp_keylen = 8;
				response->dec_rsp.E_kp_offset =
						sizeof(response->dec_rsp);
			} else  {
				response->dec_rsp.E_kp_keylen = 0;
			}

			if (params->mac_alg) {
				response->dec_rsp.A_kp_keylen = 8;
				response->dec_rsp.A_kp_offset =
						sizeof(response->dec_rsp);
			} else  {
				response->dec_rsp.A_kp_keylen	= 0;
			}
		}
		break;

	default:
		fprintf(stderr, "Unsupported message (%d) received\n",
				dp->msgtype);
		*size = 0;
	}
}

static void
process_request(int fd)
{
	char			req[MGTCSIZE], rsp[MGTCSIZE];
	int			len, size;

	printf("\nWaiting... "); fflush(stdout);

	if ((len = skip_keymgr_getmsg(fd, req, MGTCSIZE)) < 0) {
		if (errno == EINTR) {
			return;
		}
		if ((errno == EAGAIN) || (errno == EWOULDBLOCK)) {
			return;
		}
		perror("skip_keymgr_getmsg");
	}

	print_time();
	print_message((union skip_messages *) req);

	make_response((union skip_messages *) req, (union skip_messages *) rsp,
		&size);

	print_time();
	print_message((union skip_messages *) rsp);

	printf("Responding... "); 

	print_time();

	fflush(stdout);

	if (skip_keymgr_putmsg(fd, rsp, size) < 0) {
		if (errno == EINTR) {
			return;
		}
		if ((errno == EAGAIN) || (errno == EWOULDBLOCK)) {
			return;
		}
		perror("putmsg");
	}
	putchar('\n');
}

/*
 * Inform SKIP driver about a  supported Kij algorithm 
 */
static void
set_kij_alg(int fd, unsigned char version, unsigned char kij_alg)
{
	struct Set_supported_kij	set;

	set.msgtype	= SKIP_SET_SUPPORTED_KIJ;
	set.version	= version;
	set.kij_alg	= kij_alg;

	printf("Setting supported kij alg=%d for version %d...\n",
							kij_alg, version);

	print_time();

	fflush(stdout);

	if (skip_keymgr_putmsg(fd, &set, sizeof(set)) < 0) {
		if (errno == EINTR) {
			return;
		}
		if ((errno == EAGAIN) || (errno == EWOULDBLOCK)) {
			return;
		}
		perror("skip_keymhr_putmsg");
	}
	printf("Done.\n");
}

int
main(int argc, char *argv[])
{
	int	c, keyfd;

 
	while ((c = getopt(argc, argv, "d")) != EOF) {
		switch (c) { 
		case 'd': 
			debug = 1; 
			break; 
		default: 
			usage(argv[0]); 
		} 
	} 

	keyfd = open_device(keymgr);

	/*
	 * Set the local Master Key ID (IPV4)
	 */
	set_nodeid(keyfd);

	/*
	 * Set the Kij algorithms supported in SunScreen mode
	 */
	set_kij_alg(keyfd, SKIP_V1, SKIP_SIMPLECRYPT);

	/*
	 * Set the Kij algorithms supported in SunScreen mode
	 */
	set_kij_alg(keyfd, SKIP_V2, SKIP_KIJ_SIMPLECRYPT);


	/*
	 * Loop on events...
	 */
	for (;;) {
		process_request(keyfd);
	}
	/*NOTREACHED*/
	return(0);
}

