#include "bsd_locl.h"

RCSID("$Id: setenv.c,v 1.1 1996/01/31 20:35:40 bg Exp $");

/*
 * This is the easy way out, use putenv to implement setenv. We might
 * leak some memory but that is ok since we are usally about to exec
 * anyway.
 */

#if !defined(HAVE_SETENV) && defined(HAVE_PUTENV)

int
setenv(const char *var, const char *val, int rewrite)
{
    char *t;

    if (!rewrite && getenv(var) != 0)
	return 0;
  
    if ((t = malloc(strlen(var) + strlen(val) + 2)) == 0)
	return -1;

    strcpy(t, var);
    strcat(t, "=");
    strcat(t, val);
    if (putenv(t) == 0)
	return 0;
    else
	return -1;
}

#endif /* !defined(HAVE_SETENV) && defined(HAVE_PUTENV) */

/*
 * We have neither setenv nor putenv but probably a getenv that is
 * causing us trouble. Solution, implement everything but getenv.
 */

#if !defined(HAVE_SETENV) && !defined(HAVE_PUTENV)

extern char **environ;

static char *__findenv __P((const char *, int *));

/*
 * putenv --
 *	String points to a string of the form name=value.
 *
 *      Makes the value of the environment variable name equal to
 *      value by altering an existing variable or creating a new one.
 *
 *      Warning: Altering the string does NOT change the environment.
 */
int putenv(const char *string)
{
    int offset;
    char *val = __findenv(string, &offset);
    if (val != 0)
	{
	    val = environ[offset];
	    /* If there is enough space use it. */
	    if (strlen(val) >= strlen(string))
		strcpy(val, string);
	    else
		{
		    char *t = strdup(string);
		    if (t == 0)
			return -1;
		    /* Shall we free(environ[offset]) first? */
		    environ[offset] = t;
		    return 0;
		}
	}
    else
	return setenv(string, strchr(string, '='), 1);
}

/*
 * Copyright (c) 1987, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * setenv --
 *	Set the value of the environmental variable "name" to be
 *	"value".  If rewrite is set, replace any current value.
 */
int setenv(char *name, char *value, int rewrite)
{
	extern char **environ;
	static int alloced;			/* if allocated space before */
	register char *c;
	int l_value, offset;

	if (*value == '=')			/* no `=' in value */
		++value;
	l_value = strlen(value);
	if ((c = __findenv(name, &offset))) {	/* find if already exists */
		if (!rewrite)
			return (0);
		if (strlen(c) >= l_value) {	/* old larger; copy over */
			while (*c++ = *value++);
			return (0);
		}
	} else {					/* create new slot */
		register int cnt;
		register char **p;

		for (p = environ, cnt = 0; *p; ++p, ++cnt);
		if (alloced) {			/* just increase size */
			environ = (char **)realloc((char *)environ,
			    (size_t)(sizeof(char *) * (cnt + 2)));
			if (!environ)
				return (-1);
		}
		else {				/* get new space */
			alloced = 1;		/* copy old entries into it */
			p = (char **)malloc((size_t)(sizeof(char *) * (cnt + 2)));
			if (!p)
				return (-1);
			memmove(p, environ, cnt * sizeof(char *));
			environ = p;
		}
		environ[cnt + 1] = NULL;
		offset = cnt;
	}
	for (c = (char *)name; *c && *c != '='; ++c);	/* no `=' in name */
	if (!(environ[offset] =			/* name + `=' + value */
	    malloc((size_t)((int)(c - name) + l_value + 2))))
		return (-1);
	for (c = environ[offset]; (*c = *name++) && *c != '='; ++c);
	for (*c++ = '='; *c++ = *value++;);
	return (0);
}

/*
 * unsetenv(name) --
 *	Delete environmental variable "name".
 */
void
unsetenv(name)
	const char *name;
{
	extern char **environ;
	register char **p;
	int offset;

	while (__findenv(name, &offset))	/* if set multiple times */
		for (p = &environ[offset];; ++p)
			if (!(*p = *(p + 1)))
				break;
}

/*
 * getenv --
 *	Returns ptr to value associated with name, if any, else NULL.
 */
#if 0
char *
getenv(name)
	const char *name;
{
	int offset;

	return (__findenv(name, &offset));
}
#endif

/*
 * __findenv --
 *	Returns pointer to value associated with name, if any, else NULL.
 *	Sets offset to be the offset of the name/value combination in the
 *	environmental array, for use by setenv(3) and unsetenv(3).
 *	Explicitly removes '=' in argument name.
 */
static char *
__findenv(name, offset)
	register const char *name;
	int *offset;
{
	extern char **environ;
	register int len;
	register const char *np;
	register char **p, *c;

	if (name == NULL || environ == NULL)
		return (NULL);
	for (np = name; *np && *np != '='; ++np)
		continue;
	len = np - name;
	for (p = environ; (c = *p) != NULL; ++p)
		if (strncmp(c, name, len) == 0 && c[len] == '=') {
			*offset = p - environ;
			return (c + len + 1);
		}
	return (NULL);
}

#endif /* !defined(HAVE_SETENV) && !defined(HAVE_PUTENV) */
