/*
 * This software may now be redistributed outside the US.
 *
 */

/* 
  Copyright (C) 1989 by the Massachusetts Institute of Technology

   Export of this software from the United States of America is assumed
   to require a specific license from the United States Government.
   It is the responsibility of any person or organization contemplating
   export to obtain such a license before exporting.

WITHIN THAT CONSTRAINT, permission to use, copy, modify, and
distribute this software and its documentation for any purpose and
without fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright notice and
this permission notice appear in supporting documentation, and that
the name of M.I.T. not be used in advertising or publicity pertaining
to distribution of the software without specific, written prior
permission.  M.I.T. makes no representations about the suitability of
this software for any purpose.  It is provided "as is" without express
or implied warranty.

  */

#include "krb_locl.h"

RCSID("$Id: send_to_kdc.c,v 1.15 1996/03/28 11:10:52 bg Exp $");

#define S_AD_SZ sizeof(struct sockaddr_in)

static int krb_udp_port = 0;

static const char *prog = "send_to_kdc";
static send_recv(KTEXT pkt, KTEXT rpkt, int f,
		 struct sockaddr_in *_to, struct sockaddr_in *addrs,
		 int h_hosts);

/*
 * This file contains two routines, send_to_kdc() and send_recv().
 * send_recv() is a static routine used by send_to_kdc().
 */

/*
 * send_to_kdc() sends a message to the Kerberos authentication
 * server(s) in the given realm and returns the reply message.
 * The "pkt" argument points to the message to be sent to Kerberos;
 * the "rpkt" argument will be filled in with Kerberos' reply.
 * The "realm" argument indicates the realm of the Kerberos server(s)
 * to transact with.  If the realm is null, the local realm is used.
 *
 * If more than one Kerberos server is known for a given realm,
 * different servers will be queried until one of them replies.
 * Several attempts (retries) are made for each server before
 * giving up entirely.
 *
 * If an answer was received from a Kerberos host, KSUCCESS is
 * returned.  The following errors can be returned:
 *
 * SKDC_CANT    - can't get local realm
 *              - can't find "kerberos" in /etc/services database
 *              - can't open socket
 *              - can't bind socket
 *              - all ports in use
 *              - couldn't find any Kerberos host
 *
 * SKDC_RETRY   - couldn't get an answer from any Kerberos server,
 *		  after several retries
 */

int
send_to_kdc(KTEXT pkt, KTEXT rpkt, char *realm)
{
    int i, f;
    int no_host; /* was a kerberos host found? */
    int retry;
    int n_hosts;
    int retval;
    struct sockaddr_in to, *addrlist;
    struct hostent *host;
    char krbhst[MAX_HSTNM];
    char lrealm[REALM_SZ];

    /*
     * If "realm" is non-null, use that, otherwise get the
     * local realm.
     */
    if (realm)
	(void) strcpy(lrealm, realm);
    else
	if (krb_get_lrealm(lrealm,1)) {
#ifndef KRB_NDEB
	    if (krb_debug)
		fprintf(stderr, "%s: can't get local realm\n", prog);
#endif
	    return(SKDC_CANT);
	}
#ifndef KRB_NDEB
    if (krb_debug)
      fprintf(stderr, "lrealm is %s\n", lrealm);
#endif
    if (krb_udp_port == 0) {
        register struct servent *sp;
        if ((sp = getservbyname("kerberos","udp")) == 0) {
#ifndef KRB_NDEB
            if (krb_debug)
                fprintf(stderr, "%s: Can't get kerberos/udp service\n",
                        prog);
#endif
            krb_udp_port = htons(KRB_PORT); /* Was return(SKDC_CANT); */
        }
        else
	    krb_udp_port = sp->s_port;
#ifndef KRB_NDEB
        if (krb_debug)
	  fprintf(stderr, "krb_udp_port is %d\n", krb_udp_port);
#endif
    }
    bzero((char *)&to, S_AD_SZ);
    addrlist = malloc(sizeof(*addrlist));
    bzero(addrlist, sizeof(*addrlist));
    if (addrlist == NULL)
        return (SKDC_CANT);
    if ((f = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
#ifndef KRB_NDEB
        if (krb_debug)
            fprintf(stderr,"%s: Can't open socket\n", prog);
#endif
        return(SKDC_CANT);
    }
    /* from now on, exit through rtn label for cleanup */

    no_host = 1;
    /* get an initial allocation */
    n_hosts = 0;
    for (i = 1; krb_get_krbhst(krbhst, lrealm, i) == KSUCCESS; ++i) {
	char *p;

#ifndef KRB_NDEB
        if (krb_debug) {
	  fprintf(stderr, "Getting host entry for %s...",krbhst);
	  /*(void) fflush(stdout);*/
        }
#endif
        host = gethostbyname(krbhst);
#ifndef KRB_NDEB
        if (krb_debug) {
	  fprintf(stderr, "%s.\n",
                   host ? "Got it" : "Didn't get it");
        }
#endif
        if (!host)
            continue;
        no_host = 0;    /* found at least one */
	while ((p = *(host->h_addr_list)++)) {
          n_hosts++;
  	  addrlist = realloc(addrlist, sizeof(*addrlist) * (n_hosts+1));
	  if (addrlist == NULL)
	    return SKDC_CANT;
	  bzero(&addrlist[n_hosts-1], sizeof(*addrlist));
	  addrlist[n_hosts-1].sin_family = host->h_addrtype;
	  addrlist[n_hosts-1].sin_port   = krb_udp_port;
	  bcopy(p, &addrlist[n_hosts-1].sin_addr, host->h_length);
	  if (send_recv(pkt, rpkt, f, &addrlist[n_hosts-1],
			addrlist, n_hosts)) {
            retval = KSUCCESS;
            goto rtn;
        }
#ifndef KRB_NDEB
        if (krb_debug) {
	  fprintf(stderr, "Timeout, error, or wrong descriptor\n");
	  /*(void) fflush(stdout);*/
        }
#endif
      }
    }
    if (no_host) {
#ifndef KRB_NDEB
	if (krb_debug)
	    fprintf(stderr, "%s: can't find any Kerberos host.\n",
		    prog);
#endif
        retval = SKDC_CANT;
        goto rtn;
    }
    /* retry each host in sequence */
    for (retry = 0; retry < CLIENT_KRB_RETRY; ++retry) {
	for (i = 0; i < n_hosts; ++i) {
	    if (send_recv(pkt, rpkt, f, &addrlist[i], addrlist,
			  n_hosts)) {
		retval = KSUCCESS;
		goto rtn;
	    }
        }
    }
    retval = SKDC_RETRY;
rtn:
    (void) close(f);
    free(addrlist);
    return(retval);
}

/*
 * try to send out and receive message.
 * return 1 on success, 0 on failure
 */

static int
send_recv(KTEXT pkt, KTEXT rpkt, int f, struct sockaddr_in *_to,
	  struct sockaddr_in *addrs, int n_hosts)
{
    fd_set readfds;
    struct sockaddr_in from;
    int sin_size;
    int numsent;
    int i;
    /* CLIENT_KRB_TIMEOUT indicates the time to wait before
     * retrying a server.  It's defined in "krb.h".
     */
    struct timeval timeout;
    timeout.tv_sec = CLIENT_KRB_TIMEOUT;
    timeout.tv_usec = 0;

#ifndef KRB_NDEB
    if (krb_debug) {
        if (_to->sin_family == AF_INET)
            fprintf(stderr, "Sending message to %s...",
                   inet_ntoa(_to->sin_addr));
        else
	  fprintf(stderr, "Sending message...");
        /*(void) fflush(stdout);*/
    }
#endif
    if ((numsent = sendto(f,(char *)(pkt->dat), pkt->length, 0, 
			  (struct sockaddr *)_to,
                          S_AD_SZ)) != pkt->length) {
#ifndef KRB_NDEB
        if (krb_debug)
            fprintf(stderr, "sent only %d/%d\n",numsent, pkt->length);
#endif
        return 0;
    }
#ifndef KRB_NDEB
    if (krb_debug) {
      fprintf(stderr, "Sent\nWaiting for reply...");
      /*(void) fflush(stdout);*/
    }
#endif
    FD_ZERO(&readfds);
    FD_SET(f, &readfds);
    errno = 0;
    /* select - either recv is ready, or timeout */
    /* see if timeout or error or wrong descriptor */
    if (select(f + 1, &readfds, 0, 0, &timeout) < 1
        || !FD_ISSET(f, &readfds)) {
#ifndef KRB_NDEB
        if (krb_debug) {
	    long rfds;
	    bcopy(&readfds, &rfds, sizeof(rfds));
            fprintf(stderr, "select failed: readfds=%lx",
                    rfds);
            perror("");
        }
#endif
        return 0;
    }
    sin_size = sizeof(from);
    if (recvfrom(f, (char *)(rpkt->dat), sizeof(rpkt->dat), 0,
		 (struct sockaddr *)&from, &sin_size)
        < 0) {
#ifndef KRB_NDEB
        if (krb_debug)
            perror("recvfrom");
#endif
        return 0;
    }
#ifndef KRB_NDEB
    if (krb_debug) {
      fprintf(stderr, "received packet from %s\n", inet_ntoa(from.sin_addr));
      /*fflush(stdout);*/
    }
#endif
    for (i = 0; i < n_hosts; ++i) {
	 if (addrs[i].sin_addr.s_addr == from.sin_addr.s_addr) {
#ifndef KRB_NDEB
            if (krb_debug) {
	      fprintf(stderr, "Received it\n");
	      /*(void) fflush(stdout);*/
            }
#endif
            return 1;
        }
#ifndef KRB_NDEB
        if (krb_debug)
            fprintf(stderr,
                    "packet not from %lx\n",
                    (long)addrs[i].sin_addr.s_addr);
#endif
    }
#ifndef KRB_NDEB
    if (krb_debug)
        fprintf(stderr, "%s: received packet from wrong host! (%x)\n",
                "send_to_kdc(send_rcv)", (int)from.sin_addr.s_addr);
#endif
    return 0;
}
