#include "krb_locl.h"

RCSID("$Id: mk_req.c,v 1.10 1996/08/04 22:55:56 assar Exp $");

static int lifetime = 255;	/* But no longer than TGT says. */


static void
build_request(KTEXT req, char *name, char *inst, char *realm, 
	      u_int32_t checksum)
{
    struct timeval tv;
    unsigned char *p = req->dat;
    
    p += put_nir(name, inst, realm, p);
    
    p += put_int(checksum, p, 4);


    /* Fill in the times on the request id */
    gettimeofday(&tv, NULL);

    *p++ = tv.tv_usec / 5000; /* 5ms */
    
    p += put_int(tv.tv_sec, p, 4);

    /* Fill to a multiple of 8 bytes for DES */
    req->length = ((p - req->dat + 7)/8) * 8;
}


/*
 * krb_mk_req takes a text structure in which an authenticator is to
 * be built, the name of a service, an instance, a realm,
 * and a checksum.  It then retrieves a ticket for
 * the desired service and creates an authenticator in the text
 * structure passed as the first argument.  krb_mk_req returns
 * KSUCCESS on success and a Kerberos error code on failure.
 *
 * The peer procedure on the other end is krb_rd_req.  When making
 * any changes to this routine it is important to make corresponding
 * changes to krb_rd_req.
 *
 * The authenticator consists of the following:
 *
 * authent->dat
 *
 * unsigned char	KRB_PROT_VERSION	protocol version no.
 * unsigned char	AUTH_MSG_APPL_REQUEST	message type
 * (least significant
 * bit of above)	HOST_BYTE_ORDER		local byte ordering
 * unsigned char	kvno from ticket	server's key version
 * string		realm			server's realm
 * unsigned char	tl			ticket length
 * unsigned char	idl			request id length
 * text			ticket->dat		ticket for server
 * text			req_id->dat		request id
 *
 * The ticket information is retrieved from the ticket cache or
 * fetched from Kerberos.  The request id (called the "authenticator"
 * in the papers on Kerberos) contains the following:
 *
 * req_id->dat
 *
 * string		cr.pname		{name, instance, and
 * string		cr.pinst		realm of principal
 * string		myrealm			making this request}
 * 4 bytes		checksum		checksum argument given
 * unsigned char	tv_local.tf_usec	time (milliseconds)
 * 4 bytes		tv_local.tv_sec		time (seconds)
 *
 * req_id->length = 3 strings + 3 terminating nulls + 5 bytes for time,
 *                  all rounded up to multiple of 8.
 */

int
krb_mk_req(register KTEXT authent, char *service, char *instance, char *realm, 
	   int32_t checksum)
{
    KTEXT_ST req_st;
    KTEXT req_id = &req_st;

    CREDENTIALS cr;             /* Credentials used by retr */
    KTEXT ticket = &(cr.ticket_st); /* Pointer to tkt_st */
    int retval;                 /* Returned by krb_get_cred */

    des_key_schedule schedule;
    char myrealm[REALM_SZ];

    unsigned char *p = authent->dat;

    p += put_int(KRB_PROT_VERSION, p, 1);
    
    p += put_int(AUTH_MSG_APPL_REQUEST, p, 1);
    
    /* Get the ticket and move it into the authenticator */
#ifndef KRB_NDEB
    if (krb_ap_req_debug)
        printf("Realm: %s\n",realm);
#endif
    /* 
     * Determine realm of these tickets.  We will send this to the
     * KDC from which we are requesting tickets so it knows what to
     * with our session key.
     */

    retval = krb_get_tf_realm(TKT_FILE, myrealm);
    
    if (retval != KSUCCESS)
	return retval;
    
    retval = krb_get_cred(service,instance,realm,&cr);

    if (retval == RET_NOTKT) {
	retval = get_ad_tkt(service, instance, realm, lifetime);
	if(retval == KSUCCESS)
	    retval = krb_get_cred(service, instance, realm, &cr);
    }

    if(retval != KSUCCESS)
	return retval;


#ifndef KRB_NDEB
    if (krb_ap_req_debug)
        printf("%s %s %s %s %s\n", service, instance, realm,
               cr.pname, cr.pinst);
#endif

    p += put_int(cr.kvno, p, 1);

    p += put_string(realm, p);

    p += put_int(ticket->length, p, 1);

    build_request(req_id, cr.pname, cr.pinst, myrealm, checksum);
    
    encrypt_ktext(req_id, &cr.session, DES_ENCRYPT);

    p += put_int(req_id->length, p, 1);

    memcpy(p, ticket->dat, ticket->length);
    
    p += ticket->length;
    
    memcpy(p, req_id->dat, req_id->length);
    
    p += req_id->length;

    authent->length = p - authent->dat;
    
    memset(&cr, 0, sizeof(cr));
    memset(&req_st, 0, sizeof(req_st));

#ifndef KRB_NDEB
    if (krb_ap_req_debug)
        printf("Authent->length = %d\n",authent->length);
#if 0
    if (krb_ap_req_debug)
        printf("idl = %d, tl = %d\n",(int) *idl, (int) *tl);
#endif
#endif

    return KSUCCESS;
}

/* 
 * krb_set_lifetime sets the default lifetime for additional tickets
 * obtained via krb_mk_req().
 * 
 * It returns the previous value of the default lifetime.
 */

int
krb_set_lifetime(int newval)
{
    int olife = lifetime;

    lifetime = newval;
    return(olife);
}
