/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)skip_ioctl.c	1.17 95/11/23 Sun Microsystems"

/*
 * System includes
 */
#ifdef SYSV
#include <skip_sysv.h>
#endif
#ifdef SUNOS
#include <skip_sunos.h>
#endif
#if !defined(SYSV) && !defined(SUNOS)
#include <skip_os.h>
#endif

/*
 * SKIP includes
 */
#include <bdcmod.h>
#include <skip_proto.h>
#include <skip_types.h>
#include <skip_keymgrmsgs.h>
#include <skip_key.h>
#include <skip_ioctl.h>
#include <skip_es.h>
#include <skip_if.h>
#include <skip_crypt.h>

extern MUTEX_T		skip_es_lock;

/*
 * skip end system global statistics
 */
extern ioctl_key_stat_t	skip_keys_stats;
extern ioctl_hdr_stat_t	skip_hdrs_stats;

/*
 * Kvm-like interface 
 */
extern int skip_client_port;
extern int skip_server_port;
extern int skip_params_debug;

static skip_sym_t	skip_var[] = 
{
	{ &skip_key_debug,	SKIP_FMT_STD, "skip_key_debug" },
	{ &skip_key_max_bytes,	SKIP_FMT_STD, "skip_key_max_bytes" },
	{ &skip_key_max_idle,	SKIP_FMT_STD, "skip_key_max_idle" },
	{ &skip_key_max_use,	SKIP_FMT_STD, "skip_key_max_use" },
	{ &skip_params_debug,	SKIP_FMT_STD, "skip_params_debug" },
	{ &skip_client_port,	SKIP_FMT_NET, "skip_client_port" },
	{ &skip_server_port,	SKIP_FMT_NET, "skip_server_port" }
};

extern int		skip_get_key_status(ioctl_key_status_t *);
extern skip_es_t	*skip_es_ifs;


/* skip_es_hash()
 *
 * Compute a hash for the packet
 *
 * Returns: an index (0 .. SKIP_ES_HASH_TABLESZ-1)
 */
static unsigned int
skip_es_hash(struct in_addr *addr)
{
	return (((unsigned char *) addr)[3]);
}


/* skip_es_find_if()
 *
 * Given an interface name, find the corresponding interface structure
 *
 * Returns: 0 on success, errno otherwise
 */
skip_es_t *
skip_es_find_if(char *if_name)
{
	register skip_es_t	*ifp;

	for (ifp = skip_es_ifs; ifp; ifp = ifp->next) {
		if (strcmp(ifp->if_name, if_name) == 0) {
			break;
		}
	}
	return (ifp);
}

/* skip_es_list_hosts()
 *
 * Return list of current skip hosts
 *
 * Returns: 0 on success, errno otherwise
 */
int
skip_es_list_hosts(skip_es_req_t *reqp, int *nbytesp)
{
	register unsigned int		nhosts, maxhosts, h;
	register skip_es_t		*ifp;
	register skip_param_t		*hostp;
	register skip_es_hash_t		*hp;

	/*
	 * check to see if attached buffer is valid
	 */
	(*nbytesp) -= (char *) &reqp->hostinfo - (char *) &reqp->if_name;

	if ((*nbytesp % sizeof (skip_param_t)) != 0) {
		/*
		 * bad parameter list
		 */
		return (EINVAL);
	}
	maxhosts = *nbytesp / sizeof (skip_param_t);

	if (maxhosts == 0) {
		return (0);
	}

	/*
	 * grab the lock on the interface table
	 */
	MUTEX_ENTER(&skip_es_lock);

	ifp = skip_es_find_if(reqp->if_name);

	if (!ifp) {
		/*
		 * interface not found...
		 */
		MUTEX_EXIT(&skip_es_lock);
		return (ENODEV);
	}

	hostp = (skip_param_t *) &reqp->hostinfo;

	*nbytesp = (char *) &reqp->hostinfo - (char *) &reqp->if_name;

	for (nhosts = 0, h = 0; h < SKIP_ES_HASH_TABLESZ; h++) {
		for (hp = ifp->hashtable[h]; hp; hp = hp->next) {
			if (nhosts == maxhosts) {
				break;
			} else {
				nhosts++;
				(*nbytesp) += sizeof(skip_param_t);
			}
			bcopy((caddr_t) &hp->params,
					(caddr_t) hostp, sizeof(*hostp));

			/*
			 * the kernel internally initialises r_mkeyid even
			 * for NSID 0.
			 * This is confusing for user-space tools so zap it.
			 */
			if (hostp->r_nsid == 0) {
				hostp->r_mkeyid.len = 0;
			}
			hostp++;
		}
	}

	/*
	 * and release the interface table
	 */
	MUTEX_EXIT(&skip_es_lock);
	return (0);
}

/* skip_es_get_host()
 *
 * Get ACL entries
 *
 * Returns: 0 on success, errno otherwise
 */
int
skip_es_get_host(skip_es_req_t *reqp, int nbytes)
{
	register unsigned int		nhosts, h;
	register skip_es_t		*ifp;
	register skip_param_t		*hostp;
	register skip_es_hash_t		*hp;

	/*
	 * check to see if valid
	 */
	nbytes -= (char *) &reqp->hostinfo - (char *) &reqp->if_name;

	if ((nbytes % sizeof (skip_param_t)) != 0) {
		/*
		 * bad parameter list
		 */
		return (EINVAL);
	}
	nhosts = nbytes / sizeof (skip_param_t);

	/*
	 * grab the lock on the interface table
	 */
	MUTEX_ENTER(&skip_es_lock);

	ifp = skip_es_find_if(reqp->if_name);

	if (!ifp) {
		/*
		 * interface not found...
		 */
		MUTEX_EXIT(&skip_es_lock);
		return (ENODEV);
	}

	hostp = (skip_param_t *) &reqp->hostinfo;

	for (; nhosts; nhosts--, hostp++) {

		h = skip_es_hash(&hostp->ip_addr);
		
		hp = ifp->hashtable[h];

		while (hp) {
			/*
			 * check collision entries
			 */
			if (SAME_ADDR(&hostp->ip_addr, &hp->params.ip_addr)) {
				/*
				 * found entry we were looking for...
				 */
				break;
			}
			hp = hp->next;
		}
		if (!hp) {
			/*
			 * does not exist...
			 */
			MUTEX_EXIT(&skip_es_lock);
			return (ENOENT);
		}
		/*
		 * Fill in ACL entry with host/skip info
		 */
		bcopy((caddr_t) &hp->params, (caddr_t) hostp, sizeof(*hostp));

		/*
		 * the kernel internally initialises r_mkeyid even for NSID 0
		 * This is confusing for user-space tools so zap it.
		 */
		if (hostp->r_nsid == 0) {
			hostp->r_mkeyid.len = 0;
		}
	}
	/*
	 * and release the interface table
	 */
	MUTEX_EXIT(&skip_es_lock);
	return (0);
}

/* skip_es_add_host()
 *
 * Add a new skip host
 *
 * Returns: 0 on success, errno otherwise
 */
int
skip_es_add_host(skip_es_req_t *reqp, int nbytes)
{
	register unsigned int		nhosts, h;
	register skip_es_t		*ifp;
	register skip_param_t		*hostp;
	register skip_es_hash_t		*hp;

	/*
	 * check to see if valid
	 */
	nbytes -= (char *) &reqp->hostinfo - (char *) &reqp->if_name;

	if ((nbytes % sizeof (skip_param_t)) != 0) {
		/*
		 * bad parameter list
		 */
		return (EINVAL);
	}
	nhosts = nbytes / sizeof (skip_param_t);

	/*
	 * grab the lock on the interface table
	 */
	MUTEX_ENTER(&skip_es_lock);

	ifp = skip_es_find_if(reqp->if_name);

	if (!ifp) {
		/*
		 * interface not found...
		 */
		MUTEX_EXIT(&skip_es_lock);
		return (ENODEV);
	}

	hostp = (skip_param_t *) &reqp->hostinfo;

	for (; nhosts; nhosts--, hostp++) {

		h = skip_es_hash(&hostp->ip_addr);
		
		hp = ifp->hashtable[h];

		while (hp) {
			/*
			 * check collision entries
			 */
			if (SAME_ADDR(&hostp->ip_addr, &hp->params.ip_addr)) {
				/*
				 * found entry we were looking for...
				 */
				break;
			}
			hp = hp->next;
		}
		if (hp) {
			/*
			 * already exists...
			 */
			MUTEX_EXIT(&skip_es_lock);
			return (EEXIST);
		}
		hp = (skip_es_hash_t *) MEM_ALLOC(sizeof (*hp));
		if (hp == NULL) {
			MUTEX_EXIT(&skip_es_lock);
			return (ENOMEM);
		}

		/*
		 * Fill in ACL entry with host/skip info
		 */
		BCOPY((caddr_t) hostp,
			(caddr_t) &hp->params, sizeof(skip_param_t));
		/*
		 * Protect against bad configuration from the lib
		 */
		if (!hp->params.kij_alg && !hp->params.kp_alg &&
				!hp->params.mac_alg && !hp->params.comp_alg) {
			/*
			 * No transform - set this entry to cleatext
			 */
			hp->params.version = SKIP_NONE;
		}

		/*
		 * add the receiver master key default for NSID 0
		 */
		if ((hp->params.kp_alg) && (hp->params.r_nsid == 0)) {
			BCOPY((caddr_t) &hostp->ip_addr,
				hp->params.r_mkeyid.buf,
				sizeof(hostp->ip_addr));
			hp->params.r_mkeyid.len = sizeof(hostp->ip_addr);
		}
			
		if (ifp->hashtable[h] == NULL) {
			ifp->hashtable[h] = hp;
		} else {
			/*
			 * hash collision found - insert in collision chain...
			 */
			hp->next = ifp->hashtable[h];
			(ifp->hashtable[h])->prev = hp;
			ifp->hashtable[h] = hp;
		}
	}

	/*
	 * and release the interface table
	 */
	MUTEX_EXIT(&skip_es_lock);
	return (0);
}

/* skip_es_del_host()
 *
 * Remove a skip host
 *
 * Returns: 0 on success, errno otherwise
 */
int
skip_es_del_host(skip_es_req_t *reqp, int nbytes)
{
	register unsigned int		nhosts, h;
	register skip_es_t		*ifp;
	register skip_param_t		*hostp;
	register skip_es_hash_t		*hp;

	/*
	 * check to see if valid
	 */
	nbytes -= (char *) &reqp->hostinfo - (char *) &reqp->if_name;

	if ((nbytes % sizeof (skip_param_t)) != 0) {
		/*
		 * bad parameter list
		 */
		return (EINVAL);
	}
	nhosts = nbytes / sizeof (skip_param_t);

	/*
	 * grab the lock on the interface table
	 */
	MUTEX_ENTER(&skip_es_lock);

	ifp = skip_es_find_if(reqp->if_name);

	if (!ifp) {
		/*
		 * interface not found...
		 */
		MUTEX_EXIT(&skip_es_lock);
		return (ENODEV);
	}

	hostp = (skip_param_t *) &reqp->hostinfo;

	for (; nhosts; nhosts--, hostp++) {

		h = skip_es_hash(&hostp->ip_addr);
		
		hp = ifp->hashtable[h];

		while (hp) {
			/*
			 * check collision entries
			 */
			if (SAME_ADDR(&hostp->ip_addr, &hp->params.ip_addr)) {
				/*
				 * found entry we were looking for...
				 */
				break;
			}
			hp = hp->next;
		}
		if (!hp) {
			/*
			 * does not exist...
			 */
			continue;
		}

		/*
		 * remove this element from the hash table
		 */
		if (hp->next != NULL) {
			hp->next->prev = hp->prev;
		}
		if (hp->prev != NULL) {
			hp->prev->next = hp->next;
		} else {
			ifp->hashtable[h] = NULL;
		}
		MEM_FREE(hp, sizeof (*hp));
	}

	/*
	 * and release the interface table
	 */
	MUTEX_EXIT(&skip_es_lock);
	return (0);
}


/* skip_get_if_stats()
 *
 *  Get SKIP interface statistics
 *
 * Returns: 0 on success, errno otherwise
 */
int
skip_get_if_stats(skip_ctrl_stats_t *reqp)
{
	register skip_es_t		*ifp;
	register int			rc = 0;
	register ioctl_if_stat_t	*if_statp;

	/*
	 * grab the lock on the interface table
	 */
	MUTEX_ENTER(&skip_es_lock);

	ifp = skip_es_find_if(reqp->if_name);

	if (!ifp) {
		/*
		 * interface not found...
		 */
		rc = ENODEV;
		MUTEX_EXIT(&skip_es_lock);
		return (rc);
	}

	if_statp = (ioctl_if_stat_t *) &(reqp->stats.if_stats);

	BCOPY((caddr_t) &ifp->stats,
		(caddr_t) if_statp, sizeof(ioctl_if_stat_t));

	MUTEX_EXIT(&skip_es_lock);
	return (rc);

}


/* skip_get_key_stats()
 *
 *  Get SKIP keys statistics
 *
 * Returns: 0 on success, errno otherwise
 */
int
skip_get_key_stats(skip_ctrl_stats_t *reqp)
{
	register ioctl_key_stat_t	*key_statp;

	key_statp = (ioctl_key_stat_t *) &(reqp->stats.key_stats);

	BCOPY((caddr_t) &skip_keys_stats,
		(caddr_t) key_statp, sizeof(ioctl_key_stat_t));

	return(0);

}

/* skip_get_hdr_stats()
 *
 *  Get SKIP headers statistics
 *
 * Returns: 0 on success, errno otherwise
 */
int
skip_get_hdr_stats(skip_ctrl_stats_t *reqp)
{
	register ioctl_hdr_stat_t	*hdr_statp;

	hdr_statp = (ioctl_hdr_stat_t *) &(reqp->stats.hdr_stats);

	BCOPY((caddr_t) &skip_hdrs_stats,
		(caddr_t) hdr_statp, sizeof(ioctl_hdr_stat_t));

	return(0);

}


/* skip_get_kp_alg_list()
 *
 * List all loaded cryptor modules 
 *
 */
void
skip_get_kp_alg_list(ioctl_alg_list_t *alg_list)
{
	int		idxalg;
	skip_cryptor_t	*cp;

	/*
	 * Scan the V1 cryptors array for installed modules
	 */
	cp = &skip_cryptors_v1[0];
	for (idxalg = 0; idxalg < SKIP_MAXCRYPTORS; idxalg++) {

		if (cp->ekp_len == -1) {
			/*
			 * Module not present - skip it
			 */
			alg_list->flag_alg_v1[idxalg] = B_FALSE;
			cp++;
			continue;
		}
		alg_list->flag_alg_v1[idxalg] = B_TRUE;
		cp++;
	}

	/*
	 * Scan the V2 cryptors array for installed modules
	 */
	cp = &skip_cryptors[0];
	for (idxalg = 0; idxalg < SKIP_MAXCRYPTORS; idxalg++) {

		if (cp->ekp_len == -1) {
			/*
			 * Module not present - skip it
			 */
			alg_list->flag_alg[idxalg] = B_FALSE;
			cp++;
			continue;
		}
		alg_list->flag_alg[idxalg] = B_TRUE;
		cp++;
	}
}

/* skip_get_kij_alg_list()
 *
 * List all supported kij algorithms
 *
 */
void
skip_get_kij_alg_list(ioctl_alg_list_t *alg_list)
{
	int		idx;

	/*
	 * Scan the array of supported Kij algorithms
	 */
	for (idx = 0; idx < SKIP_MAXCRYPTORS; idx++) {
		alg_list->flag_alg_v1[idx] = skip_kmgr_kij_alg_v1[idx];
		alg_list->flag_alg[idx]    = skip_kmgr_kij_alg[idx];
	}
}

/* skip_cryptor_stats()
 *
 * Get cryptor modules statistics
 *
 */
void
skip_cryptor_stats(ioctl_crypt_stat_t *p_stats)
{
	int			idxalg;
	skip_cryptor_t		*alg;
	crypt_mod_stat_t	*statp = &(p_stats->crypt_stats[0]);
	unsigned int		count = 0;

	if (p_stats->version == SKIP_V2) { 
		alg = &skip_cryptors[0];
	} else {
		alg = &skip_cryptors_v1[0];
	}
	/*
	 * Scan the cryptors array for installed modules
	 */
	for (idxalg = 0; idxalg < SKIP_MAXCRYPTORS; idxalg++) {

		if (alg->ekp_len == -1) {
			/*
			 * Module not present - skip it
			 */
			alg++;
			continue;
		}

		statp->module_id	= idxalg;
		statp->encrypts         = alg->encrypts;
		statp->encrypterrs      = alg->encrypterrs;
		statp->decrypts         = alg->decrypts;
		statp->decrypterrs      = alg->decrypterrs;

		statp++;
		alg++;
		count++;
	}

	p_stats->nb_algs = count; 

}

/* skip_find_var()
 *
 * Find a named variable index
 *
 */
static int
skip_find_var(ioctl_skip_sym_t *p_symb)
{

	int	idx;
	int	count = sizeof(skip_var) / sizeof(skip_sym_t);

	/*
	 * Scan the variable array
	 */
	for (idx = 0; idx < count; idx++) {

		if (!bcmp(skip_var[idx].p_name, 
			p_symb->var_name, p_symb->var_lg)) {
			/*
			 * Name found - return its index in array
			 */
			return(idx);
		}
	}

	/*
	 * Name not found - error
	 */
	return(-1);
}

/* skip_get_var()
 *
 * Get named variable value
 *
 */
int
skip_get_var(ioctl_skip_sym_t *p_symb)
{
	int	idx;

	/*
	 * Lookup on variable name
	 */
	idx = skip_find_var(p_symb);
	if (idx < 0) {
		return(EINVAL);
	}

	/*
	 * Get the new value
	 */
	if (skip_var[idx].v_fmt == SKIP_FMT_NET) {
		p_symb->var_value = ntohs(*(skip_var[idx].p_value));
	} else {
		p_symb->var_value = *(skip_var[idx].p_value);
	}

	return(0);
}

/* skip_set_var()
 *
 * Set named variable value
 *
 */
int
skip_set_var(ioctl_skip_sym_t *p_symb)
{
	int	idx;

	/*
	 * Lookup on variable name
	 */
	idx = skip_find_var(p_symb);
	if (idx < 0) {
		return(EINVAL);
	}

	/*
	 * Set the new value
	 */
	if (skip_var[idx].v_fmt == SKIP_FMT_NET) {
		*(skip_var[idx].p_value) = htons(p_symb->var_value);
	} else {
		*(skip_var[idx].p_value) = p_symb->var_value;
	}

	return(0);
}


/* skip_es_ok()
 *
 * Permit/deny host access
 *
 * Returns: B_TRUE if the host is in list, B_FALSE otherwise
 */
boolean_t
skip_es_ok(skip_es_t *ifp,
		struct in_addr *addr,
		struct ip *iphdr,
		int size,
		skip_param_t *params
)
{
	register unsigned int		h;
	register skip_es_hash_t		*hp;

	h = skip_es_hash(addr);

	hp = ifp->hashtable[h];

	while (hp) {
		/*
		 * check collision entries
		 */
		if (SAME_ADDR(addr, &hp->params.ip_addr)) {
			/*
			 * found entry we were looking for...
			 */
			break;
		}
		hp = hp->next;
	}
	if (!hp) {
		/*
		 * Unknown remote. Check for possible candidate for bypass
		 * but only if we are in interactive mode.
		 */
		if ((iphdr->ip_p == IPPROTO_UDP) &&
					skip_es_bypass(iphdr, size)) {
			/*
			 * bypass, allow in the clear
			 */
			params->kij_alg	= 0;
			params->kp_alg	= 0;
			params->mac_alg	= 0;
			params->comp_alg= 0;
			BCOPY((caddr_t) addr,
				(caddr_t) &params->ip_addr,
				sizeof(struct in_addr));
			ifp->stats.skip_if_bypasses++;
			return (B_TRUE);
		}
		return (B_FALSE);
	}

	BCOPY((caddr_t) &hp->params, (caddr_t) params, sizeof(skip_param_t));

	if (params->kp_alg != 0) {
		/*
		 * Encrypted remote. Check for possible candidate for bypass.
		 */
		if ((iphdr->ip_p == IPPROTO_UDP) &&
					skip_es_bypass(iphdr, size)) {
			/*
			 * bypass, allow in the clear
			 */
			bzero((caddr_t) params, sizeof(skip_param_t));
			BCOPY((caddr_t) addr,
				(caddr_t) &params->ip_addr,
				sizeof(struct in_addr));
			ifp->stats.skip_if_bypasses++;
		}
	}
	return (B_TRUE);
}

/* skip_es_set_mode()
 *
 * Turn on/off access control
 *
 * Returns: 0 on success, errno otherwise
 */
int
skip_es_set_mode(SKIP_NOTIFYQ_T wq, skip_es_mode_t *reqp)
{
	register skip_es_t		*ifp;
	register int			rc = 0;

	/*
	 * grab the lock on the interface table
	 */
	MUTEX_ENTER(&skip_es_lock);

	ifp = skip_es_find_if(reqp->if_name);

	if (!ifp) {
		/*
		 * interface not found...
		 */
		MUTEX_EXIT(&skip_es_lock);
		return(ENODEV);
	}

	switch (reqp->if_mode) {

	case SkipInteractive:
		if (ifp->notifier) {
			MUTEX_EXIT(&skip_es_lock);
			return (EEXIST);
		}
		SKIP_SET_NOTIFIER(ifp->notifier, wq);
		/*FALLTHROUGH*/

	case SkipAccessControlOn:
	case SkipAccessControlOff:
		ifp->if_mode = reqp->if_mode;
		break;

	default:
		rc = EINVAL;
	}
	MUTEX_EXIT(&skip_es_lock);
	return (rc);
}

/* skip_es_get_mode()
 *
 * Test if access control is on or off
 *
 * Returns: 0 on success, errno otherwise
 */
int
skip_es_get_mode(skip_es_mode_t *reqp)
{
	register skip_es_t		*ifp;
	register int			rc = 0;

	/*
	 * grab the lock on the interface table
	 */
	MUTEX_ENTER(&skip_es_lock);

	ifp = skip_es_find_if(reqp->if_name);

	if (!ifp) {
		/*
		 * interface not found...
		 */
		rc = ENODEV;
	} else { 
		reqp->if_mode = ifp->if_mode;
	}

	MUTEX_EXIT(&skip_es_lock);
	return (rc);
}
