/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)skip_names.c	1.18 95/11/22 Sun Microsystems"

#include <stdio.h>
#include <fcntl.h>
#include <stropts.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <string.h>
#include <netdb.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>

#include <skip_types.h>
#include <skip_proto.h>
#include <skip_ioctl.h>
#include <skip_lib.h>

#define	STRSZ				(256)
#define	MAX_ALGS			(256)

/*
 * Name Space Identifiers
 */
const int	skip_max_nsids = SKIP_NSID_MD5_NIS + 1;
skip_nsid_inf_t	skip_nsids[] =
		{
			{ "Not present", 		 0 },	/* 0 */
			{ "IPv4 Address", 		 4 },	/* 1 */
			{ "POSIX/XOPEN User ID",	 4 }, 	/* 2 */
			{ "IPv6 Address",		16 },	/* 3 */
			{ "MD5 (DNS name)",		16 },	/* 4 */
			{ "MD5 (ASN.1 DN)",		16 },	/* 5 */
			{ "MD5 (Social Sec. #)",	16 },	/* 6 */
			{ "802.x Address",		 6 },	/* 7 */
			{ "MD5 (DH Pub.Value)",		16 },	/* 8 */
			{ "MD5 (Email Address)",	16 },	/* 9 */
			{ "MD5 (Bank Account #)",	16 },	/* 10 */
			{ "MD5 (NIS Name)",		16 }	/* 11 */

		};

const int	skip_max_algs = MAX_ALGS;
/*
 * Key encryption algorithms
 */
char		*skip_kij_algs[MAX_ALGS] = 
		{
				SKIP_NO_ALG,	/* 0 */
				"DES-CBC",	/* 1 */
				"DES-EDE-K3",	/* 2 */
				"IDEA-CBC"	/* 3 */
		};
boolean_t	skip_supported_kij_algs[MAX_ALGS] =  { 0 };
/*
 * Traffic encryption algorithms
 */
char		*skip_kp_algs[MAX_ALGS] = 
		{
				SKIP_NO_ALG,	/* 0 */
				"DES-CBC",	/* 1 */
				"DES-EDE-K3"	/* 2 */
		};
boolean_t	skip_supported_kp_algs[MAX_ALGS] =  { 0 };
/*
 * V1/SunScreen Key encryption algorithms
 */
char		*skip_kij_algs_v1[MAX_ALGS] = 
		{
					SKIP_NO_ALG,	/* 0 */
					"DES-CBC",	/* 1 */
					"RC2-CBC",	/* 2 */
					"RC4-40",	/* 3 */
					"RC4-128",	/* 4 */
					"DES-EDE-K2",	/* 5 */
					"DES-EDE-K3",	/* 6 */
					"",		/* 7 */
					"",		/* 8 */
					"",		/* 9 */
					"simplecrypt" 	/* 10 */ 
		};
boolean_t	skip_supported_kij_algs_v1[MAX_ALGS] =  { 0 };
/*
 * V1/SunScreen Traffic encryption algorithms
 */
char		*skip_kp_algs_v1[MAX_ALGS] = 
		{
					SKIP_NO_ALG,	/* 0 */
					"DES-CBC",	/* 1 */
					"RC2-CBC",	/* 2 */
					"RC4-40",	/* 3 */
					"RC4-128",	/* 4 */
					"DES-EDE-K2",	/* 5 */
					"DES-EDE-K3",	/* 6 */
					"",		/* 7 */
					"",		/* 8 */
					"",		/* 9 */
					"simplecrypt" 	/* 10 */ 
		};
boolean_t	skip_supported_kp_algs_v1[MAX_ALGS] =  { 0 };

/*
 * translate from a key encryption algorithm name
 */
int
skip_name_to_kij_alg(char *s, unsigned char version)
{
	int		alg;
	char		**algnames;

	switch (version) {
	case SKIP_NONE:
		if (strcasecmp(s, SKIP_NO_ALG) == 0) {
			return (0);
		}
		break;

	case SKIP_V1:
		algnames = skip_kij_algs_v1;
		break;

	case SKIP_V2:
		algnames = skip_kij_algs;
		break;

	default:
		return (-1);
	}
	for (alg = 0; alg < skip_max_algs; alg++) {
		if (algnames[alg] == NULL) {
			continue;
		}
		if (strcasecmp(algnames[alg], s) == 0) {
			return (alg);
		}
	}
	return (-1);
}

/*
 * translate to a key encryption algorithm name
 */
char *
skip_kij_alg_to_name(int alg, unsigned char version)
{
	char		**algnames;

	if ((alg >= skip_max_algs) || (alg < 0)) {
		return ("<invalid>");
	}
	switch (version) {
	case SKIP_V2:
		algnames = skip_kij_algs;
		break;

	case SKIP_V1:
		algnames = skip_kij_algs_v1;
		break;

	default:
		return ("<invalid version>");
	}
	if ((algnames[alg] == NULL) || (strlen(algnames[alg]) == 0)) {
		return ("<unknown>");
	}
	return (algnames[alg]);
}

/*
 * return B_TRUE if a particular key encryption algorithm is supported
 */
boolean_t
skip_supported_kij_alg(int alg, unsigned char version)
{
	boolean_t	*algsupported;

	if ((alg >= skip_max_algs) || (alg < 0)) {
		return (B_FALSE);
	}
	algsupported = (version == SKIP_V2 ? skip_supported_kij_algs :
						skip_supported_kij_algs_v1);
	return (algsupported[alg]);
}

/*
 * translate from a traffic encryption algorithm name
 */
int
skip_name_to_kp_alg(char *s, unsigned char version)
{
	int		alg;
	char		**algnames;

	switch (version) {
	case SKIP_NONE:
		if (strcasecmp(s, SKIP_NO_ALG) == 0) {
			return (0);
		}
		break;

	case SKIP_V1:
		algnames = skip_kp_algs_v1;
		break;

	case SKIP_V2:
		algnames = skip_kp_algs;
		break;

	default:
		return (-1);
	}
	for (alg = 0; alg < skip_max_algs; alg++) {
		if (algnames[alg] == NULL) {
			continue;
		}
		if (strcasecmp(algnames[alg], s) == 0) {
			return (alg);
		}
	}
	return (-1);
}

/*
 * translate to a key encryption algorithm name
 */
char *
skip_kp_alg_to_name(int alg, unsigned char version)
{
	char		**algnames;

	if ((alg >= skip_max_algs) || (alg < 0)) {
		return ("<invalid>");
	}
	switch (version) {
	case SKIP_V2:
		algnames = skip_kp_algs;
		break;

	case SKIP_V1:
		algnames = skip_kp_algs_v1;
		break;

	default:
		return ("<invalid version>");
	}
	if ((algnames[alg] == NULL) || (strlen(algnames[alg]) == 0)) {
		return ("<unknown>");
	}
	return (algnames[alg]);
}

/*
 * return B_TRUE if a particular key encryption algorithm is supported
 */
boolean_t
skip_supported_kp_alg(int alg, unsigned char version)
{
	boolean_t	*algsupported;

	if ((alg >= skip_max_algs) || (alg < 0)) {
		return (B_FALSE);
	}
	algsupported = (version == SKIP_V2 ? skip_supported_kp_algs :
						skip_supported_kp_algs_v1);
	return (algsupported[alg]);
}

/* skip_host_to_addr()
 * 
 * translate a hostname to a struct in_addr
 *       
 * Returns: 0 on success, -1 otherwise 
 */
int
skip_host_to_addr(char *host, struct in_addr *addr)
{
	struct hostent	*hp = NULL;
	long		in_addr; 
 
	if (strlen(host) == 0) {
		return (-1);
	}

	in_addr = inet_addr(host); 
	if (in_addr != -1) { 
		memcpy((char *) addr, (char *) &in_addr, 4);
		return (0); 
	} 
	hp = gethostbyname(host);
	if (hp == NULL) {
		return (-1);
	}
	memcpy((char *) addr, hp->h_addr, 4);
	return (0);
}


/* skip_addr_to_host()
 * 
 * translate a struct in_addr to hostname
 *       
 * Returns: 0 on success, -1 otherwise 
 */
int
skip_addr_to_host(struct in_addr *addr, char *host)
{
	struct hostent		*hp;
 
	if ((addr == NULL) || (host == NULL)) {
		return (-1);
	}

	hp = gethostbyaddr((char *) addr, sizeof(*addr), AF_INET);
	if (hp == NULL) {
		/*
		 * failed to translate
		 */
		strcpy(host, (char *) inet_ntoa(*addr));
		return (0);
	}
	strcpy(host, hp->h_name);
	return (0);
}

/* skip_s_to_keyid()
 *
 * translate a key id string to a binary value
 *       
 * Returns: 0 on success, -1 otherwise 
 */
int
skip_s_to_keyid(char *keyidstr, unsigned char nsid, skip_key_var_t *keyid)
{
	struct hostent		*hp = NULL;
	long			ip_addr;
	char			*s;
	int			i;

	if ((keyidstr == NULL) || ((int) nsid >= skip_max_nsids)) {
		return (-1);
	}

	memset(keyid->buf, 0, MAXVARSZ);
	keyid->len = 0;

	switch (nsid) {

	case SKIP_NSID_NONE:
		/*
		 * no key - string should be empty
		 */
		if (strlen(keyidstr) == 0) {
			return (0);
		}
		break;

	case SKIP_NSID_IPV4:
		/*
		 * try to process as a dot format IP address
		 */
		if (strchr(keyidstr, '.')) {
			ip_addr = inet_addr(keyidstr);
			if (ip_addr == -1) {
				return (-1);
			}
			memcpy((char *) keyid->buf, (char *) &ip_addr,
							sizeof(ip_addr));
			keyid->len = sizeof(ip_addr);
			return (0);
		}

		/*
		 * try to process as a hostname
	 	*/
		hp = gethostbyname(keyidstr);
		if (hp) {
			memcpy((char *) keyid->buf, hp->h_addr, sizeof ip_addr);
			keyid->len = sizeof(ip_addr);
			return (0);
		}

		/*
		 * try to process as an integer
		 */
		/*FALLTHROUGH*/

	default:

		/*
		 * get storage for strtok()
		 */
		if ((keyidstr = strdup(keyidstr)) == NULL) {
			return (-1);
		}
		s = strtok(keyidstr, " 	");
		if (s == NULL) {
			free(keyidstr);
			return (-1);
		}
		/*
		 * strip any leading 0x, 0X
		 */
		if (((int) strlen(s) > 2) && (s[0] == '0') &&
			((s[1] == 'x') || s[1] == 'X')) {
			s += 2;
		}
		/*
		 * MUST have 2 hex digits per byte
		 */
		if (strlen(s) != skip_nsids[nsid].len * 2) {
			free(keyidstr);
			return (-1);
		}
		/*
		 * convert hex string to binary
		 */
		for (i = 0; i < skip_nsids[nsid].len; i++) {
			if (isxdigit(s[0]) && isxdigit(s[1])) {
				keyid->buf[i] =
					(isdigit(*s) ? (*s) - '0' :
                        		 islower(*s) ? (*s) + 10 - 'a' :
					 (*s) + 10 - 'A') << 4;
				s++;
				keyid->buf[i] +=
					(isdigit(*s) ? (*s) - '0' :
                        		 islower(*s) ? (*s) + 10 - 'a' :
					 (*s) + 10 - 'A');
				s++;
			} else {
				free(keyidstr);
				return (-1);
			}
		}
		keyid->len = skip_nsids[nsid].len;
		return (0);
	}
		
	return (-1);
}

/* skip_keyid_to_s()
 *
 * translate a key id to a hex string
 *
 * Returns: 0 on success, -1 otherwise 
 */
int
skip_keyid_to_s(skip_key_var_t *keyid, unsigned char nsid, char *keyidstr)
{
	int		i;
	unsigned long	keyint;
	
	if ((int) nsid >= skip_max_nsids) {
		return (-1);
	}
	switch (nsid) {

	case SKIP_NSID_NONE:
		strcpy(keyidstr, "");
		return(0);

	default:
		if (keyid->len != skip_nsids[nsid].len) {
			return (-1);
		}
		sprintf(keyidstr, "0x");
		keyidstr += strlen(keyidstr);
		for (i = 0; i < keyid->len; i++) {
			sprintf(keyidstr, "%02x", (unsigned char)keyid->buf[i]);
			keyidstr += 2;
		}
	}
	return (0);
}

#define MAXHOSTNAME	80
/*
 * Print an ACL entry..
 */
void
skip_host_print(skip_param_t *params)
{
	char			host[MAXHOSTNAME];
	char			keyid[MAXVARSZ*2];
	extern char		*skip_kij_algs[];
	extern char		*skip_kp_algs[];
	extern char		*skip_kij_algs_v1[];
	extern char		*skip_kp_algs_v1[];

	if (skip_addr_to_host(&params->ip_addr, host) < 0) {
		printf("Get host name failed!\n");
	}

	switch (params->version) {

	case SKIP_NONE:
		printf("%s\n", host);
		break;

	case SKIP_V1:
		printf("%s: (SunScreen compatibility) params:\n", host);

		printf("\tKij alg:\t\t%s\n",
					skip_kij_algs_v1[params->kij_alg]);
		printf("\tCrypt alg:\t\t%s\n",
					skip_kp_algs_v1[params->kp_alg]);

		if (params->r_nsid == SKIP_NSID_IPV4) {
			if (skip_keyid_to_s(&params->r_mkeyid,
						params->r_nsid, keyid) < 0) {
				printf("\t*** Error in key id\n");
			} else {
				printf("\tReceiver node id:\t%s\n", keyid);
			}
		} else {
			printf("*** : Invalid NSID/node id\n");
		}
		break;

	case SKIP_V2 :
		printf("%s: SKIP params:\n", host);

		printf("\tKij alg:\t\t%s\n",
					skip_kij_algs[params->kij_alg]);
		printf("\tCrypt alg:\t\t%s\n",
					skip_kp_algs[params->kp_alg]);

		printf("\tReceiver NSID:\t\t%s\n",
					skip_nsids[params->r_nsid].name);

		if (params->r_nsid && params->r_mkeyid.len) {
			if (skip_keyid_to_s(&params->r_mkeyid,
						params->r_nsid, keyid) < 0) {
				printf("\t*** Error in key id\n");
			} else {
				printf("\tReceiver key id:\t%s\n", keyid);
			}
		}

		printf("\tSender NSID:\t\t%s\n",
				skip_nsids[params->s_nsid].name);
		break;

	default:
		printf("%s:\t\tbad version %d\n", host, params->version);

	}

}
