/* proxy.c
 *
 * $Id$ */

/* lsh, an implementation of the ssh protocol
 *
 * Copyright (C) 1999 Balazs Scheidler
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "proxy.h"
#include "lsh.h"
#include "xalloc.h"
#include "werror.h"

#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>

#define GABA_DEFINE
#include "proxy.h.x"
#undef GABA_DEFINE

#include "proxy.c.x"

/* GABA:
   (class
     (name notify_proxy_connection)
     (super command)
     (vars
       (proxy object proxy_connection)))
*/

static int do_notify_server_conn(struct command *s, struct lsh_object *x, struct command_continuation *c)
{
  CAST(notify_proxy_connection, self, s);
  CAST(ssh_connection, conn, x);
  
  SET_SERVER_CONN(self->proxy, conn);
  return COMMAND_RETURN(c, (struct lsh_object *) self->proxy);
}

static int do_notify_client_conn(struct command *s, struct lsh_object *x, struct command_continuation *c)
{
  CAST(notify_proxy_connection, self, s);
  CAST(ssh_connection, conn, x);
  
  SET_CLIENT_CONN(self->proxy, conn);
  return COMMAND_RETURN(c, (struct lsh_object *) self->proxy);
}

struct command *make_notify_server_conn(struct proxy_connection *proxy)
{
  NEW(notify_proxy_connection, self);
  self->super.call = do_call_simple_command;
  self->proxy = proxy;
  return &self->super;
}

struct command *make_notify_client_conn(struct proxy_connection *proxy)
{
  NEW(notify_proxy_connection, self);
  self->super.call = do_notify_client_conn;
  self->proxy = proxy;
  return &self->super;
}

static void do_set_client_conn(struct proxy_connection *c, struct ssh_connection *conn)
{
  c->client = conn;
}

static void do_set_server_conn(struct proxy_connection *c, struct ssh_connection *conn)
{
  c->server = conn;
}

struct proxy_connection *make_proxy_connection(void)
{
  NEW(proxy_connection, c);
  c->set_client_conn = do_set_client_conn;
  c->set_server_conn = do_set_server_conn;
  return c;
}

/* converts a listen_value to an address_info 
 * uses getsockname to get the original destination */
static struct lsh_object *do_get_trans_proxy_dest(struct command_simple *c UNUSED, struct lsh_object *x)
{
  CAST_SUBTYPE(lsh_fd, fd, x);
  
  if (fd) {
    char sockaddr_buf[256];
    struct sockaddr *sa = (struct sockaddr *) &sockaddr_buf;
    int salen = sizeof(sockaddr_buf);
    
    if (getsockname(fd->fd, sa, &salen) < -1) {
      werror("do_get_trans_proxy_dest(): getsockname() failed: %z\n", STRERROR(errno));
      return NULL;
    }
    return (struct lsh_object *) sockaddr2info(salen, sa);
  }
  return NULL;
}

struct command_simple get_trans_proxy_dest =
STATIC_COMMAND_SIMPLE(do_get_trans_proxy_dest);
