/*
 * RFC2367 PF_KEYv2 Key management API domain socket I/F
 * Copyright (C) 1999  Richard Guy Briggs.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * RCSID $Id: pfkey_v2.c,v 1.24 2000/01/10 16:38:23 rgb Exp $
 */

/*
 *		Template from /usr/src/linux-2.0.36/net/unix/af_unix.c.
 *		Hints from /usr/src/linux-2.0.36/net/ipv4/udp.c.
 */

#define __NO_VERSION__
#include <linux/module.h>
#include <linux/version.h>

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/major.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/stat.h>
#include <linux/socket.h>
#include <linux/un.h>
#include <linux/fcntl.h>
#include <linux/termios.h>
#include <linux/socket.h>
#include <linux/sockios.h>
#include <linux/net.h>
#include <linux/in.h>
#include <linux/fs.h>
#include <linux/malloc.h>
#include <asm/segment.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <net/sock.h>
/* #include <net/tcp.h> */
#include <net/af_unix.h>
#ifdef CONFIG_PROC_FS
#include <linux/proc_fs.h>
#endif /* CONFIG_PROC_FS */

#include <linux/types.h>
 
#include <freeswan.h>
#ifdef NET_21
#include <asm/uaccess.h>
#include <linux/in6.h>
#endif /* NET_21 */
#include <pfkeyv2.h>
#include <pfkey.h>
#include "radij.h"
#include "ipsec_encap.h"
#include "ipsec_netlink.h"

#ifdef DEBUG_IPSEC
int debug_pfkey = 0;
#endif

#define min(a,b)	(((a)<(b))?(a):(b))

#define SENDERR(_x) do { len = -(_x); goto errlab; } while (0)

#ifdef NET_21_
static atomic_t pfkey_nr_socks = ATOMIC_INI(0);
#endif /* NET_21 */

struct proto_ops pfkey_ops;
struct sock *pfkey_sock_list = NULL;

struct socket_list *pfkey_open_sockets = NULL;
struct socket_list *pfkey_registered_sockets = NULL;

int pfkey_msg_interp(struct sock *, struct sadb_msg *);

void
pfkey_list_remove_socket(struct socket *socketp, struct socket_list* sockets)
{
	struct socket_list *socket_listp,*prev;
	socket_listp = sockets;
	prev = NULL;
	
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_list_remove_socket: 0x%p\n",socketp);
	
	while(socket_listp != NULL) {
		if(socket_listp->socketp == socketp) {
			if(prev != NULL) {
				prev->next = socket_listp->next;
			} else {
				sockets = socket_listp->next;
			}
			
			kfree_s((void*)socket_listp, sizeof(struct socket_list));
			
			break;
		}
		prev = socket_listp;
		socket_listp = socket_listp->next;
	}
}


void
pfkey_list_insert_socket(struct socket *socketp, struct socket_list* sockets)
{
	struct socket_list *socket_listp;
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_list_insert_socket: sock=%p\n",socketp);
	
	socket_listp = (struct socket_list *)kmalloc(sizeof(struct socket_list), GFP_KERNEL);
	if(socket_listp == NULL)	{
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_list_insert_socket: memory allocation error.\n");
		return;
	}
	
	socket_listp->socketp = socketp;
	socket_listp->next = sockets;
	sockets = socket_listp;
}
  
#ifdef NET_21_
static void pfkey_destruct(struct sock *sk)
{
	return;
}
#endif /* NET_21 */

#ifndef NET_21
static void
pfkey_state_change(struct sock *sk)
{
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_state_change: .\n");
	if(!sk->dead) {
		wake_up_interruptible(sk->sleep);
	}
}
#endif /* !NET_21 */

#ifndef NET_21
static void
pfkey_data_ready(struct sock *sk, int len)
{
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_data_ready: .sk=%p len=%d\n", sk, len);
	if(!sk->dead) {
		wake_up_interruptible(sk->sleep);
		sock_wake_async(sk->socket, 1);
	}
}

static void
pfkey_write_space(struct sock *sk)
{
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_write_space: .\n");
	if(!sk->dead) {
		wake_up_interruptible(sk->sleep);
		sock_wake_async(sk->socket, 2);
	}
}

static void
pfkey_insert_socket(struct sock *sk)
{
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_insert_socket: sk=%p\n", sk);
	cli();
	sk->next=pfkey_sock_list;
	pfkey_sock_list=sk;
	sti();
}

static void
pfkey_remove_socket(struct sock *sk)
{
	struct sock **s;
	
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_remove_socket: .\n");
	cli();
	s=&pfkey_sock_list;

	while(*s!=NULL) {
		if(*s==sk) {
			*s=sk->next;
			/* XXX rgb */ sk->next=NULL;
			sti();
			KLIPS_PRINT(debug_pfkey,
				    "klips_debug:pfkey_remove_socket: succeeded.\n");
			return;
		}
		s=&((*s)->next);
	}
	sti();
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_remove_socket: not found.\n");
	return;
}
#endif /* !NET_21 */

/* static */ void
pfkey_destroy_socket(struct sock *sk)
{
	struct sk_buff *skb;

	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_destroy_socket: .\n");
#ifdef NET_21
	/* sklist_destroy_socket(&pfkey_sock_list, sk); */
	sklist_remove_socket(&pfkey_sock_list, sk);
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_destroy_socket: sklist_remove_socket called.\n");
#else /* NET_21 */
	pfkey_remove_socket(sk);
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_destroy_socket: pfkey_remove_socket called.\n");
#endif /* NET_21 */
	
#ifdef NET_21_
#if 0
	skb_queue_purge(&sk->receive_queue);
	skb_queue_purge(&sk->write_queue);
#endif
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_destroy_socket: "
		    "sk(%p)->receive_queue(%p) not dequeued, not freeing, "
		    "allowing memory to leak.\n", sk, sk->receive_queue.next);
#else /* NET_21 */
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_destroy_socket: "
		    "sk(%p)->(&%p)receive_queue.{next=%p,prev=%p}.\n",
		    sk,
		    &(sk->receive_queue),
		    sk->receive_queue.next,
		    sk->receive_queue.prev);
	while(sk && ((skb=skb_dequeue(&(sk->receive_queue)))!=NULL)) {
#ifdef NET_21
if(0 && debug_pfkey) {		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_destroy_socket: skb=%p dequeued.\n", skb);
		printk(KERN_INFO "klips_debug: pfkey_skb contents:");
		printk(" next:%p", skb->next);
		printk(" prev:%p", skb->prev);
		printk(" list:%p", skb->list);
		printk(" sk:%p", skb->sk);
		printk(" stamp:%ld.%ld", skb->stamp.tv_sec, skb->stamp.tv_usec);
		printk(" dev:%p", skb->dev);
		if(skb->dev) {
			if(skb->dev->name) {
				printk(" dev->name:%s", skb->dev->name);
			} else {
				printk(" dev->name:NULL?");
			}
		} else {
			printk(" dev:NULL");
		}
/*		printk(" dev->name:%s", skb->dev ? (skb->dev->name ? skb->dev->name : "NULL") : "NULL"); */
		printk(" h:%p", skb->h.raw);
		printk(" nh:%p", skb->nh.raw);
		printk(" mac:%p", skb->mac.raw);
		printk(" dst:%p", skb->dst);
#if 0
		{ int i;
		printk(" cb");
		for(i=0; i<48; i++) {
			printk(":%2x", skb->cb[i]);
		}
		}
#endif
		printk(" len:%d", skb->len);
		printk(" csum:%d", skb->csum);
		printk(" used:%d", skb->used);
		printk(" is_clone:%d", skb->is_clone);
		printk(" cloned:%d", skb->cloned);
		printk(" pkt_type:%d", skb->pkt_type);
		printk(" pkt_bridged:%d", skb->pkt_bridged);
		printk(" ip_summed:%d", skb->ip_summed);
		printk(" priority:%d", skb->priority);
		printk(" protocol:%d", skb->protocol);
		printk(" security:%d", skb->security);
		printk(" truesize:%d", skb->truesize);
		printk(" head:%p", skb->head);
		printk(" data:%p", skb->data);
		printk(" tail:%p", skb->tail);
		printk(" end:%p", skb->end);
#if 0
		{ unsigned int i;
		printk(" data");
		for(i=(unsigned int)(skb->head); i<(unsigned int)(skb->end); i++) {
			printk(":%2x", (unsigned char)(*(char*)(i)));
		}
		}
#endif
		printk(" destructor:%p", skb->destructor);
		printk("\n");
}
#if 1
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_destroy_socket: skb=%p freed.\n", skb);
		kfree_skb(skb);
#else
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_destroy_socket: skb=%p leaking.\n", skb);
#endif

#ifdef THIS_IS_A_QUICKREF_OF_THE_STRUCT_SK_BUFF
	struct sk_buff	* next;			/* Next buffer in list 				*/
	struct sk_buff	* prev;			/* Previous buffer in list 			*/
	struct sk_buff_head * list;		/* List we are on				*/
	struct sock	*sk;			/* Socket we are owned by 			*/
	struct timeval	stamp;			/* Time we arrived				*/
	struct device	*dev;			/* Device we arrived on/are leaving by		*/

	/* Transport layer header */
	union
	{
		struct tcphdr	*th;
		struct udphdr	*uh;
		struct icmphdr	*icmph;
		struct igmphdr	*igmph;
		struct iphdr	*ipiph;
		struct spxhdr	*spxh;
		unsigned char	*raw;
	} h;

	/* Network layer header */
	union
	{
		struct iphdr	*iph;
		struct ipv6hdr	*ipv6h;
		struct arphdr	*arph;
		struct ipxhdr	*ipxh;
		unsigned char	*raw;
	} nh;
  
	/* Link layer header */
	union 
	{	
	  	struct ethhdr	*ethernet;
	  	unsigned char 	*raw;
	} mac;

	struct  dst_entry *dst;

	char		cb[48];	 

	unsigned int 	len;			/* Length of actual data			*/
	unsigned int	csum;			/* Checksum 					*/
	volatile char 	used;			/* Data moved to user and not MSG_PEEK		*/
	unsigned char	is_clone,		/* We are a clone				*/
			cloned, 		/* head may be cloned (check refcnt to be sure). */
  			pkt_type,		/* Packet class					*/
  			pkt_bridged,		/* Tracker for bridging 			*/
  			ip_summed;		/* Driver fed us an IP checksum			*/
	__u32		priority;		/* Packet queueing priority			*/
	atomic_t	users;			/* User count - see datagram.c,tcp.c 		*/
	unsigned short	protocol;		/* Packet protocol from driver. 		*/
	unsigned short	security;		/* Security level of packet			*/
	unsigned int	truesize;		/* Buffer size 					*/

	unsigned char	*head;			/* Head of buffer 				*/
	unsigned char	*data;			/* Data head pointer				*/
	unsigned char	*tail;			/* Tail pointer					*/
	unsigned char 	*end;			/* End pointer					*/
	void 		(*destructor)(struct sk_buff *);	/* Destruct function		*/
#endif

#else /* NET_21 */
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_destroy_socket: skb=%p dequeued and freed.\n", skb);
		kfree_skb(skb, FREE_WRITE);

#endif /* NET_21 */
	}
#endif /* NET_21 */

	sk->dead = 1;
	sk_free(sk);

	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_destroy_socket: destroyed.\n");
}

/* static */ int
pfkey_upmsg(struct socket *sock, struct sadb_msg *pfkey_msg)
{
	int error;
	struct sk_buff * skb = NULL;
	struct sock *sk;

	if(sock == NULL) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_upmsg: NULL socket passed in.\n");
		return -EINVAL;
	}

	if(pfkey_msg == NULL) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_upmsg: NULL pfkey_msg passed in.\n");
		return -EINVAL;
	}

#ifdef NET_21
	sk = sock->sk;
#else /* NET_21 */
	sk = sock->data;
#endif /* NET_21 */

	if(sk == NULL) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_upmsg: NULL sock passed in.\n");
		return -EINVAL;
	}

	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_upmsg: allocating %d bytes.\n",
		    pfkey_msg->sadb_msg_len * IPSEC_PFKEYv2_ALIGN);
	if(!(skb = alloc_skb(pfkey_msg->sadb_msg_len * IPSEC_PFKEYv2_ALIGN, GFP_ATOMIC) )) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_upmsg: no buffers left to send up a message.\n");
		return -ENOBUFS;
	}
	
#if 0
	memset(skb, 0, sizeof(*skb));  /* this is a bad idea because it resets the data pointers */
#else
	skb->dev = NULL;
#endif
	
	if(skb_tailroom(skb) < pfkey_msg->sadb_msg_len * IPSEC_PFKEYv2_ALIGN) {
		printk(KERN_WARNING "klips_error:pfkey_upmsg: "
		       "tried to skb_put %ld, %d available.  "
		       "This should never happen, please report.\n",
		       (unsigned long int)pfkey_msg->sadb_msg_len * IPSEC_PFKEYv2_ALIGN,
		       skb_tailroom(skb));
#ifdef NET_21
		kfree_skb(skb);
#else /* NET_21 */
		kfree_skb(skb, FREE_WRITE);
#endif /* NET_21 */
		return -ENOBUFS;
	}
	skb->h.raw = skb_put(skb, pfkey_msg->sadb_msg_len * IPSEC_PFKEYv2_ALIGN);
	memcpy(skb->h.raw, pfkey_msg, pfkey_msg->sadb_msg_len * IPSEC_PFKEYv2_ALIGN);

#ifndef NET_21
	skb->free = 1;
#endif /* !NET_21 */

	if((error = sock_queue_rcv_skb(sk, skb)) < 0) {
		skb->sk=NULL;
#ifdef NET_21
		kfree_skb(skb);
#else /* NET_21 */
		kfree_skb(skb, FREE_WRITE);
#endif /* NET_21 */
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_upmsg: error calling sock_queue_rcv_skb.\n");
		return error;
	}
	return 0;
}

/* static */ int
pfkey_create(struct socket *sock, int protocol)
{
	struct sock *sk;

	if(sock == NULL) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_create: socket NULL.\n");
		return -EINVAL;
	}

	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_create: sock=%p type:%d state:%d flags:%ld protocol:%d\n",
		    sock,
		    sock->type,
		    (unsigned int)(sock->state),
		    sock->flags, protocol);

	if(sock->type != SOCK_RAW) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_create: only SOCK_RAW supported.\n");
		return -ESOCKTNOSUPPORT;
	}

	if(protocol != PF_KEY_V2) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_create: protocol not PF_KEY_V2.\n");
		return -EPROTONOSUPPORT;
	}

	if((current->uid != 0)) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_create: must be root to open pfkey sockets.\n");
		return -EACCES;
	}

#ifdef NET_21
	sock->state = SS_UNCONNECTED;
#endif /* NET_21 */
	MOD_INC_USE_COUNT;
#ifdef NET_21
	if((sk=(struct sock *)sk_alloc(PF_KEY, GFP_KERNEL, 1)) == NULL)
#else /* NET_21 */
	if((sk=(struct sock *)sk_alloc(GFP_KERNEL)) == NULL)
#endif /* NET_21 */
	{
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_create: Out of memory trying to allocate.\n");
		MOD_DEC_USE_COUNT;
		return -ENOMEM;
	}

#ifndef NET_21
	memset(sk, 0, sizeof(*sk));
#endif /* !NET_21 */

#ifdef NET_21
	sock_init_data(sock, sk);

	sk->destruct = NULL; /* pfkey_destruct; */
	sk->reuse = 1;
	sock->ops = &pfkey_ops;

	sk->zapped=0;
	sk->family = PF_KEY;
/*	sk->num = protocol; */
	sk->protocol = protocol;
	key_pid(sk) = current->pid;
	sklist_insert_socket(&pfkey_sock_list, sk);
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_create: sock->fasync_list=%p sk->sleep=%p.\n",
		    sock->fasync_list,
		    sk->sleep);
#else /* NET_21 */
	sk->type=sock->type;
	init_timer(&sk->timer);
	skb_queue_head_init(&sk->write_queue);
	skb_queue_head_init(&sk->receive_queue);
	skb_queue_head_init(&sk->back_log);
	sk->rcvbuf=SK_RMEM_MAX;
	sk->sndbuf=SK_WMEM_MAX;
	sk->allocation=GFP_KERNEL;
	sk->state=TCP_CLOSE;
	sk->priority=SOPRI_NORMAL;
	sk->state_change=pfkey_state_change;
	sk->data_ready=pfkey_data_ready;
	sk->write_space=pfkey_write_space;
	sk->error_report=pfkey_state_change;
	sk->mtu=4096;
	sk->socket=sock;
	sock->data=(void *)sk;
	sk->sleep=sock->wait;
	pfkey_insert_socket(sk);
#endif /* NET_21 */

	pfkey_list_insert_socket(sock, pfkey_open_sockets);
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_create: Socket sock=%p sk=%p initialised.\n", sock, sk);
	return 0;
}

#ifndef NET_21
static int
pfkey_dup(struct socket *newsock, struct socket *oldsock)
{
	struct sock *sk;

	if(newsock==NULL) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_dup: No new socket attached.\n");
		return -EINVAL;
	}
		
	if(oldsock==NULL) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_dup: No old socket attached.\n");
		return -EINVAL;
	}
		
#ifdef NET_21
	sk=oldsock->sk;
#else /* NET_21 */
	sk=oldsock->data;
#endif /* NET_21 */
	
	/* May not have data attached */
	if(sk==NULL) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_dup: No sock attached to old socket.\n");
		return -EINVAL;
	}
		
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_dup: .\n");

	return pfkey_create(newsock, sk->protocol);
}
#endif /* !NET_21 */

/* static */ int
#ifdef NETDEV_23
pfkey_release(struct socket *sock)
#else
pfkey_release(struct socket *sock, struct socket *peersock)
#endif
{
	struct sock *sk;

	if(sock==NULL) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_release: No socket attached.\n");
		return 0; /* -EINVAL; */
	}
		
#ifdef NET_21
	sk=sock->sk;
#else /* NET_21 */
	sk=sock->data;
#endif /* NET_21 */
	
	/* May not have data attached */
	if(sk==NULL) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_release: No sk attached to sock=%p.\n", sock);
		return 0; /* -EINVAL; */
	}
		
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_release: .sock=%p sk=%p\n", sock, sk);

	/* sklist_remove_socket(&pfkey_sock_list, sk); */

#ifdef NET_21
	if(!sk->dead)
#endif /* NET_21 */
		if(sk->state_change) {
			sk->state_change(sk);
		}

#ifdef NET_21
	sock->sk = NULL;
#endif /* NET_21 */

	/* Try to flush out this socket. Throw out buffers at least */
#ifdef NET_21_
	sklist_destroy_socket(&pfkey_sock_list, sk);
#else /* NET_21 */
	pfkey_destroy_socket(sk);
#endif /* NET_21 */

	MOD_DEC_USE_COUNT;
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_release: succeeded.\n");

	pfkey_list_remove_socket(sock, pfkey_open_sockets);
	pfkey_list_remove_socket(sock, pfkey_registered_sockets);
	return 0;
}

#ifndef NET_21
static int
pfkey_bind(struct socket *sock, struct sockaddr *uaddr, int addr_len)
{
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_bind: operation not supported.\n");
	return -EINVAL;
}

static int
pfkey_connect(struct socket *sock, struct sockaddr *uaddr, int addr_len, int flags)
{
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_connect: operation not supported.\n");
	return -EINVAL;
}

static int
pfkey_socketpair(struct socket *a, struct socket *b)
{
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_socketpair: operation not supported.\n");
	return -EINVAL;
}

static int
pfkey_accept(struct socket *sock, struct socket *newsock, int flags)
{
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_aaccept: operation not supported.\n");
	return -EINVAL;
}

static int
pfkey_getname(struct socket *sock, struct sockaddr *uaddr, int *uaddr_len,
		int peer)
{
	struct sockaddr *ska = (struct sockaddr*)uaddr;
	
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_getname: .\n");
	ska->sa_family = PF_KEY;
	*uaddr_len = sizeof(*ska);
	return 0;
}

static int
pfkey_select(struct socket *sock, int sel_type, select_table *wait)
{
	
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_select: .sock=%p sk=%p sel_type=%d\n",
		    sock,
		    sock->data,
		    sel_type);
	if(sock == NULL) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_select: Null socket passed in.\n");
		return -EINVAL;
	}
	return datagram_select(sock->data, sel_type, wait);
}

static int
pfkey_ioctl(struct socket *sock, unsigned int cmd, unsigned long arg)
{
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_ioctl: not supported.\n");
	return -EINVAL;
}

static int
pfkey_listen(struct socket *sock, int backlog)
{
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_listen: not supported.\n");
	return -EINVAL;
}
#endif /* !NET_21 */

/* static */ int
pfkey_shutdown(struct socket *sock, int mode)
{
	struct sock *sk;

	if(sock == NULL) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_shutdown: NULL socket passed in.\n");
		return -EINVAL;
	}

#ifdef NET_21
	sk=sock->sk;
#else /* NET_21 */
	sk=sock->data;
#endif /* NET_21 */
	
	if(sk == NULL) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_shutdown: No sock attached to socket.\n");
		return -EINVAL;
	}

	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_shutdown: mode=%x.\n", mode);
	mode++;
	
	if(mode&SEND_SHUTDOWN) {
		sk->shutdown|=SEND_SHUTDOWN;
		sk->state_change(sk);
	}

	if(mode&RCV_SHUTDOWN) {
		sk->shutdown|=RCV_SHUTDOWN;
		sk->state_change(sk);
	}
	return 0;
}

#ifndef NET_21
static int
pfkey_setsockopt(struct socket *sock, int level, int optname, char *optval, int optlen)
{
#ifndef NET_21
	struct sock *sk;

	if(sock == NULL) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_setsockopt: Null socket passed in.\n");
		return -EINVAL;
	}
	
	sk=sock->data;
	
	if(sk == NULL) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_setsockopt: Null sock passed in.\n");
		return -EINVAL;
	}
#endif /* !NET_21 */
	
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_setsockopt: .\n");
	if(level!=SOL_SOCKET) {
		return -EOPNOTSUPP;
	}
#ifdef NET_21
	return sock_setsockopt(sock, level, optname, optval, optlen);
#else /* NET_21 */
	return sock_setsockopt(sk, level, optname, optval, optlen);
#endif /* NET_21 */
}

static int
pfkey_getsockopt(struct socket *sock, int level, int optname, char *optval, int *optlen)
{
#ifndef NET_21
	struct sock *sk;

	if(sock == NULL) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_setsockopt: Null socket passed in.\n");
		return -EINVAL;
	}
	
	sk=sock->data;
	
	if(sk == NULL) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_setsockopt: Null sock passed in.\n");
		return -EINVAL;
	}
#endif /* !NET_21 */

	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_getsockopt: .\n");
	if(level!=SOL_SOCKET) {
		return -EOPNOTSUPP;
	}
#ifdef NET_21
	return sock_getsockopt(sock, level, optname, optval, optlen);
#else /* NET_21 */
	return sock_getsockopt(sk, level, optname, optval, optlen);
#endif /* NET_21 */
}

static int
pfkey_fcntl(struct socket *sock, unsigned int cmd, unsigned long arg)
{
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_fcntl: not supported.\n");
	return -EINVAL;
}
#endif /* !NET_21 */

/*
 *	Send PF_KEY data down.
 */
		
/* static */ int
#ifdef NET_21
pfkey_sendmsg(struct socket *sock, struct msghdr *msg, int len, struct scm_cookie *scm)
#else /* NET_21 */
pfkey_sendmsg(struct socket *sock, struct msghdr *msg, int len, int nonblock, int flags)
#endif /* NET_21 */
{
	struct sock *sk;
	int error = 0;
#ifdef NET_21
#if 0
	struct sockaddr*ska;
#endif
#endif /* NET_21 */
#if 0
	struct sk_buff *skb;
#endif /* 0 */
	struct sadb_msg * pfkey_msg = NULL;

	if(sock == NULL) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_sendmsg: Null socket passed in.\n");
		SENDERR(EINVAL);
	}
	
#ifdef NET_21
	sk = sock->sk;
#else /* NET_21 */
	sk = sock->data;
#endif /* NET_21 */

	if(sk == NULL) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_sendmsg: Null sock passed in.\n");
		SENDERR(EINVAL);
	}
	
	if(msg == NULL) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_sendmsg: Null msghdr passed in.\n");
		SENDERR(EINVAL);
	}

#if 0
#ifdef NET_21
	ska = (struct sockaddr*)msg->msg_name;
#endif /* NET_21 */
#endif

	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_sendmsg: .\n");
	if(sk->err) {
		error = sock_error(sk);
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_sendmsg: sk->err is non-zero, returns %d.\n",
			    error);
		SENDERR(-error);
	}

	if((current->uid != 0)) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_sendmsg: must be root to send messages to pfkey sockets.\n");
		SENDERR(EACCES);
	}

#ifdef NET_21
	if(msg->msg_control)
#else /* NET_21 */
	if(flags || msg->msg_control)
#endif /* NET_21 */
	{
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_sendmsg: can't set flags or set msg_control.\n");
		SENDERR(EINVAL);
	}
		
	if(sk->shutdown & SEND_SHUTDOWN) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_sendmsg: shutdown.\n");
		send_sig(SIGPIPE, current, 0);
		SENDERR(EPIPE);
	}
	
	if(len < sizeof(struct sadb_msg)) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_sendmsg: bogus msg len of %d, too small.\n", len);
		SENDERR(EINVAL);
	}

	if((pfkey_msg = (struct sadb_msg*)kmalloc(len, GFP_KERNEL)) == NULL) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_sendmsg: memory allocation error.\n");
		SENDERR(ENOBUFS);
	}

	memcpy_fromiovec((void *)pfkey_msg, msg->msg_iov, len);

	if(pfkey_msg->sadb_msg_version != PF_KEY_V2) {
		KLIPS_PRINT(1 || debug_pfkey,
			    "klips_debug:pfkey_sendmsg: "
			    "not PF_KEY_V2 msg, found %d, should be %d.\n",
			    pfkey_msg->sadb_msg_version,
			    PF_KEY_V2);
		kfree((void*)pfkey_msg);
		return -EINVAL;
	}

	if(len != pfkey_msg->sadb_msg_len * IPSEC_PFKEYv2_ALIGN) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_sendmsg: "
			    "bogus msg len of %d, not %d byte aligned.\n",
			    len, IPSEC_PFKEYv2_ALIGN);
		SENDERR(EMSGSIZE);
	}

	/* XXX when this becomes a lib, keying daemons must be able to receive errors */
	if(pfkey_msg->sadb_msg_errno) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_sendmsg: errno set to %d.\n",
			    pfkey_msg->sadb_msg_errno);
		SENDERR(EINVAL);
	}

	/* check PID */
	if(pfkey_msg->sadb_msg_pid != current->pid) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_sendmsg: pid (%d) does not equal sending process pid (%d).\n",
			    pfkey_msg->sadb_msg_pid, current->pid);
		SENDERR(EINVAL);
	}

	if(pfkey_msg->sadb_msg_reserved) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_sendmsg: reserved field must be zero, set to %d.\n",
			    pfkey_msg->sadb_msg_reserved);
		SENDERR(EINVAL);
	}
	
	if((pfkey_msg->sadb_msg_type > SADB_MAX) || (!pfkey_msg->sadb_msg_type)){
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_sendmsg: msg type too large or small:%d.\n",
			    pfkey_msg->sadb_msg_type);
		SENDERR(EINVAL);
	}
	
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_sendmsg: msg sent for parsing.\n");
	
	if((error = pfkey_msg_interp(sk, pfkey_msg))) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_sendmsg: pfkey_msg_parse returns %d.\n",
			    error);
		SENDERR(-error);
	}
 errlab:
	if(pfkey_msg) {
		struct sock *sockp = pfkey_sock_list;

		pfkey_msg->sadb_msg_errno = error;
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_sendmsg: send up msg.\n");
		while(sockp != NULL) {
			pfkey_upmsg(sockp->socket, pfkey_msg);
			sockp = sockp->next;
		}
		kfree_s((void*)pfkey_msg, len);
	}

	return len;
}

/*
 *	Receive PF_KEY data up.
 */
		
/* static */ int
#ifdef NET_21
pfkey_recvmsg(struct socket *sock, struct msghdr *msg, int size, int flags, struct scm_cookie *scm)
#else /* NET_21 */
pfkey_recvmsg(struct socket *sock, struct msghdr *msg, int size, int noblock, int flags, int *addr_len)
#endif /* NET_21 */
{
	struct sock *sk;
#ifdef NET_21
	int noblock = flags & MSG_DONTWAIT;
#endif /* NET_21 */
#if 0
	struct sockaddr *ska;
#endif
	struct sk_buff *skb;
	int error;
#if 0
	unsigned char *sp;
	int len;
	int num;
#endif /* 0 */

	if(sock == NULL) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_recvmsg: Null socket passed in.\n");
		return -EINVAL;
	}

#ifdef NET_21
	sk = sock->sk;
#else /* NET_21 */
	sk = sock->data;
#endif /* NET_21 */

	if(sk == NULL) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_recvmsg: Null sock passed in for sock=%p.\n", sock);
		return -EINVAL;
	}

	if(msg == NULL) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_recvmsg: Null msghdr passed in for sock=%p, sk=%p.\n",
			    sock, sk);
		return -EINVAL;
	}

#if 0
#ifdef NET_21
	ska = (struct sockaddr*)msg->msg_name;
#endif /* NET_21 */
#endif

	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_recvmsg: sock=%p sk=%p msg=%p size=%d.\n",
		    sock, sk, msg, size);
	if(flags) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_sendmsg: flags (%d) not supported.\n",
			    flags);
		return -EOPNOTSUPP;
	}
		
#ifdef NET_21
	msg->msg_namelen = 0; /* sizeof(*ska); */
#else /* NET_21 */
	if(addr_len) {
		*addr_len = 0; /* sizeof(*ska); */
	}
#endif /* NET_21 */
		
	if(sk->err) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_sendmsg: sk->err=%d.\n", sk->err);
		return sock_error(sk);
	}

	if((skb = skb_recv_datagram(sk, flags, noblock, &error) ) == NULL) {
                return error;
	}

	if(size > skb->len) {
		size = skb->len;
	}
#ifdef NET_21
	else if(size <skb->len) {
		msg->msg_flags |= MSG_TRUNC;
	}
#endif /* NET_21 */

	skb_copy_datagram_iovec(skb, 0, msg->msg_iov, size);
        sk->stamp=skb->stamp;

#if 0
	if(ska) {
		memset(ska, 0, sizeof(* ska));
		ska->sa_family = PF_KEY;
	}
#endif

	skb_free_datagram(sk, skb);
	return size;
}

#ifdef NET_21
struct net_proto_family pfkey_family_ops = {
	PF_KEY,
	pfkey_create
};

#ifndef SOCKOPS_WRAPPED
#define SOCKOPS_WRAPPED(name) name
#endif

struct proto_ops SOCKOPS_WRAPPED(pfkey_ops) = {
	PF_KEY,
#ifndef NETDEV_23
	sock_no_dup,
#endif
	pfkey_release,
	sock_no_bind,
	sock_no_connect,
	sock_no_socketpair,
	sock_no_accept,
	sock_no_getname,
	datagram_poll,
	sock_no_ioctl,
	sock_no_listen,
	pfkey_shutdown,
	sock_no_setsockopt,
	sock_no_getsockopt,
	sock_no_fcntl,
	pfkey_sendmsg,
	pfkey_recvmsg
#ifdef NETDEV_23
	, sock_no_mmap
#endif
};

#ifdef NETDEV_23
#include <linux/smp_lock.h>
SOCKOPS_WRAP(pfkey, PF_KEY);
#endif

#else /* NET_21 */
struct proto_ops pfkey_proto_ops = {
	PF_KEY,
	pfkey_create,
	pfkey_dup,
	pfkey_release,
	pfkey_bind,
	pfkey_connect,
	pfkey_socketpair,
	pfkey_accept,
	pfkey_getname,
	pfkey_select,
	pfkey_ioctl,
	pfkey_listen,
	pfkey_shutdown,
	pfkey_setsockopt,
	pfkey_getsockopt,
	pfkey_fcntl,
	pfkey_sendmsg,
	pfkey_recvmsg
};
#endif /* NET_21 */
   
#ifdef CONFIG_PROC_FS
#ifndef PROC_FS_2325
static
#endif /* PROC_FS_2325 */
int
pfkey_get_info(char *buffer, char **start, off_t offset, int length
#ifndef  PROC_NO_DUMMY
, int dummy
#endif /* !PROC_NO_DUMMY */
)
{
	off_t pos=0;
	off_t begin=0;
	int len=0;
	struct sock *sk=pfkey_sock_list;
	
	len+= sprintf(buffer,"sock     pid  d sleep    socket   next     prev     e destruct r z fa n p r  w o sndbf stamp    Flags    Type     St\n");
	
	while(sk!=NULL) {
#ifdef NET_21
		len+=sprintf(buffer+len,"%8p %4d %d %8p %8p %8p %8p %d %8p %d %d %2d %d %d %d %d %d %2d %d.%06d %08lX %08X %02X\n",
			     sk,
			     
			     key_pid(sk),
			     sk->dead,
			     sk->sleep,
			     sk->socket,
			     sk->next,
			     sk->prev,
			     sk->err,
			     sk->destruct,
			     sk->reuse,
			     sk->zapped,
			     sk->family,
			     sk->num,
			     sk->protocol,
			     atomic_read(&sk->rmem_alloc),
			     atomic_read(&sk->wmem_alloc),
			     atomic_read(&sk->omem_alloc),
			     sk->sndbuf,
			     (unsigned int)sk->stamp.tv_sec,
			     (unsigned int)sk->stamp.tv_usec,

			     sk->socket->flags,
			     sk->socket->type,
			     sk->socket->state);
#else /* NET_21 */
		len+=sprintf(buffer+len,"%8p %4d %d %8p %8p %8p %8p %d %d %d %d %d %2d %d.%06d %08lX %08X %02X\n",
			     sk,
			     
			     key_pid(sk),
			     sk->dead,
			     sk->sleep,
			     sk->socket,
			     sk->next,
			     sk->prev,
			     sk->err,
			     sk->reuse,
			     sk->zapped,
			     sk->num,
			     sk->protocol,
			     sk->sndbuf,
			     (unsigned int)sk->stamp.tv_sec,
			     (unsigned int)sk->stamp.tv_usec,

			     sk->socket->flags,
			     sk->socket->type,
			     sk->socket->state);
#endif /* NET_21 */
		
		pos=begin+len;
		if(pos<offset)
		{
			len=0;
			begin=pos;
		}
		if(pos>offset+length)
			break;
		sk=sk->next;
	}
	*start=buffer+(offset-begin);
	len-=(offset-begin);
	if(len>length)
		len=length;
	return len;
}

#ifndef PROC_FS_2325
struct proc_dir_entry proc_net_pfkey =
{
	0,
	6, "pf_key",
	S_IFREG | S_IRUGO, 1, 0, 0,
	0, &proc_net_inode_operations,
	pfkey_get_info
};
#endif /* !PROC_FS_2325 */
#endif /* CONFIG_PROC_FS */

/* void */
int
pfkey_init(void)
{
	int error = 0;
	
        printk(KERN_INFO "FreeS/WAN: initialising PF_KEY domain sockets.\n");

#ifdef NET_21
        sock_register(&pfkey_family_ops);
#else /* NET_21 */
        sock_register(pfkey_proto_ops.family, &pfkey_proto_ops);
#endif /* NET_21 */

#ifdef CONFIG_PROC_FS
#  ifndef PROC_FS_2325
#    ifdef PROC_FS_21
	proc_register(proc_net, &proc_net_pfkey);
#    else /* PROC_FS_21 */
	proc_register_dynamic(&proc_net, &proc_net_pfkey);
#    endif /* PROC_FS_21 */
#  else /* !PROC_FS_2325 */
	proc_net_create ("pf_key", 0, pfkey_get_info);
#  endif /* !PROC_FS_2325 */
#endif          /* CONFIG_PROC_FS */

	return error;
}

/* void */
int
pfkey_cleanup(void)
{
	int error = 0;
	
        printk(KERN_INFO "FreeS/WAN: shutting down PF_KEY domain sockets.\n");
#ifdef NET_21
        sock_unregister(PF_KEY);
#else /* NET_21 */
        sock_unregister(pfkey_proto_ops.family);
#endif /* NET_21 */
#ifdef CONFIG_PROC_FS
#  ifndef PROC_FS_2325
	if (proc_net_unregister(proc_net_pfkey.low_ino) != 0)
		printk("klips_debug:pfkey_cleanup: cannot unregister /proc/net/pf_key\n");
#  else /* !PROC_FS_2325 */
	proc_net_remove ("pf_key");
#  endif /* !PROC_FS_2325 */
#endif          /* CONFIG_PROC_FS */

	/* other module unloading cleanup happens here */
	return error;
}

#ifdef MODULE
#if 0
int
init_module(void)
{
	pfkey_init();
	return 0;
}

void
cleanup_module(void)
{
	pfkey_cleanup();
}
#endif /* 0 */
#else /* MODULE */
void
pfkey_proto_init(struct net_proto *pro)
{
	pfkey_init();
}
#endif /* MODULE */

/*
 * $Log: pfkey_v2.c,v $
 * Revision 1.24  2000/01/10 16:38:23  rgb
 * MB fixups for 2.3.x.
 *
 * Revision 1.23  1999/12/09 23:22:16  rgb
 * Added more instrumentation for debugging 2.0 socket
 * selection/reading.
 * Removed erroneous 2.0 wait==NULL check bug in select.
 *
 * Revision 1.22  1999/12/08 20:32:16  rgb
 * Tidied up 2.0.xx support, after major pfkey work, eliminating
 * msg->msg_name twiddling in the process, since it is not defined
 * for PF_KEYv2.
 *
 * Revision 1.21  1999/12/01 22:17:19  rgb
 * Set skb->dev to zero on new skb in case it is a reused skb.
 * Added check for skb_put overflow and freeing to avoid upmsg on error.
 * Added check for wrong pfkey version and freeing to avoid upmsg on
 * error.
 * Shut off content dumping in pfkey_destroy.
 * Added debugging message for size of buffer allocated for upmsg.
 *
 * Revision 1.20  1999/11/27 12:11:00  rgb
 * Minor clean-up, enabling quiet operation of pfkey if desired.
 *
 * Revision 1.19  1999/11/25 19:04:21  rgb
 * Update proc_fs code for pfkey to use dynamic registration.
 *
 * Revision 1.18  1999/11/25 09:07:17  rgb
 * Implemented SENDERR macro for propagating error codes.
 * Fixed error return code bug.
 *
 * Revision 1.17  1999/11/23 23:07:20  rgb
 * Change name of pfkey_msg_parser to pfkey_msg_interp since it no longer
 * parses. (PJO)
 * Sort out pfkey and freeswan headers, putting them in a library path.
 *
 * Revision 1.16  1999/11/20 22:00:22  rgb
 * Moved socketlist type declarations and prototypes for shared use.
 * Renamed reformatted and generically extended for use by other socket
 * lists pfkey_{del,add}_open_socket to pfkey_list_{remove,insert}_socket.
 *
 * Revision 1.15  1999/11/18 04:15:09  rgb
 * Make pfkey_data_ready temporarily available for 2.2.x testing.
 * Clean up pfkey_destroy_socket() debugging statements.
 * Add Peter Onion's code to send messages up to all listening sockets.
 * Changed all occurrences of #include "../../../lib/freeswan.h"
 * to #include <freeswan.h> which works due to -Ilibfreeswan in the
 * klips/net/ipsec/Makefile.
 * Replaced all kernel version macros to shorter, readable form.
 * Added CONFIG_PROC_FS compiler directives in case it is shut off.
 *
 * Revision 1.14  1999/11/17 16:01:00  rgb
 * Make pfkey_data_ready temporarily available for 2.2.x testing.
 * Clean up pfkey_destroy_socket() debugging statements.
 * Add Peter Onion's code to send messages up to all listening sockets.
 * Changed #include "../../../lib/freeswan.h" to #include <freeswan.h>
 * which works due to -Ilibfreeswan in the klips/net/ipsec/Makefile.
 *
 * Revision 1.13  1999/10/27 19:59:51  rgb
 * Removed af_unix comments that are no longer relevant.
 * Added debug prink statements.
 * Added to the /proc output in pfkey_get_info.
 * Made most functions non-static to enable oops tracing.
 * Re-enable skb dequeueing and freeing.
 * Fix skb_alloc() and skb_put() size bug in pfkey_upmsg().
 *
 * Revision 1.12  1999/10/26 17:05:42  rgb
 * Complete re-ordering based on proto_ops structure order.
 * Separated out proto_ops structures for 2.0.x and 2.2.x for clarity.
 * Simplification to use built-in socket ops where possible for 2.2.x.
 * Add shorter macros for compiler directives to visually clean-up.
 * Add lots of sk skb dequeueing debugging statements.
 * Added to the /proc output in pfkey_get_info.
 *
 * Revision 1.11  1999/09/30 02:55:10  rgb
 * Bogus skb detection.
 * Fix incorrect /proc/net/ipsec-eroute printk message.
 *
 * Revision 1.10  1999/09/21 15:22:13  rgb
 * Temporary fix while I figure out the right way to destroy sockets.
 *
 * Revision 1.9  1999/07/08 19:19:44  rgb
 * Fix pointer format warning.
 * Fix missing member error under 2.0.xx kernels.
 *
 * Revision 1.8  1999/06/13 07:24:04  rgb
 * Add more debugging.
 *
 * Revision 1.7  1999/06/10 05:24:17  rgb
 * Clarified compiler directives.
 * Renamed variables to reduce confusion.
 * Used sklist_*_socket() kernel functions to simplify 2.2.x socket support.
 * Added lots of sanity checking.
 *
 * Revision 1.6  1999/06/03 18:59:50  rgb
 * More updates to 2.2.x socket support.  Almost works, oops at end of call.
 *
 * Revision 1.5  1999/05/25 22:44:05  rgb
 * Start fixing 2.2 sockets.
 *
 * Revision 1.4  1999/04/29 15:21:34  rgb
 * Move log to the end of the file.
 * Eliminate min/max redefinition in #include <net/tcp.h>.
 * Correct path for pfkey #includes
 * Standardise an error return method.
 * Add debugging instrumentation.
 * Move message type checking to pfkey_msg_parse().
 * Add check for errno incorrectly set.
 * Add check for valid PID.
 * Add check for reserved illegally set.
 * Add check for message out of bounds.
 *
 * Revision 1.3  1999/04/15 17:58:07  rgb
 * Add RCSID labels.
 *
 * Revision 1.2  1999/04/15 15:37:26  rgb
 * Forward check changes from POST1_00 branch.
 *
 * Revision 1.1.2.2  1999/04/13 20:37:12  rgb
 * Header Title correction.
 *
 * Revision 1.1.2.1  1999/03/26 20:58:55  rgb
 * Add pfkeyv2 support to KLIPS.
 *
 *
 * RFC 2367
 * PF_KEY_v2 Key Management API
 */
