/* whack communicating routines
 * Copyright (C) 1997 Angelos D. Keromytis.
 * Copyright (C) 1998, 1999  D. Hugh Redelmeier.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * RCSID $Id: kernel_comm.c,v 1.43 2000/01/06 22:15:34 dhr Exp $
 */

#include <stdio.h>
#include <stddef.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <freeswan.h>

#include "constants.h"
#include "defs.h"
#include "id.h"
#include "connections.h"	/* needs id.h */
#include "whack.h"	/* needs connections.h */
#include "packet.h"
#include "demux.h"	/* needs packet.h */
#include "state.h"
#include "ipsec_doi.h"	/* needs demux.h and state.h */
#include "kernel.h"
#include "kernel_comm.h"
#include "log.h"
#include "preshared.h"
#include "server.h"

/*
 * XXX Eventually it should use PF_KEY or what kernel API is available
 * XXX but for now this will have to do.
 */

/* helper variables and function to decode strings from whack message */

static char
    *next_str,
    *str_roof;

static bool
unpack_str(char **p)
{
    char *end = memchr(next_str, '\0', str_roof - next_str);

    if (end == NULL)
    {
	return FALSE;	/* fishy: no end found */
    }
    else
    {
	*p = next_str == end? NULL : next_str;
	next_str = end + 1;
	return TRUE;
    }
}

/* Handle a kernel request. Supposedly, there's a message in
 * the kernelsock socket.
 */
void
whack_handle(int whackctlfd)
{
    struct whack_message msg;
    struct sockaddr_un whackaddr;
    int whackaddrlen = sizeof(whackaddr);
    int whackfd = accept(whackctlfd, (struct sockaddr *)&whackaddr, &whackaddrlen);
    ssize_t n;

    if (whackfd < 0)
    {
	log_errno((e, "accept() failed in whack_handle()"));
	return;
    }
    n = read(whackfd, &msg, sizeof(msg));
    if (n == -1)
    {
	log_errno((e, "read() failed in whack_handle()"));
	close(whackfd);
	return;
    }

    whack_log_fd = whackfd;

    /* sanity check message */
    {
	int ughno = RC_BADWHACKMESSAGE;
	char ugh[200];

	next_str = msg.string;
	str_roof = (char *)&msg + n;

	if (next_str > str_roof)
	{
	    snprintf(ugh, sizeof(ugh)
		, "truncated message from whack: got %d bytes; expected %d.  Message ignored."
		, n, (int) sizeof(msg));
	}
	else if (msg.magic != WHACK_MAGIC)
	{
	    snprintf(ugh, sizeof(ugh)
		, "message from whack has bad magic %d; should be %d; probably wrong version.  Message ignored"
		, msg.magic, WHACK_MAGIC);
	}
	else if (!unpack_str(&msg.name)	/* string 1 */
	|| !unpack_str(&msg.left.id)	/* string 2 */
	|| !unpack_str(&msg.left.updown)	/* string 3 */
	|| !unpack_str(&msg.right.id)	/* string 4 */
	|| !unpack_str(&msg.right.updown)	/* string 5 */
	|| !unpack_str(&msg.keyid)	/* string 6 */
	|| str_roof - next_str != (ptrdiff_t)msg.keyval.len)	/* check chunk */
	{
	    snprintf(ugh, sizeof(ugh)
		, "message from whack contains bad string");
	}
	else
	{
	    msg.keyval.ptr = next_str;	/* grab chunk */
	    ughno = 0;	/* ran through the gauntlet -- success */
	}

	if (ughno != 0)
	{
	    whack_log_fd = NULL_FD;	/* early so next log message is suppressed */
	    log("%s", ugh);
	    whack_log(whackfd, ughno, "%s", ugh);
	    close(whackfd);
	    return;
	}
    }

    if (msg.whack_options)
    {
#ifdef DEBUG
	debugging = msg.debugging;
	DBG(DBG_CONTROL,
	    DBG_log("debugging = %s", bitnamesof(debug_bit_names, debugging)));
#endif
    }

    /* Deleting combined with adding a connection works as replace.
     * To make this more useful, in only this combination,
     * delete will silently ignore the lack of the connection.
     */
    if (msg.whack_delete)
    {
	struct connection *c = con_by_name(msg.name
	    , msg.whack_connection? NULL_FD : whackfd);

	/* note: this is a "while" because road warrior
	 * leads to multiple connections with the same name.
	 */
	for (; c != NULL; c = con_by_name(msg.name, NULL_FD))
	    delete_connection(c);
    }

    if (msg.whack_connection)
	add_connection(&msg, whackfd);

    /* process "listen" before any operation that could require it */
    if (msg.whack_listen)
    {
	log("listening for IKE messages");
	listening = TRUE;
	find_ifaces();
	load_preshared_secrets();
    }
    if (msg.whack_unlisten)
    {
	log("no longer listening for IKE messages");
	listening = FALSE;
    }

    if (msg.whack_key)
    {
	/* add a public key */
	struct id keyid;
	const char *ugh = atoid(msg.keyid, &keyid);

	if (ugh != NULL)
	{
	    log("bad --keyid: %s", ugh);
	    whack_log(whackfd, RC_BADID, "bad --keyid: %s", ugh);
	}
	else
	{
	    add_public_key(&keyid, msg.pubkey_alg, &msg.keyval);
	}
    }

    if (msg.whack_route)
    {
	if (!listening)
	    whack_log(whackfd, RC_DEAF, "need --listen before --route");
	else
	{
	    struct connection *c = con_by_name(msg.name, whackfd);

	    cur_connection = c;
	    if (c == NULL)
		;	/* already whack_logged */
	    else if (!orient(c, TRUE))
		whack_log(whackfd, RC_ORIENT, "could not orient connection");
	    else if (!route_connection(c))
		whack_log(whackfd, RC_ROUTE, "could not route");
	    cur_connection = NULL;
	}
    }

    if (msg.whack_unroute)
    {
	struct connection *c = con_by_name(msg.name, whackfd);

	if (c != NULL)
	{
	    cur_connection = c;
	    if (c->eroute_owner != SOS_NOBODY)
		whack_log(whackfd, RC_RTBUSY, "cannot unroute: route busy");
	    else
		unroute_connection(c);
	    cur_connection = NULL;
	}
    }

    if (msg.whack_initiate)
    {
	if (!listening)
	    whack_log(whackfd, RC_DEAF, "need --listen before --initiate");
	else
	    initiate_connection(msg.name, msg.whack_async? NULL_FD : whackfd);
    }

    if (msg.whack_terminate)
	terminate_connection(msg.name, whackfd);

    if (msg.whack_status)
    {
	show_connections_status(whackfd);
	whack_log(whackfd, RC_COMMENT, BLANK_FORMAT);	/* spacer */
	show_states_status(whackfd);
    }

    if (msg.whack_shutdown)
    {
	log("shutting down");
	exit_pluto(0);	/* delete lock and leave, with 0 status */
    }

    whack_log_fd = NULL_FD;
    if (whackfd != NULL_FD)
	close(whackfd);
}
