#!/bin/sh

# James Galvin <galvin@tis.com>					4/94
# 
# This script expects to be invoked by a MIME user agent with arguments
# corresponding to a message digitally signed with PEM.  A common
# configuration is for a MIME user agent to collect the parts of a
# multipart/signed content and invoke this script with those parts.
# 
# When using the Rand MH Version 6.8.3, the following addition to the user's
# .mh_profile file is recommended:
#
# 	mhn-show-application/signature: pemverify.sh %s %f %a -dquery

# Exactly 4 arguments are required, in the order indicated, as follows:
# 
# 	type	-- the MIME content type of the data to be verified;
#		   this is required because multipart, message, and text
#		   content types must be canonicalized
# 	subtype	-- the MIME sub-content type; this information is
#		   currently unused
#	headers -- the file with the PEM header information
#	data	-- the file with the signed data
# 
# These 4 arguments may be preceded, followed, or separated by the
# following optional arguments:
#
# 	-verify	-- always verify the signature on the data; default
#	-vquery	-- query user as to whether to verify signature or just
#		   display the data; make verification default when the
#		   response is RETURN
#	-dquery	-- query user as to whether to verify signature or just
#		   display the data; make display default when the
#		   response is RETURN
# 
# If the stdin input is not attached to a terminal, the default action
# is executed.  If there are multiple occurances of any optional
# argument the last occurance will override all other occurances.
# 
# If signature verification is not selected the data is output to the
# standard output exactly as it is input.
# 
# Upon successful completion of the signature verification, the data
# will be output to the standard output (file descriptor 1) and the
# annotations (e.g., who signed the message) will be output to the
# standard error (file descriptor 2).
# 
# If the signature verification is unsuccessful, an appropriate error
# message will be output to the standard error and a non-zero exit code
# will be returned.  Nothing will be output to the standard output.


umask 077


# a helpful message is printed and then we exit; exit code argument required

usage() {
	echo Usage: $0: type subtype headers data 1>&2
	exit $1
}


# filchk is a convenience for doargs; filename argument is required

filchk() {
	if [ ! -f $1 ]; then
		echo $0: $1: not a file 1>&2
		usage 3
	fi
	if [ ! -r $1 ]; then
		echo $0: $1: unable to open for reading 1>&2
		usage 4
	fi
}

# parses the 4 required arguments

doargs() {
	if [ $# -ne 4 ]; then
		usage 2
	fi

	TYPE=$1
	STYP=$2
	HDRS=$3
	DATA=$4

	filchk $HDRS
	filchk $DATA
}


# performs the signature verification, including canonicalization

dov() {
	cat $DATA | canon | verify header-in $HDRS
	status=$?
}


# reads a user's response and processes accordingly; a single argument
# specifying the default action is required

doans() {
	if [ -t 0 ]; then
		echo Verify signature or just Display \[$1\]\? > /dev/tty
		read ans
	fi

	if [ -z "$ans" ]; then
		ans=$1
	fi

	case $ans in
	V|v)	dov ;;
	D|d)	echo Signature not verified at this time. 1>&2
		status=0 ;;
	esac
}


ARGS=
DOV=0

if [ $# -lt 4 ]; then
	usage 2
fi

while [ $# -ge 1 ]; do
	case $1 in
	protocol=pem)
		;;

	micalg=md5)
		;;

	-vquery)
		DOV=1 ;;
	-dquery)
		DOV=-1 ;;
	-verify)
		DOV=0 ;;

	protocol=*)
		echo $0: unknown option \"$1\" 1>&2
		usage 1 ;;
	micalg=*)
		echo $0: unknown option \"$1\" 1>&2
		usage 1 ;;
	-*)
		echo $0: unknown option \"$1\" 1>&2
		usage 1 ;;

	*)	ARGS="$ARGS $1" ;;
	esac
	shift
done

doargs $ARGS

case $DOV in
	1)	doans "V"
		;;

	0)	dov
		;;

	-1)	doans "D"
		;;
esac

if [ $status -ne 0 ]; then
	exit $status
fi

cat $DATA

exit 0
