/*********************************************************************
**
**     File name:               ssh_smsg.c
**
**                              Copyright 1997 Tadayoshi Kohno.
**				All rights reserved.
**                              See the LICENSE file.
**
**     Purpose:                 encode and decoode server messages
**
**     Author/Date:             Tadayoshi Kohno, 22 November 1997
**
**     References:              draft-ylonen-ssh-protocol-00.txt
**
**     Notes:
**	Handle coding of server messages SSH_SMSG_*.  Additional
**	comments can be found in ssh_cmsg.c.
**
**	Timeout send()s, recv()s.
**
**     Functions:
**	ssh_smsg_public_key_decode	decode SSH_SMSG_PUBLIC_KEY
**	ssh_smsg_merge_data_decode	decode SSH_SMSG_{STDIN,STDOUT}_DATA
**	ssh_smsg_stdin_data_decode	decode SSH_SMSG_STDIN_DATA
**	ssh_smsg_stdout_data_decode		decode SSH_SMSG_STDOUT_DATA
**	ssh_smsg_auth_rsa_challenge_decode	decode SSH_SMSG_AUTH_RSA_CHALLENGE
**	ssh_smsg_exitstatus_decode	decode SSH_SMSG_EXITSTATUS
**
*********************************************************************/

#ifndef lint
static char *RCSid="$Header: /home/cia/kohno/libssh/libssh/RCS/ssh_smsg.c,v 3.3 1998/03/14 18:17:43 kohno Exp $";
#endif

#include "ssh.h"
#include "ssh_msg.h"
#include "ssh_smsg.h"

#include "ssh_mp_stream.h"

#include "ssh_util.h"


/*********************************************************************
**
**     Function:                ssh_smsg_public_key_decode
**
**     Purpose:                 decode SSH_SMSG_PUBLIC_KEY
**
**     Entry (pre) conditions:  data is valid (not a bogus packet).
**
**     Parameters:              data		data to decode
**				data_len	length of data
**
**				< Based off SSH_SMSG_PUBLIC_KEY specs>
**				cookie
**				server_key_bits
**				server_key_public_exponent
**				server_key_public_modulus
**				host_key_bits
**				host_key_public_exponent
**				host_key_public_modulus
**				protocol_flags
**				supported_cipher_mask
**				supported_authentications_mask
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		error
**
**				ssh_errno set for SSH_ERRNO_LOW_MEM
**				and SSH_ERRNO_MSG_PARSE.
**
**     Side effects:            cookie, server_key_bits, ... set
**
**     Author/Date:             Tadayoshi Kohno, 22 November 199
**
**     Notes:
**	I really need to make sure that if I have a bogus packet with some
**	weird length or something, that I don't read off the end of the
**	packet.  Thus, I need to check to make sure I'm reading less than
**	data_len (xxx).
**
**	I'm not sure why we need the server_key_bits and host_key_bits.
**	These two values seem to be encoded in the *_key_public_modulus
**	MP_Ints (see draft-ylonen-ssh-protocol-00.txt).
**
*********************************************************************/
int ssh_smsg_public_key_decode
(
	uint8_t * data,		/* data to decode */
	uint32_t data_len,	/* length of data */

	uint8_t * cookie,	/* random cookie */
	uint32_t * server_key_bits,		/* bits for server key */
	MP_Int * server_key_public_exponent,	/* server key exponent */
	MP_Int * server_key_public_modulus,	/* server key modulus */
	uint32_t * host_key_bits,		/* bits for host key */
	MP_Int * host_key_public_exponent,	/* host key exponent */
	MP_Int * host_key_public_modulus,	/* host key modulus */
	uint32_t * protocol_flags,		/* protocol flags */
	uint32_t * supported_cipher_mask,		/* cipher mask */
	uint32_t * supported_authentications_mask	/* auth mask */
)
{
	uint32_t run = 0;	/* running count of offset */

	/*
	**	Lets make sure we at least have a minimal size
	*/
	if (data_len < SSH_SMSG_PUBLIC_KEY_MIN)
	{
		ssh_errno_set(SSH_ERRNO_LOW_MEM);
		return(S_BAD);
	}

	/*
	** First grab the cookie
	*/
	my_bcopy((void *) data, (void *) cookie, SSH_COOKIE_SIZE);
	run += SSH_COOKIE_SIZE;

	/*
	**	Now get server key bits
	*/
	my_bcopy((void *) (data + run), (void *) server_key_bits,
		SSH_SERVER_KEY_SIZE);
	*server_key_bits = ntohl(*server_key_bits);
	run += SSH_SERVER_KEY_SIZE;

	/*
	**	And now lets grab server_key_public_*
	*/
	run += mp_int_from_stream(data + run, server_key_public_exponent);
	run += mp_int_from_stream(data + run, server_key_public_modulus);

	/*
	**	Grab the host_key_bits
	*/
	my_bcopy((void *) (data + run), (void *) host_key_bits,
		SSH_HOST_KEY_SIZE);
	*host_key_bits = ntohl(*host_key_bits);
	run += SSH_HOST_KEY_SIZE;

	/*
	**	Grab host_key_public_*
	*/
	run += mp_int_from_stream(data + run, host_key_public_exponent);
	run += mp_int_from_stream(data + run, host_key_public_modulus);

	/*
	**	Get Protocol Flags
	*/
	my_bcopy((void *) (data + run), (void *) protocol_flags,
		SSH_PROTO_SIZE);
	*protocol_flags = ntohl(*protocol_flags);
	run += SSH_PROTO_SIZE;

	/*
	**	Get cipher masks
	*/
	my_bcopy((void *) (data + run), (void *) supported_cipher_mask,
		SSH_CIPHER_SIZE);
	*supported_cipher_mask = ntohl(*supported_cipher_mask);
	run += SSH_CIPHER_SIZE;

	/*
	**	Get authentications mask
	*/
	my_bcopy((void *) (data + run),
		(void *) supported_authentications_mask, SSH_AUTH_SIZE);
	*supported_authentications_mask
		= ntohl(*supported_authentications_mask);
	run += SSH_AUTH_SIZE;

	/*
	**	Lets try to do a simple "double-check" to make sure packet
	**	parsed okay (athough CRC should garontee this)
	*/
	if (run != data_len)
	{
		ssh_errno_set(SSH_ERRNO_MSG_PARSE);
		return(S_BAD);
	}

	return(S_GOOD);
}

/*********************************************************************
**
**     Functions:               ssh_smsg_merge_data_decode
**				ssh_smsg_stdout_data_decode
**				ssh_smsg_stderr_data_decode
**
**     Purpose:                 decode SSH_SMSG_STDOUT_DATA and
**				SSH_SMSG_STDERR_DATA messages
**
**     Entry (pre) conditions:  pointers valid, data a valid data
**				field in a received packet of
**				types SSH_SMSG_{STDOUT,STDERR}_DATA
**
**     Parameters:              data		data in packet
**				data_len	length of data
**
**				*str		output string for data
**
**				str_size	size of *str
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		error in string_from_stream
**				ssh_errno set in ""
**
**     Side effects:            *str contains string of length
**				data_len - sizeof(uint32_t)
**
**     Author/Date:             Tadayoshi Kohno, 27 November 1997
**
**     Notes:
**	Use this to decode stdout/stderr messages from server.
**
**	Should I return the length of the string read?
**
**	Some people would #define ssh_smsg_stdout_data_decode ssh_smsg_merge...
**	Would that be "better"?
**
*********************************************************************/

int ssh_smsg_merge_data_decode
(
	uint8_t * data,		/* data to decode */
	uint32_t data_len,	/* length of data */

	uint8_t * str,		/* output string */

	int str_size		/* size of *str */
)
{
	if (string_from_stream(data, str, str_size)
		!= (int) (data_len - sizeof(uint32_t)))
	{
		ssh_errno_set(SSH_ERRNO_MSG_PARSE);
		return(S_BAD);
	}
	return(S_GOOD);
}

int ssh_smsg_stdout_data_decode
(
	uint8_t * data,		/* data to decode */
	uint32_t data_len,	/* length of data */

	uint8_t * str,		/* output string */

	int str_size		/* size of *str */
)
{
	return(ssh_smsg_merge_data_decode(data, data_len, str, str_size));
}

int ssh_smsg_stderr_data_decode
(
	uint8_t * data,		/* data to decode */
	uint32_t data_len,	/* length of data */

	uint8_t * str,		/* output string */

	int str_size		/* size of *str */
)
{
	return(ssh_smsg_merge_data_decode(data, data_len, str, str_size));
}

/*********************************************************************
**
**     Function:                ssh_smsg_auth_rsa_challenge_decode
**
**     Purpose:                 decode an SSH_SMSG_AUTH_RSA_CHALLENGE
**
**     Entry (pre) conditions:  pointers valid
**
**     Parameters:		data		data to decode
**				data_len	length of data to decode
**
**				*challenge	server's challenge
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		error extrapolating MP_Int
**
**     Side effects:            *challenge contains server's encrypted
**				challenge
**
**     Author/Date:             Tadayoshi Kohno, 9 December 1997
**
**     Notes:
**	I really should do bounds checking, but because we have a garonteed
**	crc, I don't need to do that much (provided the server wrapped things
**	correctly) (xxx).
**
*********************************************************************/

int ssh_smsg_auth_rsa_challenge_decode
(
	uint8_t * data,		/* data to decode */
	uint32_t data_len,	/* length of data */

	MP_Int * challenge	/* encrypted challenge */
)
{
	if (mp_int_from_stream(data, challenge) != (int) data_len)
	{
		ssh_errno_set(SSH_ERRNO_MSG_PARSE);
		return(S_BAD);
	}
	return(S_GOOD);
}

/******************************************************************************
**
**     File/Function name:	ssh_smsg_exitstatus_decode
**
**     Purpose:			decode an SSH_SMSG_EXITSTATUS
**
**     Preconditions:		pointers valid
**
**     Parameters:		data		data to decode
**				data_len	length of data to decode
**
**				*exit_status	exit status from server
**
**     Exit (post) conditions:	S_GOOD
**
**     Error conditions:	S_BAD
**
**     Side effects:		*exit_status contains exit status sent from
**				server
**
**     Author/Date:		Tadayoshi Kohno, 7 March 1998
**
**     Notes:
**
******************************************************************************/

int ssh_smsg_exitstatus_decode
(
	uint8_t * data,		/* data to decode */
	uint32_t data_len,	/* length of data to decode */

	uint32_t * exit_status	/* exit status  */
)
{
	if (data_len != sizeof(uint32_t))
	{
		ssh_errno_set(SSH_ERRNO_MSG_PARSE);
		return(S_BAD);
	}

	my_bcopy((void *) (data), (void *) exit_status,
		sizeof(uint32_t));
	*exit_status = ntohl(*exit_status);

	return(S_GOOD);
}

