/*********************************************************************
**
**     File name:               ssh_ident.c
**                                  
**                              Copyright 1997, 1998 Tadayoshi Kohno.
**				All rights reserved.
**                              See the LICENSE file.
**
**     Purpose:                 saves/loads ssh identities
**
**     Author/Date:             Tadayoshi Kohno, 14 December 1997
**
**     Notes:
**	The functions in this file can be used to load and generate
**	RSA identities.
**
**	If SSH_USE_KOHNO_IDENTITY is defined, then a homebrew file
**	format is defined.  This is out-dated.  Without defining
**	SSH_USE_KOHNO_IDENTITY, Tatu Ylonen's file format 1.1 is used.
**
**     Functions:
**	ssh_identity_gen		generates an RSA identity
**	ssh_identity_load		loads an RSA identity
**
**	static ssh_identity_save	saves an RSA identity
**
*********************************************************************/

#ifndef lint
static char *RCSid="$Header: /home/kohno/LibSSH/libssh.0.0.1beta/libssh/RCS/ssh_ident.c,v 3.15 1998/05/22 16:21:31 kohno Exp $";
#endif

#include <string.h>
#include <assert.h>

#include "ssh.h"
#include "ssh_debug.h"
#include "ssh_crypt.h"
#include "ssh_ident.h"
#include "ssh_mp_int.h"
#include "ssh_mp_int_dec.h"
#include "ssh_random.h"

#include "ssh_util.h"

#include <rsa.h>

/*
**	des3 encryption/decryption values
*/
static struct des3_values_identity
{
	des_cblock iv_encrypt_1;	/* initialization vertor (enc) */
	des_cblock iv_encrypt_2;	/* 2nd enc initialization vector */
	des_cblock iv_encrypt_3;	/* 3d enc initialization vector */
	des_cblock iv_decrypt_1;	/* init vertor (decrypt) */
	des_cblock iv_decrypt_2;	/* 2nd decrypt init vector */
	des_cblock iv_decrypt_3;	/* 3d decrypt init vector */
	des_key_schedule key_1;		/* First DES key */
	des_key_schedule key_2;		/* Second DES key */
	des_key_schedule key_3;		/* Third DES key */
} DES3ValuesIdentity;

/*
**	save our ssh identity
*/
static int ssh_identity_save(struct ssh_struct * ssh_info,
	const char * priv_file, const char * pub_file, const char * msg,
	const char * passphrase, uint8_t cipher_type,  MP_Int modulus,
	MP_Int private_exp, MP_Int public_exp, MP_Int prime_p, MP_Int prime_q,
	MP_Int p_mod_q);


/*********************************************************************
**
**     Function:                ssh_identity_gen
**
**     Purpose:                 generate and save an RSA identity
**
**     Entry (pre) conditions:  priv_file, pub_file are valid files
**
**     Parameters:              ssh_info	information about ssh
**				
**				priv_file	private  filename
**				pub_file	public filename
**				msg		public trailing message
**				passphrase	passphrase to use
**
**				bits		bits for modulus
**
**				cipher_type	type of cipher to use
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		error on generation
**						error on saving
**				ssh_errno set
**
**     Side effects:            RSA identity generated and saved in
**				specified file
**
**     Author/Date:             Tadayoshi Kohno, 14 December 1997
**
**     References:                  
**	This routine also makes use of Eric Young's libeay library.
**
**     Notes:
**	I want to double check that my conversion from Eric Young's BN
**	to my MP_Ints are correct.  I'm pretty sure it is. xxx
**
**	In the ssh_identity case, the ssh_info variable is used
**	to determine if and how to display debug information
**
**	not sure if p_mod_q is generated correctly. xxx
**
*********************************************************************/

int ssh_identity_gen
(
	struct ssh_struct * ssh_info, 	/* information about ssh */
	const char * priv_file,	/* location to store RSA key (private) */
	const char * pub_file,	/* location to store RSA key (public) */
	const char * msg,	/* message to tail public file */
	const char * passphrase,/* passphrase to decrypt RSA key */
	int bits,		/* number of bits to use */
	uint8_t cipher_type	/* type of cipher to use */
)
{	
	RSA * private_key;	/* our generated private key */

	MP_Int modulus;		/* our modulus */
	MP_Int public_exp;	/* our public exponent */
	MP_Int private_exp;	/* our private exponent */

	MP_Int prime_p;		/* p, prime number */
	MP_Int prime_q;		/* q, prime number */
	MP_Int p_mod_q;		/* p mod q */

	BN_CTX * bn_context;	/* big number context for p mod q */
	BIGNUM * bn_p_mod_q;	/* bignum of p mod q */

	/*
	**	Initialize our MP_Ints
	*/
	mp_int_new(&modulus);
	mp_int_new(&public_exp);
	mp_int_new(&private_exp);

	mp_int_new(&prime_p);
	mp_int_new(&prime_q);
	mp_int_new(&p_mod_q);

	if ((bn_context = BN_CTX_new()) == (BN_CTX *) NULL)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"Error mallocing memory for bn_context",
			"ssh_identity_gen");
		ssh_errno_set(SSH_ERRNO_MALLOC);
		return(S_BAD);
	}

	bn_p_mod_q = (BIGNUM *) NULL;

	/*
	**	Generate our private key
	*/
	ssh_debugger_new(&(ssh_info->debug_info),
		"calling RSA_generate_key", "ssh_identity_gen");
	private_key = RSA_generate_key(bits, SSH_IDENT_EXP, NULL);

	/*
	**	And convert those private keys to MP_Ints (allocate memory)
	*/
	modulus.data_bytes = BN_num_bytes(private_key->n);
	if ((modulus.num
		= (uint8_t *) malloc(sizeof(uint8_t)*(modulus.data_bytes + 1)))
		== NULL)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"Error mallocing memory for modulus",
			"ssh_identity_gen");
		ssh_errno_set(SSH_ERRNO_MALLOC);
		return(S_BAD);
	}

	public_exp.data_bytes = BN_num_bytes(private_key->e);
	if ((public_exp.num
		= (uint8_t *) malloc(sizeof(uint8_t)
			* (public_exp.data_bytes + 1)))
		== NULL)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"Error mallocing memory for public exp",
			"ssh_identity_gen");

		ssh_errno_set(SSH_ERRNO_MALLOC);
		return(S_BAD);
	}

	private_exp.data_bytes = BN_num_bytes(private_key->d);
	if ((private_exp.num
		= (uint8_t *) malloc(sizeof(uint8_t)
			* (private_exp.data_bytes + 1)))
		== NULL)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"Error mallocing memory for private exp",
			"ssh_identity_gen");

		ssh_errno_set(SSH_ERRNO_MALLOC);
		return(S_BAD);
	}

	prime_p.data_bytes = BN_num_bytes(private_key->p);
	if ((prime_p.num
		= (uint8_t *) malloc(sizeof(uint8_t)
			* (prime_p.data_bytes + 1)))
		== NULL)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"Error mallocing memory for prime p",
			"ssh_identity_gen");

		ssh_errno_set(SSH_ERRNO_MALLOC);
		return(S_BAD);
	}

	prime_q.data_bytes = BN_num_bytes(private_key->q);
	if ((prime_q.num
		= (uint8_t *) malloc(sizeof(uint8_t)
			* (prime_q.data_bytes + 1)))
		== NULL)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"Error mallocing memory for prime q",
			"ssh_identity_gen");

		ssh_errno_set(SSH_ERRNO_MALLOC);
		return(S_BAD);
	}

	/*
	**	now take the multiplicative inverse of p mod q
	*/
	if ((bn_p_mod_q = BN_mod_inverse(private_key->p, private_key->q,
		bn_context)) == (BIGNUM *) NULL)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"Error finding mult. inverse of p mod q",
			"ssh_identity_gen");

		ssh_errno_set(SSH_ERRNO_MALLOC);
		return(S_BAD);
	}

	p_mod_q.data_bytes = BN_num_bytes(bn_p_mod_q);
	if ((p_mod_q.num
		= (uint8_t *) malloc(sizeof(uint8_t)
			* (p_mod_q.data_bytes + 1)))
		== NULL)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"Error mallocing memory for mult. inverse of p mod q",
			"ssh_identity_gen");

		ssh_errno_set(SSH_ERRNO_MALLOC);
		return(S_BAD);
	}


	/*
	**	and convert the actual numbers to bytes for MP_Ints
	*/
	assert(modulus.data_bytes == BN_bn2bin(private_key->n, modulus.num));
	modulus.nbits = modulus.data_bytes * BITS_PER_BYTE;

	assert(public_exp.data_bytes
		== BN_bn2bin(private_key->e, public_exp.num));
	public_exp.nbits = public_exp.data_bytes * BITS_PER_BYTE;

	assert(private_exp.data_bytes 
		== BN_bn2bin(private_key->d, private_exp.num));
	private_exp.nbits = private_exp.data_bytes * BITS_PER_BYTE;

	assert(prime_p.data_bytes 
		== BN_bn2bin(private_key->p, prime_p.num));
	prime_p.nbits = prime_p.data_bytes * BITS_PER_BYTE;

	assert(prime_q.data_bytes 
		== BN_bn2bin(private_key->q, prime_q.num));
	prime_q.nbits = prime_q.data_bytes * BITS_PER_BYTE;

	assert(p_mod_q.data_bytes 
		== BN_bn2bin(bn_p_mod_q, p_mod_q.num));
	p_mod_q.nbits = p_mod_q.data_bytes * BITS_PER_BYTE;

	/*
	**	Now save our identity
	*/
	if (ssh_identity_save(ssh_info, priv_file, pub_file, msg,
		passphrase, cipher_type, modulus, private_exp, public_exp,
		prime_p, prime_q, p_mod_q) == S_BAD)
	{

		free(private_exp.num);
		free(public_exp.num);
		free(modulus.num);

	/* commented out 9 May 1998 -- application needs to clear this

		my_bzero(passphrase, strlen(passphrase));
	*/

		return(S_BAD);
	}

	/*
	**	Cleanup
	*/
	/* commented out 9 May 1998 -- applicaton needs to clear this
		my_bzero(passphrase, strlen(passphrase));
	*/
	my_bzero(private_exp.num, private_exp.data_bytes);
	my_bzero(public_exp.num, public_exp.data_bytes);
	my_bzero(modulus.num, modulus.data_bytes);

	free(private_exp.num);
	free(public_exp.num);
	free(modulus.num);

	return(S_GOOD);
}




#ifdef SSH_USE_KOHNO_IDENTITY

/*
**	Use Tadayoshi Kohno's identity file format
*/

/*********************************************************************
**
**     Function:                ssh_identity_load		(kohno)
**
**     Purpose:                 read private/public keys from kohno-format
**				identity file
**
**     Entry (pre) conditions:  identity previously stored
**				MP_Ints initialized
**
**				file at filename is a valid private
**				identity
**
**     Parameters:              ssh_info		info about ssh
**
**				filename		name of file to load
**				passphrase		decryption passphrase
**
**				*public_key_modulus	public modulus
**				*public_key_exponent	public exponent
**				*private_key_modulus	private modulus
**				*private_key_exponent	private exponent
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD			(ssh_errno set)
**
**     Side effects:            users' RSA key returned through paramaters
**
**     Author/Date:             Tadayoshi Kohno, 14 December 1997
**				Tadayoshi Kohno, 23 February 1998
**
**     Notes:
**	In the ssh_identity case, ssh_info is used to determine how
**	to display debug information
**
**	February, 1998.  Modified to support only new-format
**	version (agreed with the University of Colorado's JavaSSH
**	version)
**
*********************************************************************/

int ssh_identity_load			/* KOHNO format */
(
	struct ssh_struct * ssh_info,	/* information about ssh config */
	const char * filename,		/* file to read from */
	const char * passphrase,	/* user's passphrase */
	MP_Int * public_key_modulus,	/* public modulus */
	MP_Int * public_key_exponent,	/* public exponent */
	MP_Int * private_key_modulus,	/* private modulus */
	MP_Int * private_key_exponent	/* private exponent */
)
{
	FILE * private_file;			/* our private identity */

	uint8_t tmp_buf[SSH_MAX_PACKET];	/* temporary buffer */

	uint8_t tmp_buf1[SSH_MAX_PACKET];	/* more temporary buffers */
	uint8_t tmp_buf2[SSH_MAX_PACKET];	/* more temporary buffers */

	char head[SSH_IDENTITY_HEAD_LEN];	/* length of identity head */

	int bytes_read;				/* bytes read from file */
	
	MD5_CTX md5_context;			/* MD5 context for decipher */
	uint8_t md5_digest[MD5_DIGEST_LEN];	/* MD5 digest for decryption */
	int ret_val;				/* decryption return value */

	int offset;				/* offset into tmp_buf */
	int count;				/* number of bytes written */

	if ((private_file = fopen(filename, "rb")) == (FILE *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_LOAD);
		return(S_BAD);
	}
	
	bytes_read = fread((void *) tmp_buf, sizeof(uint8_t), SSH_MAX_PACKET,
		private_file);
	if (ferror(private_file))
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_LOAD);
		return(S_BAD);
	}
	
	if (feof(private_file))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"have EOF on identity file", "ssh_identity_load");
	}

	(void) fclose(private_file);

	if (bytes_read <= 0)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_LOAD);
		return(S_BAD);
	}

	/*
	**	notice if we really read a multiple of SSH_UNITS (8)
	**	bytes
	*/
	if (bytes_read % SSH_UNITS)
	{
		ssh_debug_int_new(&(ssh_info->debug_info),
			"didn\'t read multiple of %d bytes",
			SSH_UNITS, "ssh_identity_load (kohno-format)");
	} else
	{
		ssh_debug_int_new(&(ssh_info->debug_info),
			"good, read a multiple of %d bytes",
			SSH_UNITS, "ssh_identity_load (kohno-format)");
	}

	ssh_debugger_new(&(ssh_info->debug_info),
		"identity file bytes read",
		"ssh_identity_load (kohno-format)");

	/* BEGIN */

	/*
	**	Let's create the key for our deciphering
	*/
	MD5_Init(&md5_context);
	MD5_Update(&md5_context, (uint8_t *) passphrase, strlen(passphrase));
	MD5_Final(md5_digest, &md5_context);
	
	/*
	**	Now let's seed our 3des cipher
	*/
	if ((ret_val = des_set_key((C_Block *) md5_digest,
		DES3ValuesIdentity.key_1)) != 0)
	{
		ssh_errno_set(SSH_ERRNO_INIT_CIPHER);
		return(S_BAD);
	}
	if ((ret_val = des_set_key((C_Block *) (md5_digest + DES_KEY_LEN),
		DES3ValuesIdentity.key_2)) != 0)
	{
		ssh_errno_set(SSH_ERRNO_INIT_CIPHER);
		return(S_BAD);
	}
	if ((ret_val = des_set_key((C_Block *) md5_digest,
		DES3ValuesIdentity.key_3)) != 0)
	{
		ssh_errno_set(SSH_ERRNO_INIT_CIPHER);
		return(S_BAD);
	}

	my_bzero(DES3ValuesIdentity.iv_encrypt_1, DES_IV_LEN);
	my_bzero(DES3ValuesIdentity.iv_encrypt_2, DES_IV_LEN);
	my_bzero(DES3ValuesIdentity.iv_encrypt_3, DES_IV_LEN);
	my_bzero(DES3ValuesIdentity.iv_decrypt_1, DES_IV_LEN);
	my_bzero(DES3ValuesIdentity.iv_decrypt_2, DES_IV_LEN);
	my_bzero(DES3ValuesIdentity.iv_decrypt_3, DES_IV_LEN);

	/*
	**	And decrypt the identity
	*/
	des_ncbc_encrypt((C_Block *)(tmp_buf),
		(C_Block *)(tmp_buf1), (long) bytes_read,
		DES3ValuesIdentity.key_1,
		(C_Block *)DES3ValuesIdentity.iv_decrypt_1, DES_DECRYPT);		
	des_ncbc_encrypt((C_Block *)(tmp_buf1),
		(C_Block *)(tmp_buf2), (long) bytes_read,
		DES3ValuesIdentity.key_2,
		(C_Block *)DES3ValuesIdentity.iv_decrypt_2, DES_ENCRYPT);		
	des_ncbc_encrypt((C_Block *)(tmp_buf2),
		(C_Block *)(tmp_buf), (long) bytes_read,
		DES3ValuesIdentity.key_3,
		(C_Block *)DES3ValuesIdentity.iv_decrypt_3, DES_DECRYPT);

	/*
	**	Now lets read our data from the stream
	*/
	offset = 0;

#ifdef OLD_IDENTITY_FORMAT
	my_bcopy(tmp_buf + offset, head, strlen(SSH_IDENTITY_HEAD));
	offset += strlen(SSH_IDENTITY_HEAD);

	head[strlen(SSH_IDENTITY_HEAD)] = 0;

	/*
	**	First make sure we have a valid identity file (should *really*
	**	check for buffer stuff [am I reading too much!] xxx)
	*/
	if (strcmp(head, SSH_IDENTITY_HEAD))
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_DECRYPT);
		return(S_BAD);
	}
#else	/* OLD_IDENTITY_FORMAT */

	my_bcopy(tmp_buf + offset, head, SSH_IDENTITY_CHECKHEAD_LEN);
	offset += SSH_IDENTITY_CHECKHEAD_LEN;

	/*
	**	check our bytes to make sure they really work out
	*/
	if (head[0] != head[2] || head[1] != head[3])
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_DECRYPT);
		return(S_BAD);
	}

#endif	/* OLD_IDENTITY_FORMAT */

	ssh_debugger_new(&(ssh_info->debug_info),
		"correctly deciphered identity file",
		"ssh_identity_load (kohno-format)");

	/*
	**	Now let's read the data
	*/
	count = mp_int_from_stream(tmp_buf + offset, public_key_modulus);
	count = mp_int_from_stream(tmp_buf + offset, private_key_modulus);
	if (count <= 0)
	{
		/* should also check not overflowing auth_offset */
		ssh_debugger_new(&(ssh_info->debug_info),
			"error reading modulus",
			"ssh_identity_load (kohno-format)");

		ssh_errno_set(SSH_ERRNO_IDENTITY_LOAD);
		return(S_BAD);
	}
	offset = offset + count;

	count = mp_int_from_stream(tmp_buf + offset, private_key_exponent);
	if (count <= 0)
	{
		/* should also check not overflowing auth_offset */
		ssh_debugger_new(&(ssh_info->debug_info),
			"error reading private exp",
			"ssh_identity_load (kohno-format)");

		ssh_errno_set(SSH_ERRNO_IDENTITY_LOAD);
		return(S_BAD);
	}
	offset = offset + count;
	
	count = mp_int_from_stream(tmp_buf + offset, public_key_exponent);
	if (count <= 0)
	{
		/* should also check not overflowing auth_offset */
		ssh_debugger_new(&(ssh_info->debug_info),
			"error reading public exp",
			"ssh_identity_load (kohno-format)");

		ssh_errno_set(SSH_ERRNO_IDENTITY_LOAD);
		return(S_BAD);
	}
	offset = offset + count;

	ssh_debugger_new(&(ssh_info->debug_info),
		"identity read", "ssh_identity_load (kohno-format)");

	return(S_GOOD);
}

/*********************************************************************
**
**     Function:                ssh_identity_save		(kohno)
**
**     Purpose:                 save an RSA identity in kohno-format
**
**     Entry (pre) conditions:  n, d, e valid, filenames valid
**
**     Parameters:              ssh_info	info about ssh config
**
**				priv_file	private file
**				pub_file	public file
**				msg		message to end pub_file
**				passphrase	passphrase to encrypt by
**
**				cipher_type	type of cipher to use (not used)
**
**				modulus		RSA modulus		(n)
**				private_exp	private exponent	(d)
**				public_exp	public exponent		(e)
**
**				prime_p		prime number		(p)
**				prime_q		prime number		(q)
**				p_mod_q		p mod q			(u)
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		(ssh_errno set)
**
**     Side effects:            RSA identity encrypted and stored in
**				the specified file.
**
**     Author/Date:             Tadayoshi Kohno, 14 December 1997
**				Tadayoshi Kohno, 23 February 1998
**					(see "Notes")
**                              Tadayoshi Kohno, 14 March 1998
**					(now really uses random bytes)
**
**     Notes:
**
**	February, 1998.  Modified to support only the new-format
**	version (agreed with the University of Colorado's JavaSSH
**	version)
**
*********************************************************************/

static int ssh_identity_save				(kohno format)
(
	struct ssh_struct * ssh_info,	/* information about ssh config */

	const char * priv_file,		/* private filename to save to */
	const char * pub_file,		/* public filename to save to */

	const char * msg,		/* message to end public file */
	const char * passphrase,	/* passphrase for the keys */

	uint8_t cipher_type,		/* type of cipher to use (not used) */

	MP_Int modulus,			/* modulus */

	MP_Int private_exp,		/* d, private exp */
	MP_Int public_exp,		/* e, public exp */

	MP_Int prime_p,			/* p, prime number */
	MP_Int prime_q,			/* q, prime number */
	MP_Int p_mod_q			/* p mod q */
)
{
	FILE * private_file;		/* private identity file */
	FILE * public_file;		/* public identity file */

	uint8_t tmp_buf[SSH_MAX_PACKET];	/* temporary buffer */

	uint8_t tmp_buf1[SSH_MAX_PACKET];	/* more temporary buffers */
	uint8_t tmp_buf2[SSH_MAX_PACKET];	/* more temporary buffers */
	
	MD5_CTX md5_context;			/* MD5 context for decipher */
	uint8_t md5_digest[MD5_DIGEST_LEN];	/* MD5 digest for decryption */
	int ret_val;				/* decryption return value */

	int offset;				/* offset into tmp_buf */
	int count;				/* number of bytes written */

	uint8_t rand_bytes[2];			/* two random bytes */

	/*
	**	Make sure our passphrase is long enough (some companies may
	**	want to make this rather large)
	*/
	/* kohno 10 May 1999 removed this -- it is up to the caller of
	   this function to make sure the passphrase is not too short
	if (strlen(passphrase) < SSH_PASSPHRASE_MIN_LEN)
	{
		ssh_errno_set(SSH_ERRNO_PHRASE_SHORT);
		return(S_BAD);
	}
	*/

	/*
	**	Open our identity file
	*/
	if ((private_file = fopen(priv_file, "wb")) == (FILE *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_SAVE);
		return(S_BAD);
	}

	ssh_debugger_new(&(ssh_info->debug_info),
		"private file open", "ssh_identity_save (kohno)");

	offset = 0;

#ifdef OLD_IDENTITY_FORMAT

	/*
	**	write an identity header to the tmp_buf to be enciphered
	*/
	my_bcopy(SSH_IDENTITY_HEAD, tmp_buf + offset,
		strlen(SSH_IDENTITY_HEAD));
	offset += strlen(SSH_IDENTITY_HEAD);

#else	/* OLD_IDENTITY_FORMAT */
	/*
	**	use four checkbytes to ensure correct decipherment
	*/
	RAND_bytes(rand_bytes, 2);
	tmp_buf[offset] = tmp_buf[offset + 2] = rand_bytes[0];
	tmp_buf[offset + 1] = tmp_buf[offset + 3] = rand_bytes[1];

	offset += SSH_IDENTITY_CHECKHEAD_LEN;

#endif	/* OLD_IDENTITY_FORMAT */

	/*
	**	Write modules and exponents in MP_Int stream notation.
	*/
	if ((count = mp_int_to_stream((uint8_t *) (tmp_buf + offset),
		modulus, SSH_MAX_PACKET - offset)) < 0)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_SAVE);
		return(S_BAD);
	}
	offset += count;
	if ((count = mp_int_to_stream((uint8_t *) (tmp_buf + offset),
		private_exp, SSH_MAX_PACKET - offset)) < 0)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_SAVE);
		return(S_BAD);
	}
	offset += count;
	if ((count = mp_int_to_stream((uint8_t *) (tmp_buf + offset),
		public_exp, SSH_MAX_PACKET - offset)) < 0)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_SAVE);
		return(S_BAD);
	}
	offset += count;

#ifndef OLD_IDENTITY_FORMAT
	/* these bytes should really be random.  xxx -- call a function */
	/* like we would for a real packet xxx */

	/*
	**	add padding to ensure tmp_buf is a multiple of
	**	SSH_UNITS (8) bytes
	*/
	offset += SSH_UNITS - (offset % SSH_UNITS);

#endif	/* OLD_IDENTITY_FORMAT */

	ssh_debug_int_new(&(ssh_info->debug_info),
		"length of offset to tmp_buf: %d", offset,
		"ssh_identity_save (kohno)");

	assert(offset < SSH_MAX_PACKET);

	ssh_debugger_new(&(ssh_info->debug_info),
		"tmp_buf created", "ssh_identity_save (kohno)");

	/*
	**	Let's create the key for our decryption
	*/
	MD5_Init(&md5_context);
	MD5_Update(&md5_context, (uint8_t *) passphrase, strlen(passphrase));
	MD5_Final(md5_digest, &md5_context);
	
	/*
	**	Now let's seed our 3des cipher
	*/
	if ((ret_val = des_set_key((C_Block *) md5_digest,
		DES3ValuesIdentity.key_1)) != 0)
	{
		ssh_errno_set(SSH_ERRNO_INIT_CIPHER);
		return(S_BAD);
	}
	if ((ret_val = des_set_key((C_Block *) (md5_digest + DES_KEY_LEN),
		DES3ValuesIdentity.key_2)) != 0)
	{
		ssh_errno_set(SSH_ERRNO_INIT_CIPHER);
		return(S_BAD);
	}
	if ((ret_val = des_set_key((C_Block *) md5_digest,
		DES3ValuesIdentity.key_3)) != 0)
	{
		ssh_errno_set(SSH_ERRNO_INIT_CIPHER);
		return(S_BAD);
	}

	my_bzero(DES3ValuesIdentity.iv_encrypt_1, DES_IV_LEN);
	my_bzero(DES3ValuesIdentity.iv_encrypt_2, DES_IV_LEN);
	my_bzero(DES3ValuesIdentity.iv_encrypt_3, DES_IV_LEN);
	my_bzero(DES3ValuesIdentity.iv_decrypt_1, DES_IV_LEN);
	my_bzero(DES3ValuesIdentity.iv_decrypt_2, DES_IV_LEN);
	my_bzero(DES3ValuesIdentity.iv_decrypt_3, DES_IV_LEN);

	/*
	**	And encrypt the identity
	*/
	des_ncbc_encrypt((C_Block *)(tmp_buf),
		(C_Block *)(tmp_buf1), (long) offset,
		DES3ValuesIdentity.key_3,
		(C_Block *)DES3ValuesIdentity.iv_encrypt_1, DES_ENCRYPT);		
	des_ncbc_encrypt((C_Block *)(tmp_buf1),
		(C_Block *)(tmp_buf2), (long) offset,
		DES3ValuesIdentity.key_2,
		(C_Block *)DES3ValuesIdentity.iv_encrypt_2, DES_DECRYPT);		
	des_ncbc_encrypt((C_Block *)(tmp_buf2),
		(C_Block *)(tmp_buf), (long) offset,
		DES3ValuesIdentity.key_1,
		(C_Block *)DES3ValuesIdentity.iv_encrypt_3, DES_ENCRYPT);

	/*
	**	and now write the encrypted file
	*/
	if (fwrite(tmp_buf, sizeof(uint8_t), offset, private_file)
		< (size_t)offset)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_SAVE);
		return(S_BAD);
	}

	my_bzero(tmp_buf, SSH_MAX_PACKET);
	my_bzero(tmp_buf1, SSH_MAX_PACKET);
	my_bzero(tmp_buf2, SSH_MAX_PACKET);
	fclose(private_file);

	ssh_debugger_new(&(ssh_info->debug_info),
		"private file written", "ssh_identity_save (kohno)");

	/*
	**	now create public file
	*/

	if ((public_file = fopen((char *) pub_file, "w")) == (FILE *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_SAVE);
		return(S_BAD);
	}

	fprintf(public_file, "%d ", modulus.nbits);

	/* xxx note here that tmp_buf is an SSH_MAX_PACKET, but mp_int_to_dec
	   xxx takes string of size MP_INT_DEC_LEN -- here we're bigger
	   xxx but nice to make changes and make sure
	*/
	(void) mp_int_to_dec((char *) tmp_buf, public_exp);
	fprintf(public_file, "%s ", tmp_buf);

	(void) mp_int_to_dec((char *) tmp_buf, modulus);
	fprintf(public_file, "%s ", tmp_buf);

	fprintf(public_file, "%s\n", msg);

	fclose(public_file);

	ssh_debugger_new(&(ssh_info->debug_info),
		"done writing identity file", "ssh_identity_save (kohno)");

	return(S_GOOD);
}



#else	/* SSH_USE_KOHNO_IDENTITY */

/*
**	Use Tatu Ylonen's identity file format
*/

/*********************************************************************
**
**     Function:                ssh_identity_load		(ylonen)
**
**     Purpose:                 read private/public keys from a ylonen-style
**				identity file
**
**     Entry (pre) conditions:  identity previously stored
**				MP_Ints initialized
**
**				file at filename is a valid private
**				identity file in ylonen's format
**
**     Parameters:              ssh_info		info about ssh
**
**				filename		name of file to load
**				passphrase		decryption passphrase
**
**				*public_key_modulus	public modulus
**				*public_key_exponent	public exponent
**				*private_key_modulus	private modulus
**				*private_key_exponent	private exponent
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD			(ssh_errno set)
**
**     Side effects:            users' RSA key returned through paramaters
**
**     Author/Date:             Tadayoshi Kohno, 7 December 1998
**     Modified:		Tadayoshi Kohno, 17 May 1998
**     Modified:		Tadayoshi Kohno, 21 May 1998
**					fixed byte order on reading pub_bits
**					(though we don't look at it, so no
**					matter)
**
**     Notes:
**	In the ssh_identity case, ssh_info is used to determine how
**	to display debug information
**
**	This was written a long time ago for an early version of WinSSH.
**	after popular demand, it is being re-incorperated into the libSSH
**	library.  Until details are figured out, however, this function
**	is #ifdef SSH_USE_YLONEN_IDENTITY'd out.  There is still my own
**	home-brew identity file format.  A conversion routine for the
**	library would be cool too.  If it turns out that the identity file
**	format is not copyright by SSH Communications, then my own file
**	format will probably disappear.
**	
**	Because the call to ssh_identity_load occurs within ssh_connect.c,
**	both the ylonen-format and kohno-format identity loads are named
**	the same.  I don't know how desirable this is, and hopefully
**	all the nitty grittys will be worked out soon.
**
**	Also, although a lot of the code for this was done with
**	WinSSH-0.0.47, it has been cleaned up by copying a lot of
**	code from the kohno-decoder.
**
*********************************************************************/

int ssh_identity_load			/* YLONEN format */
(
	struct ssh_struct * ssh_info,	/* information about ssh config */

	const char * filename,		/* file to read from */
	const char * passphrase,	/* user's passphrase */

	MP_Int * public_key_modulus,	/* public modulus */
	MP_Int * public_key_exponent,	/* public exponent */
	MP_Int * private_key_modulus,	/* private modulus */
	MP_Int * private_key_exponent	/* private exponent */
)
{
	FILE * private_file;			/* our private identity */

	uint8_t tmp_buf[SSH_MAX_PACKET];	/* temporary buffer */

	uint8_t tmp_buf1[SSH_MAX_PACKET];	/* more temporary buffers */
	uint8_t tmp_buf2[SSH_MAX_PACKET];	/* more temporary buffers */

	char head[SSH_IDENT_YLONEN_FILE_ID_LEN];	/* file format header */

	int bytes_read;				/* bytes read from file */
	
	MD5_CTX md5_context;			/* MD5 context for decipher */
	uint8_t md5_digest[MD5_DIGEST_LEN];	/* MD5 digest for decryption */
	int ret_val;				/* decryption return value */

	int offset;				/* offset into tmp_buf */
	int count;				/* number of bytes written */

	/*
	**	ylonen-format specific stuff
	*/
	uint8_t cipher_type;				/* cipher type */
	uint32_t reserved;			/* reserved info */
	uint32_t pub_bits;			/* public key bits */

	int decrypt_len;			/* length to decrypt */

	if ((private_file = fopen(filename, "rb")) == (FILE *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_LOAD);
		return(S_BAD);
	}
	
	bytes_read = fread((void *) tmp_buf, sizeof(uint8_t), SSH_MAX_PACKET,
		private_file);
	if (ferror(private_file))
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_LOAD);
		return(S_BAD);
	}
	
	if (feof(private_file))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"have EOF on identity file", "ssh_identity_load");
	}

	(void) fclose(private_file);

	if (bytes_read < (int) (strlen(SSH_IDENT_YLONEN_FILE_ID) + 1))
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_LOAD);
		return(S_BAD);
	}
	ssh_debugger_new(&(ssh_info->debug_info),
		"identity file bytes read",
		"ssh_identity_load (ylonen-format)");

	/*
	**	First let's read the id string at the head of the file
	*/
	offset = 0;
	my_bcopy(tmp_buf, head, strlen(SSH_IDENT_YLONEN_FILE_ID) + 1);
	offset += strlen(SSH_IDENT_YLONEN_FILE_ID) + 1;

	if (strncmp(SSH_IDENT_YLONEN_FILE_ID, head,
		strlen(SSH_IDENT_YLONEN_FILE_ID)) != 0)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"identity file format version incorrect",
			"ssh_identity_load (ylonen-format)");

		ssh_errno_set(SSH_ERRNO_ID_STRING);
		return(S_BAD);
	}
	ssh_debugger_new(&(ssh_info->debug_info),
		"format header correct",
		"ssh_identity_load (ylonen-format)");

	/*
	**	get the cipher type, reserved word, and public key bits
	*/
	my_bcopy((void *) (tmp_buf + offset), (void *) &cipher_type,
		sizeof(cipher_type));
	offset += sizeof(cipher_type);

	my_bcopy((void *) (tmp_buf + offset), (void *) &reserved,
		sizeof(reserved));
	offset += sizeof(reserved);
	reserved = ntohl(reserved);

	my_bcopy((void *) (tmp_buf + offset), (void *) &pub_bits,
		sizeof(pub_bits));
	offset += sizeof(pub_bits);
	pub_bits = ntohl(pub_bits);

	/*
	**	now read the modulus's and public key exponent
	*/
	count = mp_int_from_stream(tmp_buf + offset, public_key_modulus);
	count = mp_int_from_stream(tmp_buf + offset, private_key_modulus);

	if (count <= 0)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"error reading private and public key modulus",
			"ssh_identity_load (ylonen-format)");

		ssh_errno_set(SSH_ERRNO_MALLOC);
		return(S_BAD);
	}
	offset += count;

	count = mp_int_from_stream(tmp_buf + offset, public_key_exponent);
	if (count <= 0)
	{
		/* should also be checking no to overflow the tmp_buf */
		ssh_debugger_new(&(ssh_info->debug_info),
			"error reading the public key exponent",
			"ssh_identity_load (ylonen-format)");

		ssh_errno_set(SSH_ERRNO_MALLOC);
		return(S_BAD);
	}
	offset += count;

	/*
	**	get the comment
	*/
	count = string_from_stream(tmp_buf + offset, tmp_buf1,
		SSH_MAX_PACKET);
	tmp_buf1[count] = 0;
	offset += count + sizeof(uint32_t);


	ssh_debugger_new(&(ssh_info->debug_info),
		"read the comment",
		"ssh_identity_load (ylonen-format)");

	/*
	**	now decrypt the rest of the identity file
	**	[it would be nice to be able to use ssh_crypt, but
	**	the functions there do packets as a whole -- anyway,
	**	we'll move this over and start supporting idea, ...
	*/

	if (cipher_type == SSH_CIPHER_3DES)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"deciphering with 3DES",
			"ssh_identity_load (ylonen-format)");
		
		/*
		**	Let's create the key for our deciphering
		*/
		MD5_Init(&md5_context);
		MD5_Update(&md5_context, (uint8_t *) passphrase,
			strlen(passphrase));
		MD5_Final(md5_digest, &md5_context);
		
		/*
		**	Now let's seed our 3des cipher
		*/
		if ((ret_val = des_set_key((C_Block *) md5_digest,
			DES3ValuesIdentity.key_1)) != 0)
		{
			ssh_errno_set(SSH_ERRNO_INIT_CIPHER);
			return(S_BAD);
		}
		if ((ret_val
			= des_set_key((C_Block *) (md5_digest + DES_KEY_LEN),
			DES3ValuesIdentity.key_2)) != 0)
		{
			ssh_errno_set(SSH_ERRNO_INIT_CIPHER);
			return(S_BAD);
		}
		if ((ret_val = des_set_key((C_Block *) md5_digest,
			DES3ValuesIdentity.key_3)) != 0)
		{
			ssh_errno_set(SSH_ERRNO_INIT_CIPHER);
			return(S_BAD);
		}
	
		my_bzero(DES3ValuesIdentity.iv_encrypt_1, DES_IV_LEN);
		my_bzero(DES3ValuesIdentity.iv_encrypt_2, DES_IV_LEN);
		my_bzero(DES3ValuesIdentity.iv_encrypt_3, DES_IV_LEN);
		my_bzero(DES3ValuesIdentity.iv_decrypt_1, DES_IV_LEN);
		my_bzero(DES3ValuesIdentity.iv_decrypt_2, DES_IV_LEN);
		my_bzero(DES3ValuesIdentity.iv_decrypt_3, DES_IV_LEN);
	
		decrypt_len = bytes_read - offset;
		if (decrypt_len % SSH_UNITS)
		{
			ssh_debug_int_new(&(ssh_info->debug_info),
				"multiple of %d bytes not left",
				SSH_UNITS, "ssh_identity_load (ylonen-format)");
		} else
		{
			ssh_debug_int_new(&(ssh_info->debug_info),
				"good, multiple of %d bytes left",
				SSH_UNITS, "ssh_identity_load (ylonen-format)");
		}
	
		/*
		**	Now decrypt what's left
		*/
		des_ncbc_encrypt((C_Block *)(tmp_buf + offset),
			(C_Block *)(tmp_buf1), (long) bytes_read,
			DES3ValuesIdentity.key_1,
			(C_Block *)DES3ValuesIdentity.iv_decrypt_1,
			DES_DECRYPT);		

		des_ncbc_encrypt((C_Block *)(tmp_buf1),
			(C_Block *)(tmp_buf2), (long) bytes_read,
			DES3ValuesIdentity.key_2,
			(C_Block *)DES3ValuesIdentity.iv_decrypt_2,
			DES_ENCRYPT);		

		des_ncbc_encrypt((C_Block *)(tmp_buf2),
			(C_Block *)(tmp_buf + offset), (long) bytes_read,
			DES3ValuesIdentity.key_3,
			(C_Block *)DES3ValuesIdentity.iv_decrypt_3,
			DES_DECRYPT);

	} else if (cipher_type == SSH_CIPHER_NONE)
	{
		/* do nothing */

		ssh_debugger_new(&(ssh_info->debug_info),
			"deciphering with NONE",
			"ssh_identity_load (ylonen-format)");
	} else
	{
		ssh_errno_set(SSH_ERRNO_UNKNOWN_CIPHER);
		return(S_BAD);
	}

	/*
	**	now make sure check bytes are correct (we decrypted correctly)
	*/
	if (*(tmp_buf + offset) != *(tmp_buf + offset + 2)
		|| *(tmp_buf + offset + 1) != *(tmp_buf + offset + 3))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"error decrypting identity file",
			"ssh_identity_load (ylonen-format)");
			
		ssh_errno_set(SSH_ERRNO_IDENTITY_DECRYPT);
		return(S_BAD);
	}
	offset += 4;		/* move past four check bytes */

	count = mp_int_from_stream(tmp_buf + offset, private_key_exponent);
	if (count <= 0)
	{
		/* should also check not overflowing tmp_buf */
		ssh_debugger_new(&(ssh_info->debug_info),
			"error reading private exp",
			"ssh_identity_load (ylonen-format)");

		ssh_errno_set(SSH_ERRNO_IDENTITY_LOAD);
		return(S_BAD);
	}
	offset = offset + count;

	/*
	**	Now are u, p, q
	*/

	ssh_debugger_new(&(ssh_info->debug_info),
		"identity read passed completely",
		"ssh_identity_load (ylonen-format)");

	return(S_GOOD);
}


/*********************************************************************
**
**     Function:                ssh_identity_save		(ylonen)
**
**     Purpose:                 save an RSA identity in ylonen-format
**
**     Entry (pre) conditions:  n, d, e, p, q, p_mod_q valid, filenames valid
**
**     Parameters:              ssh_info	info about ssh config
**
**				priv_file	private file
**				pub_file	public file
**				msg		message to end pub_file
**				passphrase	passphrase to encrypt by
**
**				cipher_type	type of cipher to use
**
**				modulus		RSA modulus		(n)
**				private_exp	private exponent	(d)
**				public_exp	public exponent		(e)
**
**				prime_p		prime number		(p)
**				prime_q		prime number		(q)
**				p_mod_q		p mod q			(u)
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		(ssh_errno set)
**
**     Side effects:            RSA identity encrypted and stored in
**				the specified file.
**
**     Author/Date:             Tadayoshi Kohno, 21 May 1998
**
**     Notes:
**	Save an RSA identity file using Tatu Ylonen's format (with
**	permission from Tatu Ylonen)
**
**	currently only SSH_CIPHER_3DES and SSH_CIPHER_NONE is supported,
**	this is something to be changed
**
*********************************************************************/

static int ssh_identity_save
(
	struct ssh_struct * ssh_info,	/* information about ssh config */

	const char * priv_file,		/* private filename to save to */
	const char * pub_file,		/* public filename to save to */

	const char * msg,		/* message to end public file */
	const char * passphrase,	/* passphrase for the keys */
	uint8_t cipher_type,		/* type of cipher to use */

	MP_Int modulus,			/* modulus */

	MP_Int private_exp,		/* d, private exp */
	MP_Int public_exp,		/* e, public exp */

	MP_Int prime_p,			/* p, prime number */
	MP_Int prime_q,			/* q, prime number */
	MP_Int p_mod_q			/* p mod q */
)
{
	FILE * private_file;		/* private identity file */
	FILE * public_file;		/* public identity file */

	uint8_t encrypt_buf[SSH_MAX_PACKET];	/* buffer for encrypted part */
	uint8_t plain_buf[SSH_MAX_PACKET];	/* plain text part of file */

	uint8_t tmp_buf1[SSH_MAX_PACKET];	/* temporary buffer */
	uint8_t tmp_buf2[SSH_MAX_PACKET];	/* temporary buffer */

	MD5_CTX md5_context;			/* MD5 context for decipher */
	uint8_t md5_digest[MD5_DIGEST_LEN];	/* MD5 digest for decryption */
	int ret_val;				/* decryption return value */

	int plain_offset;			/* offset into plain_buf */
	int encrypt_offset;			/* offset into encrypt_buf */
	int count;				/* number of bytes written */

	uint8_t rand_bytes[2];			/* two random bytes */

	uint32_t reserved;			/* reserved info */
	uint32_t pub_bits;			/* modulus size in bits */

	/*
	**	Open our identity file
	*/
	if ((private_file = fopen(priv_file, "wb")) == (FILE *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_SAVE);
		return(S_BAD);
	}

	ssh_debugger_new(&(ssh_info->debug_info),
		"private file open", "ssh_identity_save (ylonen)");

	plain_offset = 0;

	/* write the header for the appropriate file format */
	my_bcopy((void *) SSH_IDENT_YLONEN_FILE_ID, (void *) plain_buf,
		(int) (strlen(SSH_IDENT_YLONEN_FILE_ID) + 1));
	plain_offset += strlen(SSH_IDENT_YLONEN_FILE_ID) + 1;

	/* write the cipher type to the plaintext buffer */
	my_bcopy((void *) &cipher_type, (void *) (plain_buf + plain_offset),
		sizeof(cipher_type));
	plain_offset += sizeof(cipher_type);

	/* write the "reserved" field of 0's  */
	reserved = 0;
	reserved = htonl(reserved);
	my_bcopy((void *) &reserved, (void *) (plain_buf + plain_offset),
		sizeof(reserved));
	plain_offset += sizeof(reserved);

	/* write the public key bits */
	pub_bits = modulus.nbits;
	pub_bits = htonl(pub_bits);
	my_bcopy((void *) &pub_bits, (void *) (plain_buf + plain_offset),
		sizeof(pub_bits));
	plain_offset += sizeof(pub_bits);

	/* Write modules in MP_Int stream notation. */
	if ((count = mp_int_to_stream((uint8_t *) (plain_buf + plain_offset),
		modulus, SSH_MAX_PACKET - plain_offset)) < 0)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_SAVE);
		return(S_BAD);
	}
	plain_offset += count;

	/* write the public key exponent in MP_Int stream notation */
	if ((count = mp_int_to_stream((uint8_t *) (plain_buf + plain_offset),
		public_exp, SSH_MAX_PACKET - plain_offset)) < 0)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_SAVE);
		return(S_BAD);
	}
	plain_offset += count;

	/* now write the comment */
	if (strlen(msg) > SSH_MAX_PACKET - plain_offset - 10)
	{
		/* make sure not overflow a buffer (10 is a random number > 4 */
		if (SSH_MAX_PACKET - plain_offset - 10 < 0)
		{
			/* and make sure indexing msg is a valid index */
			ssh_errno_set(SSH_ERRNO_IDENTITY_SAVE);
			return(S_BAD);
		}
		strncpy(tmp_buf1, msg, SSH_MAX_PACKET - plain_offset - 10),
		tmp_buf1[SSH_MAX_PACKET - plain_offset - 10] = 0;
	} else
	{
		strcpy(tmp_buf1, msg);
	}
	if ((count = string_to_stream((uint8_t *) (plain_buf + plain_offset),
		tmp_buf1)) < 0)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_SAVE);
		return(S_BAD);
	}
	plain_offset += count;

	/*
	**	now generate the encrypted portion of the file
	*/

	encrypt_offset = 0;

	/* use four checkbytes to ensure correct decipherment */
	RAND_bytes(rand_bytes, 2);
	encrypt_buf[encrypt_offset]
		= encrypt_buf[encrypt_offset + 2] = rand_bytes[0];
	encrypt_buf[encrypt_offset + 1]
		= encrypt_buf[encrypt_offset + 3] = rand_bytes[1];

	encrypt_offset += SSH_IDENTITY_CHECKHEAD_LEN;

	/* write private exponent */
	if ((count = mp_int_to_stream((uint8_t *)(encrypt_buf + encrypt_offset),
		private_exp, SSH_MAX_PACKET - encrypt_offset)) < 0)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_SAVE);
		return(S_BAD);
	}
	encrypt_offset += count;

	/* write multiplicative inverse of p mod q*/
	if ((count = mp_int_to_stream((uint8_t *)(encrypt_buf + encrypt_offset),
		p_mod_q, SSH_MAX_PACKET - encrypt_offset)) < 0)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_SAVE);
		return(S_BAD);
	}
	encrypt_offset += count;

	/* write the prime p */
	if ((count = mp_int_to_stream((uint8_t *)(encrypt_buf + encrypt_offset),
		prime_p, SSH_MAX_PACKET - encrypt_offset)) < 0)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_SAVE);
		return(S_BAD);
	}
	encrypt_offset += count;

	/* write the prime q */
	if ((count = mp_int_to_stream((uint8_t *)(encrypt_buf + encrypt_offset),
		prime_q, SSH_MAX_PACKET - encrypt_offset)) < 0)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_SAVE);
		return(S_BAD);
	}
	encrypt_offset += count;


	/* make sure encrypt_buf is a multiple of SSH_UNITS (8) bytes */
	encrypt_offset += SSH_UNITS - (encrypt_offset % SSH_UNITS);

	/*
	**	now encrypt the to-be encrypted portion of the file
	*/
	if (cipher_type == SSH_CIPHER_3DES)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"enciphering with 3DES",
			"ssh_identity_save (ylonen)");

		/*
		**	Let's create the key for our decryption
		*/
		MD5_Init(&md5_context);
		MD5_Update(&md5_context, (uint8_t *) passphrase,
			strlen(passphrase));
		MD5_Final(md5_digest, &md5_context);
		
		/*
		**	Now let's seed our 3des cipher
		*/
		if ((ret_val = des_set_key((C_Block *) md5_digest,
			DES3ValuesIdentity.key_1)) != 0)
		{
			ssh_errno_set(SSH_ERRNO_INIT_CIPHER);
			return(S_BAD);
		}
		if ((ret_val = des_set_key((C_Block *) (md5_digest+DES_KEY_LEN),
			DES3ValuesIdentity.key_2)) != 0)
		{
			ssh_errno_set(SSH_ERRNO_INIT_CIPHER);
			return(S_BAD);
		}
		if ((ret_val = des_set_key((C_Block *) md5_digest,
			DES3ValuesIdentity.key_3)) != 0)
		{
			ssh_errno_set(SSH_ERRNO_INIT_CIPHER);
			return(S_BAD);
		}
	
		my_bzero(DES3ValuesIdentity.iv_encrypt_1, DES_IV_LEN);
		my_bzero(DES3ValuesIdentity.iv_encrypt_2, DES_IV_LEN);
		my_bzero(DES3ValuesIdentity.iv_encrypt_3, DES_IV_LEN);
		my_bzero(DES3ValuesIdentity.iv_decrypt_1, DES_IV_LEN);
		my_bzero(DES3ValuesIdentity.iv_decrypt_2, DES_IV_LEN);
		my_bzero(DES3ValuesIdentity.iv_decrypt_3, DES_IV_LEN);
	
		/*
		**	And encrypt the identity
		*/
		des_ncbc_encrypt((C_Block *)(encrypt_buf),
			(C_Block *)(tmp_buf1), (long) encrypt_offset,
			DES3ValuesIdentity.key_3,
			(C_Block *)DES3ValuesIdentity.iv_encrypt_1,DES_ENCRYPT);		
		des_ncbc_encrypt((C_Block *)(tmp_buf1),
			(C_Block *)(tmp_buf2), (long) encrypt_offset,
			DES3ValuesIdentity.key_2,
			(C_Block *)DES3ValuesIdentity.iv_encrypt_2,DES_DECRYPT);		
		des_ncbc_encrypt((C_Block *)(tmp_buf2),
			(C_Block *)(encrypt_buf), (long) encrypt_offset,
			DES3ValuesIdentity.key_1,
			(C_Block *)DES3ValuesIdentity.iv_encrypt_3,DES_ENCRYPT);
		
	} else if (cipher_type == SSH_CIPHER_NONE)
	{
		/* do nothing */

		ssh_debugger_new(&(ssh_info->debug_info),
			"not enciphering identity file",
			"ssh_identity_save (ylonen)");
		
	} else
	{
		/* currently there are no other cipher types to choose from */
		ssh_errno_set(SSH_ERRNO_UNKNOWN_CIPHER);

		return(S_BAD);
	}

	/*
	**	now concatenate the encryted buffer onto the end of the plain
	**	text buffer
	*/
	my_bcopy((void *) encrypt_buf, (void *) (plain_buf + plain_offset),
		encrypt_offset);
	plain_offset += encrypt_offset;

	/*
	**	and now write the encrypted file
	*/
	if (fwrite(plain_buf, sizeof(uint8_t), plain_offset, private_file)
		< (size_t) plain_offset)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_SAVE);
		return(S_BAD);
	}

	my_bzero(plain_buf, SSH_MAX_PACKET);
	my_bzero(encrypt_buf, SSH_MAX_PACKET);
	my_bzero(tmp_buf1, SSH_MAX_PACKET);
	my_bzero(tmp_buf2, SSH_MAX_PACKET);
	my_bzero(md5_digest, MD5_DIGEST_LEN);
	fclose(private_file);

	ssh_debugger_new(&(ssh_info->debug_info),
		"private file written", "ssh_identity_save");

	/*
	**	now create public file
	*/

	if ((public_file = fopen((char *) pub_file, "w")) == (FILE *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_IDENTITY_SAVE);
		return(S_BAD);
	}

	fprintf(public_file, "%d ", modulus.nbits);

	/* xxx note here that tmp_buf is an SSH_MAX_PACKET, but mp_int_to_dec
	   xxx takes string of size MP_INT_DEC_LEN -- here we're bigger
	   xxx but nice to make changes and make sure
	*/
	(void) mp_int_to_dec((char *) tmp_buf1, public_exp);
	fprintf(public_file, "%s ", tmp_buf1);

	(void) mp_int_to_dec((char *) tmp_buf1, modulus);
	fprintf(public_file, "%s ", tmp_buf1);

	fprintf(public_file, "%s\n", msg);

	fclose(public_file);

	ssh_debugger_new(&(ssh_info->debug_info),
		"done writing identity file", "ssh_identity_save");

	return(S_GOOD);
}


#endif /* not use SSH_USE_KOHNO_IDENTITY (use Tatu Ylonen's) */

