/*
 *  linux/fs/cryptfs/inode.c
 *  $Id: inode.c,v 1.1.1.1 1998/11/05 21:06:44 ezk Exp $
 *
 *  Based on lofs/inode.c, which is:
 *  -----------------------------------------------------------------------------
 *  Copyright (C) 1996  Benjamin LaHaise <blah@dot.superaje.com>
 *
 *  cryptfs v0.7 - a loopback filesystem for linux 2.0.something
 *
 *  Based on linux/fs/nfs/inode.c and others.
 *
 *  This is FREE software with NO WARRANTY.  See the file COPYING for details.
 *  -----------------------------------------------------------------------------
 *
 */

#include <linux/module.h>

#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/mman.h>
#include <linux/swap.h>
#include <linux/string.h>
#include <linux/stat.h>
#include <linux/errno.h>
#include <linux/locks.h>
#include <linux/smp.h>
#include <linux/smp_lock.h>
#include <linux/file.h>
#include <linux/malloc.h>
#include <linux/fcntl.h>

#include <asm/system.h>
#include <asm/segment.h>

#include "cryptfs.h"

#define USE_GENERIC_READ
#define USE_GENERIC_WRITE

extern struct vm_operations_struct file_shared_mmap;
extern struct vm_operations_struct file_private_mmap;

static struct vm_operations_struct cryptfs_shared_vm_ops;
static struct vm_operations_struct cryptfs_private_vm_ops;

static struct file_operations cryptfs_fops;
static struct inode_operations cryptfs_iops;
static struct super_operations cryptfs_sops;

static void cryptfs_put_super(struct super_block *);
static int cryptfs_interpose(struct inode *hidden_inode, struct inode **result, struct super_block *super);

static void
checkinode(struct inode *inode, char *msg)
{
    if (inode) {
	if (!itopd(inode)) {
	    printk(KERN_WARNING "cryptfs: checkinode(%ld) - no private data (%s)\n", inode->i_ino, msg);
	    return;
	}
	if (!itohi(inode)) {
	    printk(KERN_WARNING "cryptfs: checkinode(%ld) - underlying is NULL! (%s)\n", inode->i_ino, msg);
	    return;
	}
	if (!inode->i_sb || !inode->i_sb->s_type) {
	    printk(KERN_WARNING "cryptfs: checkinode(%ld) - inode->i_sb is NULL! (%s)\n", inode->i_ino, msg);
	    return;
	}
	fist_dprint(7, "inode->i_count = %d, hidden_inode->i_count = %d, inode = %d\n",
		    inode->i_count, itohi(inode)->i_count, inode->i_ino);
    } else
	printk(KERN_WARNING "cryptfs: checkinode - inode is NULL! (%s)\n", msg);
}

static void
cryptfs_copy_inode(struct inode *dest, struct inode *src)
{
    print_entry_location();
    /* dest->i_dev = src->i_dev; */ /* methinks so? */

    dest->i_mode = src->i_mode;
    dest->i_nlink = src->i_nlink;
    dest->i_uid = src->i_uid;
    dest->i_gid = src->i_gid;
    dest->i_rdev = src->i_rdev;
    dest->i_size = src->i_size;
    dest->i_atime = src->i_atime;
    dest->i_mtime = src->i_mtime;
    dest->i_ctime = src->i_ctime;
    dest->i_blksize = src->i_blksize;
    dest->i_blocks = src->i_blocks;

    print_exit_location();
}

static int
cryptfs_interpose(struct inode *hidden_inode, struct inode **inode, struct super_block *super)
{
    int err = 0;

    print_entry_location();

    if (!(*inode = iget(super, hidden_inode->i_ino))) {
	err = -EACCES;		/* should be impossible??? */
	goto out;
    }
    if ((*inode)->i_lock)
	printk(KERN_WARNING "INTERPOSE: inode (%ld use %d) is locked (uh-oh)\n", (*inode)->i_ino, (int) (*inode)->i_count);

    if (itopd(*inode)) {
	/* Already interposed */
	iput(hidden_inode);
	if (itohi(*inode) != hidden_inode) {
	    printk("INTERPOSE PANIC: inode already interposed with different hidden_inode!\n");
	    err = -EPERM;
	    goto out;
	}
	err = 0;
	goto out;
    }
    cryptfs_copy_inode((*inode), hidden_inode);
    (*inode)->i_op = &cryptfs_iops;
    itopd(*inode) = kmalloc(sizeof(struct cryptfs_inode_info), GFP_KERNEL);
    itohi(*inode) = hidden_inode;

    checkinode(*inode, "cryptfs_interpose: OUT");

out:
    print_exit_status(err);
    return err;
}

static int
cryptfs_notify_change(struct inode *inode, struct iattr *ia)
{
    int err = 0;
    struct inode *hidden_inode = itohi(inode);

    print_entry_location();

    checkinode(inode, "cryptfs_notify_change");
    if (hidden_inode->i_sb &&
	hidden_inode->i_sb->s_op &&
	hidden_inode->i_sb->s_op->notify_change) {
	err = hidden_inode->i_sb->s_op->notify_change(hidden_inode, ia);
	if (err)
	    goto out;
    } else {
	if ((err = inode_change_ok(hidden_inode, ia)) != 0)
	    goto out;

	inode_setattr(hidden_inode, ia);
    }

    cryptfs_copy_inode(inode, hidden_inode);
    checkinode(inode, "post cryptfs_notify_change");
out:
    print_exit_status(err);
    return err;
}

static int
cryptfs_create(struct inode *dir, const char *name, int namelen, int mode, struct inode **inode)
{
    int err = -EACCES;
    struct inode *hidden_inode, *hidden_dir = itohi(dir);
    char *encoded_name;
    int encoded_namelen;
    void *key;

    print_entry_location();

    if ((key = fist_get_userpass(dir->i_sb)) == NULL) {
	err = -EACCES;
	goto out;
    }
    checkinode(dir, "cryptfs_create");
    if (hidden_dir->i_op &&
	hidden_dir->i_op->create) {
	encoded_namelen = fist_crypt_encodefilename(name, namelen, &encoded_name, key, SKIP_DOTS);
	hidden_dir->i_count++;
	down(&hidden_dir->i_sem);
	err = hidden_dir->i_op->create(hidden_dir, encoded_name, encoded_namelen - 1, mode, &hidden_inode);
	up(&hidden_dir->i_sem);
	kfree_s(encoded_name, encoded_namelen);
	if (!err && (hidden_inode->i_sb == hidden_dir->i_sb)) {
	    fist_dprint(6, "CREATE: \"%s\" -> inode 0x%x\n", name, (int) hidden_inode);
	    err = cryptfs_interpose(hidden_inode, inode, dir->i_sb);
	    fist_dprint(6, "CREATE: inode 0x%x, hidden_inode 0x%x, inode->i_count %d, hidden_inode->i_count %d\n",
			(int) (*inode), (int) hidden_inode, (*inode)->i_count, hidden_inode->i_count);
	}
    }
out:
    checkinode(dir, "post cryptfs_create");
    iput(dir);

    print_exit_status(err);
    return err;
}

static int
cryptfs_link(struct inode *oldinode, struct inode *dir, const char *name, int namelen)
{
    int err = -EPERM;
    struct inode *hidden_dir = itohi(dir);
    char *encoded_name;
    int encoded_namelen;
    void *key;

    print_entry_location();

    /* MUST make sure we only hard link into our own file system! */
    if (oldinode->i_op != &cryptfs_iops) {
	err = -EXDEV;
	goto out;
    }

    if ((key = fist_get_userpass(dir->i_sb)) == NULL) {
	err = -EACCES;
	goto out;
    }
    checkinode(dir, "cryptfs_link-dir");
    checkinode(oldinode, "cryptfs_link-oldinode");
    if (hidden_dir->i_op &&
	hidden_dir->i_op->link) {
	encoded_namelen = fist_crypt_encodefilename(name, namelen, &encoded_name, key, SKIP_DOTS);
	hidden_dir->i_count++;
	itohi(oldinode)->i_count++;
	down(&hidden_dir->i_sem);
	err = hidden_dir->i_op->link(itohi(oldinode), hidden_dir, encoded_name, encoded_namelen - 1);
	up(&hidden_dir->i_sem);
	kfree_s(encoded_name, encoded_namelen);
    }
out:
    checkinode(dir, "post cryptfs_link-dir");
    checkinode(oldinode, "post cryptfs_link-oldinode");
    iput(oldinode);
    iput(dir);

    print_exit_status(err);
    return err;
}

static int
cryptfs_unlink(struct inode *dir, const char *name, int namelen)
{
    int err = -EPERM;
    struct inode *hidden_dir = itohi(dir);
    char *encoded_name;
    int encoded_namelen;
    void *key;

    print_entry_location();

    if ((key = fist_get_userpass(dir->i_sb)) == NULL) {
	err = -EACCES;
	goto out;
    }
    checkinode(dir, "cryptfs_unlink-dir");
    if (hidden_dir->i_op &&
	hidden_dir->i_op->unlink) {
	encoded_namelen = fist_crypt_encodefilename(name, namelen, &encoded_name, key, SKIP_DOTS);
	hidden_dir->i_count++;
	err = hidden_dir->i_op->unlink(hidden_dir, encoded_name, encoded_namelen - 1);
	kfree_s(encoded_name, encoded_namelen);
    }
out:
    checkinode(dir, "post cryptfs_unlink-dir");
    iput(dir);

    print_exit_status(err);
    return err;
}

static int
cryptfs_symlink(struct inode *dir, const char *name, int namelen, const char *oldname)
{
    int err = -EPERM;
    struct inode *hidden_dir = itohi(dir);
    char *encoded_name, *encoded_oldname;
    int encoded_namelen, encoded_oldnamelen;
    void *key;

    print_entry_location();

    if ((key = fist_get_userpass(dir->i_sb)) == NULL) {
	err = -EACCES;
	goto out;
    }
    checkinode(dir, "cryptfs_symlink-dir");
    if (hidden_dir->i_op &&
	hidden_dir->i_op->symlink) {
	encoded_oldnamelen = fist_crypt_encodefilename(oldname, strlen(oldname), &encoded_oldname, key, DO_DOTS);
	encoded_namelen = fist_crypt_encodefilename(name, namelen, &encoded_name, key, SKIP_DOTS);
	hidden_dir->i_count++;
	err = hidden_dir->i_op->symlink(hidden_dir, encoded_name, encoded_namelen - 1, encoded_oldname);
	kfree_s(encoded_name, encoded_namelen);
	kfree_s(encoded_oldname, encoded_oldnamelen);
    }
out:
    checkinode(dir, "post cryptfs_symlink-dir");
    iput(dir);

    print_exit_status(err);
    return err;
}

static int
cryptfs_mkdir(struct inode *dir, const char *name, int namelen, int mode)
{
    int err = -EPERM;
    struct inode *hidden_dir = itohi(dir);
    char *encoded_name;
    int encoded_namelen;
    void *key;

    print_entry_location();

    if ((key = fist_get_userpass(dir->i_sb)) == NULL) {
	err = -EACCES;
	goto out;
    }
    checkinode(dir, "cryptfs_mkdir-dir");
    if (hidden_dir->i_op &&
	hidden_dir->i_op->mkdir) {
	encoded_namelen = fist_crypt_encodefilename(name, namelen, &encoded_name, key, SKIP_DOTS);
	hidden_dir->i_count++;
	down(&hidden_dir->i_sem);
	err = hidden_dir->i_op->mkdir(hidden_dir, encoded_name, encoded_namelen - 1, mode);
	up(&hidden_dir->i_sem);
	kfree_s(encoded_name, encoded_namelen);
    }
out:
    checkinode(dir, "post cryptfs_mkdir-dir");
    iput(dir);

    print_exit_status(err);
    return err;
}

static int
cryptfs_rmdir(struct inode *dir, const char *name, int namelen)
{
    int err = -EPERM;
    struct inode *hidden_dir = itohi(dir);
    char *encoded_name;
    int encoded_namelen;
    void *key;

    print_entry_location();

    if ((key = fist_get_userpass(dir->i_sb)) == NULL) {
	err = -EACCES;
	goto out;
    }
    checkinode(dir, "cryptfs_rmdir-dir");
    if (hidden_dir->i_op &&
	hidden_dir->i_op->rmdir) {
	encoded_namelen = fist_crypt_encodefilename(name, namelen, &encoded_name, key, SKIP_DOTS);
	hidden_dir->i_count++;
	err = hidden_dir->i_op->rmdir(hidden_dir, encoded_name, encoded_namelen - 1);
	kfree_s(encoded_name, encoded_namelen);
    }
out:
    checkinode(dir, "post cryptfs_rmdir-dir");
    iput(dir);

    print_exit_status(err);
    return err;
}

static int
cryptfs_mknod(struct inode *dir, const char *name, int namelen, int mode, int dev)
{
    int err = -EPERM;
    struct inode *hidden_dir = itohi(dir);
    char *encoded_name;
    int encoded_namelen;
    void *key;

    print_entry_location();

    if ((key = fist_get_userpass(dir->i_sb)) == NULL) {
	err = -EACCES;
	goto out;
    }
    checkinode(dir, "cryptfs_mknod-dir");
    if (hidden_dir->i_op &&
	hidden_dir->i_op->mknod) {
	encoded_namelen = fist_crypt_encodefilename(name, namelen, &encoded_name, key, SKIP_DOTS);
	hidden_dir->i_count++;
	down(&hidden_dir->i_sem);
	err = hidden_dir->i_op->mknod(hidden_dir, encoded_name, encoded_namelen - 1, mode, dev);
	up(&hidden_dir->i_sem);
	kfree_s(encoded_name, encoded_namelen);
    }
out:
    checkinode(dir, "post cryptfs_mknod-dir");
    iput(dir);

    print_exit_status(err);
    return err;
}

static int
cryptfs_rename(struct inode *old_dir, const char *old_name, int old_len,
	       struct inode *new_dir, const char *new_name, int new_len, int must_be_dir)
{
    int err = -EPERM;
    struct inode *hidden_old_dir = itohi(old_dir), *hidden_new_dir = itohi(new_dir);
    char *encoded_oldname, *encoded_newname;
    int encoded_oldnamelen, encoded_newnamelen;
    void *key;

    print_entry_location();

    if ((key = fist_get_userpass(new_dir->i_sb)) == NULL) {
	err = -EACCES;
	goto out;
    }
    checkinode(old_dir, "cryptfs_rename-old_dir");
    checkinode(new_dir, "cryptfs_rename-new_dir");
    if (hidden_old_dir->i_op &&
	hidden_old_dir->i_op->rename) {
	hidden_old_dir->i_count++;
	hidden_new_dir->i_count++;
	encoded_oldnamelen = fist_crypt_encodefilename(old_name, old_len, &encoded_oldname, key, SKIP_DOTS);
	encoded_newnamelen = fist_crypt_encodefilename(new_name, new_len, &encoded_newname, key, SKIP_DOTS);
	down(&hidden_new_dir->i_sem);
	err = hidden_old_dir->i_op->rename(hidden_old_dir, encoded_oldname, encoded_oldnamelen - 1, hidden_new_dir, encoded_newname, encoded_newnamelen - 1, must_be_dir);
	up(&hidden_new_dir->i_sem);
	kfree_s(encoded_newname, encoded_newnamelen);
	kfree_s(encoded_oldname, encoded_oldnamelen);
    }
out:
    checkinode(old_dir, "post cryptfs_rename-old_dir");
    checkinode(new_dir, "post cryptfs_rename-new_dir");
    iput(old_dir);
    iput(new_dir);

    print_exit_status(err);
    return err;
}

static int
cryptfs_readlink(struct inode *inode, char *buf, int bufsiz)
{
    int err = -EINVAL;
    struct inode *hidden_inode = itohi(inode);
    char *hidden_buf;
    unsigned long old_fs;
    char *decoded_name;
    void *key;

    print_entry_location();

    if ((key = fist_get_userpass(inode->i_sb)) == NULL) {
	err = -EACCES;
	goto out;
    }
    hidden_buf = kmalloc(bufsiz, GFP_KERNEL);
    if (hidden_buf == NULL) {
	printk("Out of memory.\n");
	err = -ENOMEM;
	goto out;
    }
    checkinode(inode, "cryptfs_readlink IN");
    if (hidden_inode->i_op &&
	hidden_inode->i_op->readlink) {
	hidden_inode->i_count++;
	old_fs = get_fs();
	set_fs(KERNEL_DS);
	err = hidden_inode->i_op->readlink(hidden_inode, hidden_buf, bufsiz);
	set_fs(old_fs);
	if (err >= 0) {
	    fist_dprint(7, "READLINK: link \"%s\", length %d\n", hidden_buf, err);
	    err = fist_crypt_decodefilename(hidden_buf, err,
					    &decoded_name, key, DO_DOTS);
	    if (err > 0) {
		memcpy_tofs(buf, decoded_name, err);
		kfree_s(decoded_name, err);
	    }
	}
    }

    kfree_s(hidden_buf, bufsiz);

out:
    checkinode(inode, "post cryptfs_readlink-inode");
    iput(inode);

    print_exit_status(err);
    return err;
}

/* kludgey - replace when up/down problem fixed */
static void
cryptfs_truncate(struct inode *inode)
{
    struct inode *hidden_inode = itohi(inode);

    print_entry_location();

    checkinode(inode, "cryptfs_truncate");
    down(&hidden_inode->i_sem);
    if (hidden_inode->i_op &&
	hidden_inode->i_op->truncate)
	hidden_inode->i_op->truncate(hidden_inode);
    up(&hidden_inode->i_sem);
#if 0
    iput(inode);
#endif
    print_exit_location();
}

static int
cryptfs_lookup(struct inode *dir, const char *name, int namelen, struct inode **inode)
{
    int err = -ENOTDIR;
    struct inode *hidden_inode, *hidden_dir = itohi(dir);
    char *encoded_name;
    int encoded_namelen;
    void *key;

    print_entry_location();

    if ((key = fist_get_userpass(dir->i_sb)) == NULL) {
	err = -EACCES;
	goto out;
    }
    checkinode(dir, "cryptfs_lookup");
    fist_dprint(8, "LOOKUP: name \"%s\", namelen %d\n", name, namelen);
    if (hidden_dir->i_op &&
	hidden_dir->i_op->lookup) {
	hidden_dir->i_count++;
	encoded_namelen = fist_crypt_encodefilename(name, namelen, &encoded_name, key, SKIP_DOTS);
	err = hidden_dir->i_op->lookup(hidden_dir, encoded_name, encoded_namelen - 1, &hidden_inode);
	kfree_s(encoded_name, encoded_namelen);
	if (!err && (hidden_inode->i_sb == hidden_dir->i_sb)) {
	    fist_dprint(6, "LOOKUP: \"%s\" -> inode 0x%x\n", name, (int) hidden_inode);
	    fist_dprint(8, "LOOKUP: inode->i_op 0x%x\n", (int)hidden_inode->i_op);
	    err = cryptfs_interpose(hidden_inode, inode, dir->i_sb);
	    fist_dprint(6, "LOOKUP: inode 0x%x, hidden_inode 0x%x, inode->i_count %d, hidden_inode->i_count %d\n",
			(int) (*inode), (int) hidden_inode, (*inode)->i_count, hidden_inode->i_count);

	    checkinode(*inode, "cryptfs_lookup OUT: inode:");
	}
    }
out:
    iput(dir);
    checkinode(dir, "cryptfs_lookup OUT: dir:");

    print_exit_status(err);
    return err;
}

#if 0
/* inode is from cryptfs, dir could be anything */
int
cryptfs_follow_link(struct inode *dir, struct inode *inode, int flag, int mode, struct inode **result)
{
    int err = 0;
    struct inode *new_hidden_inode, *hidden_inode = itohi(inode);

    print_entry_location();

    checkinode(inode, "cryptfs_follow_link IN: inode:");
    fist_dprint(6, "FOLLOW_LINK1: inode 0x%x, hidden_inode 0x%x, inode->i_count %d, hidden_inode->i_count %d\n",
		(int) inode, (int) hidden_inode, inode->i_count, hidden_inode->i_count);
    if (hidden_inode->i_op &&
	hidden_inode->i_op->follow_link) {
	dir->i_count++;
	hidden_inode->i_count++;
	err = hidden_inode->i_op->follow_link(dir, hidden_inode, flag, mode, &new_hidden_inode);
	if (!err) {
	    if (new_hidden_inode->i_sb == hidden_inode->i_sb)
		/*
		 * the destination of the link is in our fs - interpose
		 */
		err = cryptfs_interpose(new_hidden_inode, result, inode->i_sb);
	    else
		/*
		 * the destination of the link is elsewhere - copy
		 */
		*result = new_hidden_inode;
	}
	iput(inode);
    } else {
	/*
	 * Underlying file system does not support links
	 */
	*result = inode;
    }
    iput(dir);
    fist_dprint(6, "FOLLOW_LINK2: inode 0x%x, hidden_inode 0x%x, inode->i_count %d, hidden_inode->i_count %d\n",
		(int) inode, (int) hidden_inode, inode->i_count, hidden_inode->i_count);

    print_exit_status(err);
    return err;
}
#else
/* inode is from cryptfs, dir could be anything */
int
cryptfs_follow_link(struct inode *dir, struct inode *inode, int flag, int mode, struct inode **result)
{
    int err = 0;
    struct inode *hidden_inode = itohi(inode);
    char *symlink;
    long old_fs;

    print_entry_location();

    checkinode(inode, "cryptfs_follow_link IN: inode:");
    if (!hidden_inode->i_op ||
	!hidden_inode->i_op->readlink) {
	*result = inode;
	iput(dir);
	goto out;
    }
    if (current->link_count > 5) {
	err = -ELOOP;
	iput(dir);
	iput(inode);
	goto out;
    }
    symlink = kmalloc(PAGE_SIZE, GFP_KERNEL);
    if (symlink == NULL) {
	printk("Out of memory.\n");
	err = -ENOMEM;
	iput(dir);
	iput(inode);
	goto out;
    }
    old_fs = get_fs();
    set_fs(KERNEL_DS);
    inode->i_count++;
    err = cryptfs_readlink(inode, symlink, PAGE_SIZE);
    if (err < 0) {
	iput(dir);
	goto out_free;
    }
    UPDATE_ATIME(inode);
    symlink[err] = 0;

    current->link_count++;
    err = open_namei(symlink, flag, mode, result, dir);
    current->link_count--;

out_free:
    iput(inode);
    set_fs(old_fs);
    kfree_s(symlink, PAGE_SIZE);

out:
    print_exit_status(err);
    return err;
}
#endif

static void
cryptfs_put_inode(struct inode *inode)
{
    struct inode *hidden_inode = itohi(inode);

    print_entry_location();

    fist_dprint(6, "PUT_INODE1: inode 0x%x, hidden_inode 0x%x, hidden_inode->i_count %d\n",
		(int) inode, (int) hidden_inode, hidden_inode->i_count);

    inode->i_lock = 1;
    checkinode(inode, "cryptfs_put_inode IN");
    if (inode->i_count != 1)
	printk(KERN_WARNING "cryptfs_put_inode(%ld): i_count is %d\n", inode->i_ino, (int) inode->i_count);
    iput(hidden_inode);
    kfree_s(itopd(inode), sizeof(struct cryptfs_inode_info));

    checkinode(inode, "cryptfs_put_inode OUT");
    inode->i_lock = 0;
    clear_inode(inode);
    wake_up(&inode->i_wait);

    fist_dprint(6, "PUT_INODE2: inode 0x%x, hidden_inode 0x%x, hidden_inode->i_count %d\n",
		(int) inode, (int) hidden_inode, hidden_inode->i_count);

    print_exit_location();
}

static void
cryptfs_statfs(struct super_block *sb, struct statfs *buf, int bufsiz)
{
    struct super_block *hidden_sb = stohs(sb);

    print_entry_location();

    if (hidden_sb->s_op && hidden_sb->s_op->statfs)
	hidden_sb->s_op->statfs(hidden_sb, buf, bufsiz);
    print_exit_location();
}

static inline void
add_to_page_cache(struct page *page,
		  struct inode *inode, unsigned long offset,
		  struct page **hash)
{
    page->count++;
    page->flags &= ~((1 << PG_uptodate) | (1 << PG_error));
    page->offset = offset;
    add_page_to_inode_queue(inode, page);
    __add_page_to_hash_queue(page, hash);
}

static int
cryptfs_readpage(struct inode *inode, struct page *page)
{
    int err;
    struct inode *hidden_inode = itohi(inode);
    struct page *hidden_page, **hash;
    unsigned long page_cache = 0, page_data, hidden_page_data;
    void *key;

    print_entry_location();

    ASSERT(hidden_inode->i_op != NULL);

    if ((key = fist_get_userpass(inode->i_sb)) == NULL) {
	err = -EACCES;
	goto out;
    }
    fist_dprint(6, "READPAGE: inode 0x%x, hidden_inode 0x%x, page->inode 0x%x\n",
		(int) inode, (int) hidden_inode, (int) page->inode);

    /* create new hidden_page (look for it in cache?) */
    hash = page_hash(hidden_inode, page->offset & PAGE_MASK);

restart_page:
    hidden_page = __find_page(hidden_inode, page->offset & PAGE_MASK, *hash);
    if (!hidden_page) {
	if (!page_cache) {		/* allocate new page */
	    page_cache = __get_free_page(GFP_KERNEL);
	    if (!page_cache) {
		err = -ENOMEM;
		goto out;		/* no need to free anything */
	    }
	    goto restart_page;
	}
	hidden_page = mem_map + MAP_NR(page_cache);
	page_cache = 0;
	add_to_page_cache(hidden_page, hidden_inode, page->offset & PAGE_MASK, hash);
    }
    /* now we have a valid hidden page */

    fist_dprint(6, "READPAGE: hidden_page %x\n", (int) hidden_page);

    fist_dprint(8, "hidden_inode->i_op 0x%x, hidden_inode->i_op->default_file_ops 0x%x\n",
		hidden_inode->i_op, hidden_inode->i_op->default_file_ops);
    fist_dprint(8, "hidden_inode->i_op->permission 0x%x, hidden_inode->i_op->readpage 0x%x\n",
		hidden_inode->i_op->permission, hidden_inode->i_op->readpage);
    if (hidden_inode->i_op->readpage) {
	/*
	 * create new hidden_page (look for it in cache?)
	 * call lower level readpage on hidden_inode and hidden_page
	 */
	err = hidden_inode->i_op->readpage(hidden_inode, hidden_page);
	if (err) {
	    goto out_free;
	}

	/* XXX: may need to wait_on_page(hidden_page) */
	wait_on_page(hidden_page);
	if (!PageUptodate(hidden_page) || PageError(hidden_page)) {
	    printk("hidden page not up-to-date or error!\n");
	    err = -EIO;
	    goto out_free;
	}
    }
#if 1
 else {
     /*
      * ==========XXXX This is a rough hack=============
      * hidden_file really should be generated by the lower layer, somehow,
      * maybe similar to open (but we don't have a filename to pass...)
      *
      * Beware of double stacking - private_data *can't* have a valid value
      * no matter how hard we try
      */
	struct file hidden_file;
	long old_fs;

	fist_dprint(7, "faking a read operation\n");

	bzero((char *)&hidden_file, sizeof(struct file));
	hidden_file.f_op = hidden_inode->i_op->default_file_ops;
	if (!hidden_file.f_op->read) {
	    err = -EIO;
	    goto out_free;
	}
	hidden_file.f_mode = 3;
	hidden_file.f_flags = 0;
	hidden_file.f_count = 1;
	hidden_file.f_inode = hidden_inode;
	hidden_file.f_pos = hidden_page->offset;
	hidden_file.f_reada = 0;

	bzero((char *)page_address(hidden_page), PAGE_SIZE);
	old_fs = get_fs();
	set_fs(KERNEL_DS);
	err = hidden_file.f_op->read(hidden_inode, &hidden_file, (char *)page_address(hidden_page), PAGE_SIZE);
	set_fs(old_fs);
	if (err < 0) {
	    goto out_free;
	}
    }
#endif

    /*
     * decrypt hidden_page data onto page
     */
    page_data = page_address(page);
    hidden_page_data = page_address(hidden_page);
    crypt_decode_block(__FUNCTION__, __LINE__, (char *) hidden_page_data, (char *) page_data, CRYPT_BLOCK_SIZE, key);

    /*
     * adjust flags and wake up processes waiting on the page
     * code shamelessly stolen from generic_readpage
     */
    clear_bit(PG_locked, &page->flags);
    set_bit(PG_uptodate, &page->flags);
    wake_up(&page->wait);

#ifdef AFTER_UNLOCK_PAGE_IS_NOT_STATIC
    after_unlock_page(page);
#else
    if (clear_bit(PG_decr_after, &page->flags))
	atomic_dec(&nr_async_pages);
# ifdef SUPPORT_SWAP_INTO_CRYPT_FILES
    if (clear_bit(PG_swap_unlock_after, &page->flags))
	swap_after_unlock_page(page->swap_unlock_entry);
# endif
#endif

out_free:
    if (page_cache)
	free_page(page_cache);
    __free_page(hidden_page);	/* release_page() in mm/filemap.c */
out:
    print_exit_status(err);
    return err;
}

static int
cryptfs_bmap(struct inode *inode, int block)
{
    int err = -EINVAL;

    print_entry_location();

    if (itohi(inode)->i_op && itohi(inode)->i_op->bmap)
	err = itohi(inode)->i_op->bmap(itohi(inode), block);

    print_exit_status(err);
    return err;
}

/*******************
 * File Operations *
 *******************/

static int
cryptfs_lseek(struct inode *inode, struct file *file, off_t offset, int origin)
{
    long tmp = -1;
    int err = -EINVAL;
    struct file *hidden_file = ftohf(file);

    print_entry_location();

    if (hidden_file->f_op && hidden_file->f_op->lseek) {
	err = hidden_file->f_op->lseek(itohi(inode),
				       hidden_file,
				       offset, origin);
	file->f_pos = err;
	goto out;
    }
    /* This code is taken from the VFS code for lseek */

    /* this is the default handler if no lseek handler is present */
    switch (origin) {
    case 0:
	tmp = offset;
	break;
    case 1:
	tmp = file->f_pos + offset;
	break;
    case 2:
	if (!file->f_inode)
	    goto out;
	tmp = file->f_inode->i_size + offset;
	break;
    }
    if (tmp < 0)
	goto out;
    if (tmp != file->f_pos) {
	file->f_pos = hidden_file->f_pos = tmp;
	file->f_reada = hidden_file->f_reada = 0;
	file->f_version = hidden_file->f_version = ++event;
    }
    err = file->f_pos;

out:
    print_exit_status(err);
    return err;
}

#ifndef USE_GENERIC_READ
#error using generic read --- should not get here...
/* this is the version ported from solaris, but we're not using it */
static int
cryptfs_read(struct inode *inode, struct file *file, char *buf, int count)
{
    int err = -EINVAL;
    struct file *hidden_file = ftohf(file);
    char *hidden_buf, *current_base;
    int hidden_count, num_blocks;
    unsigned long old_fs;
    int i;
    loff_t hidden_start, hidden_end;
    loff_t start, end;
    void *key;

    print_entry_location();

    if ((key = fist_get_userpass(inode->i_sb)) == NULL) {
	err = -EACCES;
	goto out;
    }
    if (!hidden_file->f_op || !hidden_file->f_op->read)
	goto out;

    start = file->f_pos;
    end = file->f_pos + count;
    hidden_start = start & ~CRYPT_BLOCK_MASK;
    hidden_end = (end + CRYPT_BLOCK_SIZE - 1) & ~CRYPT_BLOCK_MASK;
    hidden_count = hidden_end - hidden_start;

    fist_dprint(6, "READ: hs=%d he=%d s=%d e=%d hc=%d\n",
		(int) hidden_start,
		(int) hidden_end,
		(int) start,
		(int) end,
		hidden_count
		);

    /* must be vmalloc() b/c hidden_count can be larger than 128KB */
    hidden_buf = vmalloc(hidden_count);

    old_fs = get_fs();
    set_fs(KERNEL_DS);
    hidden_file->f_pos = hidden_start;
    err = hidden_file->f_op->read(itohi(inode), hidden_file, hidden_buf, hidden_count);
    set_fs(old_fs);

    if (err < 0)
	goto out_free;

    /* we have read 0 or more bytes: adjust file pointer */
    if (hidden_file->f_pos >= end) {
	err = count;
	file->f_pos = end;
    } else {
	/* if hidden_file->f_pos < start, need to return 0 */
	if (hidden_file->f_pos < start) {
	    err = 0;
	    file->f_pos = start;
	    goto out_free;
	} else {
	    /* short read */
	    err = count - (end - hidden_file->f_pos);
	    file->f_pos = hidden_file->f_pos;
	}
    }

    /* decode chunks and copy back to user buffer */
    num_blocks = (hidden_file->f_pos - hidden_start + CRYPT_BLOCK_SIZE - 1) >> CRYPT_BLOCK_SHIFT;
    for (i = 0, current_base = hidden_buf;
	 i < num_blocks;
	 current_base += CRYPT_BLOCK_SIZE, i++)
	crypt_decode_block(__FUNCTION__, __LINE__, current_base, current_base, CRYPT_BLOCK_SIZE, key);
    memcpy_tofs(buf, hidden_buf + start - hidden_start, err);

out_free:
    vfree(hidden_buf);

out:
    print_exit_status(err);
    return err;
}
#endif /* not USE_GENERIC_READ */


#ifdef USE_GENERIC_WRITE
/*
 * Adapted from 2.1 generic_file_write
 */
static int
cryptfs_write(struct inode *inode, struct file *file, const char *buf, int count)
{
    struct inode *hidden_inode;
    struct file *hidden_file, tmp;
    struct page *page, **hash;
    unsigned long page_cache = 0;
    unsigned long pgpos, pgpos2, offset;
    unsigned long bytes, bytes_to_write, written;
    unsigned long pos, pos2;
    long status, didread, old_fs;
    char *hidden_buffer;
    void *key;

    print_entry_location();

    if ((key = fist_get_userpass(inode->i_sb)) == NULL) {
	status = -EACCES;
	goto out;
    }
    pos = file->f_pos;
    written = 0;
    status = 0;

    old_fs = get_fs();
    hidden_inode = itohi(inode);
    hidden_file = ftohf(file);
    if (hidden_file == NULL) {
	/*
	 * we are called from unmap and the private data does not exist
	 * so we have to "fabricate" one
	 */
	tmp.f_op = hidden_inode->i_op->default_file_ops;
	tmp.f_mode = 3;
	tmp.f_flags = 0;
	tmp.f_count = 1;
	tmp.f_inode = hidden_inode;
	tmp.f_reada = 0;
	hidden_file = &tmp;
    }
    hidden_buffer = kmalloc(PAGE_SIZE, GFP_KERNEL);
    if (!hidden_buffer) {
	printk("Out of memory.\n");
	status = -ENOMEM;
	goto out;
    }
    /* we don't want anybody to do updates while we write, so lock the inode */
    down(&hidden_inode->i_sem);

    if (file->f_flags & O_APPEND)
	pos = inode->i_size;

    fist_dprint(6, "WRITE: initial values: pos %ld, count %d\n", pos, count);

    /* we need to handle sparse files differently */
    pos2 = pos;
    pgpos2 = pos2 & PAGE_MASK;
    if (pos > inode->i_size)
	pos = inode->i_size;

    while (count) {
	fist_dprint(7, "WRITE: pos %ld, written %ld, count %d, buf 0x%lx\n",
		    pos, written, count, (long) buf);
	/*
	 * Try to find the page in the cache. If it isn't there,
	 * allocate a free page.
	 */
	offset = (pos & ~PAGE_MASK);
	pgpos = pos & PAGE_MASK;

	if ((bytes = PAGE_SIZE - offset) > count && pgpos >= pgpos2)
	    bytes = count;

	hash = page_hash(inode, pgpos);
	if (!(page = __find_page(inode, pgpos, *hash))) {
	    if (!page_cache) {
		page_cache = __get_free_page(GFP_KERNEL);
		if (page_cache)
		    continue;
		status = -ENOMEM;
		break;
	    }
	    page = mem_map + MAP_NR(page_cache);
	    add_to_page_cache(page, inode, pgpos, hash);
	    page_cache = 0;
	}
	/*
	 * Note: setting of the PG_locked bit is handled
	 * below the i_op->xxx interface.
	 */
	didread = 0;
    page_wait:
	wait_on_page(page);
	if (PageUptodate(page))
	    goto do_update_page;

	/*
	 * The page is not up-to-date ... if we're writing less
	 * than a full page of data, we may have to read it first.
	 * But if the page is past the current end of file, we must
	 * clear it before updating.
	 */
	if (bytes < PAGE_SIZE) {
	    if (pgpos < inode->i_size) {
		status = -EIO;
		if (didread >= 2)
		    goto done_with_page;
		status = cryptfs_readpage(inode, page);
		if (status < 0)
		    goto done_with_page;
		didread++;
		goto page_wait;
	    } else {
		/* Must clear for partial writes */
		fist_dprint(7, "WRITE: clearing page at offset 0x%x\n", pgpos);
		memset((void *) page_address(page), 0, PAGE_SIZE);
	    }
	} else if (pos < pos2) {
	    /* Must clear for sparse files */
	    fist_dprint(7, "WRITE: clearing page at offset 0x%x\n", pgpos);
	    memset((void *) page_address(page), 0, PAGE_SIZE);
	}
	/*
	 * N.B. We should defer setting PG_uptodate at least until
	 * the data is copied. A failure in i_op->updatepage() could
	 * leave the page with garbage data.
	 */
	set_bit(PG_uptodate, &page->flags);

    do_update_page:
	/* Alright, the page is there.  Now update it. */
#if 0
	status = inode->i_op->updatepage(file, page, buf,
					 offset, bytes, sync);
#endif
	fist_dprint(7, "WRITE: count %d, pos %d, offset %d, pgpos 0x%x, bytes %d\n",
		    count, pos, offset, pgpos, bytes);

	if (pgpos < pgpos2)
	    /* nothing to do, not even memcpy_fromfs */
	    goto encode_page;
	if (pgpos == pgpos2) {
	    /*
	     * this is the only interesting case,
	     * we have to shift from pos2 to pos somehow
	     */
	    pos = pos2;
	    offset = pos & ~PAGE_MASK;
	    if ((bytes = PAGE_SIZE - offset) > count)
		bytes = count;
	}
	fist_dprint(7, "WRITE: copying %ld bytes at offset %ld\n", bytes, offset);
	memcpy_fromfs((char *) (page_address(page) + offset), buf, bytes);
    encode_page:
	crypt_encode_block(__FUNCTION__, __LINE__, (char *) page_address(page), hidden_buffer, CRYPT_BLOCK_SIZE, key);

	hidden_file->f_pos = pgpos;
	if (inode->i_size < pgpos + PAGE_SIZE)
	    if (inode->i_size > pos + bytes)
		bytes_to_write = inode->i_size - pgpos;
	    else
		bytes_to_write = offset + bytes;
	else
	    bytes_to_write = PAGE_SIZE;

	fist_dprint(7, "WRITE: writing at f_pos %ld, size %ld\n", pgpos, bytes_to_write);
	/* switch to kernel space */
	set_fs(KERNEL_DS);
	status = hidden_file->f_op->write(hidden_inode, hidden_file,
					  hidden_buffer, bytes_to_write);
	/* switch back to user space */
	set_fs(old_fs);

	/* the rest of the code must not see that we are writing extra bytes */
	/* do not adjust status if only filling up holes */
	if (status > 0 && pgpos >= pgpos2)
	    if (status < offset)
		status = 0;
	    else {
		status -= offset;
		if (status > bytes)
		    status = bytes;
	    }
    done_with_page:
	__free_page(page);
	if (status < 0)
	    break;

	/* do not adjust these variables if only filling up holes */
	if (pgpos >= pgpos2) {
	    written += status;
	    count -= status;
	    buf += status;
	}
	pos += status;
    }
    file->f_pos = pos;
    if (pos > inode->i_size)
	inode->i_size = pos;

    up(&hidden_inode->i_sem);

    kfree_s(hidden_buffer, PAGE_SIZE);

    if (page_cache)
	free_page(page_cache);

    status = written ? written : status;

out:
    print_exit_status(status);
    return status;
}

#else /* not USE_GENERIC_WRITE */
#error using different write function --- should not get here...
/* this is the version ported from solaris, but we're not using it */
/* it doesn't use the page cache */
static int
cryptfs_write(struct inode *inode, struct file *file, const char *buf, int count)
{
    int error = -EPERM;
    struct inode *hidden_inode;
    struct file *hidden_file, tmp;
    int i;
    char *hidden_buf, *current_block;
    unsigned long old_fs;
    int hidden_count, bytes_read, num_blocks, first_page_bytes, bytes_to_write;
    loff_t hidden_start, hidden_end;
    loff_t start, end;
    void *key;

    print_entry_location();

    if ((key = fist_get_userpass(inode->i_sb)) == NULL) {
	err = -EACCES;
	goto out;
    }
    hidden_inode = itohi(inode);
    hidden_file = ftohf(file);
    if (hidden_file == NULL) {
	/*
	 * we are called from unmap and the private data does not exist
	 * so we have to "fabricate" one
	 */
	tmp.f_op = hidden_inode->i_op->default_file_ops;
	tmp.f_mode = 3;
	tmp.f_flags = 0;
	tmp.f_count = 1;
	tmp.f_inode = hidden_inode;
	tmp.f_reada = 0;
	hidden_file = &tmp;
    }
    fist_dprint(6, "WRITE: file->f_pos %d, file->f_flags %x, inode->i_mode %x, inode->i_size %d\n",
		(int) file->f_pos, file->f_flags, inode->i_mode, (int) inode->i_size);
    fist_dprint(6, "WRITE: hidden_file->f_pos %d, hidden_file->f_flags %x, hidden_inode->i_mode %x, hidden_inode->i_size %d\n",
		(int) hidden_file->f_pos, hidden_file->f_flags, hidden_inode->i_mode, (int) hidden_inode->i_size);

    /* we don't want anybody to do updates while we write, so lock the inode */
    down(&hidden_inode->i_sem);

    /* switch to kernel space */
    old_fs = get_fs();
    set_fs(KERNEL_DS);

    /* just in case someone tries to pull a fast one */
    if (count == 0) {
	error = 0;
	goto out;
    }
    start = file->f_pos;
    end = file->f_pos + count;
    if (file->f_flags & O_APPEND) {
	fist_dprint(6, "WRITE: turning off append flag\n");
	start += inode->i_size;
	end += inode->i_size;
    }
    hidden_start = MIN(start, inode->i_size) & ~CRYPT_BLOCK_MASK;
    first_page_bytes = MIN(start, inode->i_size) - hidden_start;
    hidden_end = (end + CRYPT_BLOCK_SIZE - 1) & ~CRYPT_BLOCK_MASK;

    ASSERT(first_page_bytes <= CRYPT_BLOCK_SIZE);

    hidden_count = hidden_end - hidden_start;
    num_blocks = hidden_count >> CRYPT_BLOCK_SHIFT;

    if (num_blocks == 1)
	first_page_bytes = CRYPT_BLOCK_SIZE;

    fist_dprint(5,
		"WRITE: hs=%d he=%d s=%d e=%d hc=%d nb=%d fpb=%d\n",
		(int) hidden_start,
		(int) hidden_end,
		(int) start,
		(int) end,
		hidden_count,
		num_blocks,
		first_page_bytes
		);
    /* must be vmalloc() b/c hidden_count can be larger than 128KB */
    hidden_buf = vmalloc(hidden_count);

    /* read first block XXX check length of file */
    hidden_file->f_pos = hidden_start;
    bytes_read = hidden_file->f_op->read(hidden_inode, hidden_file, hidden_buf, first_page_bytes);
    if (bytes_read < 0) {
	fist_dprint(4, "VOP_READ returned error - not good\n");
	error = bytes_read;
	/* XXX to be checked */
	goto out_free;
    }
    /* decrypt block read */
    crypt_decode_block(__FUNCTION__, __LINE__,
		       hidden_buf, hidden_buf, bytes_read, key);
    /* fill in with zeroes: fills in zfod pages */
    if ((start - hidden_start) - bytes_read > 0)
	memset(hidden_buf + bytes_read, 0, (start - hidden_start) - bytes_read);

    /*
     * if num_pages == 1, we already read the page... don't clobber it
     * if num_pages > 1, then we must read the last page, and decrypt it
     * completely, before clobbering it.
     * XXX: if end offset is on page boundary, we don't have to do this.
     */
    if (num_blocks > 1) {
	/* read last block XXX check length of file */
	hidden_file->f_pos = hidden_end - CRYPT_BLOCK_SIZE;
	bytes_read = hidden_file->f_op->read(hidden_inode, hidden_file,
					     hidden_buf + (hidden_count - CRYPT_BLOCK_SIZE),
					     CRYPT_BLOCK_SIZE);

	if (bytes_read < 0) {
	    fist_dprint(4, "VOP_READ returned error - not good\n");
	    error = bytes_read;
	    /* XXX to be checked */
	    goto out_free;
	}
	/* decrypt block read */
	crypt_decode_block(__FUNCTION__, __LINE__,
			   hidden_buf + (hidden_count - CRYPT_BLOCK_SIZE),
			   hidden_buf + (hidden_count - CRYPT_BLOCK_SIZE),
			   bytes_read, key);
	if (bytes_read < CRYPT_BLOCK_SIZE) {
	    /* short read: should not happen normally (unless we're extending the
	     * file) */
	    bzero((hidden_buf + hidden_count) - (CRYPT_BLOCK_SIZE - bytes_read),
		  CRYPT_BLOCK_SIZE - bytes_read);
	}
    }
    /* switch back to user space */
    set_fs(old_fs);

    /*
     * Now we are ready to write the bytes within the start/end
     * cleartext offsets in the buffers we allocated.
     */
    memcpy_fromfs(hidden_buf + (start - hidden_start), buf, count);
    for (i = 0, current_block = hidden_buf; i < num_blocks; i++, current_block += CRYPT_BLOCK_SIZE)
	/* encode block before writing */
	crypt_encode_block(__FUNCTION__, __LINE__, current_block, current_block, CRYPT_BLOCK_SIZE, inode.key);

    bytes_to_write = hidden_count;
    if (inode->i_size < hidden_end)	/* avoid extending the file
					 * unnecessarily */
	if (inode->i_size < end)
	    bytes_to_write -= hidden_end - end;
	else
	    bytes_to_write -= hidden_end - inode->i_size;

    /* switch to kernel space */
    set_fs(KERNEL_DS);

    /*
     * pass operation to hidden filesystem, and return status
     */
    hidden_file->f_pos = hidden_start;
    error = hidden_file->f_op->write(hidden_inode, hidden_file, hidden_buf, bytes_to_write);
    if (error < 0)
	goto out_free;

    if (hidden_file->f_pos < end) {
	/* incomplete write: this case is an error and should not happen */
	if (hidden_file->f_pos < start) {
	    /* if hidden_file->f_pos < start, need to return 0 */
	    error = 0;
	    /* file->f_pos = start; */ /* not really needed */
	} else {
	    error = count - (end - hidden_file->f_pos);
	    file->f_pos = hidden_file->f_pos;
	}
    } else {
	error = count;
	file->f_pos = end;
    }

    if (file->f_flags & O_APPEND) {
	file->f_pos = 0;
    }
    inode->i_size = hidden_inode->i_size;

out_free:
    vfree(hidden_buf);

out:
    /* switch back to user space */
    set_fs(old_fs);

    up(&hidden_inode->i_sem);

    print_exit_status(error);
    return (error);
}
#endif /* not USE_GENERIC_WRITE */


struct cryptfs_getdents_callback {
    void *dirent;
    struct super_block *super;
    filldir_t filldir;
};

/* copied from generic filldir in fs/readir.c */
static int
cryptfs_filldir(void *dirent, const char *name, int namlen, off_t offset, ino_t ino)
{
    struct cryptfs_getdents_callback *buf = (struct cryptfs_getdents_callback *) dirent;
    int err;
    char *decoded_name;
    int decoded_length;
    void *key;

    key = fist_get_userpass(buf->super);

    if ((decoded_length = fist_crypt_decodefilename(name, namlen, &decoded_name, key, SKIP_DOTS)) < 0)
	return 0;			/* no error, just skip the entry */

    err = buf->filldir(buf->dirent, decoded_name, decoded_length, offset, ino);
    kfree_s(decoded_name, decoded_length);
    return err;
}

static int
cryptfs_readdir(struct inode *inode, struct file *file, void *dirent, filldir_t filldir)
{
    int err = -EOPNOTSUPP;
    struct file *hidden_file = ftohf(file);
    struct cryptfs_getdents_callback buf;

    print_entry_location();

    checkinode(inode, "cryptfs_readdir");
    if (!file) {
	printk(KERN_WARNING "cryptfs_readdir: What???? file is NULL!!!\n");
	goto out;
    }
    if (hidden_file && hidden_file->f_op && hidden_file->f_op->readdir) {
	buf.dirent = dirent;
	buf.super = inode->i_sb;
	buf.filldir = filldir;
	err = hidden_file->f_op->readdir(itohi(inode), hidden_file, (void *) &buf, cryptfs_filldir);
	file->f_pos = hidden_file->f_pos;
    }
    checkinode(inode, "post cryptfs_readdir");

out:
    print_exit_status(err);
    return err;
}

static int
cryptfs_select(struct inode *inode, struct file *file, int flag, select_table * wait)
{
    int err = 0;
    struct file *hidden_file = ftohf(file);

    print_entry_location();

    if (hidden_file->f_op && hidden_file->f_op->select) {
	err = hidden_file->f_op->select(itohi(inode), hidden_file, flag, wait);
	goto out;
    }
    if (flag != SEL_EX)
	err = 1;

out:
    print_exit_status(err);
    return err;
}

static int
cryptfs_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
    int err = 0, val;
    unsigned char cbc_key[16];
    struct file *hidden_file;

    print_entry_location();

    /* check if asked for local commands */
    switch (cmd) {
#ifdef FIST_DEBUG
    case FIST_IOCTL_GET_DEBUG_VALUE:
	err = verify_area(VERIFY_WRITE, (int *) arg, sizeof(int));
	if (err)
	    goto out;
	val = fist_get_debug_value();
	put_user(val, (int *) arg);
	goto out;

    case FIST_IOCTL_SET_DEBUG_VALUE:
	err = verify_area(VERIFY_READ, (int *) arg, sizeof(int));
	if (err)
	    goto out;
	val = get_user((int *) arg);
	if (val < 0 || val > 20) {
	    err = EINVAL;
	    goto out;
	}
	fist_dprint(6, "IOCTL: got arg %d\n", val);
	fist_set_debug_value(val);
	goto out;

#endif
    case FIST_IOCTL_SET_KEY:
	err = verify_area(VERIFY_READ, (int *) arg, 16);
	if (err)
	    goto out;
	memcpy_fromfs((char *) cbc_key, (char *) arg, 16);
	fist_set_userpass(inode->i_sb, cbc_key);
	goto out;
    }

    hidden_file = ftohf(file);

    /* pass operation to hidden filesystem, and return status */
    if (hidden_file->f_op && hidden_file->f_op->ioctl)
	err = hidden_file->f_op->ioctl(itohi(inode), hidden_file, cmd, arg);

out:
    print_exit_status(err);
    return err;
}

static int
cryptfs_mmap(struct inode *inode, struct file *file, struct vm_area_struct *vma)
{
    int err = -ENODEV;
    struct file *hidden_file = ftohf(file);
    struct inode *hidden_inode = itohi(inode);
    struct vm_area_struct *hidden_vma;

    print_entry_location();

    fist_dprint(6, "MMAP1: inode 0x%x, hidden_inode 0x%x, inode->i_count %d, hidden_inode->i_count %d\n",
		(int) inode, (int) hidden_inode, (int) inode->i_count, (int) hidden_inode->i_count);

    if (!hidden_file->f_op || !hidden_file->f_op->mmap)
	goto out;

    /*
     * Most of this code comes straight from generic_mmap
     */
    if ((vma->vm_flags & VM_SHARED) && (vma->vm_flags & VM_MAYWRITE)) {
	vma->vm_ops = &cryptfs_shared_vm_ops;
	/* share_page() can only guarantee proper page sharing if * the offsets
	 * are all page aligned. */
	if (vma->vm_offset & (PAGE_SIZE - 1)) {
	    err = -EINVAL;
	    goto out;
	}
    } else {
	vma->vm_ops = &cryptfs_private_vm_ops;
	if (vma->vm_offset & (hidden_inode->i_sb->s_blocksize - 1)) {
	    err = -EINVAL;
	    goto out;
	}
    }
    if (!inode->i_sb || !S_ISREG(inode->i_mode))
	return -EACCES;
    if (!hidden_inode->i_op || !hidden_inode->i_op->readpage)
	return -ENOEXEC;
    UPDATE_ATIME(inode)
	vma->vm_inode = inode;
    inode->i_count++;

    hidden_vma = kmalloc(sizeof(struct vm_area_struct), GFP_KERNEL);
    if (!hidden_vma) {
	printk("MMAP: Out of memory\n");
	err = -ENOMEM;
	goto out;
    }
    memcpy(hidden_vma, vma, sizeof(struct vm_area_struct));
    vmatohvma(vma) = hidden_vma;

    err = hidden_file->f_op->mmap(hidden_inode, hidden_file, hidden_vma);
    iput(hidden_inode);

out:
    fist_dprint(6, "MMAP2: inode 0x%x, hidden_inode 0x%x, inode->i_count %d, hidden_inode->i_count %d\n",
		(int) inode, (int) hidden_inode, (int) inode->i_count, (int) hidden_inode->i_count);
    print_exit_status(err);
    return err;
}

static void
cryptfs_unmap(struct vm_area_struct *vma, unsigned long start, size_t len)
{
    struct vm_area_struct *hidden_vma = vmatohvma(vma);

    print_entry_location();
    fist_dprint(6, "UNMAP1: inode 0x%x, hidden_inode 0x%x, inode->i_count %d, hidden_inode->i_count %d\n",
		(int) vma->vm_inode, (int) hidden_vma->vm_inode,
		(int) vma->vm_inode->i_count, (int) hidden_vma->vm_inode->i_count);

    if (vma->vm_ops->sync)
	vma->vm_ops->sync(vma, start, len, MS_ASYNC);
    if (hidden_vma->vm_ops->unmap)
	hidden_vma->vm_ops->unmap(hidden_vma, start, len);
    kfree_s(hidden_vma, sizeof(struct vm_area_struct));

    fist_dprint(6, "UNMAP2: inode 0x%x, hidden_inode 0x%x, inode->i_count %d, hidden_inode->i_count %d\n",
		(int) vma->vm_inode, (int) hidden_vma->vm_inode,
		(int) vma->vm_inode->i_count, (int) hidden_vma->vm_inode->i_count);
    print_exit_location();
}

static void
cryptfs_vm_open(struct vm_area_struct *vma)
{
    struct vm_area_struct *hidden_vma = vmatohvma(vma), *hidden_vma2;

    print_entry_location();
    fist_dprint(6, "VM_OPEN: inode 0x%x, hidden_inode 0x%x, inode->i_count %d, hidden_inode->i_count %d\n",
		(int) vma->vm_inode, (int) hidden_vma->vm_inode,
		(int) vma->vm_inode->i_count, (int) hidden_vma->vm_inode->i_count);

    if (hidden_vma->vm_ops->open)
	hidden_vma->vm_ops->open(hidden_vma);

    /* We need to duplicate the private data */
    hidden_vma2 = kmalloc(sizeof(struct vm_area_struct), GFP_KERNEL);
    /* XXX:EZK where is hidden_vma2 being free'ed ??? */
    if (!hidden_vma2) {
	printk("MMAP: Out of memory\n");
	goto out;
    }
    memcpy(hidden_vma2, hidden_vma, sizeof(struct vm_area_struct));
    vmatohvma(vma) = hidden_vma2;

out:
    fist_dprint(6, "UNMAP2: inode 0x%x, hidden_inode 0x%x, inode->i_count %d, hidden_inode->i_count %d\n",
		(int) vma->vm_inode, (int) hidden_vma->vm_inode,
		(int) vma->vm_inode->i_count, (int) hidden_vma->vm_inode->i_count);
    print_exit_location();
}

static int
cryptfs_open(struct inode *inode, struct file *file)
{
    int err = 0;
    struct file *hidden_file;
    struct inode *hidden_inode;

    print_entry_location();

    if (!file) {
	printk(KERN_WARNING "cryptfs_open: Doh! file is NULL...\n");
	err = -EINVAL;
	goto out;
    }
    file->private_data = NULL;

    fist_dprint(6, "OPEN: file->f_pos %d, file->f_flags %x, inode->i_mode %x, inode->i_size %d\n",
		(int) file->f_pos, file->f_flags, inode->i_mode, (int) inode->i_size);

    hidden_file = get_empty_filp();
    hidden_inode = itohi(inode);

    if (hidden_file) {
	fist_dprint(7, "cryptfs_open: got empty hidden_file\n");
	file->private_data = hidden_file;
	if (!file->f_inode)
	    printk(KERN_WARNING "cryptfs_open: what is happenning? - f_inode is NULL!!!\n");
	if (file->f_inode != inode)
	    printk(KERN_WARNING "cryptfs_open: what is happenning? - f_inode is not the same as inode!!!\n");
	hidden_file->f_inode = hidden_inode;

	fist_dprint(7, "cryptfs_open: got f_inode\n");

	/* we need read access no matter what */
	hidden_file->f_mode = file->f_mode | FMODE_READ;

	hidden_file->f_flags = file->f_flags;
	if (file->f_flags & O_APPEND) {
	    fist_dprint(5, "***WARNING*** file is opened in append-only mode!!!\n");
	    hidden_file->f_flags &= ~O_APPEND;	/* turn off O_APPEND flag */
	}
	hidden_file->f_pos = file->f_pos;
	hidden_file->f_reada = file->f_reada;
	fist_dprint(8, "OPEN: hidden_inode->i_op 0x%x\n", hidden_inode->i_op);
	if (hidden_inode->i_op)
	    hidden_file->f_op = hidden_inode->i_op->default_file_ops;
	else
	    hidden_file->f_op = NULL;

	fist_dprint(6, "OPEN1: inode->i_count %d, hidden_inode->i_count %d, file->f_count %d, hidden_file->f_count %d\n", inode->i_count, hidden_inode->i_count, file->f_count, hidden_file->f_count);

	if (hidden_file->f_op && hidden_file->f_op->open)
	    err = hidden_file->f_op->open(hidden_inode, hidden_file);
	fist_dprint(6, "OPEN2: inode->i_count %d, hidden_inode->i_count %d, file->f_count %d, hidden_file->f_count %d\n", inode->i_count, hidden_inode->i_count, file->f_count, hidden_file->f_count);
	hidden_file->f_flags &= ~(O_CREAT | O_EXCL | O_NOCTTY | O_TRUNC);

	if (err) {
	    fput(hidden_file, hidden_inode);
	    printk(KERN_WARNING "cryptfs_open: error(%d)\n", err);
	}
    } else
	err = -ENFILE;
    checkinode(inode, "post cryptfs_open");

out:
    print_exit_status(err);
    return err;
}

static void
cryptfs_release(struct inode *inode, struct file *file)
{
    struct file *hidden_file = ftohf(file);
    struct inode *hidden_inode = itohi(inode);

    print_entry_location();

    checkinode(inode, "cryptfs_release");
    if (file) {
	fist_dprint(6, "RELEASE1: inode->i_count %d, hidden_inode->i_count %d, file->f_count %d, hidden_file->f_count %d\n", inode->i_count, hidden_inode->i_count, file->f_count, hidden_file->f_count);
	if (file->f_count == 1) {
	    hidden_inode->i_count++;
	    fput(hidden_file, itohi(inode));
	}
	fist_dprint(6, "RELEASE2: inode->i_count %d, hidden_inode->i_count %d, file->f_count %d, hidden_file->f_count %d\n", inode->i_count, hidden_inode->i_count, file->f_count, hidden_file->f_count);
    }
    checkinode(inode, "post cryptfs_release");

    print_exit_location();
}

static int
cryptfs_fsync(struct inode *inode, struct file *file)
{
    int err = -EINVAL;
    struct file *hidden_file = ftohf(file);

    print_entry_location();

    if (hidden_file->f_op && hidden_file->f_op->fsync)
	err = hidden_file->f_op->fsync(itohi(inode), hidden_file);

    print_exit_status(err);
    return err;
}

static int
cryptfs_fasync(struct inode *inode, struct file *file, int flag)
{
    int err = 0;
    struct file *hidden_file = ftohf(file);

    print_entry_location();

    if (hidden_file->f_op && hidden_file->f_op->fasync)
	err = hidden_file->f_op->fasync(itohi(inode), hidden_file, flag);

    print_exit_status(err);
    return err;
}

static int
cryptfs_parse_options(struct super_block *sb, char *options)
{
    int err = -1;

    print_entry_location();

    while (*options) {
	if (!strncmp("dir=", options, 4)) {
	    fist_dprint(4, "cryptfs: using directory: %s\n", options + 4);
	    err = _namei(options + 4, NULL, 1, &stopd(sb)->s_rootinode);
	} else {
	    printk(KERN_WARNING "cryptfs: unrecognized options '%s'\n", options);
	}
	while (*options && *options != ',')
	    options++;
    }

    print_exit_status(err);
    return err;
}

#if 0
/*
 * cryptfs needs to be able to 'automount' filesystems that are underlying whatever
 * is being looped back... i.e. my /home has /home/dot1 and /home/dot2 mounted
 * beneath it, so using cryptfs to loopback mount /home has to deal with the filesystem
 * from /, /home/dot1 and /home/dot2.  This whole mess is dealt with by automagically
 * creating a new superblock for the superblocks found as we're traversing the fs.
 * It would be nice to use different inodes for the underlying inodes, but that would
 * be difficult.
 */
static int
cryptfs_automount(struct super_block *sb)
{
    int err = -EMFILE;
    struct super_block *s;
    kdev_t dev;

    print_entry_location();

    if (!(dev = get_unnamed_dev())) {
	printk(KERN_ERROR "cryptfs: no unnamed devices\n");
	goto out;
    }
    for (s = 0 + super_blocks;; s++) {
	if (s >= NR_SUPER + super_blocks) {
	    printk(KERN_ERROR "cryptfs: no free super blocks\n");
	    goto out;
	}
	if (!(s->s_dev))
	    break;
    }

    s->s_dev = dev;
    s->s_flags = sb->s_flags;
    s->s_covered = NULL;
    s->s_rd_only = 0;
    s->s_dirt = 0;
    s->s_type = &cryptfs_fs_type;
    err = 0;

out:
    print_exit_status(err);
    return err;
}
#endif

/* mount a file system */
struct super_block *
cryptfs_read_super(struct super_block *sb, void *raw_data, int silent)
{
    int ret;

    print_entry_location();

    MOD_INC_USE_COUNT;
    lock_super(sb);
    if (!raw_data) {
	printk(KERN_WARNING "cryptfs_read_super: missing data argument\n");
	unlock_super(sb);
	MOD_DEC_USE_COUNT;
	print_exit_location();
	return NULL;
    }
    /*
     * Allocate superblock private data
     */
    stopd(sb) = kmalloc(sizeof(struct cryptfs_sb_info), GFP_KERNEL);
    stohs(sb) = NULL;
    stopd(sb)->s_rootinode = NULL;
    bzero((char *) stopd(sb)->fwi_user, sizeof(fist_key_t *) * FIST_HASH_SIZE);

    ret = cryptfs_parse_options(sb, raw_data);
    if (ret) {
	kfree_s(stopd(sb), sizeof(struct cryptfs_sb_info));
	unlock_super(sb);
	printk(KERN_WARNING "cryptfs_read_super: namei failed? (ret = %d)\n", ret);
	MOD_DEC_USE_COUNT;
	print_exit_location();
	return NULL;
    }
    stohs(sb) = stopd(sb)->s_rootinode->i_sb;

    /*
     * This is a huge hack to avoid making public a whole bunch of
     * functions in mm/filemap.c. Basically, we copy the whole generic structure
     * into ours and then change only the function(s) we're interested in.
     *
     * XXX This hack needs to be reviewed for 2.1 and later
     */
    memcpy(&cryptfs_shared_vm_ops, &file_shared_mmap, sizeof(struct vm_operations_struct));
    memcpy(&cryptfs_private_vm_ops, &file_private_mmap, sizeof(struct vm_operations_struct));
    cryptfs_shared_vm_ops.unmap = cryptfs_unmap;
    cryptfs_shared_vm_ops.open = cryptfs_vm_open;
    cryptfs_private_vm_ops.unmap = cryptfs_unmap;
    cryptfs_private_vm_ops.open = cryptfs_vm_open;

    unlock_super(sb);
    sb->s_op = &cryptfs_sops;
    sb->s_blocksize_bits = CRYPT_BLOCK_SHIFT;
    sb->s_blocksize = CRYPT_BLOCK_SIZE;

    cryptfs_interpose(stopd(sb)->s_rootinode, &sb->s_mounted, sb);
    if (!sb->s_mounted) {
	kfree_s(stopd(sb), sizeof(struct cryptfs_sb_info));
	printk(KERN_WARNING "cryptfs_read_super: iget failed\n");
	iput(stopd(sb)->s_rootinode);
	MOD_DEC_USE_COUNT;
	print_exit_location();
	return NULL;
    }
    /*
     * For the special case of the root inode, we need an extra reference
     */
    stopd(sb)->s_rootinode->i_count++;

    print_exit_location();
    return sb;
}

/* unmount a file system */
void
cryptfs_put_super(struct super_block *sb)
{
    print_entry_location();

    lock_super(sb);
    iput(stopd(sb)->s_rootinode);
    sb->s_dev = 0;
    if (stopd(sb)) {
	/* free user key hash table */
	fist_free_userpass(sb);
	kfree_s(stopd(sb), sizeof(struct cryptfs_sb_info));
    }
    unlock_super(sb);
    MOD_DEC_USE_COUNT;

    print_exit_location();
}

/*----*/

static struct super_operations cryptfs_sops =
{
    NULL,				/* read inode */
    cryptfs_notify_change,	/* notify change */
    NULL,				/* write inode */
    cryptfs_put_inode,		/* put inode */
    cryptfs_put_super,		/* put superblock */
    NULL,				/* write superblock */
    cryptfs_statfs,		/* stat filesystem */
    NULL
};

static struct file_operations cryptfs_fops =
{
    cryptfs_lseek,		/* lseek */
#ifdef USE_GENERIC_READ
    generic_file_read,		/* read */
#else /* not USE_GENERIC_READ */
    cryptfs_read,			/* read */
#endif /* not USE_GENERIC_READ */
    cryptfs_write,		/* write */
    cryptfs_readdir,		/* readdir */
    cryptfs_select,		/* select */
    cryptfs_ioctl,		/* ioctl */
    cryptfs_mmap,			/* mmap */
    cryptfs_open,			/* open */
    cryptfs_release,		/* release */
    cryptfs_fsync,		/* fsync */
    cryptfs_fasync,		/* fasync */
    NULL,				/* check_media_change */
    NULL				/* revalidate */
};

static struct inode_operations cryptfs_iops =
{
    &cryptfs_fops,		/* default file operations */
    cryptfs_create,		/* create */
    cryptfs_lookup,		/* lookup */
    cryptfs_link,			/* link */
    cryptfs_unlink,		/* unlink */
    cryptfs_symlink,		/* symlink */
    cryptfs_mkdir,		/* mkdir */
    cryptfs_rmdir,		/* rmdir */
    cryptfs_mknod,		/* mknod */
    cryptfs_rename,		/* rename */
    cryptfs_readlink,		/* readlink */
    cryptfs_follow_link,		/* follow_link */
    cryptfs_readpage,		/* readpage */
    NULL,				/* writepage */
    cryptfs_bmap,			/* bmap */
    cryptfs_truncate,		/* truncate */
    NULL,				/* permission */
    NULL				/* smap */
};

static struct file_system_type cryptfs_fs_type =
{
    cryptfs_read_super, "cryptfs", 0, NULL
};

int
init_cryptfs_fs(void)
{
    return register_filesystem(&cryptfs_fs_type);
}

/* Every kernel module contains stuff like this. */

#ifdef MODULE
int
init_module(void)
{
    int status;

    printk("Inserting module cryptfs version $Id: inode.c,v 1.1.1.1 1998/11/05 21:06:44 ezk Exp $\n");
    if ((status = init_cryptfs_fs()) == 0)
	register_symtab(0);
    return status;
}

void
cleanup_module(void)
{
    printk("Removing module cryptfs version $Id: inode.c,v 1.1.1.1 1998/11/05 21:06:44 ezk Exp $\n");
    unregister_filesystem(&cryptfs_fs_type);
}
#endif /* MODULE */

/*
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
