/*
 * This Java version of Eric Young's code created from the
 * original source by Jeremy Allison. <jra@cygnus.com>.
 * Version 1.0.
 */

/* Copyright (C) 1995 Eric Young (eay@mincom.oz.au)
 * All rights reserved.
 *
 * This file is part of an SSL implementation written
 * by Eric Young (eay@mincom.oz.au).
 * The implementation was written so as to conform with Netscapes SSL
 * specification.  This library and applications are
 * FREE FOR COMMERCIAL AND NON-COMMERCIAL USE
 * as long as the following conditions are aheared to.
 *
 * Copyright remains Eric Young's, and as such any Copyright notices in
 * the code are not to be removed.  If this code is used in a product,
 * Eric Young should be given attribution as the author of the parts used.
 * This can be in the form of a textual message at program startup or
 * in documentation (online or textual) provided with the package.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    This product includes software developed by Eric Young (eay@mincom.oz.au)
 *
 * THIS SOFTWARE IS PROVIDED BY ERIC YOUNG ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * The licence and distribution terms for any publically available version or
 * derivative of this code cannot be changed.  i.e. this code cannot simply be
 * copied and put under another distribution licence
 * [including the GNU Public Licence.]
 */

/**
 * Int32Manipulator class.
 * Provides byte manipulation functions used by the Des and
 * TripleDes code.
 *
 * Written by Jeremy Allison (jra@cygnus.com) based on
 * C code from Eric Young (eay@mincom.oz.au).
 */
class Int32Manipulator {
  /**
   *
   * Convert 4 bytes from a byte array to a 32-bit int 
   * (read from the array as little-endian format).
   * Returns the 32 bit int.
   * @param b             Byte array to convert from.
   * @param start         offset in byte array to begin.
   *
   */
  public static int bytes_to_int(byte [] b, int start) {
    return (((int)b[start]) & 0xff)|
      ((((int)b[start+1]) & 0xff) << 8) |
      ((((int)b[start+2]) & 0xff) << 16) |
      ((((int)b[start+3]) & 0xff) << 24);
  }

  /**
   *
   * Write a 32-bit int into 4 bytes of a byte array 
   * (write into the array as little-endian format).
   * @param bytes        Byte aray to write into.
   * @param offset       Offset into array to begin.
   * @param val          32-bit int to write.
   *
   */
  public static void set_int(byte [] bytes, int offset, int val) {
    bytes[offset++] = (byte)(val & 0xff);
    bytes[offset++] = (byte)((val >>> 8) & 0xff);
    bytes[offset++] = (byte)((val >>> 16) & 0xff);
    bytes[offset++] = (byte)((val >>> 24) & 0xff);
  }

  /**
   * c2ln. Internal routine used by Des code.
   *
   */
  /*
     #define c2ln(c,l1,l2,n)	{ \
     c+=n; \
     l1=l2=0; \
     switch (n) { \
     case 8: l2 =((DES_LONG)(*(--(c))))<<24L; \
     case 7: l2|=((DES_LONG)(*(--(c))))<<16L; \
     case 6: l2|=((DES_LONG)(*(--(c))))<< 8L; \
     case 5: l2|=((DES_LONG)(*(--(c))));     \
     case 4: l1 =((DES_LONG)(*(--(c))))<<24L; \
     case 3: l1|=((DES_LONG)(*(--(c))))<<16L; \
     case 2: l1|=((DES_LONG)(*(--(c))))<< 8L; \
     case 1: l1|=((DES_LONG)(*(--(c))));     \
     } \
     }
     */

  public static int c2ln(byte [] input, int offset, int length, int [] ref_to_in01) {
    int orig_length = length;
    ref_to_in01[0] = 0;
    ref_to_in01[1] = 0;
    switch (length) {
    case 8:
      ref_to_in01[1] = (((int)input[offset + (--length)]) & 0xff) << 24;
    case 7:
      ref_to_in01[1] |= (((int)input[offset + (--length)]) & 0xff) << 16;
    case 6:
      ref_to_in01[1] |= (((int)input[offset + (--length)]) & 0xff) << 8;
    case 5:
      ref_to_in01[1] |= ((int)input[offset + (--length)]) & 0xff;
    case 4:
      ref_to_in01[0] = (((int)input[offset + (--length)]) & 0xFF) << 24;
    case 3:
      ref_to_in01[0] |= (((int)input[offset + (--length)]) & 0xff) << 16;
    case 2:
      ref_to_in01[0] |= (((int)input[offset + (--length)]) & 0xff) << 8;
    case 1:
      ref_to_in01[0] |= ((int)input[offset + (--length)]) & 0xff;
    }
    return orig_length;
  }

  /**
   * l2cn. Internal routine used by Des Code.
   *
   */
  /*
    #define l2cn(l1,l2,c,n)	{ \
    c+=n; \
    switch (n) { \
    case 8: *(--(c))=(unsigned char)(((l2)>>24L)&0xff); \
    case 7: *(--(c))=(unsigned char)(((l2)>>16L)&0xff); \
    case 6: *(--(c))=(unsigned char)(((l2)>> 8L)&0xff); \
    case 5: *(--(c))=(unsigned char)(((l2)     )&0xff); \
    case 4: *(--(c))=(unsigned char)(((l1)>>24L)&0xff); \
    case 3: *(--(c))=(unsigned char)(((l1)>>16L)&0xff); \
    case 2: *(--(c))=(unsigned char)(((l1)>> 8L)&0xff); \
    case 1: *(--(c))=(unsigned char)(((l1)     )&0xff); \
    } \
    }
    */
  public static int l2cn(byte [] output, int offset, int length, int out0, int out1) {
    int orig_length = length;
    switch(length) {
    case 8:
      output[offset + (--length)] = (byte)((out1>>>24)&0xff);
    case 7:
      output[offset + (--length)] = (byte)((out1>>>16)&0xff);
    case 6:
      output[offset + (--length)] = (byte)((out1>>>8)&0xff);
    case 5:
      output[offset + (--length)] = (byte)(out1 & 0xff);
    case 4:
      output[offset + (--length)] = (byte)((out0>>>24)&0xff);
    case 3:
      output[offset + (--length)] = (byte)((out0>>>16)&0xff);
    case 2:
      output[offset + (--length)] = (byte)((out0>>>8)&0xff);
    case 1:
      output[offset + (--length)] = (byte)(out0 & 0xff);
    }
    return orig_length;
  }

  /**
   * PERM_OP. Internal routine used by Des code.
   *
   */
  /*
   * Do a PERM_OP. Defined as ..
   *
   #define PERM_OP(a,b,t,n,m) ((t)=((((a)>>(n))^(b))&(m)),\
   (b)^=(t),\
   (a)^=((t)<<(n)))
   */
  public static void PERM_OP(int [] ref_to_a, int [] ref_to_b, int tmp, int n, int m) {
    int a = ref_to_a[0];
    int b = ref_to_b[0];
    tmp = ((a>>>n)^b)&m;
    b ^= tmp;
    a ^= tmp<<n;
    ref_to_a[0] = a;
    ref_to_b[0] = b;
  }

  /**
   * HPERM_OP. Internal routine used by Des code.
   *
   */

  /*
   * Do a HPERM_OP. Defined as ..
   #define HPERM_OP(a,t,n,m) ((t)=((((a)<<(16-(n)))^(a))&(m)),\
   (a)=(a)^(t)^(t>>(16-(n))))

   */
  public static void HPERM_OP(int [] ref_to_a, int tmp, int n, int m) {
    int a = ref_to_a[0];
    tmp = ((a<<(16-n))^a)&m;
    a = a^tmp^(tmp>>>(16-n));
    ref_to_a[0] = a;
  }
}

/**
 * Class DesKey.
 * Create a key object used by the Des and TripleDes code.
 * DesKey objects can be created from a String object, or from
 * a byte array containing 8 bytes.
 * An optional check is done for weak keys.
 * 
 * Written by Jeremy Allison (jra@cygnus.com) based on
 * C code from Eric Young (eay@mincom.oz.au).
 */

class DesKey {

  private byte [] keysced_ = new byte [128];
  /* True if key was weak */
  private boolean is_weak_key_;
  private boolean is_parity_ok_;

  /**
   * Function to return the internal byte representation of
   * the DesKey. Used internally by the Des and TripleDes
   * classes.
   * Returns a byte [].
   */
  public byte [] get_keysced() { return keysced_; }

  /**
   * Function to check if a DesKey is weak.
   * returns a boolean, true if this is a weak key.
   */
  public boolean is_weak_key() { return is_weak_key_; }

  /**
   * Function to check is the DesKey parity is correct.
   * returns a boolean, true if the parity is correct.
   */
  public boolean is_parity_ok() { return is_parity_ok_; }

  private static final byte[] odd_parity_array = {
    (byte)1,  (byte)1,  (byte)2,  (byte)2, (byte)4,  (byte)4,  (byte)7,  (byte)7,
    (byte)8,  (byte)8, (byte)11, (byte)11, (byte)13, (byte)13, (byte)14, (byte)14,
    (byte)16, (byte)16, (byte)19, (byte)19, (byte)21, (byte)21, (byte)22, (byte)22,
    (byte)25, (byte)25, (byte)26, (byte)26, (byte)28, (byte)28, (byte)31, (byte)31,
    (byte)32, (byte)32, (byte)35, (byte)35, (byte)37, (byte)37, (byte)38, (byte)38,
    (byte)41, (byte)41, (byte)42, (byte)42, (byte)44, (byte)44, (byte)47, (byte)47,
    (byte)49, (byte)49, (byte)50, (byte)50, (byte)52, (byte)52, (byte)55, (byte)55,
    (byte)56, (byte)56, (byte)59, (byte)59, (byte)61, (byte)61, (byte)62, (byte)62,
    (byte)64, (byte)64, (byte)67, (byte)67, (byte)69, (byte)69, (byte)70, (byte)70,
    (byte)73, (byte)73, (byte)74, (byte)74, (byte)76, (byte)76, (byte)79, (byte)79,
    (byte)81, (byte)81, (byte)82, (byte)82, (byte)84, (byte)84, (byte)87, (byte)87,
    (byte)88, (byte)88, (byte)91, (byte)91, (byte)93, (byte)93, (byte)94, (byte)94,
    (byte)97, (byte)97, (byte)98, (byte)98,(byte)100,(byte)100,(byte)103,(byte)103,
    (byte)104,(byte)104,(byte)107,(byte)107,(byte)109,(byte)109,(byte)110,(byte)110,
    (byte)112,(byte)112,(byte)115,(byte)115,(byte)117,(byte)117,(byte)118,(byte)118,
    (byte)121,(byte)121,(byte)122,(byte)122,(byte)124,(byte)124,(byte)127,(byte)127,
    (byte)128,(byte)128,(byte)131,(byte)131,(byte)133,(byte)133,(byte)134,(byte)134,
    (byte)137,(byte)137,(byte)138,(byte)138,(byte)140,(byte)140,(byte)143,(byte)143,
    (byte)145,(byte)145,(byte)146,(byte)146,(byte)148,(byte)148,(byte)151,(byte)151,
    (byte)152,(byte)152,(byte)155,(byte)155,(byte)157,(byte)157,(byte)158,(byte)158,
    (byte)161,(byte)161,(byte)162,(byte)162,(byte)164,(byte)164,(byte)167,(byte)167,
    (byte)168,(byte)168,(byte)171,(byte)171,(byte)173,(byte)173,(byte)174,(byte)174,
    (byte)176,(byte)176,(byte)179,(byte)179,(byte)181,(byte)181,(byte)182,(byte)182,
    (byte)185,(byte)185,(byte)186,(byte)186,(byte)188,(byte)188,(byte)191,(byte)191,
    (byte)193,(byte)193,(byte)194,(byte)194,(byte)196,(byte)196,(byte)199,(byte)199,
    (byte)200,(byte)200,(byte)203,(byte)203,(byte)205,(byte)205,(byte)206,(byte)206,
    (byte)208,(byte)208,(byte)211,(byte)211,(byte)213,(byte)213,(byte)214,(byte)214,
    (byte)217,(byte)217,(byte)218,(byte)218,(byte)220,(byte)220,(byte)223,(byte)223,
    (byte)224,(byte)224,(byte)227,(byte)227,(byte)229,(byte)229,(byte)230,(byte)230,
    (byte)233,(byte)233,(byte)234,(byte)234,(byte)236,(byte)236,(byte)239,(byte)239,
    (byte)241,(byte)241,(byte)242,(byte)242,(byte)244,(byte)244,(byte)247,(byte)247,
    (byte)248,(byte)248,(byte)251,(byte)251,(byte)253,(byte)253,(byte)254,(byte)254
  };

  private static final boolean shifts2[] = {
    false, false, true, true,
    true, true, true, true,
    false, true, true, true,
    true, true, true, false
  };

  private static final int des_skb[][] = {
    {
      /* for C bits (numbered as per FIPS 46) 1 2 3 4 5 6 */
      0x00000000,0x00000010,0x20000000,0x20000010,
      0x00010000,0x00010010,0x20010000,0x20010010,
      0x00000800,0x00000810,0x20000800,0x20000810,
      0x00010800,0x00010810,0x20010800,0x20010810,
      0x00000020,0x00000030,0x20000020,0x20000030,
      0x00010020,0x00010030,0x20010020,0x20010030,
      0x00000820,0x00000830,0x20000820,0x20000830,
      0x00010820,0x00010830,0x20010820,0x20010830,
      0x00080000,0x00080010,0x20080000,0x20080010,
      0x00090000,0x00090010,0x20090000,0x20090010,
      0x00080800,0x00080810,0x20080800,0x20080810,
      0x00090800,0x00090810,0x20090800,0x20090810,
      0x00080020,0x00080030,0x20080020,0x20080030,
      0x00090020,0x00090030,0x20090020,0x20090030,
      0x00080820,0x00080830,0x20080820,0x20080830,
      0x00090820,0x00090830,0x20090820,0x20090830,
    },{
      /* for C bits (numbered as per FIPS 46) 7 8 10 11 12 13 */
      0x00000000,0x02000000,0x00002000,0x02002000,
      0x00200000,0x02200000,0x00202000,0x02202000,
      0x00000004,0x02000004,0x00002004,0x02002004,
      0x00200004,0x02200004,0x00202004,0x02202004,
      0x00000400,0x02000400,0x00002400,0x02002400,
      0x00200400,0x02200400,0x00202400,0x02202400,
      0x00000404,0x02000404,0x00002404,0x02002404,
      0x00200404,0x02200404,0x00202404,0x02202404,
      0x10000000,0x12000000,0x10002000,0x12002000,
      0x10200000,0x12200000,0x10202000,0x12202000,
      0x10000004,0x12000004,0x10002004,0x12002004,
      0x10200004,0x12200004,0x10202004,0x12202004,
      0x10000400,0x12000400,0x10002400,0x12002400,
      0x10200400,0x12200400,0x10202400,0x12202400,
      0x10000404,0x12000404,0x10002404,0x12002404,
      0x10200404,0x12200404,0x10202404,0x12202404,
    },{
      /* for C bits (numbered as per FIPS 46) 14 15 16 17 19 20 */
      0x00000000,0x00000001,0x00040000,0x00040001,
      0x01000000,0x01000001,0x01040000,0x01040001,
      0x00000002,0x00000003,0x00040002,0x00040003,
      0x01000002,0x01000003,0x01040002,0x01040003,
      0x00000200,0x00000201,0x00040200,0x00040201,
      0x01000200,0x01000201,0x01040200,0x01040201,
      0x00000202,0x00000203,0x00040202,0x00040203,
      0x01000202,0x01000203,0x01040202,0x01040203,
      0x08000000,0x08000001,0x08040000,0x08040001,
      0x09000000,0x09000001,0x09040000,0x09040001,
      0x08000002,0x08000003,0x08040002,0x08040003,
      0x09000002,0x09000003,0x09040002,0x09040003,
      0x08000200,0x08000201,0x08040200,0x08040201,
      0x09000200,0x09000201,0x09040200,0x09040201,
      0x08000202,0x08000203,0x08040202,0x08040203,
      0x09000202,0x09000203,0x09040202,0x09040203,
    },{
      /* for C bits (numbered as per FIPS 46) 21 23 24 26 27 28 */
      0x00000000,0x00100000,0x00000100,0x00100100,
      0x00000008,0x00100008,0x00000108,0x00100108,
      0x00001000,0x00101000,0x00001100,0x00101100,
      0x00001008,0x00101008,0x00001108,0x00101108,
      0x04000000,0x04100000,0x04000100,0x04100100,
      0x04000008,0x04100008,0x04000108,0x04100108,
      0x04001000,0x04101000,0x04001100,0x04101100,
      0x04001008,0x04101008,0x04001108,0x04101108,
      0x00020000,0x00120000,0x00020100,0x00120100,
      0x00020008,0x00120008,0x00020108,0x00120108,
      0x00021000,0x00121000,0x00021100,0x00121100,
      0x00021008,0x00121008,0x00021108,0x00121108,
      0x04020000,0x04120000,0x04020100,0x04120100,
      0x04020008,0x04120008,0x04020108,0x04120108,
      0x04021000,0x04121000,0x04021100,0x04121100,
      0x04021008,0x04121008,0x04021108,0x04121108,
    },{
      /* for D bits (numbered as per FIPS 46) 1 2 3 4 5 6 */
      0x00000000,0x10000000,0x00010000,0x10010000,
      0x00000004,0x10000004,0x00010004,0x10010004,
      0x20000000,0x30000000,0x20010000,0x30010000,
      0x20000004,0x30000004,0x20010004,0x30010004,
      0x00100000,0x10100000,0x00110000,0x10110000,
      0x00100004,0x10100004,0x00110004,0x10110004,
      0x20100000,0x30100000,0x20110000,0x30110000,
      0x20100004,0x30100004,0x20110004,0x30110004,
      0x00001000,0x10001000,0x00011000,0x10011000,
      0x00001004,0x10001004,0x00011004,0x10011004,
      0x20001000,0x30001000,0x20011000,0x30011000,
      0x20001004,0x30001004,0x20011004,0x30011004,
      0x00101000,0x10101000,0x00111000,0x10111000,
      0x00101004,0x10101004,0x00111004,0x10111004,
      0x20101000,0x30101000,0x20111000,0x30111000,
      0x20101004,0x30101004,0x20111004,0x30111004,
    },{
      /* for D bits (numbered as per FIPS 46) 8 9 11 12 13 14 */
      0x00000000,0x08000000,0x00000008,0x08000008,
      0x00000400,0x08000400,0x00000408,0x08000408,
      0x00020000,0x08020000,0x00020008,0x08020008,
      0x00020400,0x08020400,0x00020408,0x08020408,
      0x00000001,0x08000001,0x00000009,0x08000009,
      0x00000401,0x08000401,0x00000409,0x08000409,
      0x00020001,0x08020001,0x00020009,0x08020009,
      0x00020401,0x08020401,0x00020409,0x08020409,
      0x02000000,0x0A000000,0x02000008,0x0A000008,
      0x02000400,0x0A000400,0x02000408,0x0A000408,
      0x02020000,0x0A020000,0x02020008,0x0A020008,
      0x02020400,0x0A020400,0x02020408,0x0A020408,
      0x02000001,0x0A000001,0x02000009,0x0A000009,
      0x02000401,0x0A000401,0x02000409,0x0A000409,
      0x02020001,0x0A020001,0x02020009,0x0A020009,
      0x02020401,0x0A020401,0x02020409,0x0A020409,
    },{
      /* for D bits (numbered as per FIPS 46) 16 17 18 19 20 21 */
      0x00000000,0x00000100,0x00080000,0x00080100,
      0x01000000,0x01000100,0x01080000,0x01080100,
      0x00000010,0x00000110,0x00080010,0x00080110,
      0x01000010,0x01000110,0x01080010,0x01080110,
      0x00200000,0x00200100,0x00280000,0x00280100,
      0x01200000,0x01200100,0x01280000,0x01280100,
      0x00200010,0x00200110,0x00280010,0x00280110,
      0x01200010,0x01200110,0x01280010,0x01280110,
      0x00000200,0x00000300,0x00080200,0x00080300,
      0x01000200,0x01000300,0x01080200,0x01080300,
      0x00000210,0x00000310,0x00080210,0x00080310,
      0x01000210,0x01000310,0x01080210,0x01080310,
      0x00200200,0x00200300,0x00280200,0x00280300,
      0x01200200,0x01200300,0x01280200,0x01280300,
      0x00200210,0x00200310,0x00280210,0x00280310,
      0x01200210,0x01200310,0x01280210,0x01280310,
    },{
      /* for D bits (numbered as per FIPS 46) 22 23 24 25 27 28 */
      0x00000000,0x04000000,0x00040000,0x04040000,
      0x00000002,0x04000002,0x00040002,0x04040002,
      0x00002000,0x04002000,0x00042000,0x04042000,
      0x00002002,0x04002002,0x00042002,0x04042002,
      0x00000020,0x04000020,0x00040020,0x04040020,
      0x00000022,0x04000022,0x00040022,0x04040022,
      0x00002020,0x04002020,0x00042020,0x04042020,
      0x00002022,0x04002022,0x00042022,0x04042022,
      0x00000800,0x04000800,0x00040800,0x04040800,
      0x00000802,0x04000802,0x00040802,0x04040802,
      0x00002800,0x04002800,0x00042800,0x04042800,
      0x00002802,0x04002802,0x00042802,0x04042802,
      0x00000820,0x04000820,0x00040820,0x04040820,
      0x00000822,0x04000822,0x00040822,0x04040822,
      0x00002820,0x04002820,0x00042820,0x04042820,
      0x00002822,0x04002822,0x00042822,0x04042822,
    }
  };

  private static final byte weak_keys[][] = {
    /* weak keys */
    {(byte)0x01,(byte)0x01,(byte)0x01,(byte)0x01,
     (byte)0x01,(byte)0x01,(byte)0x01,(byte)0x01},
    {(byte)0xFE,(byte)0xFE,(byte)0xFE,(byte)0xFE,
     (byte)0xFE,(byte)0xFE,(byte)0xFE,(byte)0xFE},
    {(byte)0x1F,(byte)0x1F,(byte)0x1F,(byte)0x1F,
     (byte)0x1F,(byte)0x1F,(byte)0x1F,(byte)0x1F},
    {(byte)0xE0,(byte)0xE0,(byte)0xE0,(byte)0xE0,
     (byte)0xE0,(byte)0xE0,(byte)0xE0,(byte)0xE0},
    /* semi-weak keys */
    {(byte)0x01,(byte)0xFE,(byte)0x01,(byte)0xFE,
     (byte)0x01,(byte)0xFE,(byte)0x01,(byte)0xFE},
    {(byte)0xFE,(byte)0x01,(byte)0xFE,(byte)0x01,
     (byte)0xFE,(byte)0x01,(byte)0xFE,(byte)0x01},
    {(byte)0x1F,(byte)0xE0,(byte)0x1F,(byte)0xE0,
     (byte)0x0E,(byte)0xF1,(byte)0x0E,(byte)0xF1},
    {(byte)0xE0,(byte)0x1F,(byte)0xE0,(byte)0x1F,
     (byte)0xF1,(byte)0x0E,(byte)0xF1,(byte)0x0E},
    {(byte)0x01,(byte)0xE0,(byte)0x01,(byte)0xE0,
     (byte)0x01,(byte)0xF1,(byte)0x01,(byte)0xF1},
    {(byte)0xE0,(byte)0x01,(byte)0xE0,(byte)0x01,
     (byte)0xF1,(byte)0x01,(byte)0xF1,(byte)0x01},
    {(byte)0x1F,(byte)0xFE,(byte)0x1F,(byte)0xFE,
     (byte)0x0E,(byte)0xFE,(byte)0x0E,(byte)0xFE},
    {(byte)0xFE,(byte)0x1F,(byte)0xFE,(byte)0x1F,
     (byte)0xFE,(byte)0x0E,(byte)0xFE,(byte)0x0E},
    {(byte)0x01,(byte)0x1F,(byte)0x01,(byte)0x1F,
     (byte)0x01,(byte)0x0E,(byte)0x01,(byte)0x0E},
    {(byte)0x1F,(byte)0x01,(byte)0x1F,(byte)0x01,
     (byte)0x0E,(byte)0x01,(byte)0x0E,(byte)0x01},
    {(byte)0xE0,(byte)0xFE,(byte)0xE0,(byte)0xFE,
     (byte)0xF1,(byte)0xFE,(byte)0xF1,(byte)0xFE},
    {(byte)0xFE,(byte)0xE0,(byte)0xFE,(byte)0xE0,
     (byte)0xFE,(byte)0xF1,(byte)0xFE,(byte)0xF1}
  };

  private boolean check_weak_key(byte [] key) {
    for (int i= 0; i < weak_keys.length; i++) {
      int j;
      for(j = 0; j < 8; j++) {
	if(key[j] != weak_keys[i][j])
	  break;
      }
      if(j == 8) {
	/*
	 * We got a match.
	 */
	is_weak_key_ = true;
	return true;
      }
    }
    is_weak_key_ = false;
    return false;
  }

  private boolean check_parity(byte [] key) {
    int i;
    for (i=0; i<8; i++) {
      if(key[i] != odd_parity_array[(int)(key[i]&0xff)]) {
	is_parity_ok_ = false;
	return false;
      }
    }
    is_parity_ok_ = true;
    return true;
  }

  /**
   * DesKey constructor from a String object.
   * Take a string and convert it into a 128
   * byte key schedule. Do so by doing a des cypher-block
   * chaining checksum operation on the string and using
   * the output as the key.
   * 
   * @param str           String object.
   * @param check_key     boolean. If true checks for weak keys.
   *
   */
  public void DesKey(String str, boolean check_key) {
    int i;
    byte j;

    byte [] key = new byte[8];
    byte [] keystr = new byte[str.length()];
    for(i = 0; i < 8; i++) {
      key[i] = 0;
    }
    for (i=0; i<str.length(); i++) {
      keystr[i] = j = (byte)str.charAt(i);
      if ((i%16) < 8) {
	key[i%8]^=(j<<1);
      } else {
	/* Reverse the bit order */
	j=(byte)(((j<<4)&0xf0)|((j>>>4)&0x0f));
	j=(byte)(((j<<2)&0xcc)|((j>>>2)&0x33));
	j=(byte)(((j<<1)&0xaa)|((j>>>1)&0x55));
	key[7-(i%8)]^=j;
      }
    }
    set_odd_parity(key);
    set_key_sced(key, false);
    Des des = new Des(this);

    /*
     * Now do the des cbc-checksum to get the 8 byte remainder.
     * This will be the real key.
     */
    des.cbc_cksum(keystr, 0, keystr.length, key, 0, key);
    set_odd_parity(key);
    set_key_sced(key, check_key);
  }

  /**
   * DesKey constructor from an 8 byte array.
   * 
   * @param key           byte [] array, length 8.
   * @param check_key     boolean. If true checks for weak keys.
   *
   */
  public DesKey(byte [] key, boolean check_key) {
    set_odd_parity(key);
    set_key_sced(key, check_key);
  }

  /**
   * Add parity bits to an 8 byte array. Use this
   * before calling the DesKey constructor with this
   * array.
   *
   * @param key           byte [] array, length 8.
   */
  public static void set_odd_parity(byte [] key) {
    int i;
    for (i=0; i<8; i++) {
      key[i]= DesKey.odd_parity_array[(int)(key[i]&0xff)];
    }
  }


  /*
   * Take an 8 byte key and transform it into a des
   * key schedule.
   */
  private void set_key_sced(byte [] key, boolean check_key) {
    if (check_key == true) {
      /*
       * Just exit if we fail tests - caller will check.
       */
      if(check_parity(key) == false) {
	return;
      }
      if(check_weak_key(key) == true) {
	return;
      }
    }

    /*
     * Convert key to two 32 bit ints.
     */

    int key_c0 = Int32Manipulator.bytes_to_int(key, 0);
    int key_d0 = Int32Manipulator.bytes_to_int(key, 4);

    int tmp = 0;
    int [] ref_to_c0 = new int[1];
    int [] ref_to_d0 = new int[1];

    ref_to_c0[0] = key_c0;
    ref_to_d0[0] = key_d0;

    Int32Manipulator.PERM_OP (ref_to_d0, ref_to_c0, tmp, 4, 0x0f0f0f0f);
    Int32Manipulator.HPERM_OP(ref_to_c0, tmp, -2, 0xcccc0000);
    Int32Manipulator.HPERM_OP(ref_to_d0, tmp, -2, 0xcccc0000);
    Int32Manipulator.PERM_OP (ref_to_d0, ref_to_c0, tmp, 1, 0x55555555);
    Int32Manipulator.PERM_OP (ref_to_c0, ref_to_d0, tmp, 8, 0x00ff00ff);
    Int32Manipulator.PERM_OP (ref_to_d0, ref_to_c0, tmp, 1, 0x55555555);

    /* Convert back to ordinary ints */
    key_c0 = ref_to_c0[0];
    key_d0 = ref_to_d0[0];

    key_d0 = (((key_d0 & 0x000000ff)<<16)|(key_d0 & 0x0000ff00)|
	      ((key_d0 & 0x00ff0000)>>>16)|
	      ((key_c0 & 0xf0000000)>>>4));
    key_c0 &= 0x0fffffff;

    int i, tmp1;
    for (i=0; i < 16; i++) {
      if (shifts2[i] == true) {
	key_c0 = ((key_c0>>>2)|(key_c0<<26));
	key_d0 = ((key_d0>>>2)|(key_d0<<26));
      } else {
	key_c0 = ((key_c0>>>1)|(key_c0<<27));
	key_d0 = ((key_d0>>>1)|(key_d0<<27));
      }
      key_c0 &= 0x0fffffff;
      key_d0 &= 0x0fffffff;

      tmp1 =	des_skb[0][ (key_c0    )&0x3f                ]|
	des_skb[1][((key_c0>>> 6)&0x03)|((key_c0>>> 7)&0x3c)]|
	des_skb[2][((key_c0>>>13)&0x0f)|((key_c0>>>14)&0x30)]|
	des_skb[3][((key_c0>>>20)&0x01)|((key_c0>>>21)&0x06) |
				       ((key_c0>>>22)&0x38)];

      tmp = des_skb[4][ (key_d0    )&0x3f                ]|
	des_skb[5][((key_d0>>> 7)&0x03)|((key_d0>> 8)&0x3c)]|
	des_skb[6][ (key_d0>>>15)&0x3f                ]|
	des_skb[7][((key_d0>>>21)&0x0f)|((key_d0>>>22)&0x30)];

      /* table contained 0213 4657 */
      Int32Manipulator.set_int(keysced_, i*8, (tmp<<16)|(tmp1&0x0000ffff));
      tmp1 = ((tmp1>>>16)|(tmp&0xffff0000));
      tmp1 = (tmp1 <<4)|(tmp1>>>28);
      Int32Manipulator.set_int(keysced_, i*8 + 4, tmp1);
    }
  }
}

/**
 *
 * Interface that both Des and TripleDes implement.
 * Created so code can be written independently of whether
 * Des or TripleDes is being used.
 *
 * Written by Jeremy Allison (jra@cygnus.com).
 */

interface DesCrypt {

  /**
   * Do the ecb (Encrypt/Decrypt 8 bytes electronic code book)
   * mode. Encrypts 8 bytes starting at offset input_start in
   * byte array input and writes them out at offset output_start in
   * byte array output.
   *
   * @param input          Input byte [] array.
   * @param input_start    Offset into input to start encryption.
   * @param output         Output byte [] array.
   * @param output_start   Offset into output to place result.
   * @param encrypt        Pass Des.ENCRYPT to encrypt, Des.DECRYPT to
   * decrypt.
   *
   */

  void ecb_encrypt(byte [] input, int input_start, 
		   byte [] output, int output_start, 
		   boolean encrypt);

  /**
   * Do the cbc (Cypher block chaining mode)
   * encrypt/decrypt. This updates the ivec array, and is equivalent
   * to the des_ncbc_encrypt in the C library.
   *
   * @param input          Input byte [] array.
   * @param input_start    Offset into input to start encryption.
   * @param length         Number of bytes to encrypt.
   * @param output         Output byte [] array.
   * @param output_start   Offset into output to place result.
   * @param ivec           Initialization vector. A byte [] array of length 8. Updated on exit.
   * @param encrypt        Pass Des.ENCRYPT to encrypt, Des.DECRYPT to
   * decrypt.
   */

  void cbc_encrypt(byte [] input, int input_start, int length,
		   byte [] output, int output_start,
		   byte [] ivec,
		   boolean encrypt);

  /**
   * Do the CFB mode with 64 bit feedback. Used to encrypt/decrypt
   * arbitrary numbers of bytes. To use this initialize num[0] to zero
   * and set input_start to the correct offset into input, and length to
   * the number of bytes following that offset that you wish to encrypt
   * before calling.
   *
   * @param input          Input byte [] array.
   * @param input_start    Offset into input to start encryption.
   * @param length         Number of bytes to encrypt.
   * @param output         Output byte [] array.
   * @param output_start   Offset into output to place result.
   * @param ivec           Initialization vector. A byte [] array of length 8. Updated on exit.
   * @param num            Reference to an int used to keep track of 'how far' we are though ivec. Updated on exit.
   * @param encrypt        Pass Des.ENCRYPT to encrypt, Des.DECRYPT to
   * decrypt.
   */

  void cfb64_encrypt(byte [] input, int input_start, int length, 
		     byte [] output, int output_start,
		     byte [] ivec,
		     int [] num, 
		     boolean encrypt);

  /**
   * Do the OFB mode with 64 bit feedback. Used to encrypt/decrypt
   * arbitrary numbers of bytes. To use this initialize num[0] to zero
   * and set input_start to the correct offset into input, and length to
   * the number of bytes following that offset that you wish to encrypt
   * before calling.
   *
   * @param input          Input byte [] array.
   * @param input_start    Offset into input to start encryption.
   * @param length         Number of bytes to encrypt.
   * @param output         Output byte [] array.
   * @param output_start   Offset into output to place result.
   * @param ivec           Initialization vector. A byte [] array of length 8. Updated on exit.
   * @param num            Reference to an int used to keep track of 'how far' we are though ivec. Updated on exit.
   */

  void ofb64_encrypt(byte [] input, int input_start, int length, 
		     byte [] output, int output_start,
		     byte [] ivec,
		     int [] num);
}

/**
 * Des class. Does all modes of Des encryption.
 * Written by Jeremy Allison (jra@cygnus.com) based on
 * C code from Eric Young (eay@mincom.oz.au).
 */

class Des implements DesCrypt {

  /**
   * Constant. Pass this to functions that have a boolean
   * encrypt parameter to tell them to encrypt the data.
   */
  public static final boolean ENCRYPT = true;

  /**
   * Constant. Pass this to functions that have a boolean
   * encrypt parameter to tell them to decrypt the data.
   */
  public static final boolean DECRYPT = false;

  private DesKey ks_;

  private static final int des_SPtrans_[][] = {
    {
      /* nibble 0 */
      0x00820200, 0x00020000, 0x80800000, 0x80820200,
      0x00800000, 0x80020200, 0x80020000, 0x80800000,
      0x80020200, 0x00820200, 0x00820000, 0x80000200,
      0x80800200, 0x00800000, 0x00000000, 0x80020000,
      0x00020000, 0x80000000, 0x00800200, 0x00020200,
      0x80820200, 0x00820000, 0x80000200, 0x00800200,
      0x80000000, 0x00000200, 0x00020200, 0x80820000,
      0x00000200, 0x80800200, 0x80820000, 0x00000000,
      0x00000000, 0x80820200, 0x00800200, 0x80020000,
      0x00820200, 0x00020000, 0x80000200, 0x00800200,
      0x80820000, 0x00000200, 0x00020200, 0x80800000,
      0x80020200, 0x80000000, 0x80800000, 0x00820000,
      0x80820200, 0x00020200, 0x00820000, 0x80800200,
      0x00800000, 0x80000200, 0x80020000, 0x00000000,
      0x00020000, 0x00800000, 0x80800200, 0x00820200,
      0x80000000, 0x80820000, 0x00000200, 0x80020200
    },{
      /* nibble 1 */
      0x10042004, 0x00000000, 0x00042000, 0x10040000,
      0x10000004, 0x00002004, 0x10002000, 0x00042000,
      0x00002000, 0x10040004, 0x00000004, 0x10002000,
      0x00040004, 0x10042000, 0x10040000, 0x00000004,
      0x00040000, 0x10002004, 0x10040004, 0x00002000,
      0x00042004, 0x10000000, 0x00000000, 0x00040004,
      0x10002004, 0x00042004, 0x10042000, 0x10000004,
      0x10000000, 0x00040000, 0x00002004, 0x10042004,
      0x00040004, 0x10042000, 0x10002000, 0x00042004,
      0x10042004, 0x00040004, 0x10000004, 0x00000000,
      0x10000000, 0x00002004, 0x00040000, 0x10040004,
      0x00002000, 0x10000000, 0x00042004, 0x10002004,
      0x10042000, 0x00002000, 0x00000000, 0x10000004,
      0x00000004, 0x10042004, 0x00042000, 0x10040000,
      0x10040004, 0x00040000, 0x00002004, 0x10002000,
      0x10002004, 0x00000004, 0x10040000, 0x00042000
    },{
      /* nibble 2 */
      0x41000000, 0x01010040, 0x00000040, 0x41000040,
      0x40010000, 0x01000000, 0x41000040, 0x00010040,
      0x01000040, 0x00010000, 0x01010000, 0x40000000,
      0x41010040, 0x40000040, 0x40000000, 0x41010000,
      0x00000000, 0x40010000, 0x01010040, 0x00000040,
      0x40000040, 0x41010040, 0x00010000, 0x41000000,
      0x41010000, 0x01000040, 0x40010040, 0x01010000,
      0x00010040, 0x00000000, 0x01000000, 0x40010040,
      0x01010040, 0x00000040, 0x40000000, 0x00010000,
      0x40000040, 0x40010000, 0x01010000, 0x41000040,
      0x00000000, 0x01010040, 0x00010040, 0x41010000,
      0x40010000, 0x01000000, 0x41010040, 0x40000000,
      0x40010040, 0x41000000, 0x01000000, 0x41010040,
      0x00010000, 0x01000040, 0x41000040, 0x00010040,
      0x01000040, 0x00000000, 0x41010000, 0x40000040,
      0x41000000, 0x40010040, 0x00000040, 0x01010000
    },{
      /* nibble 3 */
      0x00100402, 0x04000400, 0x00000002, 0x04100402,
      0x00000000, 0x04100000, 0x04000402, 0x00100002,
      0x04100400, 0x04000002, 0x04000000, 0x00000402,
      0x04000002, 0x00100402, 0x00100000, 0x04000000,
      0x04100002, 0x00100400, 0x00000400, 0x00000002,
      0x00100400, 0x04000402, 0x04100000, 0x00000400,
      0x00000402, 0x00000000, 0x00100002, 0x04100400,
      0x04000400, 0x04100002, 0x04100402, 0x00100000,
      0x04100002, 0x00000402, 0x00100000, 0x04000002,
      0x00100400, 0x04000400, 0x00000002, 0x04100000,
      0x04000402, 0x00000000, 0x00000400, 0x00100002,
      0x00000000, 0x04100002, 0x04100400, 0x00000400,
      0x04000000, 0x04100402, 0x00100402, 0x00100000,
      0x04100402, 0x00000002, 0x04000400, 0x00100402,
      0x00100002, 0x00100400, 0x04100000, 0x04000402,
      0x00000402, 0x04000000, 0x04000002, 0x04100400
    },{
      /* nibble 4 */
      0x02000000, 0x00004000, 0x00000100, 0x02004108,
      0x02004008, 0x02000100, 0x00004108, 0x02004000,
      0x00004000, 0x00000008, 0x02000008, 0x00004100,
      0x02000108, 0x02004008, 0x02004100, 0x00000000,
      0x00004100, 0x02000000, 0x00004008, 0x00000108,
      0x02000100, 0x00004108, 0x00000000, 0x02000008,
      0x00000008, 0x02000108, 0x02004108, 0x00004008,
      0x02004000, 0x00000100, 0x00000108, 0x02004100,
      0x02004100, 0x02000108, 0x00004008, 0x02004000,
      0x00004000, 0x00000008, 0x02000008, 0x02000100,
      0x02000000, 0x00004100, 0x02004108, 0x00000000,
      0x00004108, 0x02000000, 0x00000100, 0x00004008,
      0x02000108, 0x00000100, 0x00000000, 0x02004108,
      0x02004008, 0x02004100, 0x00000108, 0x00004000,
      0x00004100, 0x02004008, 0x02000100, 0x00000108,
      0x00000008, 0x00004108, 0x02004000, 0x02000008
    },{
      /* nibble 5 */
      0x20000010, 0x00080010, 0x00000000, 0x20080800,
      0x00080010, 0x00000800, 0x20000810, 0x00080000,
      0x00000810, 0x20080810, 0x00080800, 0x20000000,
      0x20000800, 0x20000010, 0x20080000, 0x00080810,
      0x00080000, 0x20000810, 0x20080010, 0x00000000,
      0x00000800, 0x00000010, 0x20080800, 0x20080010,
      0x20080810, 0x20080000, 0x20000000, 0x00000810,
      0x00000010, 0x00080800, 0x00080810, 0x20000800,
      0x00000810, 0x20000000, 0x20000800, 0x00080810,
      0x20080800, 0x00080010, 0x00000000, 0x20000800,
      0x20000000, 0x00000800, 0x20080010, 0x00080000,
      0x00080010, 0x20080810, 0x00080800, 0x00000010,
      0x20080810, 0x00080800, 0x00080000, 0x20000810,
      0x20000010, 0x20080000, 0x00080810, 0x00000000,
      0x00000800, 0x20000010, 0x20000810, 0x20080800,
      0x20080000, 0x00000810, 0x00000010, 0x20080010
    },{
      /* nibble 6 */
      0x00001000, 0x00000080, 0x00400080, 0x00400001,
      0x00401081, 0x00001001, 0x00001080, 0x00000000,
      0x00400000, 0x00400081, 0x00000081, 0x00401000,
      0x00000001, 0x00401080, 0x00401000, 0x00000081,
      0x00400081, 0x00001000, 0x00001001, 0x00401081,
      0x00000000, 0x00400080, 0x00400001, 0x00001080,
      0x00401001, 0x00001081, 0x00401080, 0x00000001,
      0x00001081, 0x00401001, 0x00000080, 0x00400000,
      0x00001081, 0x00401000, 0x00401001, 0x00000081,
      0x00001000, 0x00000080, 0x00400000, 0x00401001,
      0x00400081, 0x00001081, 0x00001080, 0x00000000,
      0x00000080, 0x00400001, 0x00000001, 0x00400080,
      0x00000000, 0x00400081, 0x00400080, 0x00001080,
      0x00000081, 0x00001000, 0x00401081, 0x00400000,
      0x00401080, 0x00000001, 0x00001001, 0x00401081,
      0x00400001, 0x00401080, 0x00401000, 0x00001001
    },{
      /* nibble 7 */
      0x08200020, 0x08208000, 0x00008020, 0x00000000,
      0x08008000, 0x00200020, 0x08200000, 0x08208020,
      0x00000020, 0x08000000, 0x00208000, 0x00008020,
      0x00208020, 0x08008020, 0x08000020, 0x08200000,
      0x00008000, 0x00208020, 0x00200020, 0x08008000,
      0x08208020, 0x08000020, 0x00000000, 0x00208000,
      0x08000000, 0x00200000, 0x08008020, 0x08200020,
      0x00200000, 0x00008000, 0x08208000, 0x00000020,
      0x00200000, 0x00008000, 0x08000020, 0x08208020,
      0x00008020, 0x08000000, 0x00000000, 0x00208000,
      0x08200020, 0x08008020, 0x08008000, 0x00200020,
      0x08208000, 0x00000020, 0x00200020, 0x08008000,
      0x08208020, 0x00200000, 0x08200000, 0x08000020,
      0x00208000, 0x00008020, 0x08008020, 0x08200000,
      0x00000020, 0x08208000, 0x00208020, 0x00000000,
      0x08000000, 0x08200020, 0x00008000, 0x00208020
    }
  };

  /**
   * Internal routine only exported for use by TripleDes code.
   */
  /*
   * Initial permutation.
   *
   #define IP(l,r) \
   { \
   register DES_LONG tt; \
   PERM_OP(r,l,tt, 4,0x0f0f0f0fL); \
   PERM_OP(l,r,tt,16,0x0000ffffL); \
   PERM_OP(r,l,tt, 2,0x33333333L); \
   PERM_OP(l,r,tt, 8,0x00ff00ffL); \
   PERM_OP(r,l,tt, 1,0x55555555L); \
   }
   */

  public void IP(int [] ref_to_l, int [] ref_to_r) {
    int tt = 0;
    Int32Manipulator.PERM_OP(ref_to_r, ref_to_l, tt, 4, 0x0f0f0f0f);
    Int32Manipulator.PERM_OP(ref_to_l, ref_to_r, tt, 16, 0x0000ffff);
    Int32Manipulator.PERM_OP(ref_to_r, ref_to_l, tt, 2, 0x33333333);
    Int32Manipulator.PERM_OP(ref_to_l, ref_to_r, tt, 8, 0x00ff00ff);
    Int32Manipulator.PERM_OP(ref_to_r, ref_to_l, tt, 1, 0x55555555);
  }

  /**
   * Internal routine only exported for use by TripleDes code.
   */
  /*
   * Final permutation.
   *
   #define FP(l,r) \
   { \
   register DES_LONG tt; \
   PERM_OP(l,r,tt, 1,0x55555555L); \
   PERM_OP(r,l,tt, 8,0x00ff00ffL); \
   PERM_OP(l,r,tt, 2,0x33333333L); \
   PERM_OP(r,l,tt,16,0x0000ffffL); \
   PERM_OP(l,r,tt, 4,0x0f0f0f0fL); \
   }
   */

  public void FP(int [] ref_to_l, int [] ref_to_r) {
    int tt = 0;
    Int32Manipulator.PERM_OP(ref_to_l, ref_to_r, tt, 1, 0x55555555);
    Int32Manipulator.PERM_OP(ref_to_r, ref_to_l, tt, 8, 0x00ff00ff);
    Int32Manipulator.PERM_OP(ref_to_l, ref_to_r, tt, 2, 0x33333333);
    Int32Manipulator.PERM_OP(ref_to_r, ref_to_l, tt,16, 0x0000ffff);
    Int32Manipulator.PERM_OP(ref_to_l, ref_to_r, tt, 4, 0x0f0f0f0f);
  }

  /*
    #define D_ENCRYPT(Q,R,S) {\
    u=(R^s[S  ]); \
    t=R^s[S+1]; \
    t=((t>>4L)+(t<<28L)); \
    Q^=     des_SPtrans[1][(t     )&0x3f]| \
    des_SPtrans[3][(t>> 8L)&0x3f]| \
    des_SPtrans[5][(t>>16L)&0x3f]| \
    des_SPtrans[7][(t>>24L)&0x3f]| \
    des_SPtrans[0][(u     )&0x3f]| \
    des_SPtrans[2][(u>> 8L)&0x3f]| \
    des_SPtrans[4][(u>>16L)&0x3f]| \
    des_SPtrans[6][(u>>24L)&0x3f]; }
    */

  private void D_ENCRYPT(int [] ref_to_Q, int R, int S, int [] ref_to_u) {
    byte [] s = ks_.get_keysced();
    S = S*4; /* Remember, S is a int offset, int C */
    int s_at_S_offset = Int32Manipulator.bytes_to_int(s, S);
    int s_at_S_plus_one_offset = Int32Manipulator.bytes_to_int(s, S+4);

    ref_to_u[0] = R^s_at_S_offset;
    int tmp = R^s_at_S_plus_one_offset;
    tmp = (tmp>>>4) + (tmp<<28);
    ref_to_Q[0] ^= des_SPtrans_[1][(tmp             )&0x3f]|
      des_SPtrans_[3][(tmp>>>8         )&0x3f]|
      des_SPtrans_[5][(tmp>>>16        )&0x3f]|
      des_SPtrans_[7][(tmp>>>24        )&0x3f]|
      des_SPtrans_[0][(ref_to_u[0]     )&0x3f]|
      des_SPtrans_[2][(ref_to_u[0]>>> 8)&0x3f]|
      des_SPtrans_[4][(ref_to_u[0]>>>16)&0x3f]|
      des_SPtrans_[6][(ref_to_u[0]>>>24)&0x3f];
  }

  /**
   * Constructor for a Des Object. Takes an already
   * existing DesKey which will be used for all future
   * encryption/decryption with this Des object.
   * The passed in key is not checked for parity of weakness.
   *
   * @param key         Existing DesKey.
   */
  public Des(DesKey key) {
    ks_ = key;
  }

  /*
   * This is the internal encrypt routine, called by all the
   * other des encrypt/decrypt functions. It is not callable
   * outside the Des class.
   */

  private void des_encrypt(int [] data, boolean encrypt) {
    int tmp = 0;
    int [] ref_to_u = new int [1];
    int [] ref_to_r = new int [1];
    int [] ref_to_l = new int [1];

    ref_to_u[0] = data[0];
    ref_to_r[0] = data[1];

    IP(ref_to_u, ref_to_r);

    ref_to_l[0]=(ref_to_r[0]<<1)|(ref_to_r[0]>>>31);
    ref_to_r[0]=(ref_to_u[0]<<1)|(ref_to_u[0]>>>31);

    if (encrypt == Des.ENCRYPT) {
      for (int i = 0; i < 32; i+=4) {
	D_ENCRYPT(ref_to_l,ref_to_r[0],i+0, ref_to_u); /*  1 */
	D_ENCRYPT(ref_to_r,ref_to_l[0],i+2, ref_to_u); /*  2 */
      }
    } else {
      /* Decrypt */
      for (int i = 30; i > 0; i-=4) {
	D_ENCRYPT(ref_to_l,ref_to_r[0],i-0, ref_to_u); /* 16 */
	D_ENCRYPT(ref_to_r,ref_to_l[0],i-2, ref_to_u); /* 15 */
      }
    }

    ref_to_l[0] = (ref_to_l[0]>>>1)|(ref_to_l[0]<<31);
    ref_to_r[0] = (ref_to_r[0]>>>1)|(ref_to_r[0]<<31);

    FP(ref_to_r, ref_to_l);

    data[0]=ref_to_l[0];
    data[1]=ref_to_r[0];
  }

  /**
   * This is the a publicly callable variant of the
   * internal encrypt routine, called by all the
   * TripleDes encrypt/decrypt functions. The only difference
   * between this and the des_encrypt function is that this
   * doesn't do the initial and final permutations. This
   * optimises the speed of TripleDes as
   * IP() des_encrypt2() des_encrypt2() des_encrypt2() FP() 
   * is the same as
   * des_encrypt() des_encrypt() des_encrypt() 
   * except faster :-).
   * 
   * No parameters listed as this is only used by the TripleDes
   * class.
   */

  public void des_encrypt2(int [] data, boolean encrypt) {
    int tmp = 0;
    int [] ref_to_u = new int [1];
    int [] ref_to_r = new int [1];
    int [] ref_to_l = new int [1];

    ref_to_u[0] = data[0];
    ref_to_r[0] = data[1];

    ref_to_l[0]=(ref_to_r[0]<<1)|(ref_to_r[0]>>>31);
    ref_to_r[0]=(ref_to_u[0]<<1)|(ref_to_u[0]>>>31);

    if (encrypt == Des.ENCRYPT) {
      for (int i = 0; i < 32; i+=4) {
	D_ENCRYPT(ref_to_l,ref_to_r[0],i+0, ref_to_u); /*  1 */
	D_ENCRYPT(ref_to_r,ref_to_l[0],i+2, ref_to_u); /*  2 */
      }
    } else {
      /* Decrypt */
      for (int i = 30; i > 0; i-=4) {
	D_ENCRYPT(ref_to_l,ref_to_r[0],i-0, ref_to_u); /* 16 */
	D_ENCRYPT(ref_to_r,ref_to_l[0],i-2, ref_to_u); /* 15 */
      }
    }

    ref_to_l[0] = (ref_to_l[0]>>>1)|(ref_to_l[0]<<31);
    ref_to_r[0] = (ref_to_r[0]>>>1)|(ref_to_r[0]<<31);

    data[0]=ref_to_l[0];
    data[1]=ref_to_r[0];
  }

  /**
   * Do the ecb (Encrypt/Decrypt 8 bytes electronic code book)
   * mode. Encrypts 8 bytes starting at offset input_start in
   * byte array input and writes them out at offset output_start in
   * byte array output.
   *
   * @param input          Input byte [] array.
   * @param input_start    Offset into input to start encryption.
   * @param output         Output byte [] array.
   * @param output_start   Offset into output to place result.
   * @param encrypt        Pass Des.ENCRYPT to encrypt, Des.DECRYPT to
   * decrypt.
   *
   */

  public void ecb_encrypt(byte [] input, int input_start,
			  byte [] output, int output_start,
			  boolean encrypt) {
    int [] ll = new int[2];

    ll[0] = Int32Manipulator.bytes_to_int(input, input_start);
    ll[1] = Int32Manipulator.bytes_to_int(input, input_start+4);
    des_encrypt(ll, encrypt);
    Int32Manipulator.set_int(output, output_start, ll[0]);
    Int32Manipulator.set_int(output, output_start+4, ll[1]);
  }

  /**
   * Do the cbc (Cypher block chaining mode)
   * encrypt/decrypt. This updates the ivec array, and is equivalent
   * to the des_ncbc_encrypt in the C library.
   *
   * @param input          Input byte [] array.
   * @param input_start    Offset into input to start encryption.
   * @param length         Number of bytes to encrypt.
   * @param output         Output byte [] array.
   * @param output_start   Offset into output to place result.
   * @param ivec           Initialization vector. A byte [] array of length 8. Updated on exit.
   * @param encrypt        Pass Des.ENCRYPT to encrypt, Des.DECRYPT to
   * decrypt.
   */

  public void cbc_encrypt(byte [] input, int input_start, int length,
			  byte [] output, int output_start,
			  byte [] ivec,
			  boolean encrypt) {

    int [] inout = new int[2];
    int tin0;
    int tin1;
    int tout0;
    int tout1;
    int chunksize;

    if(encrypt == Des.ENCRYPT) {
      /*
       * Encrypt.
       */
      tout0 = Int32Manipulator.bytes_to_int(ivec, 0);
      tout1 = Int32Manipulator.bytes_to_int(ivec, 4);

      /*
       * Deal with input 8 bytes at a time.
       */
      for (; length > 0; length -=8) {
	if(length >= 8) {
	  /*
	   * We have a full 8 byte chunk.
	   */
	  tin0 = Int32Manipulator.bytes_to_int(input, input_start);
	  tin1 = Int32Manipulator.bytes_to_int(input, input_start+4);
	  chunksize = 8;
	} else {
	  /*
	   * We only have a less than 8 byte fragment.
	   */
	  int [] ref_to_tin01 = new int [2];

	  /* c2ln(in,tin0,tin1,length); */
	  chunksize = Int32Manipulator.c2ln(input, input_start, 
					    length, ref_to_tin01);
	  tin0 = ref_to_tin01[0];
	  tin1 = ref_to_tin01[1];
	}
	tin0 ^= tout0;
	tin1 ^= tout1;
	inout[0] = tin0;
	inout[1] = tin1;
	des_encrypt(inout, encrypt);
	tout0 = inout[0];
	tout1 = inout[1];
	Int32Manipulator.set_int(output, output_start,   tout0);
	Int32Manipulator.set_int(output, output_start+4, tout1);
	input_start += chunksize;
	output_start += chunksize;
      }
      Int32Manipulator.set_int(ivec, 0, tout0);
      Int32Manipulator.set_int(ivec, 4, tout1);
    }else {
      /*
       * Decrypt.
       */
      int xor0 = Int32Manipulator.bytes_to_int(ivec, 0);
      int xor1 = Int32Manipulator.bytes_to_int(ivec, 4);

      /*
       * Deal with input 8 bytes at a time.
       */
      for (; length > 0; length -=8) {
	tin0 = Int32Manipulator.bytes_to_int(input, input_start);
	tin1 = Int32Manipulator.bytes_to_int(input, input_start+4);
	inout[0] = tin0;
	inout[1] = tin1;
	des_encrypt(inout, encrypt);
	tout0 = inout[0] ^ xor0;
	tout1 = inout[1] ^ xor1;
	if(length >= 8) {
	  Int32Manipulator.set_int(output, output_start,   tout0);
	  Int32Manipulator.set_int(output, output_start+4, tout1);
	  chunksize = 8;
	} else {
	  chunksize = Int32Manipulator.l2cn(output, output_start, 
					    length, tout0, tout1);
	}
	xor0 = tin0;
	xor1 = tin1;
	input_start += chunksize;
	output_start += chunksize;
      }
      Int32Manipulator.set_int(ivec, 0, xor0);
      Int32Manipulator.set_int(ivec, 4, xor1);
    }
  }

  /**
   * This is Propagating Cipher Block Chaining mode of DES.  It is used
   * by Kerberos v4.  It's parameters are the same as cbc_encrypt().
   *
   * @param input          Input byte [] array.
   * @param input_start    Offset into input to start encryption.
   * @param length         Number of bytes to encrypt.
   * @param output         Output byte [] array.
   * @param output_start   Offset into output to place result.
   * @param ivec           Initialization vector. A byte [] array of length 8. Updated on exit.
   * @param encrypt        Pass Des.ENCRYPT to encrypt, Des.DECRYPT to
   * decrypt.
   */
  public void pcbc_encrypt(byte [] input, int input_start, int length, 
			   byte [] output, int output_start,
			   byte [] ivec,
			   boolean encrypt) {
    int [] inout = new int[2];
    int sin0;
    int sin1;
    int tout0;
    int tout1;
    int chunksize;

    int xor0 = Int32Manipulator.bytes_to_int(ivec, 0);
    int xor1 = Int32Manipulator.bytes_to_int(ivec, 4);

    if(encrypt == Des.ENCRYPT) {
      /*
       * Encrypt.
       */
      /*
       * Deal with input 8 bytes at a time.
       */
      for (; length > 0; length -=8) {
	if(length >= 8) {
	  /*
	   * We have a full 8 byte chunk.
	   */
	  sin0 = Int32Manipulator.bytes_to_int(input, input_start);
	  sin1 = Int32Manipulator.bytes_to_int(input, input_start+4);
	  chunksize = 8;
	} else {
	  /*
	   * We only have a less than 8 byte fragment.
	   */
	  int [] ref_to_sin01 = new int[2];

	  /* c2ln(in,sin0,sin1,length); */
	  chunksize = Int32Manipulator.c2ln(input,input_start,
					    length,ref_to_sin01);
	  sin0 = ref_to_sin01[0];
	  sin1 = ref_to_sin01[1];
	}
	inout[0] = sin0^xor0;
	inout[1] = sin1^xor1;
	des_encrypt(inout, encrypt);
	tout0 = inout[0];
	tout1 = inout[1];
	xor0 = sin0^tout0;
	xor1 = sin1^tout1;
	Int32Manipulator.set_int(output, output_start,   tout0);
	Int32Manipulator.set_int(output, output_start+4, tout1);
	input_start += chunksize;
	output_start += chunksize;
      }
    } else {
      /*
       * Decrypt.
       */
      for (; length > 0; length -=8) {
	sin0 = Int32Manipulator.bytes_to_int(input, input_start);
	sin1 = Int32Manipulator.bytes_to_int(input, input_start+4);
	inout[0] = sin0;
	inout[1] = sin1;
	des_encrypt(inout, encrypt);
	tout0 = inout[0]^xor0;
	tout1 = inout[1]^xor1;
	if (length >= 8) {
	  Int32Manipulator.set_int(output, output_start,   tout0);
	  Int32Manipulator.set_int(output, output_start+4, tout1);
	  chunksize = 8;
	} else {
	  chunksize = Int32Manipulator.l2cn(output, output_start, 
					    length, tout0, tout1);
	}
	xor0 = tout0^sin0;
	xor1 = tout1^sin1;
	input_start += chunksize;
	output_start += chunksize;
      }
    }
  }

  /**
   * Cipher Feedback Back mode of DES.  This implementation 'feeds back'
   * in numbit blocks.  The input (and output) is in multiples of numbits
   * bits.  numbits should to be a multiple of 8 bits.  Length is the
   * number of bytes input.  If numbits is not a multiple of 8 bits,
   * the extra bits in the bytes will be considered padding.  So if
   * numbits is 12, for each 2 input bytes, the 4 high bits of the
   * second byte will be ignored.  So to encode 72 bits when using
   * a numbits of 12 take 12 bytes.  To encode 72 bits when using
   * numbits of 9 will take 16 bytes.  To encode 80 bits when using
   * numbits of 16 will take 10 bytes. etc, etc.  This padding will
   * apply to both input and output.
   *
   * @param input          Input byte [] array.
   * @param input_start    Offset into input to start encryption.
   * @param length         Number of bytes to encrypt.
   * @param output         Output byte [] array.
   * @param output_start   Offset into output to place result.
   * @param numbits        Multiple of 8 bits as described above.
   * @param ivec           Initialization vector. A byte [] array of length 8. Updated on exit.
   * @param encrypt        Pass Des.ENCRYPT to encrypt, Des.DECRYPT to
   * decrypt.
   */
  public void cfb_encrypt(byte [] input, int input_start, int length,
			  byte [] output, int output_start,
			  int numbits, byte [] ivec,
			  boolean encrypt) {

    if(numbits > 64)
      numbits = 64;

    int n = (numbits+7)/8;
    int mask0;
    int mask1;

    if ( numbits > 32 ) {
      mask0 = 0xffffffff;
      if ( numbits == 64 ) {
	mask1 = mask0;
      } else {
	mask1 = (1<<(numbits-32))-1;
      }
    } else {
      if ( numbits  == 32 ) {
	mask0 = 0xffffffff;
      }	else {
	mask0 = (1<<numbits)-1;
      }
      mask1 = 0x00000000;
    }

    int v0 = Int32Manipulator.bytes_to_int(ivec, 0);
    int v1 = Int32Manipulator.bytes_to_int(ivec, 4);
    
    int [] ref_to_d01 = new int[2];
    int [] ti = new int [2];

    if(encrypt == Des.ENCRYPT) {
      /*
       * Encrypt.
       */
      while( length >= n) {
	length -= n;
	ti[0] = v0;
	ti[1] = v1;
	des_encrypt(ti, encrypt);

	/* c2ln(in,d0,d1,n); */
	Int32Manipulator.c2ln(input, input_start, n, ref_to_d01);
	input_start += n;
	ref_to_d01[0] = (ref_to_d01[0] ^ ti[0]) & mask0;
	ref_to_d01[1] = (ref_to_d01[1] ^ ti[1]) & mask1;
	/* l2cn(d0,d1,out,n); */
        Int32Manipulator.l2cn(output, output_start, n, 
			      ref_to_d01[0], ref_to_d01[1]);
	output_start += n;
	if (numbits == 32){ 
	  v0 = v1; 
	  v1 = ref_to_d01[0]; 
	} else if (numbits == 64) { 
	  v0 = ref_to_d01[0]; 
	  v1 = ref_to_d01[1]; 
	} else if (numbits > 32) { /* && numbits != 64 */
	  v0 = ((v1>>>(numbits-32))|(ref_to_d01[0]<<(64-numbits)));
	  v1 = ((ref_to_d01[0]>>>(numbits-32))|(ref_to_d01[1]<<(64-numbits)));
	} else {/* numbits < 32 */
	  v0 = ((v0>>>numbits)|(v1<<(32-numbits)));
	  v1 = ((v1>>>numbits)|(ref_to_d01[0]<<(32-numbits)));
	}
      }
    } else {
      /*
       * Decrypt.
       */
      while (length >= n) {
	length -= n;
	ti[0] = v0;
	ti[1] = v1;
	des_encrypt(ti, Des.ENCRYPT);
	/* c2ln(in,d0,d1,n); */
	Int32Manipulator.c2ln(input, input_start, n, ref_to_d01);
	if (numbits == 32) { 
	  v0 = v1; 
	  v1 = ref_to_d01[0]; 
	} else if (numbits == 64) { 
	  v0 = ref_to_d01[0]; 
	  v1 = ref_to_d01[1]; 
	} else if (numbits > 32) { /* && numbits != 64 */
	  v0 = ((v1>>>(numbits-32))|(ref_to_d01[0]<<(64-numbits)));
	  v1 = ((ref_to_d01[0]>>>(numbits-32))|(ref_to_d01[1]<<(64-numbits)));
	} else { /* numbits < 32 */
	  v0 = ((v0>>>numbits)|(v1<<(32-numbits)));
	  v1 = ((v1>>>numbits)|(ref_to_d01[0]<<(32-numbits)));
	}
	ref_to_d01[0] = (ref_to_d01[0] ^ ti[0]) & mask0;
	ref_to_d01[1] = (ref_to_d01[1] ^ ti[1]) & mask1;
	/* l2cn(d0,d1,out,n); */
	Int32Manipulator.l2cn(output, output_start, n, 
			      ref_to_d01[0], ref_to_d01[1]);
	input_start += n;
	output_start += n;
      }
    }
    Int32Manipulator.set_int( ivec, 0, v0);
    Int32Manipulator.set_int( ivec, 4, v1);
  }
      
  /**
   * This is a implementation of Output Feed Back mode of DES.  It is
   * the same as cfb_encrypt() in that numbits is the size of the
   * units dealt with during input and output (in bits).
   *
   * @param input          Input byte [] array.
   * @param input_start    Offset into input to start encryption.
   * @param length         Number of bytes to encrypt.
   * @param output         Output byte [] array.
   * @param output_start   Offset into output to place result.
   * @param numbits        Multiple of 8 bits as described above.
   * @param ivec           Initialization vector. A byte [] array of length 8. Updated on exit.
   */
  public void ofb_encrypt(byte [] input, int input_start, int length,
			  byte [] output, int output_start, 
			  int numbits, byte [] ivec) {

    if(numbits > 64)
      numbits = 64;

    int n = (numbits+7)/8;
    int mask0;
    int mask1;

    if ( numbits > 32 ) {
      mask0 = 0xffffffff;
      if ( numbits == 64 ) {
	mask1 = mask0;
      } else {
	mask1 = (1<<(numbits-32))-1;
      }
    } else {
      if ( numbits  == 32 ) {
	mask0 = 0xffffffff;
      }	else {
	mask0 = (1<<numbits)-1;
      }
      mask1 = 0x00000000;
    }

    int v0 = Int32Manipulator.bytes_to_int(ivec, 0);
    int v1 = Int32Manipulator.bytes_to_int(ivec, 4);
    
    int [] ti = new int [2];
    ti[0] = v0;
    ti[1] = v1;
    int [] ref_to_d01 = new int[2];

    while( length-- > 0) {
      des_encrypt(ti, Des.ENCRYPT);
      /* c2ln(in,d0,d1,n); */
      Int32Manipulator.c2ln(input, input_start, n, ref_to_d01);
      ref_to_d01[0] = (ref_to_d01[0] ^ ti[0]) & mask0;
      ref_to_d01[1] = (ref_to_d01[1] ^ ti[1]) & mask1;
      /* l2cn(d0,d1,out,n); */
      Int32Manipulator.l2cn(output, output_start, n, 
			    ref_to_d01[0], ref_to_d01[1]);
      input_start += n;
      output_start += n;
    }
    v0 = ti[0];
    v1 = ti[1];
    Int32Manipulator.set_int( ivec, 0, v0);
    Int32Manipulator.set_int( ivec, 4, v1);
  }

  /**
   * Do the CFB mode with 64 bit feedback. Used to encrypt/decrypt
   * arbitrary numbers of bytes. To use this initialize num[0] to zero
   * and set input_start to the correct offset into input, and length to
   * the number of bytes following that offset that you wish to encrypt
   * before calling.
   *
   * @param input          Input byte [] array.
   * @param input_start    Offset into input to start encryption.
   * @param length         Number of bytes to encrypt.
   * @param output         Output byte [] array.
   * @param output_start   Offset into output to place result.
   * @param ivec           Initialization vector. A byte [] array of length 8. Updated on exit.
   * @param num            Reference to an int used to keep track of 'how far' we are though ivec. Updated on exit.
   * @param encrypt        Pass Des.ENCRYPT to encrypt, Des.DECRYPT to
   * decrypt.
   */

  public void cfb64_encrypt(byte [] input, int input_start, int length, 
			    byte [] output, int output_start,
			    byte [] ivec,
			    int [] num, 
			    boolean encrypt) {

    byte c;
    byte cc;
    int n = num[0];
    int [] ti = new int [2];

    if( encrypt == Des.ENCRYPT) {
      /*
       * Encrypt.
       */
      while ((length--) != 0) {
	if (n == 0) {
	  ti[0] = Int32Manipulator.bytes_to_int(ivec, 0); 
	  ti[1] = Int32Manipulator.bytes_to_int(ivec, 4); 
	  des_encrypt(ti, encrypt);
	  Int32Manipulator.set_int(ivec, 0, ti[0]);
	  Int32Manipulator.set_int(ivec, 4, ti[1]);
	}
	c = (byte)(input[input_start] ^ ivec[n]);
	output[output_start] = c;
	input_start++;
	output_start++;
	ivec[n] = c;
	n = (n + 1) & 0x7;
      }
    } else {
      /*
       * Decrypt.
       */
      while ((length--) != 0) {
	if (n == 0) {
	  ti[0] = Int32Manipulator.bytes_to_int(ivec, 0);
	  ti[1] = Int32Manipulator.bytes_to_int(ivec, 4);
	  des_encrypt(ti, Des.ENCRYPT);
	  Int32Manipulator.set_int(ivec, 0, ti[0]);
	  Int32Manipulator.set_int(ivec, 4, ti[1]);
	}
	cc = input[input_start];
	c = ivec[n];
	ivec[n] = cc;
	output[output_start] = (byte)(c ^ cc);
	input_start++;
	output_start++;
	n = (n + 1) & 0x7;
      }
    }
    num[0] = n;
  }

  /**
   * Do the OFB mode with 64 bit feedback. Used to encrypt/decrypt
   * arbitrary numbers of bytes. To use this initialize num[0] to zero
   * and set input_start to the correct offset into input, and length to
   * the number of bytes following that offset that you wish to encrypt
   * before calling.
   *
   * @param input          Input byte [] array.
   * @param input_start    Offset into input to start encryption.
   * @param length         Number of bytes to encrypt.
   * @param output         Output byte [] array.
   * @param output_start   Offset into output to place result.
   * @param ivec           Initialization vector. A byte [] array of length 8. Updated on exit.
   * @param num            Reference to an int used to keep track of 'how far' we are though ivec. Updated on exit.
   */

  public void ofb64_encrypt(byte [] input, int input_start, int length, 
			    byte [] output, int output_start,
			    byte [] ivec,
			    int [] num) {
    byte [] dp = new byte [8];
    int [] ti = new int [2];
    int n = num[0];
    boolean save = false;

    ti[0] = Int32Manipulator.bytes_to_int(ivec, 0);
    ti[1] = Int32Manipulator.bytes_to_int(ivec, 4);
    Int32Manipulator.set_int(dp, 0, ti[0]);
    Int32Manipulator.set_int(dp, 4, ti[1]);
    while((length--) != 0) {
      if (n == 0) { 
	des_encrypt(ti,Des.ENCRYPT);
	Int32Manipulator.set_int(dp, 0, ti[0]);
	Int32Manipulator.set_int(dp, 4, ti[1]);
	save = true;
      }
      output[output_start] = (byte)(input[input_start] ^ dp[n]);
      input_start++;
      output_start++;
      n = (n + 1) & 0x7;
    }
    if (save) {
      Int32Manipulator.set_int(ivec, 0, ti[0]);
      Int32Manipulator.set_int(ivec, 4, ti[1]);
    }
    num[0] = n;
  }

  /**
   * This function produces an 8 byte checksum from input that it puts in
   * output and returns the last 4 bytes as an int.  The checksum is
   * generated via cbc mode of DES in which only the last 8 byes are
   * kept.  I would recommend not using this function but instead using
   * the EVP_Digest routines, or at least using MD5 or SHA.  This
   * function is used by Kerberos v4 so that is why it stays in the
   * library.
   * 
   * @param input               byte [] array to checksum.
   * @param input_start         Offset into input array to begin.
   * @param length              Number of bytes to process.
   * @param output              byte [] array to write into.
   * @param output_start        Offset into output to write into.
   * @param ivec                Initialization vector. Not modified.
   */

  public int cbc_cksum(byte [] input, int input_start, int length,
		       byte [] output, int output_start,
		       byte [] ivec) {
    int tin0;
    int tin1;
    int [] inout = new int [2];

    int tout0 = Int32Manipulator.bytes_to_int(ivec, 0);
    int tout1 = Int32Manipulator.bytes_to_int(ivec, 4);

    int offset = 0;
    int chunksize;
    for (; length > 0; length -= 8) {
      if (length >= 8) {
	/*
	 * We have a full 8 byte chunk.
	 */
	tin0 = Int32Manipulator.bytes_to_int(input, input_start);
	tin1 = Int32Manipulator.bytes_to_int(input, input_start+4);
	chunksize = 8;
      } else {
	/*
	 * We only have a less than 8 byte fragment.
	 */
	int [] ref_to_tin01 = new int[2];

	/* c2ln(in,tin0,tin1,length); */
	chunksize = Int32Manipulator.c2ln(input,input_start,
					  length,ref_to_tin01);
	tin0 = ref_to_tin01[0];
	tin1 = ref_to_tin01[1];
      }
      tin0 ^= tout0;
      tin1 ^= tout1;
      inout[0]= tin0;
      inout[1]= tin1;
      des_encrypt(inout, Des.ENCRYPT);
      /* fix 15/10/91 eay - thanks to keithr@sco.COM */
      tout0 = inout[0];
      tout1 = inout[1];
      input_start += chunksize;
    }
    Int32Manipulator.set_int(output, output_start, tout0);
    Int32Manipulator.set_int(output, output_start+4, tout1);
    return tout1;
  }
}

/**
 * TripleDes class. Does all modes of TripleDes encryption.
 * Written by Jeremy Allison (jra@cygnus.com) based on
 * C code from Eric Young (eay@mincom.oz.au).
 */

class TripleDes implements DesCrypt {

  /**
   * Constant. Pass this to functions that have a boolean
   * encrypt parameter to tell them to encrypt the data.
   */
  public static final boolean ENCRYPT = true;

  /**
   * Constant. Pass this to functions that have a boolean
   * encrypt parameter to tell them to decrypt the data.
   */
  public static final boolean DECRYPT = false;

  private DesKey ks1_;
  private DesKey ks2_;
  private DesKey ks3_;

  /**
   * TripleDes constructor. Takes two DesKey objects. The first
   * key is used as the third key internally. Does not check
   * either key for parity or weakness.
   *
   * @param key1             First (and third) key
   * @param key2             Second key.
   */

  public TripleDes(DesKey key1, DesKey key2) {
    ks1_ = key1;
    ks2_ = key2;
    ks3_ = ks1_;
  }

  /**
   * TripleDes constructor. Takes three DesKey objects.
   * Does not check keys for parity or weakness.
   *
   * @param key1             First key.
   * @param key2             Second key.
   * @param key3             Third key.
   */

  public TripleDes(DesKey key1, DesKey key2, DesKey key3) {
    ks1_ = key1;
    ks2_ = key2;
    ks3_ = key3;
  }

  /**
   * Do the ecb (Encrypt/Decrypt 8 bytes electronic code book)
   * mode. Encrypts 8 bytes starting at offset input_start in
   * byte array input and writes them out at offset output_start in
   * byte array output.
   *
   * @param input          Input byte [] array.
   * @param input_start    Offset into input to start encryption.
   * @param output         Output byte [] array.
   * @param output_start   Offset into output to place result.
   * @param encrypt        Pass Des.ENCRYPT to encrypt, Des.DECRYPT to
   * decrypt.
   *
   */

  public void ecb_encrypt(byte [] input, int input_start,
			  byte [] output, int output_start,
			  boolean encrypt) {
    int [] ref_to_ll = new int [2];
    int [] ref_to_l0 = new int [1];
    int [] ref_to_l1 = new int [1];

    ref_to_l0[0] = Int32Manipulator.bytes_to_int(input, input_start);
    ref_to_l1[0] = Int32Manipulator.bytes_to_int(input, input_start+4);

    Des des1 = new Des(ks1_);
    Des des2 = new Des(ks2_);
    Des des3 = new Des(ks3_);

    des1.IP(ref_to_l0, ref_to_l1);

    ref_to_ll[0] = ref_to_l0[0];
    ref_to_ll[1] = ref_to_l1[0];

    des1.des_encrypt2(ref_to_ll, encrypt);
    des2.des_encrypt2(ref_to_ll, !encrypt);
    des3.des_encrypt2(ref_to_ll, encrypt);

    ref_to_l0[0] = ref_to_ll[0];
    ref_to_l1[0] = ref_to_ll[1];
    des1.FP(ref_to_l1, ref_to_l0);

    Int32Manipulator.set_int(output, output_start, ref_to_l0[0]);
    Int32Manipulator.set_int(output, output_start+4, ref_to_l1[0]);
  }

  /**
   * Do the cbc (Cypher block chaining mode)
   * encrypt/decrypt. This updates the ivec array, and is equivalent
   * to the des_ncbc_encrypt in the C library.
   *
   * @param input          Input byte [] array.
   * @param input_start    Offset into input to start encryption.
   * @param length         Number of bytes to encrypt.
   * @param output         Output byte [] array.
   * @param output_start   Offset into output to place result.
   * @param ivec           Initialization vector. A byte [] array of length 8. Updated on exit.
   * @param encrypt        Pass Des.ENCRYPT to encrypt, Des.DECRYPT to
   * decrypt.
   */

  public void cbc_encrypt(byte [] input, int input_start, int length,
			  byte [] output, int output_start,
			  byte [] ivec,
		   boolean encrypt) {
    int [] ref_to_tin0 = new int [1];
    int [] ref_to_tin1 = new int [1];
    int [] inout = new int[2];
    int [] ref_to_tout0 = new int [1];
    int [] ref_to_tout1 = new int [1];
    int chunksize;

    Des des1 = new Des(ks1_);
    Des des2 = new Des(ks2_);
    Des des3 = new Des(ks3_);

    if(encrypt == TripleDes.ENCRYPT) {
      /*
       * Encrypt.
       */
      ref_to_tout0[0] = Int32Manipulator.bytes_to_int(ivec,0);
      ref_to_tout1[0] = Int32Manipulator.bytes_to_int(ivec,4);

      /*
       * Deal with input 8 bytes at a time.
       */
      for (; length > 0; length -= 8) {
	if (length >= 8) {
	  /*
	   * We have a full 8 byte chunk.
	   */
	  ref_to_tin0[0] = Int32Manipulator.bytes_to_int(input, input_start);
	  ref_to_tin1[0] = Int32Manipulator.bytes_to_int(input, input_start+4);
	  chunksize = 8;
	} else {
	  /*
	   * We only have a less than 8 byte fragment.
	   */
	  int [] ref_to_tin01 = new int [2];

	  /* c2ln(in,tin0,tin1,length); */
	  chunksize = Int32Manipulator.c2ln(input, input_start, 
					    length, ref_to_tin01);
	  ref_to_tin0[0] = ref_to_tin01[0];
	  ref_to_tin1[0] = ref_to_tin01[1];
	}
	ref_to_tin0[0] ^= ref_to_tout0[0];
	ref_to_tin1[0] ^= ref_to_tout1[0];

	des1.IP(ref_to_tin0,ref_to_tin1);

	inout[0] = ref_to_tin0[0];
	inout[1] = ref_to_tin1[0];
	des1.des_encrypt2(inout,Des.ENCRYPT);
	des2.des_encrypt2(inout,Des.DECRYPT);
	des3.des_encrypt2(inout,Des.ENCRYPT);
	ref_to_tout0[0] = inout[0];
	ref_to_tout1[0] = inout[1];

	des1.FP(ref_to_tout1,ref_to_tout0);
	
	Int32Manipulator.set_int(output, output_start,   ref_to_tout0[0]);
	Int32Manipulator.set_int(output, output_start+4, ref_to_tout1[0]);
	input_start += chunksize;
	output_start += chunksize;
      }
      Int32Manipulator.set_int(ivec, 0, ref_to_tout0[0]);
      Int32Manipulator.set_int(ivec, 4, ref_to_tout1[0]);
    } else {
      /*
       * Decrypt.
       */
      int xor0 = Int32Manipulator.bytes_to_int(ivec, 0);
      int xor1 = Int32Manipulator.bytes_to_int(ivec, 4);

      /*
       * Deal with input 8 bytes at a time.
       */
      for (; length > 0; length -=8) {
	int t0;
	int t1;

	ref_to_tin0[0] = Int32Manipulator.bytes_to_int(input, input_start);
	ref_to_tin1[0] = Int32Manipulator.bytes_to_int(input, input_start+4);
	
	t0 = ref_to_tin0[0];
	t1 = ref_to_tin1[0];
	des1.IP(ref_to_tin0,ref_to_tin1);  

	inout[0] = ref_to_tin0[0];
	inout[1] = ref_to_tin1[0];

	des3.des_encrypt2(inout,Des.DECRYPT);
	des2.des_encrypt2(inout,Des.ENCRYPT);
	des1.des_encrypt2(inout,Des.DECRYPT);
	ref_to_tout0[0] = inout[0];
	ref_to_tout1[0] = inout[1];

	des1.FP(ref_to_tout1,ref_to_tout0);

	ref_to_tout0[0] ^= xor0;
	ref_to_tout1[0] ^= xor1;
	if(length >= 8) {
	  Int32Manipulator.set_int(output, output_start,   ref_to_tout0[0]);
	  Int32Manipulator.set_int(output, output_start+4, ref_to_tout1[0]);
	  chunksize = 8;
	} else {
	  chunksize = Int32Manipulator.l2cn(output, output_start, length, 
					    ref_to_tout0[0], ref_to_tout1[0]);
	}

	xor0 = t0;
	xor1 = t1;
	input_start += chunksize;
	output_start += chunksize;
      }
      Int32Manipulator.set_int(ivec, 0, xor0);
      Int32Manipulator.set_int(ivec, 4, xor1);
    }
  }

  /**
   * Do the CFB mode with 64 bit feedback. Used to encrypt/decrypt
   * arbitrary numbers of bytes. To use this initialize num[0] to zero
   * and set input_start to the correct offset into input, and length to
   * the number of bytes following that offset that you wish to encrypt
   * before calling.
   *
   * @param input          Input byte [] array.
   * @param input_start    Offset into input to start encryption.
   * @param length         Number of bytes to encrypt.
   * @param output         Output byte [] array.
   * @param output_start   Offset into output to place result.
   * @param ivec           Initialization vector. A byte [] array of length 8. Updated on exit.
   * @param num            Reference to an int used to keep track of 'how far' we are though ivec. Updated on exit.
   * @param encrypt        Pass Des.ENCRYPT to encrypt, Des.DECRYPT to
   * decrypt.
   */

  public void cfb64_encrypt(byte [] input, int input_start, int length, 
			    byte [] output, int output_start,
			    byte [] ivec,
			    int [] num, 
			    boolean encrypt) {
    int n = num[0];
    int [] ti = new int [2];
    Des des1 = new Des(ks1_);
    Des des2 = new Des(ks2_);
    Des des3 = new Des(ks3_);
    int [] ref_to_v0 = new int [1];
    int [] ref_to_v1 = new int [1];
    int v0;
    int v1;

    byte c;
    byte cc;

    if( encrypt == Des.ENCRYPT) {
      /*
       * Encrypt.
       */
      while ((length--) != 0) {
	if (n == 0) {
	  v0 = Int32Manipulator.bytes_to_int(ivec, 0); 
	  v1 = Int32Manipulator.bytes_to_int(ivec, 4); 

	  ref_to_v0[0] = v0;
	  ref_to_v1[0] = v1;
	  des1.IP(ref_to_v0,ref_to_v1);
	  v0 = ref_to_v0[0];
	  v1 = ref_to_v1[0];

	  ti[0] = v0;
	  ti[1] = v1;

	  des1.des_encrypt2(ti,Des.ENCRYPT);
	  des2.des_encrypt2(ti,Des.DECRYPT);
	  des3.des_encrypt2(ti,Des.ENCRYPT);
	  v0 = ti[0];
	  v1 = ti[1];

	  ref_to_v0[0] = v0;
	  ref_to_v1[0] = v1;
	  des1.FP(ref_to_v1,ref_to_v0);
	  v0 = ref_to_v0[0];
	  v1 = ref_to_v1[0];

	  Int32Manipulator.set_int(ivec, 0, v0);
	  Int32Manipulator.set_int(ivec, 4, v1);
	}
	c = (byte)(input[input_start] ^ ivec[n]);
	output[output_start] = c;
	input_start++;
	output_start++;
	ivec[n] = c;
	n = (n + 1) & 0x7;
      }
    } else {
      /*
       * Decrypt.
       */
      while ((length--) != 0) {
	if (n == 0) {
	  v0 = Int32Manipulator.bytes_to_int(ivec, 0);
	  v1 = Int32Manipulator.bytes_to_int(ivec, 4);

	  ref_to_v0[0] = v0;
	  ref_to_v1[0] = v1;
	  des1.IP(ref_to_v0, ref_to_v1);
	  v0 = ref_to_v0[0];
	  v1 = ref_to_v1[0];

	  ti[0] = v0;
	  ti[1] = v1;
	  des1.des_encrypt2(ti,Des.ENCRYPT);
	  des2.des_encrypt2(ti,Des.DECRYPT);
	  des3.des_encrypt2(ti,Des.ENCRYPT);

	  v0 = ti[0];
	  v1 = ti[1];

	  ref_to_v0[0] = v0;
	  ref_to_v1[0] = v1;
	  des1.FP(ref_to_v1, ref_to_v0);
	  v0 = ref_to_v0[0];
	  v1 = ref_to_v1[0];

	  Int32Manipulator.set_int(ivec, 0, v0);
	  Int32Manipulator.set_int(ivec, 4, v1);
	}
	cc = input[input_start];
	c = ivec[n];
	ivec[n] = cc;
	output[output_start] = (byte)(c ^ cc);
	input_start++;
	output_start++;
	n = (n + 1) & 0x7;
      }
    }
    num[0] = n;
  }

  /**
   * Do the OFB mode with 64 bit feedback. Used to encrypt/decrypt
   * arbitrary numbers of bytes. To use this initialize num[0] to zero
   * and set input_start to the correct offset into input, and length to
   * the number of bytes following that offset that you wish to encrypt
   * before calling.
   *
   * @param input          Input byte [] array.
   * @param input_start    Offset into input to start encryption.
   * @param length         Number of bytes to encrypt.
   * @param output         Output byte [] array.
   * @param output_start   Offset into output to place result.
   * @param ivec           Initialization vector. A byte [] array of length 8. Updated on exit.
   * @param num            Reference to an int used to keep track of 'how far' we are though ivec. Updated on exit.
   */

  public void ofb64_encrypt(byte [] input, int input_start, int length, 
			    byte [] output, int output_start,
			    byte [] ivec,
			    int [] num) {
    int [] ref_to_v0 = new int [1];
    int [] ref_to_v1 = new int [1];
    int v0;
    int v1;
    int n = num[0];
    int [] ti = new int [2];
    boolean save = false;
    byte [] dp = new byte [8];
    Des des1 = new Des(ks1_);
    Des des2 = new Des(ks2_);
    Des des3 = new Des(ks3_);

    v0 = Int32Manipulator.bytes_to_int(ivec, 0);
    v1 = Int32Manipulator.bytes_to_int(ivec, 4);
    ti[0] = v0;
    ti[1] = v1;
    Int32Manipulator.set_int(dp, 0, ti[0]);
    Int32Manipulator.set_int(dp, 4, ti[1]);
    while((length--) != 0) {
      if (n == 0) { 
	ref_to_v0[0] = v0;
	ref_to_v1[0] = v1;
	des1.IP(ref_to_v0, ref_to_v1);
	v0 = ref_to_v0[0];
	v1 = ref_to_v1[0];
	ti[0] = v0;
	ti[1] = v1;

	des1.des_encrypt2(ti,Des.ENCRYPT);
	des2.des_encrypt2(ti,Des.DECRYPT);
	des3.des_encrypt2(ti,Des.ENCRYPT);
	v0 = ti[0];
	v1 = ti[1];

	ref_to_v0[0] = v0;
	ref_to_v1[0] = v1;
	des1.FP(ref_to_v1, ref_to_v0);
	v0 = ref_to_v0[0];
	v1 = ref_to_v1[0];

	Int32Manipulator.set_int(dp, 0, v0);
	Int32Manipulator.set_int(dp, 4, v1);
	save = true;
      }
      output[output_start] = (byte)(input[input_start] ^ dp[n]);
      input_start++;
      output_start++;
      n = (n + 1) & 0x7;
    }
    if (save) {
      Int32Manipulator.set_int(ivec, 0, v0);
      Int32Manipulator.set_int(ivec, 4, v1);
    }
    num[0] = n;
  }

}
