/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"
#include "mutt_curses.h"
#include "rfc822.h"

#include <pwd.h>
#include <string.h>

static LIST *Alias_list = NULL;

static ADDRESS *lookup_alias (const char *s)
{
  ALIAS *t = Aliases;

  while (t) {
    if (!strcmp(s, t->name)) return(t->addr);
    t = t->next;
  }
  return (NULL);   /* no such alias */
}

static ADDRESS *mutt_expand_aliases_r (ADDRESS *a)
{
  ADDRESS *head = 0, *last = 0, *t, *w;
  LIST *u;
  char i;

  while (a)
  {
    if (a->host && *a->host == '@' && !a->personal)
    {
      t = lookup_alias(a->mailbox);

      if (!t)
      {	
	if (!a->personal)
	{
	  struct passwd *pw = getpwnam(a->mailbox);
	  char buffer[256], *p;

	  if (pw)
	  {
	    strcpy(buffer, pw->pw_gecos);
	    if ((p = strchr(buffer, ','))) *p = 0;
	    a->personal = safe_strdup(buffer);
	  }
	}

        if (last)
	{
          last->next = a;
          last = last->next;
          a = a->next;
          last->next = 0;
        }
        else
	{
          head = last = a;
          a = a->next;
          last->next = 0;
        }
        continue;
      }
      else
      {
        u = Alias_list;
        i = 0;
        while (u)
	{
          if (strcmp(a->mailbox, u->data) == 0) { /* alias already found */
            i = 1;
            break;
          }
          u = u->next;
        }
        if (!i)
	{
          u = (LIST *) safe_malloc(sizeof(LIST));
          u->data = safe_strdup(a->mailbox);
          u->next = Alias_list;
          Alias_list = u;
	  w = rfc822_cpy_adr(t);
	  w = mutt_expand_aliases_r(w);
	  if (head)
	    last->next = w;
	  else
	    head = last = w;
	  while (last->next) last = last->next;
        }
	t = a;
	a = a->next;
	t->next = 0;
	mutt_free_address(&t);
      }
    }
    else
    {
      if (head)
      {
	last->next = a;
	last = last->next;
	a = a->next;
	last->next = 0;
      }
      else
      {
	head = last = a;
	a = a->next;
	head->next = 0;
      }
    }
  }

  /*
   * now qualify all local addresses
   */
  t = head;
  while (t)
  {
    if (t->host && *t->host == '@')
    {
      safe_free((void **)&t->host);
      t->host = safe_strdup(Fqdn);
    }
    t = t->next;
  }
  
  return (head);
}

ADDRESS *mutt_expand_aliases (ADDRESS *a)
{
  ADDRESS *t;

  t = mutt_expand_aliases_r (a);

  /* free up Alias_list */

  mutt_free_list(&Alias_list);
  return(t);
}

void ci_mkalias (ENVELOPE *cur)
{
  ALIAS *new, *t;
  char buffer[LONG_STRING];
  char prompt[SHORT_STRING];
  FILE *rc;
  int r;

  strcpy(buffer, cur->reply_to ? cur->reply_to->mailbox : cur->from->mailbox);
  if (ci_get_field("Alias as: ", buffer, sizeof(buffer), 0) != 0 || !buffer[0])
    return;

  /* check to see if the user already has an alias defined */
  if (lookup_alias (buffer))
  {
    beep();
    mutt_error ("You already have an alias defined with that name!");
    return;
  }

  new = (ALIAS *) safe_malloc (sizeof (ALIAS));
  new->name = safe_strdup (buffer);
  new->next = 0;
  new->addr = 0;

  mutt_simple_address (buffer, sizeof(buffer), cur->reply_to ? cur->reply_to :
		       cur->from);

  do {
    mutt_free_address (&new->addr);
    if (ci_get_field ("Address: ", buffer, sizeof(buffer), 0) != 0 || !buffer[0])
    {
      mutt_free_alias (&new);
      return;
    }
    r = mutt_parse_adrlist (&new->addr, buffer, Fqdn);
    if (r == -1)
      mutt_error (ParseError);
  } while (r == -1);

  if (cur->reply_to && cur->reply_to->personal)
    strfcpy (buffer, cur->reply_to->personal, sizeof(buffer));
  else if (cur->from->personal)
    strfcpy(buffer, cur->from->personal, sizeof(buffer));
  else
    buffer[0] = 0;

  if (ci_get_field("Personal name: ", buffer, sizeof(buffer), 0) != 0)
  {
    mutt_free_alias(&new);
    return;
  }
  new->addr->personal = safe_strdup(buffer);

  mutt_write_address(buffer, sizeof(buffer), new->addr);
  snprintf(prompt, sizeof(prompt), "[%s = %s] Accept?", new->name, buffer);
  if (mutt_yesorno (buffer, 1) != 1)
  {
    mutt_free_alias (&new);
    return;
  }

  if ((t = Aliases))
  {
    while (t->next) t = t->next;
    t->next = new;
  }
  else
    Aliases = new;

  strfcpy (buffer, AliasFile, sizeof (buffer));
  if (ci_get_field ("Save to file: ", buffer, sizeof(buffer), M_FILE) != 0)
    return;
  mutt_expand_path (buffer, sizeof (buffer));
  if ((rc = fopen(buffer, "a")))
  {
    buffer[0] = 0;
    mutt_write_address(buffer, sizeof(buffer), new->addr);
    fprintf(rc, "alias %s %s\n", new->name, buffer);
    fclose (rc);
    mutt_error ("Alias added.");
  }
  else
    mutt_perror("fopen");
}

/*
 * This routine looks to see if the user has an alias defined for the given
 * address.
 */
ADDRESS *alias_reverse_lookup (ADDRESS *a)
{
  ALIAS *t = Aliases;
  ADDRESS *ap;

  if (!a || !a->mailbox || !a->host)
    return NULL;

  while (t)
  {
    /* cycle through all addresses if this is a group alias */
    ap = t->addr;
    while (ap)
    {
      if (ap->mailbox && ap->host &&
	  strcasecmp (ap->mailbox, a->mailbox) == 0 &&
	  strcasecmp (ap->host, a->host) == 0)
	return ap;
      ap = ap->next;
    }
    t = t->next;
  }
  return 0;
}
