/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"
#include "mutt_curses.h"
#include "keymap.h"

#include <string.h>
#include <ctype.h>

#include "functions.h"

/* contains the last key the user pressed */
int LastKey;

struct keymap_t *Keymaps[MENU_MAX];

static struct keymap_t *findkey (struct keymap_t *k, int c)
{
  while (k)
  {
    if (k->key == c)
      return k;
    k = k->next;
  }
  return NULL;
}

static struct keymap_t *allocKey (void)
{
  struct keymap_t *p = (struct keymap_t *)safe_malloc(sizeof(struct keymap_t));

  memset(p, 0, sizeof(struct keymap_t));
  return p;
}

int parsekey (char **s)
{
  char *p = *s;

  if (*p == '\\')
  {
    p++;
    if (tolower (*p) == 'c') /* control char */
    {
      *s += 3;
      return (toupper (*(p+1)) - '@');
    }
    else if (*p == 'n')
    {
      *s += 2;
      return '\n';
    }
    else if (*p == 'r')
    {
      *s += 2;
      return '\r';
    }
    else if (*p == 't')
    {
      *s += 2;
      return '\t';
    }
    else if (*p == 'e')
    {
      *s += 2;
      return '\033';
    }
    else
    {
      *s += 2;
      return (*p);
    }
  }
  else if (strcasecmp ("pageup", p) == 0)
  {
    (*s) += 6;
    return KEY_PPAGE;
  }
  else if (strcasecmp ("pagedown", p) == 0)
  {
    *s += 8;
    return KEY_NPAGE;
  }
  else if (strcasecmp ("up", p) == 0)
  {
    *s += 2;
    return KEY_UP;
  }
  else if (strcasecmp ("down", p) == 0)
  {
    *s += 4;
    return KEY_DOWN;
  }
  else if (strcasecmp ("right", p) == 0)
  {
    *s += 5;
    return KEY_RIGHT;
  }
  else if (strcasecmp ("left", p) == 0)
  {
    *s += 4;
    return KEY_LEFT;
  }
  else if (strcasecmp ("delete", p) == 0)
  {
    *s += 6;
    return KEY_DC;
  }
  else if (strcasecmp ("backspace", p) == 0)
  {
    *s += 9;
    return KEY_BACKSPACE;
  }
  else if (!strcasecmp ("home", p))
  {
    *s += 4;
    return KEY_HOME;
  }
  else if (!strcasecmp ("end", p))
  {
    *s += 3;
    return KEY_END;
  }
  else
  {
    (*s)++;
    return (*p);
  }
}

static struct keymap_t *genkey (char **s)
{
  struct keymap_t *r = NULL;
  struct keymap_t *tmp = NULL;

  while (**s)
  {
    if (r)
    {
      tmp->seq = allocKey();
      tmp = tmp->seq;
    }
    else
      r = tmp = allocKey();
    tmp->key = parsekey(s);
  }
  return r;
}

void bindkey (char *s, int menu, int op, char *macro)
{
  struct keymap_t *top = Keymaps[menu];
  struct keymap_t *tmp;
  int key;

  if (!top)
  {
    Keymaps[menu] = tmp = genkey(&s);
    while (tmp->seq) tmp = tmp->seq;
    tmp->op = op;
    tmp->macro = safe_strdup(macro);
    return;
  }

  while (*s)
  {
    key = parsekey (&s);
    tmp = findkey (top, key);
    if (!tmp)
    {
      /* the rest of the sequence needs to be allocated */
      while (top->next) top = top->next;
      top->next = allocKey ();
      top = top->next;
      top->key = key;
      top->seq = genkey (&s);
      while (top->seq) top = top->seq;
      top->op = op;
      top->macro = safe_strdup (macro);
      return;
    }
    if (*s)
    {
      if (!tmp->seq)
      {
	if (tmp->op != OP_NULL)
	  return; /* looks like the user already bound the prefix! */

	/* just a null-op, so generate the rest of the sequence */
	tmp->seq = genkey (&s);
	tmp = tmp->seq;
	while (tmp->seq) tmp = tmp->seq;
	tmp->op = op;
	tmp->macro = safe_strdup (macro);
	return;
      }
      top = tmp->seq;
    }
    else
    {
      /* override the previous value */
      tmp->op = op;
      safe_free ((void **)&tmp->macro);
      tmp->macro = safe_strdup (macro);
      return;
    }
  }
}

static struct keymap_t *lookupkey (int key, struct keymap_t *map)
{
  while (map)
  {
    if (map->key == key)
      return map;
    map = map->next;
  }
  return NULL;
}

static void push_string (char *s)
{
  char *p = s + strlen (s) - 1;

  while (p >= s) ungetch (*p--);
}

int dokey (int menu)
{
  struct keymap_t *map = Keymaps[menu];

  FOREVER
  {
    if ((LastKey = ci_getch ()) == ERR)
      return (-1);
    map = lookupkey (LastKey, map);
    if (!map)
      return OP_NULL;
    if (!map->seq)
    {
      if (map->op == OP_MACRO)
      {
	push_string (map->macro);
	map = Keymaps[menu];
	continue;
      }
      return map->op;
    }
    map = map->seq;
  }
}

static void create_bindings (struct binding_t *map, int menu)
{
  int i;

  for (i = 0 ; map[i].name ; i++)
    if (map[i].seq) bindkey (map[i].seq, menu, map[i].op, NULL);
}

static void generic_bindings (int menu)
{
  create_bindings (OpGeneric, menu);

  bindkey ("pagedown", menu, OP_NEXT_PAGE, NULL);
  bindkey ("right", menu, OP_NEXT_PAGE, NULL);
  bindkey ("pageup", menu, OP_PREVIOUS_PAGE, NULL);
  bindkey ("left", menu, OP_PREVIOUS_PAGE, NULL);
  bindkey ("up", menu, OP_GENERIC_PREVIOUS_ENTRY, NULL);
  bindkey ("down", menu, OP_GENERIC_NEXT_ENTRY, NULL);
}

void keymap_init (void)
{
  memset (Keymaps, 0, sizeof (struct keymap_t *) * MENU_MAX);

  create_bindings (OpMain, MENU_MAIN);
  create_bindings (OpPager, MENU_PAGER);
  create_bindings (OpSend, MENU_SEND);

  generic_bindings (MENU_POST);
  create_bindings (OpPost, MENU_POST);

  generic_bindings (MENU_SENDATTACH);
  create_bindings (OpSendAttach, MENU_SENDATTACH);

  generic_bindings (MENU_ATTACH);
  create_bindings (OpAttach, MENU_ATTACH);

  generic_bindings (MENU_FOLDER);
  generic_bindings (MENU_ALIAS);
  generic_bindings (MENU_URL);

  /* Miscellaneous extra bindings */

  bindkey (" ",	MENU_MAIN, OP_DISPLAY_MESSAGE, NULL);
  bindkey ("1", MENU_MAIN, OP_MAIN_JUMP, NULL);
  bindkey ("2", MENU_MAIN, OP_MAIN_JUMP, NULL);
  bindkey ("3", MENU_MAIN, OP_MAIN_JUMP, NULL);
  bindkey ("4", MENU_MAIN, OP_MAIN_JUMP, NULL);
  bindkey ("5", MENU_MAIN, OP_MAIN_JUMP, NULL);
  bindkey ("6", MENU_MAIN, OP_MAIN_JUMP, NULL);
  bindkey ("7", MENU_MAIN, OP_MAIN_JUMP, NULL);
  bindkey ("8", MENU_MAIN, OP_MAIN_JUMP, NULL);
  bindkey ("9", MENU_MAIN, OP_MAIN_JUMP, NULL);
  bindkey ("right", MENU_MAIN, OP_NEXT_PAGE, NULL);
  bindkey ("pagedown", MENU_MAIN, OP_NEXT_PAGE, NULL);
  bindkey ("left", MENU_MAIN, OP_PREVIOUS_PAGE, NULL);
  bindkey ("pageup", MENU_MAIN, OP_PREVIOUS_PAGE, NULL);
  bindkey ("up", MENU_MAIN, OP_MAIN_PREVIOUS_UNDELETED, NULL);
  bindkey ("down", MENU_MAIN, OP_MAIN_NEXT_UNDELETED, NULL);

  bindkey ("n", MENU_PAGER, OP_MAIN_NEXT_UNDELETED, NULL);
  bindkey ("x", MENU_PAGER, OP_PAGER_EXIT, NULL);
  bindkey ("q", MENU_PAGER, OP_PAGER_EXIT, NULL);
  bindkey ("backspace", MENU_PAGER, OP_PAGER_PREVIOUS_LINE, NULL);
  bindkey ("pagedown", MENU_PAGER, OP_NEXT_PAGE, NULL);
  bindkey ("pageup", MENU_PAGER, OP_PREVIOUS_PAGE, NULL);
  bindkey ("up", MENU_PAGER, OP_MAIN_PREVIOUS_UNDELETED, NULL);
  bindkey ("right", MENU_PAGER, OP_MAIN_NEXT_UNDELETED, NULL);
  bindkey ("down", MENU_PAGER, OP_MAIN_NEXT_UNDELETED, NULL);
  bindkey ("left", MENU_PAGER, OP_MAIN_PREVIOUS_UNDELETED, NULL);

  bindkey ("y",  MENU_SEND, OP_SEND_SEND_MESSAGE, NULL);
}
