/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"
#include "mutt_curses.h"
#include <string.h>
#include <stdlib.h>

enum {
  /* flags to doString() */
  M_TO = 1,
  M_CC,
  M_SUBJECT,
  M_FROM,
  M_DATE
};

static const struct m_flag_t {
  char *str;
  short flag;
} FlagInfo[] = {
  { "all", M_ALL },
  { "new", M_NEW },
  { "old", M_OLD },
  { "replied", M_REPLIED },
  { "flagged", M_FLAG },
  { "tagged", M_TAG },
  { "none", M_NONE },
  { "unread", M_UNREAD },
  { "read", M_READ },
  { NULL, 0 }
};

/* macros to select a single bit from a multi-byte vector */
#define setBit(B,V)   V[(B)/8] |= (1 << ((B) % 8))
#define isSet(B, V) (V[(B)/8] & (1 << ((B) % 8))) != 0

/* global variables used in this file */
static struct {

  short VecLen;                /* how many bytes in a search vector */

  /* vars used by ci_search_command() */

  char LastSearch[STRING];     /* last pattern searched for */
  unsigned char *SearchVector; /* last search vector */

} LOCAL = { 0, "", 0 };

static unsigned char *allocVec (void)
{
  unsigned char *p;

  LOCAL.VecLen = (Context->msgcount + 7) / 8;
  p = (unsigned char *)safe_malloc (LOCAL.VecLen);
  memset(p, 0, LOCAL.VecLen);
  return p;
}

static int matchesCrit (HEADER *h, int crit, int not)
{
  switch (crit) {
  case M_ALL:
    return (!not);
  case M_FLAG:
    return (not ? !h->flagged : h->flagged);
  case M_TAG:
    return (not ? !h->tagged : h->tagged);
  case M_NEW:
    return (not ? h->old || h->read : !(h->old || h->read));
  case M_UNREAD:
    return (not ? h->read : !h->read);
  case M_REPLIED:
    return (not ? !h->replied : h->replied);
  case M_OLD:
    return (not ? !h->old : h->old);
  case M_READ:
    return (not ? !h->read : h->read);
  }
  return 0;
}

static unsigned char *doKeyword (int crit, int not)
{
  unsigned char *p = allocVec();
  int i;

  for (i = 0; i < Context->msgcount; i++)
    if (matchesCrit (Context->hdrs[i], crit, not))
      setBit (i, p);
  return p;
}

static int getDate (char *s, int *day, int *month, int *year)
{
  char buf[SHORT_STRING];
  char *p = buf;
  time_t now = time (NULL);
  struct tm *tm = gmtime (&now);

  while (*s && *s != '/')
    *p++ = *s++;
  *p = 0;
  *day = atoi(buf);
  if (*day < 1 || *day > 31)
  {
    beep();
    mutt_error("Invalid day of month: %s", buf);
    return(-1);
  }
  if (!*s)
  {
    /* fill in today's month and year */
    *month = tm->tm_mon;
    *year = tm->tm_year + 1900 - EPOCH;
    return 0;
  }
  s++;
  p = buf;
  while (*s && *s != '/') *p++ = *s++;
  *month = atoi(buf);
  if (*month < 1 || *month > 12)
  {
    beep();
    mutt_error("Invalid month: %s", buf);
    return(-1);
  }
  (*month)--;
  if (!*s)
  {
    *year = tm->tm_year + 1900 - EPOCH;
    return 0;
  }
  s++;
  *year = atoi(s);
  if (*year < 1900) *year += 1900;
  *year -= EPOCH;
  return 0;
}

/*
 * special cases:
 *
 * DD/MM/YY	all messages on DD/MM/YY
 * -DD/MM/YY	all messages on/before DD/MM/YY
 * DD/MM/YY-	all messages on/after DD/MM/YY
 */

static unsigned char *doDate (char *s, int not)
{
  int i;
  int minday=0, minmonth=0, minyear=0;
  int maxday=31, maxmonth=11;
  int maxyear=50; /* Arbitrary year in the future.  Don't set this too high
		     or mutt_gmsecs() returns something larger than will
		     fit in a time_t on some systems */
  char *p;
  unsigned char *vec;
  time_t now, mint, maxt;

  if ((p = strchr(s, '-'))) *p++ = 0;
  if (*s)
  {
    /* mininum date specified */
    if (getDate(s, &minday, &minmonth, &minyear) != 0) return NULL;
  }
  if (p)
  {
    if (*p)
    {
      /* max date */
      if (getDate(p, &maxday, &maxmonth, &maxyear) != 0) return NULL;
    }
  }
  else
  {
    /* search for messages on a specific day */
    maxday = minday;
    maxmonth = minmonth;
    maxyear = minyear;
  }
  mint = mutt_gmsecs(minyear, minmonth, minday, 0, 0, 0);
  maxt = mutt_gmsecs(maxyear, maxmonth, maxday, 23, 59, 59);
  vec = allocVec();
  for (i=0; i < Context->msgcount; i++)
  {
    now = mutt_gmtime(Context->hdrs[i]);
    if (now >= mint && (now <= maxt))
    {
      if (!not) setBit(i, vec);
    }
    else if (not)
      setBit(i, vec);
  }
  return vec;
}

static int isFlag (char **s)
{
  int i;

  for (i=0; FlagInfo[i].str; i++)
  {
    if (!strcasecmp (*s, FlagInfo[i].str))
    {
      *s += strlen (FlagInfo[i].str);
      return (FlagInfo[i].flag);
    }
  }
  return 0;
}

#ifdef HAVE_REGCOMP
/* if no uppercase letters are given, do a case-insensitive search */
static int whichCase (const char *s)
{
  while (*s)
  {
    if (isalpha (*s) && isupper (*s))
      return 0; /* case-sensitive */
    s++;
  }
  return REG_ICASE; /* case-insensitive */
}
#endif

static unsigned char *doString (int field, char *s, int not)
{
  unsigned char *vec = allocVec ();
  char buf[LONG_STRING];
  int i, err;
  RE_TYPE (exp);

  if (field == M_DATE)
    return doDate (s, not);

  if ((err = REGCOMP (exp, s, whichCase (s))))
  {
#ifdef HAVE_REGCOMP
    regerror (err, &exp, buf, sizeof (buf));
    mutt_error ("%s: %s", s, buf);
#elif defined(HAVE_REGCMP)
    mutt_error ("%s: %s", s, exp);
#else
    mutt_error ("Error in regexp: %s", s);
#endif
    safe_free ((void **)&vec);
    return NULL;
  }
  for (i = 0; i < Context->msgcount; i++)
  {
    buf[0] = 0;
    switch (field)
    {
      case M_FROM:
	mutt_write_address(buf, sizeof(buf), Context->hdrs[i]->env->from);
	break;
      case M_TO:
	mutt_write_address(buf, sizeof(buf), Context->hdrs[i]->env->to);
	break;
      case M_CC:
	mutt_write_address(buf, sizeof(buf), Context->hdrs[i]->env->cc);
	break;
      case M_SUBJECT:
	if (Context->hdrs[i]->env->subject)
	  strfcpy(buf, Context->hdrs[i]->env->subject, sizeof(buf));
	break;
    }
    if (REGEXEC(exp, buf) == 0) {
      if (!not) setBit(i, vec);
    }
    else if (not)
      setBit(i, vec);
  }
  REGFREE(exp);
  return vec;
}

static void doThread_r (HEADER *hdr, unsigned char *p)
{
  while (hdr)
  {
    setBit(hdr->msgno - 1, p);
    doThread_r(hdr->child, p);
    hdr = hdr->next;
  }
}

static unsigned char *doThread (HEADER *hdr)
{
  unsigned char *vec = allocVec();

  while (hdr->parent) hdr = hdr->parent;
  setBit(hdr->msgno - 1, vec);
  doThread_r(hdr->child, vec);
  return vec;
}

static void doOR (unsigned char *a, unsigned char *b)
{
  int i;

  for (i=0; i < LOCAL.VecLen; i++) a[i] |= b[i];
}

static void doAND (unsigned char *a, unsigned char *b)
{
  int i;

  for (i=0; i < LOCAL.VecLen; i++) a[i] &= b[i];
}

/*
 * Returns a vector with bits set to indicate which messages matched the given
 * pattern or NULL if there was an error in the pattern.
 *
 * The ``hdr'' argument is a pointer to the current message (required for
 * handling of the ``thread'' keyword)
 *
 * This routine gets called recursively to handle parentheses in the expression
 */

unsigned char *mutt_match_pattern (char *s, HEADER *hdr)
{
  int i;
  int not = 0; /* Do logical NOT */
  int or = 1;  /* Do logical OR.  Initialized to 1 so the first pattern we
		  match gets OR'd with the zero-initialized ``curvec'' */
  char buf[STRING];
  unsigned char *tmpvec = NULL;
  unsigned char *curvec = allocVec();
  char *p;

  while (*(s = skip_whitespace (s)))
  {
    if (*s == '!')
    {
      not= !not;
      s++;
    }
    else if (*s == '~')
    {
      int op;

      s++;
      if (*s == 't')
	op = M_TO;
      else if (*s == 'f')
	op = M_FROM;
      else if (*s == 'c')
	op = M_CC;
      else if (*s == 's')
	op = M_SUBJECT;
      else if (*s == 'd')
	op = M_DATE;
      else
      {
	mutt_error ("Unknown operator: %c", *s);
	safe_free ((void **)&curvec);
	return NULL;
      }

      /* get the string to match */
      s = skip_whitespace(s+1);
      p = buf;
      if (*s == '\'' || *s == '\"') /* handle quotes (for spaces) */
      {
	char q = *s++;

	while (*s && *s != q)
	{
	  if (*s == '\\') s++; /* quote the next character */
	  *p++ = *s++;
	}
	s++; /* move past the end quote char */
      }
      else
        while (*s && strchr(" !|~(", *s) == NULL)
	{
	  if (*s == '\\') s++; /* quote the next character */
	  *p++ = *s++;
	}
      *p = 0;

      if ((tmpvec = doString (op, buf, not)) == NULL)
      {
	safe_free((void **)&curvec);
	break;
      }

      if (or)
	doOR(curvec, tmpvec);
      else
	doAND(curvec, tmpvec);
      safe_free((void **)&tmpvec);
      not=0;
      or=0;
    }
    else if ((i = isFlag (&s)))
    {
      tmpvec = doKeyword (i, not);
      if (or)
	doOR (curvec, tmpvec);
      else
	doAND (curvec, tmpvec);
      safe_free ((void **)&tmpvec);
      not = 0;
      or = 0;
    }
    else if (!strcasecmp (s, "thread"))
    {
      s += 6;
      tmpvec = doThread (hdr);
      if (not)
      {
	for (i=0; i<LOCAL.VecLen; i++)
	  tmpvec[i] = ~tmpvec[i];
      }
      if (or)
	doOR(curvec, tmpvec);
      else
	doAND(curvec, tmpvec);
      safe_free((void **)&tmpvec);
      not = 0;
      or = 0;
    }
    else if (isdigit (*s)) /* range of messages */
    {
      int min, max;

      if (sscanf(s, "%d-%d", &min, &max) != 2 || min < 1 ||
	  max > Context->msgcount)
      {
	mutt_error("Invalid range at: %s", s);
	safe_free((void **)&curvec);
	return NULL;
      }
      tmpvec = allocVec();
      for (i=0; i<Context->msgcount; i++)
      {
	if (i >= min-1 && i < max)
	{
	  if (!not) setBit(i, tmpvec);
	}
	else if (not)
	  setBit(i, tmpvec);
      }
      if (or)
	doOR(curvec, tmpvec);
      else
	doAND(curvec, tmpvec);
      not=0;
      or=0;
      safe_free((void **)&tmpvec);

      /* move to the next token */
      while (*s && strchr(" |~!(", *s) == NULL) s++;
    }
    else if (*s == '(')
    {
      int count = 1; /* parenthesis level */

      s = skip_whitespace(s+1);
      p = buf;
      while (*s && count)
      {
	if (*s == '(')
	  count++;
	else if (*s == ')')
	  count--;
	*p++ = *s++;
      }
      if (count)
      {
	beep();
	mutt_error("Mismatched parentheses.");
	safe_free((void **)&curvec);
	return NULL;
      }
      *--p = 0;
      if ((tmpvec = mutt_match_pattern(buf, hdr)) == NULL)
      {
	safe_free((void **)&curvec);
	break;
      }
      if (not)
      {
	for (i=0; i < LOCAL.VecLen; i++) tmpvec[i] = ~tmpvec[i];
      }
      if (or)
	doOR (curvec, tmpvec);
      else
	doAND (curvec, tmpvec);
      safe_free ((void **)&tmpvec);
      not=0;
      or=0;
    }
    else if (*s == '|')
    {
      s++;
      or=1;
      not=0;
    }
    else
    {
      beep();
      mutt_error("Error in pattern at: %s", s);
      safe_free((void **)&curvec);
      break;
    }
  }
  return curvec;
}

/* convert a simple search into a real request */
static void check_simple (char *s, size_t len)
{
  char tmp[SHORT_STRING];
  int i, j;

  if (!strpbrk (s, "!()~|")) /* yup, so spoof a real request */
  {
    if (strchr (s, '\'')) /* check for quotes that need quoting */
    {
      for (i=0,j=0; s[i] && j<sizeof(tmp)-1; i++)
      {
	if (s[i] == '\'')
	  tmp[j++] = '\\';
	tmp[j++] = s[i];
      }
      tmp[j] = 0;
    }
    else
      strfcpy (tmp, s, sizeof (tmp));

    snprintf (s, len, "~f '%s'|~s '%s'", tmp, tmp);
  }
}

int ci_pattern_function (int op, char *prompt, HEADER *hdr)
{
  char buf[STRING] = "";
  unsigned char *vec;
  int i;

  if (ci_get_field (prompt, buf, sizeof (buf), 0) != 0 || !buf[0])
    return (-1);

  if ((vec = mutt_match_pattern (buf, hdr)) == NULL)
    return (-1);

  if (op == M_LIMIT)
  {
    for (i=0; i<Context->msgcount; i++)
      Context->hdrs[i]->virtual = -1;
    Context->vcount=0;
  }

  for (i=0; i<Context->msgcount; i++)
    if (isSet (i, vec))
    {
      switch (op)
      {
	case M_DELETE:
	  Context->hdrs[i]->deleted = 1;
	  break;
	case M_UNDELETE:
	  Context->hdrs[i]->deleted = 0;
	  break;
	case M_TAG:
	  if (!Context->hdrs[i]->tagged)
	  {
	    Context->hdrs[i]->tagged = 1;
	    Context->tagged++;
	  }
	  break;
	case M_UNTAG:
	  if (Context->hdrs[i]->tagged)
	  {
	    Context->hdrs[i]->tagged = 0;
	    Context->tagged--;
	  }
	  break;
	case M_LIMIT:
	  Context->hdrs[i]->virtual = Context->vcount;
	  Context->v2r[Context->vcount] = i;
	  Context->vcount++;
	  break;
      }
    }

  if (op == M_LIMIT && !Context->vcount)
  {
    beep ();
    mutt_error ("No messages matched criteria.");
    /* restore full display */
    Context->vcount = 0;
    for (i=0; i<Context->msgcount; i++)
    {
      Context->hdrs[i]->virtual = i;
      Context->v2r[i] = i;
    }
    Context->vcount = Context->msgcount;
  }
  safe_free ((void **)&vec);
  return 0;
}

int ci_search_command (int cur, int reverse)
{
  int i, j;
  char buf[STRING];
  
  strfcpy (buf, LOCAL.LastSearch, sizeof(buf));
  if (ci_get_field (reverse ? "Reverse search for: " : "Search for: ",
		    buf, sizeof (buf), 0) != 0 || !buf[0])
    return (-1);

  /*
   * see if the search pattern changed.  if so, recompute the pattern vector.
   * otherwise use the previously computed pattern to speed things up.
   */
  if (!LOCAL.SearchVector || strcmp (buf, LOCAL.LastSearch))
  {
    safe_free ((void **)&LOCAL.SearchVector);
    strfcpy (LOCAL.LastSearch, buf, sizeof(LOCAL.LastSearch));
    check_simple (buf, sizeof (buf));
    if (!(LOCAL.SearchVector = mutt_match_pattern (buf, Context->hdrs[Context->v2r[cur]])))
      return(-1);
  }
  
  i = reverse ? cur - 1 : cur + 1;
  for (j=0; j != Context->vcount; j++)
  {
    if (i > Context->vcount - 1)
    {
      i = 0;
      mutt_error ("Search wrapped to top.");
    }
    else if (i < 0)
    {
      i = Context->vcount - 1;
      mutt_error ("Search wrapped to bottom.");
    }

    if (isSet(Context->v2r[i], LOCAL.SearchVector))
      return i;

    if (reverse)
      i--;
    else
      i++;
  }
  beep ();
  mutt_error ("Not found.");
  return (-1);
}

/* search for a regexp in the body of a message starting with "msgno" */
int mutt_body_search (int msgno, const char *pat)
{
  int i;
  long bytes;
  char buf[LONG_STRING];
  RE_TYPE (re);

  if ((i = REGCOMP (re, pat, whichCase (pat))) != 0)
  {
    beep ();
#ifdef HAVE_REGCOMP
    regerror (i, &re, buf, sizeof (buf));
    mutt_error (buf);
#elif HAVE_REGCMP
    mutt_error (re);
#else
    mutt_error ("Error in regular expression.");
#endif
    return (-1);
  }

  buf[sizeof (buf) - 1] = 0;

  for (i = msgno + 1; i < Context->vcount; i++)
  {
    fseek (Context->folder->fp, Context->hdrs[Context->v2r[i]]->content->offset, 0);
    for (bytes = Context->hdrs[Context->v2r[i]]->content->length; bytes > 0; )
    {
      fgets (buf, sizeof (buf) - 1, Context->folder->fp);
      if (REGEXEC (re, buf) == 0)
      {
	REGFREE (re);
	return (i);
      }
      bytes -= strlen (buf);
    }
  }

  REGFREE (re);

  beep ();
  mutt_error ("Not found.");

  return (-1);
}
