/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

/* WARNING!!!  WARNING!!!  WARNING!!!  WARNING!!!  WARNING!!!  WARNING!!!
 *
 * Legal for distribution in the U.S./Canada ONLY!  Distributing this file
 * outside of the U.S./Canada may be in violation of ITAR regulations!
 */

/*
 * This file contains all of the PGP routines necessary to sign, encrypt,
 * verify and decrypt PGP messages in either the new PGP/MIME format, or
 * in the older Application/Pgp format.  It also contains some code to
 * cache the user's passphrase for repeat use when decrypting or signing
 * a message.
 */

#include "mutt.h"
#include "mutt_curses.h"
#include "rfc822.h"

#include <sys/wait.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <errno.h>
#include <ctype.h>

#ifdef _PGPPATH

struct keyinfo {
  char *keyid;
  ADDRESS *address;
  struct keyinfo *next;
};
typedef struct keyinfo KEYINFO;

char pgp_passphrase[STRING];
static time_t pgp_exptime = 0; /* when does the cached passphrase expire? */
static time_t pgp_timeout = 300; /* default to 5 minutes. */

void pgp_void_passphrase (void)
{
  memset(pgp_passphrase, 0, sizeof(pgp_passphrase));
  pgp_exptime = 0;
}

int pgp_valid_passphrase (void)
{
  time_t now = time(0);

  if (now < pgp_exptime)
    return 1; /* just use the cached copy. */
  pgp_void_passphrase ();

  if (mutt_get_password ("Enter PGP passphrase:", pgp_passphrase, sizeof (pgp_passphrase)) == 0)
  {
    pgp_exptime = time (NULL) + pgp_timeout;
    return(1);
  }
  else
  {
    pgp_exptime = 0;
    return (0);
  }
  /* not reached */
}

/*
 * The following routines are used to parse the output of "pgp -kv" to build
 * a list of the public keys in the user's database.
 */
static KEYINFO *ki_parse_line (char *s)
{
  char *keyid, *address;
  KEYINFO *p;

  if ((s = strchr(s, '/')) == 0)
    return 0;
  s++;
  keyid = s;
  while (*s && *s != ' ') s++;
  if (!*s) return 0;
  *s++ = 0;
  while (*s && isspace(*s)) s++;
  if ((s = strchr(s, ' ')) == 0) return 0;
  while (*s && isspace(*s)) s++;
  address = s;
  while (*s && *s != '\n') s++;
  *s = 0;

  p = (KEYINFO *)malloc(sizeof(KEYINFO));
  p->keyid = safe_strdup(keyid);
  p->address = 0;
  p->next = 0;
  rfc822_parse_adrlist (&p->address, address, "@");
  return(p);
}

static KEYINFO *ki_getdb (void)
{
  FILE *in;
  KEYINFO *db = NULL, *end = 0;
  char buffer[STRING], *c;

  sprintf(buffer, "%s +verbose=0 -kv", Pgp);
  mutt_create_filter(buffer, 0, &in);
  while (fgets(buffer, STRING, in) != NULL)
  {
    if (strncmp(buffer, "pub", 3) == 0 || strncmp(buffer, "sec", 3) == 0)
    {
      if (db)
      {
        end->next = ki_parse_line(buffer);
        end = end->next;
      }
      else
        db = end = ki_parse_line(buffer);
    }
    else if (buffer[0] == ' ') /* alias */
    {
      c = buffer;
      while (*c && (*c == ' ' || *c == '\t')) c++;
      rfc822_parse_adrlist (&end->address, c, "@");
    }
  }
  fclose (in);
  wait (NULL);
  return (db);
}

static void ki_closedb (KEYINFO *k)
{
  KEYINFO *tmp;

  while (k)
  {
    if (k->keyid)
      free (k->keyid);
    mutt_free_address (&k->address);
    tmp = k;
    k = k->next;
    free (tmp);
  }
}

static char *ki_getkeyidbyaddr (ADDRESS *a, KEYINFO *k)
{
  ADDRESS *p;

  while (k)
  {
    p = k->address;
    while (p)
    {
      if (p->mailbox && p->host && a->mailbox && a->host &&
          strcasecmp(p->mailbox, a->mailbox) == 0 && 
          strcasecmp(p->host, a->host) == 0)
        return k->keyid;
      if (a->personal && p->personal && strcasecmp(a->personal, p->personal) == 0)
        return k->keyid;
      p = p->next;
    }
    k = k->next;
  }
  return 0;
}

static char *ki_getkeybystr (char *p, KEYINFO *k)
{
  while (k)
  {
    if (strcasecmp(p, k->keyid) == 0 ||
	(k->address->personal && strcasecmp(p, k->address->personal) == 0) ||
	strcasecmp(p, k->address->mailbox) == 0)
      return k->keyid;
    k = k->next;
  }
  return (NULL);
}

/* ----------------------------------------------------------------------------
 * Support for the Application/PGP Content Type.  This will fail if the CTE
 * is something other than 7BIT or 8BIT, but I've never seen a message like
 * that.  Hopefully more people will start using Multipart/Signed and
 * Multipart/Encrypted instead, because it's much more robust.
 */
void application_pgp_handler (BODY *m, STATE *s)
{
  int status, needpass = -1;
  long len, bytes;
  char buffer[LONG_STRING];
  FILE *pgpout, *pgpin;

  fseek (s->fpin, m->offset, 0);
  for (bytes = m->length; bytes > 0;)
  {
    if (fgets (buffer, sizeof (buffer)-1, s->fpin) == NULL)
      return;
    if (!strncmp ("-----BEGIN PGP ", buffer, 15))
    {
      if (!strcmp ("MESSAGE-----\n", buffer+15))
        needpass = 1;
      else
        needpass = 0;
      break;
    }
  }

  if (needpass < 0)
  {
    state_puts ("[Error!  Could not find beginning of PGP message!]\n\n", s);
    mutt_text_handler (m, s); /* just display as text */
    return;
  }

  if (s->displaying)
  {
    if (needpass)
      state_puts ("[-----BEGIN PGP MESSAGE-----]\n\n", s);
    else
      state_puts ("[-----BEGIN PGP SIGNED MESSAGE-----]\n\n", s);
  }

  snprintf (buffer, sizeof (buffer), "%s%s +verbose=0 +batchmode -f",
	    needpass ? "PGPPASSFD=0; export PGPPASSFD; " : "",
	    Pgp);
  mutt_create_filter (buffer, &pgpin, &pgpout);
  fseek (s->fpin, m->offset, 0);

  if (needpass)
  {
    fputs (pgp_passphrase, pgpin);
    fputc ('\n', pgpin);
  }
  mutt_copy_bytes (s->fpin, pgpin, m->length);
  fclose (pgpin);
  if (s->prefix)
  {
    while (fgets (buffer, sizeof(buffer), pgpout) != NULL)
    {
      state_puts (s->prefix, s);
      state_puts (buffer, s);
    }
  }
  else
  {
    /*
     * Since we don't need to prepend a string to each line, just copy the
     * stream in as big chunks as we can handler.  This will speed things
     * up somewhat.
     */
    while ((len = fread (buffer, 1, sizeof (buffer), pgpout)) > 0)
      fwrite (buffer, 1, len, s->fpout);
  }
  fclose (pgpout);
  wait (&status);
  if (s->displaying)
  {
    if (needpass)
      state_puts ("[-----END PGP MESSAGE-----]\n", s);
    else
      state_puts ("[-----END PGP SIGNED MESSAGE-----]\n", s);
  }
  /*
   * Pause to allow the user to read any PGP error messages.
   */
  ci_any_key_to_continue (0);
}

int pgp_query (BODY *m)
{
  char *p;

  /* Check for old-style APPLICATION/PGP messages */
  if (m->type == TYPEAPPLICATION)
  {
    if (strcasecmp (m->subtype, "pgp") == 0 || strcasecmp (m->subtype, "x-pgp-message") == 0)
    {
      if ((p = mutt_get_parameter ("x-action", m->parameter)))
      {
	if (!strcasecmp (p, "sign") || !strcasecmp (p, "signclear"))
	  return PGPSIGN;
      }
      return PGPENCRYPT;  /* not necessarily correct, but... */
    }

    if (strcasecmp (m->subtype, "pgp-signed") == 0)
      return PGPSIGN;
  }

  /* Check for PGP/MIME messages. */
  if (m->type == TYPEMULTIPART)
  {
    if (strcasecmp(m->subtype, "signed") == 0)
    {
      if ((p = mutt_get_parameter("protocol", m->parameter)) &&
	  strcasecmp(p, "application/pgp-signature") == 0)
	return PGPSIGN;
    }
    else if (strcasecmp(m->subtype, "encrypted") == 0)
    {
      if ((p = mutt_get_parameter("protocol", m->parameter)) &&
	  strcasecmp(p, "application/pgp-encrypted") == 0)
	return PGPENCRYPT;
    }
  }
  return 0;
}

/*
 * This routine verfies a PGP/MIME signed body.
 */
void pgp_signed_handler (BODY *a, STATE *s)
{
  FILE *fp;
  char tempfile[_POSIX_PATH_MAX], sigfile[STRING], buffer[LONG_STRING];
  int bytes, len;

  a = a->parts;

  /* First do some error checking to make sure that this looks like a valid
   * multipart/signed body.
   */
  if (a && a->next && a->next->type == TYPEAPPLICATION && a->next->subtype &&
      strcasecmp(a->next->subtype, "pgp-signature") == 0)
  {
    mutt_mktemp(tempfile);
    fp = fopen(tempfile, "w");
    
    fseek(s->fpin, a->hdr_offset, 0);
    bytes = a->length + a->offset - a->hdr_offset;
    while (bytes > 0) {
      fgets(buffer, sizeof(buffer)-1, s->fpin);
      len = strlen(buffer);
      /* Convert LF=>CRLF when required (probably always) */
      if (buffer[len-2] != '\r')
      {
        buffer[len-1] = '\r';
        buffer[len] = '\n';
        buffer[len+1] = 0;
      }
      fputs(buffer, fp);
      bytes -= len;
    }
    fclose(fp);
    
    /* Now grab the signature.  Since signature data is required to be 7bit,
     * we don't have to worry about doing CTE decoding...
     */
    sprintf(sigfile, "%s.asc", tempfile);
    fp = fopen (sigfile, "w");
    fseek(s->fpin, a->next->offset, 0);
    mutt_copy_bytes(s->fpin, fp, a->next->length);
    fclose(fp);
    
    snprintf(buffer, sizeof(buffer), "%s +batchmode +verbose=0 %s", Pgp, tempfile);
    printf("Executing: %s\n", buffer);
    mutt_system(buffer);
    mutt_unlink(tempfile);
    mutt_unlink(sigfile);
    ci_any_key_to_continue(0);

    /* Now display the signed body */
    if (s->displaying)
      state_puts ("[The following data is PGP/MIME signed.]\n\n", s);
    mutt_body_handler (a, s);
    if (s->displaying)
      state_puts ("\n[End of PGP/MIME signed data.]\n", s);
  }
  else
  {
    dprint (1,(debugfile, "pgp_signed_handler: invalid format!\n"));
    state_puts ("[Error!  This message does not comply with the PGP/MIME specification!]\n\n", s);
    mutt_text_handler (a, s); /* just treat the data as text/plain... */
  }
}

void pgp_encrypted_handler (BODY *a, STATE *s)
{
  char tempfile[_POSIX_PATH_MAX], buffer[2048];
  FILE *pgpin, *pgpout, *fpout;
  BODY *tattach;
  int len;
  struct stat info;

  a = a->parts;
  if (!a || a->type != TYPEAPPLICATION || !a->subtype || 
      strcasecmp("pgp-encrypted", a->subtype) != 0 ||
      !a->next || a->next->type != TYPEAPPLICATION || !a->next->subtype ||
      strcasecmp ("octet-stream", a->next->subtype) != 0)
  {
    state_puts ("[Error!  Malformed PGP/MIME message.  Could not process it!]\n\n", s);
    return;
  }

  snprintf (buffer, sizeof (buffer), "PGPPASSFD=0; export PGPPASSFD; %s +verbose=0 +batchmode -f", Pgp);

  printf ("Executing: %s\n", buffer);
  fflush (stdout);
  if (mutt_create_filter (buffer, &pgpin, &pgpout) == -1)
  {
    state_puts ("[Error!  Could not create a PGP subprocess!]\n\n", s);
    return;
  }

  /* send the PGP passphrase to the subprocess */
  fputs (pgp_passphrase, pgpin);
  fputc ('\n', pgpin);

  /*
   * Move forward to the application/pgp-encrypted body.
   */
  a = a->next;

  /*
   * Position the stream at the beginning of the body, and send the data to
   * the PGP subprocess.
   */
  fseek (s->fpin, a->offset, 0);
  mutt_copy_bytes (s->fpin, pgpin, a->length);
  fclose (pgpin);

  mutt_mktemp (tempfile);
  fpout = fopen (tempfile, "w");

  /*
   * Read the output from PGP, and make sure to change CRLF to LF, otherwise
   * read_mime_header has a hard time parsing the message.
   */
  while (fgets (buffer, sizeof (buffer)-1, pgpout) != NULL)
  {
    len = strlen (buffer);
    if (buffer[len-2] == '\r')
      strcpy(buffer+len-2, "\n");
    fputs(buffer, fpout);
  }

  fclose (pgpout);
  fclose (fpout);

  wait (NULL);

  fpout = s->fpin;
  s->fpin = fopen (tempfile, "r");
  /*
   * unlink the file now so it disappears.  we can still read it, and it
   * will provide some amount of security for the decrypted data.
   */
  unlink (tempfile);

  /* allow the user to read the PGP output in case there was output. */
  ci_any_key_to_continue (NULL);

  tattach = mutt_read_mime_header (s->fpin);

  /*
   * Need to set the length of this body part.
   */
  fstat(fileno (s->fpin), &info);
  tattach->length = info.st_size - tattach->offset;

  /*
   * See if we need to recurse on this MIME part.
   */
  if (tattach->type == TYPEMULTIPART)
  {
    fseek (s->fpin, tattach->offset, 0);
    tattach->parts = parse_multipart (s->fpin, mutt_get_parameter ("boundary", tattach->parameter));
  }
  else if (tattach->type == TYPEMESSAGE)
  {
    fseek(s->fpin, tattach->offset, 0);
    tattach->parts = parse_messageRFC822(s->fpin, tattach);
  }

  state_puts ("[The following data is PGP/MIME encrypted.]\n\n", s);
  mutt_body_handler (tattach, s);
  state_puts ("\n[End of PGP/MIME encrypted data.]\n", s);
  mutt_free_body (&tattach);
  fclose (s->fpin);
  s->fpin = fpout;
}

/* ----------------------------------------------------------------------------
 * Routines for sending PGP/MIME messages.
 */

static void convert_to_7bit (BODY *a)
{
  while (a)
  {
    if (a->type == TYPEMULTIPART)
    {
      if (a->encoding != ENC7BIT)
      {
        a->encoding = ENC7BIT;
        convert_to_7bit (a->parts);
      }
    }
    else if (a->encoding == ENC8BIT)
      a->encoding = ENCQUOTEDPRINTABLE;
    else if (a->encoding == ENCBINARY)
      a->encoding = ENCBASE64;
    else if (a->content->from)
      a->encoding = ENCQUOTEDPRINTABLE;
    a = a->next;
  }
}

BODY *pgp_signMessage (BODY *a)
{
  PARAMETER *p;
  BODY *t;
  char cmd[STRING], sigfile[_POSIX_PATH_MAX], buffer[LONG_STRING];
  FILE *pgpin, *pgpout, *fp;
  int err = 0;

  convert_to_7bit (a); /* Signed data _must_ be in 7-bit format. */

  mutt_mktemp (sigfile);
  if ((fp = fopen (sigfile, "w")) == NULL)
  {
    safe_free ((void **)&sigfile);
    return 0;
  }

  snprintf (cmd, sizeof(cmd), "%s%s +verbose=0 -abfst",
	    pgp_passphrase[0] ? "PGPPASSFD=0; export PGPPASSFD; " : "",
	    Pgp);
  printf ("Executing: %s\n", cmd);
  fflush (stdout);

  if (mutt_create_filter (cmd, &pgpin, &pgpout) == -1)
  {
    safe_free((void **)&sigfile);
    fclose(fp);
    return 0;
  }

  if (pgp_passphrase[0])
  {
    fputs (pgp_passphrase, pgpin);
    fputc ('\n', pgpin);
  }

  mutt_write_mime_header(a, pgpin);
  fputc('\n', pgpin);
  mutt_write_mime_body(a, pgpin);

  if (fclose (pgpin) != 0)
  {
    dprint(1,(debugfile, "pgp_sign_message: fclose() returned -1; errno=%d.\n",
              errno));
    fclose (pgpin);
    fclose (pgpout);
    fclose (fp);
    unlink (sigfile);
    return 0;
  }

  /*
   * Read back the PGP signature.  Also, change MESSAGE=>SIGNATURE as
   * recommended for future releases of PGP.
   */
  while (fgets (buffer, sizeof (buffer)-1, pgpout) != NULL)
  {
    if (strcmp("-----BEGIN PGP MESSAGE-----\n", buffer) == 0)
      fputs("-----BEGIN PGP SIGNATURE-----\n", fp);
    else if (strcmp("-----END PGP MESSAGE-----\n", buffer) == 0)
      fputs("-----END PGP SIGNATURE-----\n", fp);
    else
      fputs(buffer, fp);
  }

  err = fclose (pgpout);
  if (fclose (fp) != 0 || err != 0)
  {
    unlink (sigfile);
    return 0;
  }

  ci_any_key_to_continue (NULL);

  t = mutt_new_body();
  t->type = TYPEMULTIPART;
  t->subtype = safe_strdup("signed");
  t->use_disp = 0;
  t->encoding = ENC7BIT;

  t->parameter = p = mutt_new_parameter();
  p->attribute = safe_strdup("protocol");
  p->value = safe_strdup("application/pgp-signature");

  p->next = mutt_new_parameter();
  p = p->next;
  p->attribute = safe_strdup("micalg");
  p->value = safe_strdup("pgp-md5");

  p->next = mutt_new_parameter();
  p = p->next;
  p->attribute = safe_strdup("boundary");
  p->value = mutt_generate_boundary();

  t->parts = a;
  a = t;

  t->parts->next = mutt_new_body();
  t = t->parts->next;
  t->type = TYPEAPPLICATION;
  t->subtype = safe_strdup("pgp-signature");
  t->filename = safe_strdup(sigfile);
  t->use_disp = 0;
  t->encoding = ENC7BIT;
  t->unlink = 1; /* ok to remove this file after sending. */

  return(a);
}

/*
 * This routine attempts to find the keyids of the recipients of a message.
 * It returns NULL if any of the keys can not be found.
 */
char *pgp_findKeys (ADDRESS *to, ADDRESS *cc, ADDRESS *bcc)
{
  KEYINFO *db = ki_getdb();
  char *key, keylist[1024];
  ADDRESS *p = to;
  ADDRESS *tmp;
  int count = 0;

  *keylist = 0;
  for (count = 0; count < 3; count++)
  {
    switch (count) {
    case 0:
      p = to;
      break;
    case 1:
      p = cc;
      break;
    case 2:
      p = bcc;
      break;
    }
    while (p) {
      key = ki_getkeyidbyaddr(p, db);
      if (!key) {
	char buf[LONG_STRING];
	char resp[SHORT_STRING];

	FOREVER
	{
	  strcpy(buf, "Enter keyID for ");
	  tmp = p->next;
	  rfc822_address(buf+strlen(buf), p);
	  strcat(buf, ": ");
	  p->next = tmp;
	  resp[0] = 0;
	  if (ci_get_field(buf, resp, sizeof(resp), 0) != 0 || !resp[0]) {
	    /* no key, so give up */
	    ki_closedb(db);
	    return NULL;
	  }
	  if ((key = ki_getkeybystr(resp, db)) != NULL)
	    break;
	  beep();
	}
      }
      sprintf(keylist+strlen(keylist), " 0x%s", key);
      p = p->next;
    }
  }
  ki_closedb(db);
  return (safe_strdup(keylist));
}

BODY *pgp_encryptMessage (BODY *a, char *keylist, int sign)
{
  char cmd[STRING], tempfile[_POSIX_PATH_MAX], buffer[LONG_STRING];
  FILE *pgpin, *pgpout, *fpout;
  int len;
  BODY *t;
  PARAMETER *p;

  if (sign) convert_to_7bit(a);
  snprintf(cmd, sizeof(cmd), "%s%s +verbose=0 +encrypttoself -aef%s %s",
          sign ? "PGPPASSFD=0; export PGPPASSFD; " : "",
	  Pgp, sign ? "s" : "", keylist);
  if (mutt_create_filter(cmd, &pgpin, &pgpout) == -1) return 0;

  if (sign) {
    fputs(pgp_passphrase, pgpin);
    fputc('\n', pgpin);
  }

  mutt_write_mime_header(a, pgpin);
  fputc('\n', pgpin);
  mutt_write_mime_body(a, pgpin);
  if (fclose(pgpin) != 0) {
    dprint(1,(debugfile,"pgp_encrypt_message: fclose() returned -1; errno=%d.\n",
              errno));
    fclose(pgpin);
    fclose(pgpout);
  }
  mutt_mktemp(tempfile);
  if ((fpout = fopen(tempfile, "w")) == NULL)
  {
    dprint(1,(debugfile,"pgp_encrypt_mesage: error opening %s for writing!\n",
              tempfile));
    fclose(pgpin);
    fclose(pgpout);
    return 0;
  }
  while ((len = fread(buffer, 1, sizeof(buffer), pgpout)) > 0)
    fwrite(buffer, 1, len, fpout);
  fclose(pgpout);
  fclose(fpout);

  wait (NULL); /* wait to let PGP finish */

  ci_any_key_to_continue (NULL); /* pause to let the user read the PGP output */

  t = mutt_new_body();
  t->type = TYPEMULTIPART;
  t->subtype = safe_strdup("encrypted");
  t->encoding = ENC7BIT;
  t->use_disp = 0;

  t->parameter = p = mutt_new_parameter();
  p->attribute = safe_strdup("protocol");
  p->value = safe_strdup("application/pgp-encrypted");

  p->next = mutt_new_parameter();
  p = p->next;
  p->attribute = safe_strdup("boundary");
  p->value = mutt_generate_boundary();

  t->parts = mutt_new_body();
  t->parts->type = TYPEAPPLICATION;
  t->parts->subtype = safe_strdup("pgp-encrypted");
  t->parts->encoding = ENC7BIT;
  t->parts->use_disp = 0;

  t->parts->next = mutt_new_body();
  t->parts->next->type = TYPEAPPLICATION;
  t->parts->next->subtype = safe_strdup("octet-stream");
  t->parts->next->encoding = ENC7BIT;
  t->parts->next->filename = safe_strdup(tempfile);
  t->parts->next->use_disp = 0;
  t->parts->next->unlink = 1; /* delete this part after sending the message. */

  mutt_free_body(&a); /* no longer needed! */
  return(t);
}
#endif /* _PGPPATH */
