/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "mutt.h"
#include "mutt_curses.h"
#include "rfc1522.h"

#include <sys/wait.h>
#include <string.h>

/*
 * See muttlib.h for the flags to pass to this function.
 */
void mutt_copy_header (FILE *in, HEADER *h, FILE *out, int flags)
{
  char buf[HUGE_STRING];
  int ignore=0, class=0, cl_found=0, lines_found=0;

  fseek (in, h->offset, 0);
  while (fgets (buf, sizeof (buf), in) != NULL)
  {
    if (buf[0] == '\n' ||
	(buf[0] != ' ' && buf[0] != '\t' && strchr (buf, ':') == NULL))
      break; /* end of headers */

    if (strncmp ("From ", buf, 5) == 0 && (flags & CH_FROM) == 0)
      continue; /* skip the msg separator */

    if (flags & CH_WEED)
    {
      if (buf[0] == ' ' || buf[0] == '\t')
      {
	if (ignore)
	  continue;
      }
      else if (mutt_matches_ignore (buf, Ignore) &&
	       !mutt_matches_ignore (buf, UnIgnore))
	ignore = 1;
      else
      {
	if (flags & CH_DECODE)
	  class = rfc1522_classify (buf);
	ignore=0;
      }
    }
    
    if (!ignore)
    {
      if (((flags & CH_UPDATE) || (flags & CH_XMIT)) &&
	  (strncasecmp ("Status:", buf, 7) == 0 ||
	   strncasecmp ("X-Status:", buf, 9) == 0))
	continue;
      if (strncasecmp ("Lines:", buf, 6) == 0)
      {
	if (flags & CH_XMIT) continue;
	lines_found=1;
      }
      else if (strncasecmp ("Content-Length:", buf, 15) == 0)
      {
	if (flags & CH_XMIT) continue;
	cl_found=1;
      }

      if (flags & CH_DECODE)
	rfc1522_decode(buf, sizeof(buf), class);

      if (flags & CH_PREFIX)
	fputs(Prefix, out);
      fputs(buf, out);
    }
  }
  if (flags & CH_UPDATE)
  {
    if (h->old || h->read)
    {
      fputs("Status: ", out);
      if (h->read)
	fputs("RO", out);
      else if (h->old)
	fputc('O', out);
      fputc('\n', out);
    }
    if (h->flagged || h->replied)
    {
      fputs ("X-Status: ", out);
      if (h->replied) fputc ('A', out);
      if (h->flagged) fputc ('F', out);
      fputc ('\n', out);
    }

    if (!cl_found)
      fprintf (out, "Content-Length: %ld\n", h->content->length);
    if (!lines_found)
      fprintf (out, "Lines: %d\n", h->lines);
  }

  if ((flags & CH_XMIT) == 0) fputc ('\n', out); /* add header terminator */
}

static void print_msg (HEADER *h, STATE *s)
{
#ifdef _PGPPATH
  if (mutt_check_pgp (h) != 0) return;
#endif

  mutt_parse_mime_message (h);
  mutt_copy_header (Context->folder->fp, h, s->fpout, CH_WEED | CH_DECODE);
  mutt_body_handler (h->content, s);

  /* add a formfeed */
  state_putc ('\f', s);
}

void mutt_print_message (HEADER *h)
{
  STATE s;
  int i, count = 0, status;
  char tmp[SHORT_STRING];

  if (option (OPTASKPRINT) &&
      mutt_yesorno (h ? "Print tagged messages?" : "Print message?", 0) != 1)
    return;

  memset (&s, 0, sizeof (STATE));
  s.fpin = Context->folder->fp;
  mutt_create_filter (Print, &s.fpout, NULL);

  if (h)
  {
    print_msg (h, &s);
    count++;
  }
  else
  {
    for (i = 0 ; i < Context->vcount ; i++)
    {
      if (Context->hdrs[Context->v2r[i]]->tagged)
      {
	print_msg (Context->hdrs[Context->v2r[i]], &s);
	count++;
      }
    }
  }

  fclose (s.fpout);
  wait (&status);
  snprintf (tmp, sizeof (tmp),
	    status ? "Error while printing message%s!" : "Message%s printed.",
	    count > 1 ? "s" : "");
  mutt_error (tmp);
}
