/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"
#include "mutt_curses.h"
#include "keymap.h"

#include <errno.h>
#include <unistd.h>
#include <string.h>

#define ATT_REDRAW_INDEX 1
#define ATT_REDRAW_INFO  2
#define ATT_REDRAW_FULL  4

#define ATT_OFFSET 1			/* where to start printing the index */
#define ATT_TYPE (LINES-8)
#define ATT_ENCODING (LINES-7)
#define ATT_DESCRIPTION (LINES-6)
#define ATT_LENGTH (ATT_TYPE-ATT_OFFSET-1)	/* how many entries/page? */

#define ATT_XOFFSET 27 /* where the data for the content- headers begins */

BODY *ci_attach_menu (BODY *att)
{
  BODY **iindex = 0, *ptr = att;
  int redraw = ATT_REDRAW_FULL;
  int iindexlen = 0, count = 0, cur = 1, top = 1, i, ch;
  char buffer[STRING];

  while (ptr)
  {
    if (iindexlen == count)
    {
      if (iindexlen == 0)
	iindex = (BODY **)safe_malloc(sizeof(BODY *)*((iindexlen += 5)));
      else
	safe_realloc((void **)&iindex, sizeof(BODY *)*((iindexlen += 5)));
    }
    iindex[count++] = ptr;
    ptr = ptr->next;
  }

  FOREVER
  {
    if (redraw == ATT_REDRAW_FULL)
    {
      clear ();
      SETCOLOR (MT_COLOR_STATUS);
      mvprintw (LINES-2, 0, "%*.*s", COLS, COLS, "");
      mvprintw (LINES-2, 0, "---Mutt: Attachment Mode");
      SETCOLOR (MT_COLOR_NORMAL);
      redraw = ATT_REDRAW_INDEX | ATT_REDRAW_INFO;
    }

    if (redraw & ATT_REDRAW_INDEX)
    {
      for (i=top; i<ATT_LENGTH; i++)
      {
	if (i <= count)
	{
	  snprintf (buffer, sizeof (buffer), "  %2d  %-*.*s", i, COLS-6, COLS-6, iindex[i-1]->filename);
	  move(i-top+ATT_OFFSET, 0);
	  if (i == cur) SETCOLOR (MT_COLOR_INDICATOR);
	  addstr (buffer);
	  if (i == cur) SETCOLOR (MT_COLOR_NORMAL);
	}
	else
	  CLEARLINE (i-top+ATT_OFFSET);
      }
      if (!count) mvaddstr (ATT_OFFSET, 6, "(no attachments)");
      redraw &= ~ATT_REDRAW_INDEX;
    }

    if (redraw & ATT_REDRAW_INFO)
    {
      if (count > 0)
	snprintf (buffer, sizeof (buffer), "%s/%s", TYPE (iindex[cur-1]->type),
		  iindex[cur-1]->subtype);
      else
	buffer[0] = 0;

      CLEARLINE (ATT_TYPE);
      printw("             Content-Type: %-*.*s",
	     COLS-ATT_XOFFSET, COLS-ATT_XOFFSET, buffer);
      CLEARLINE (ATT_ENCODING);
      printw("Content-Transfer-Encoding: %-*.*s",
	     COLS-ATT_XOFFSET, COLS-ATT_XOFFSET,
	     count > 0 ? ENCODING(iindex[cur-1]->encoding) : "");
      CLEARLINE (ATT_DESCRIPTION);
      printw("      Content-Description: %-*.*s",
	     COLS-ATT_XOFFSET, COLS-ATT_XOFFSET,
	     count > 0 ? (iindex[cur-1]->description ? iindex[cur-1]->description : "") : "");
      redraw &= ~ATT_REDRAW_INFO;
    }

    move (cur-top+ATT_OFFSET, COLS-1);
    refresh ();
    ch = dokey (MENU_SENDATTACH);
    ci_clear_error ();

    switch (ch)
    {
      case OP_GENERIC_FIRST_ENTRY:

	cur = 1;
	redraw = ATT_REDRAW_INDEX | ATT_REDRAW_INFO;
	break;

      case OP_GENERIC_LAST_ENTRY:

	cur = count;
	redraw = ATT_REDRAW_INDEX | ATT_REDRAW_INFO;
	break;

      case OP_GENERIC_NEXT_ENTRY:

	if (cur < count)
	{
	  cur++;
	  redraw = ATT_REDRAW_INDEX | ATT_REDRAW_INFO;
	}
	else
	{
	  beep ();
	  mutt_error("You are on the last attachment.");
	}
	break;

      case OP_GENERIC_PREVIOUS_ENTRY:

	if (cur > 1)
	{
	  cur--;
	  redraw = ATT_REDRAW_INDEX | ATT_REDRAW_INFO;
	}
	else
	{
	  beep ();
	  mutt_error("You are on the first attachment.");
	}
	break;

      case OP_NEXT_PAGE:

	if (cur + ATT_LENGTH < count)
	{
	  while (cur + ATT_LENGTH < count)
	    cur += ATT_LENGTH;
	  redraw = ATT_REDRAW_INDEX | ATT_REDRAW_INFO;
	}
	else
	{
	  beep ();
	  mutt_error ("You are on the last page.");
	}
	break;

      case OP_PREVIOUS_PAGE:

	if (top > 1)
	{
	  while (top > 0) top -= ATT_LENGTH;
	  cur = top;
	  redraw = ATT_REDRAW_INDEX | ATT_REDRAW_INFO;
	}
	else
	{
	  beep ();
	  mutt_error ("You are on the first page.");
	}
	break;

      case OP_DELETE:

	if (count)
	{
	  if (cur == count)
	  {
	    cur--;
	    count--;
	    mutt_free_body(&iindex[count]);
	    if (count > 0) iindex[count-1]->next = 0;
	  }
	  else
	  {
	    /* Don't clobber the rest of the attachments! */
	    iindex[cur-1]->next = 0;
	    mutt_free_body(&iindex[cur-1]);
	    for (i=cur-1; i<count-1; i++)
	      iindex[i] = iindex[i+1];
	    iindex[count-1] = 0;
	    count--;
	    for (i=1; i<=count; i++)
	      iindex[i-1]->next = iindex[i];
	  }
	  redraw = ATT_REDRAW_INDEX | ATT_REDRAW_INFO;
	}
	else
	  mutt_error ("There are no attachments.");
	break;

      case OP_GENERIC_EXIT:

	goto done;

      case OP_REDRAW:

	clearok (stdscr, TRUE);
	redraw = ATT_REDRAW_FULL;
	break;

      case OP_HELP:

	ci_help (MENU_SENDATTACH);
	redraw = ATT_REDRAW_FULL;
	break;

      case OP_SENDATTACH_ADD_ATTACHMENT:

	CLEARLINE (LINES-1);
	addstr ("Attach file (? for list): ");
	if ((i = ci_getch ()) == ERR)
	  break;
	else if (i == '?')
	{
	  mutt_select_file (buffer, sizeof (buffer));
	  redraw = ATT_REDRAW_FULL;
	  if (!buffer[0]) continue;
	}
	else
	{
	  ungetch (i);
	  buffer[0] = 0;
	  if (ci_get_field ("Attach file: ", buffer, sizeof (buffer), M_FILE) != 0 ||
	      !buffer[0])
	    continue;
	  mutt_expand_path (buffer, sizeof (buffer));
	}
	if (access (buffer, F_OK | R_OK) == -1)
	{
	  beep ();
	  mutt_perror (buffer);
	  continue;
	}
	if (iindexlen == count)
	{
	  if (iindexlen == 0)
	    iindex = (BODY **)safe_malloc(sizeof(BODY *) * (iindexlen += 5) );
	  else
	    safe_realloc((void **)&iindex, sizeof(BODY *) * (iindexlen += 5));
	}

	iindex[count] = mutt_make_attach(buffer);
	if (count > 0) iindex[count-1]->next = iindex[count];
	count++;
	cur = count;

	/* if the folder-menu was used, redraw will already be set above */
	if (!redraw)
	  redraw = ATT_REDRAW_INDEX | ATT_REDRAW_INFO;

	break;

      case OP_SENDATTACH_ENTER_DESCRIPTION:

	if (iindex)
	{
	  if (iindex[cur-1]->description)
	    strcpy (buffer, iindex[cur-1]->description);
	  else
	    buffer[0] = 0;

	  if (ci_get_field ("Description: ", buffer, sizeof (buffer), 0) == 0)
	  {
	    if (buffer[0])
	    {
	      safe_free((void **)&iindex[cur-1]->description);
	      iindex[cur-1]->description = safe_strdup(buffer);
	    }
	    else
	      safe_free((void **)&iindex[cur-1]->description);
	    redraw = ATT_REDRAW_INFO;
	  }
	}
	else
	{
	  beep ();
	  mutt_error ("There are no attachments.");
	}
	break;

      case OP_SENDATTACH_EDIT_TYPE:

	if (iindex)
	{
	  snprintf (buffer, sizeof (buffer), "%s/%s",
		    TYPE (iindex[cur-1]->type),
		    iindex[cur-1]->subtype);
	  if (ci_get_field ("Content-Type: ", buffer, sizeof (buffer), 0) == 0 && 
	      buffer[0])
	  {
	    char *p = strchr (buffer, '/');

	    if (p)
	    {
	      *p++ = 0;
	      if ((i = mutt_check_mime_type (buffer)) != TYPEOTHER)
	      {
		iindex[cur-1]->type = i;
		safe_free ((void **)&iindex[cur-1]->subtype);
		iindex[cur-1]->subtype = safe_strdup (p);
		redraw = ATT_REDRAW_INFO;
	      }
	      else
	      {
		beep ();
		mutt_error ("Invalid Content-Type");
	      }
	    }
	    else
	    {
	      beep ();
	      mutt_error ("Invalid Content-Type!");
	    }
	  }
	}
	else
	{
	  beep ();
	  mutt_error ("There are no attachments!");
	}
	break;

      case OP_SENDATTACH_EDIT_ENCODING:

	if (iindex)
	{
	  strfcpy (buffer, ENCODING (iindex[cur-1]->encoding), sizeof (buffer));
	  if (ci_get_field ("Content-Transfer-Encoding: ", buffer, sizeof (buffer), 0) == 0 &&
	      buffer[0])
	  {
	    if ((i = mutt_check_encoding (buffer)) != ENCOTHER)
	    {
	      iindex[cur-1]->encoding = i;
	      redraw = ATT_REDRAW_INFO;
	    }
	    else
	    {
	      beep ();
	      mutt_error ("Invalid Content-Transfer-Encoding!");
	    }
	  }
	}
	else
	{
	  beep ();
	  mutt_error ("There are no attachments!");
	}

	break;

      default:

	beep ();
	mutt_error ("Key is unbound.  Press '?' for help.");
    }
  }

done:

  if (iindex)
  {
    att = iindex[0];
    safe_free ((void **)&iindex);
  }
  return (att);
}
