/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "muttlib.h"
#include "rfc822.h"
#include "rfc1522.h"

#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>

#define DISPOSITION(X) X==DISPATTACH?"attachment":"inline"

extern char *tspecials;

char Base64_chars[64] = {
  'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O',
  'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 'a', 'b', 'c', 'd',
  'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's',
  't', 'u', 'v', 'w', 'x', 'y', 'z', '0', '1', '2', '3', '4', '5', '6', '7',
  '8', '9', '+', '/'
};

static void encode_quoted (FILE * fin, FILE *fout, int istext)
{
  int c, linelen = 0;
  char line[77], savechar;

  while ((c = fgetc(fin)) != EOF) {
    /* Escape lines that begin with "the message separator". */
    if (linelen == 5 && !strncmp("From ", line, 5)) {
      strcpy(line, "=46rom ");
      linelen = 7;
    }

    /* Wrap the line if needed. */
    if (linelen == 76 && ((istext && c != '\n') || !istext)) {
      /* If the last character is "quoted", then be sure to move all three
       * characters to the next line.  Otherwise, just move the last
       * character...
       */
      if (line[linelen-3] == '=') {
        line[linelen-3] = 0;
        fputs(line, fout);      
        fputs("=\n", fout);
        line[linelen] = 0;
        line[0] = '=';
        line[1] = line[linelen-2];
        line[2] = line[linelen-1];
        linelen = 3;
      }
      else {
        savechar = line[linelen-1];
        line[linelen-1] = '=';
        line[linelen] = 0;
        fputs(line, fout);
        fputc('\n', fout);
        line[0] = savechar;
        linelen = 1;
      }
    }

    if (c == '\n' && istext) {
      /* Check to make sure there is no trailing space on this line. */
      if (line[linelen-1] == ' ' || line[linelen-1] == '\t') {
        if (linelen < 74) {
          sprintf(line+linelen-1, "=%2.2X", line[linelen-1]);
          fputs(line, fout);
        }
        else {
          int savechar = line[linelen-1];

          line[linelen-1] = '=';
          line[linelen] = 0;
          fputs(line, fout);
          fprintf(fout, "\n=%2.2X", savechar);
        }
      }
      else {
        line[linelen] = 0;
        fputs(line, fout);
      }
      fputc('\n', fout);
      linelen = 0;
    }
    else if (c != 9 && (c < 32 || c > 126 || c == '=')) {
      /* Check to make sure there is enough room for the quoted character.
       * If not, wrap to the next line.
       */
      if (linelen > 73) {
        line[linelen++] = '=';
        line[linelen] = 0;
        fputs(line, fout);
        fputc('\n', fout);
        linelen = 0;
      }
      sprintf(line+linelen,"=%2.2X", c);
      linelen += 3;
    }
    else {
      /* Don't worry about wrapping the line here.  That will happen during
       * the next iteration when I'll also know what the next character is.
       */
      line[linelen++] = c;
    }
  }

  /* Take care of anything left in the buffer */
  if (linelen > 0) {
    if (line[linelen-1] == ' ' || line[linelen-1] == '\t') {
      /* take care of trailing whitespace */
      if (linelen < 74)
        sprintf(line+linelen-1, "=%2.2X", line[linelen-1]);
      else {
        savechar = line[linelen-1];
        line[linelen-1] = '=';
        line[linelen] = 0;
        fputs(line, fout);
        fputc('\n', fout);
        sprintf(line, "=%2.2X", savechar);
      }
    }
    else
      line[linelen] = 0;
    fputs(line, fout);
  }
}

static void encode_base64 (FILE * fin, FILE *fout, int istext)
{
  int c1, c2, c3, ch;
  int insert_newline = 0;
  int linelen = 0;

  for (;;) {
    if (istext) {
      if (insert_newline) {      
        c1 = '\n';
        insert_newline = 0;
  
        c2 = fgetc(fin);
        if (c2 == '\n') {
          c2 = '\r';
          c3 = '\n';
        }
        else {
          c3 = fgetc(fin);
          if (c3 == '\n') {
            c3 = '\r';
            insert_newline = 1;
          }
        }
      }
      else {
        c1 = fgetc(fin);
        if (c1 == '\n') {
          c1 = '\r';
          c2 = '\n';
          c3 = fgetc(fin);
          if (c3 == '\n') {
            c3 = '\r';
            insert_newline = 1;
          }
        }
        else {
          c2 = fgetc(fin);
          if (c2 == '\n') {
            c2 = '\r';
            c3 = '\n';
          }
          else {
            c3 = fgetc(fin);
            if (c3 == '\n') {
              c3 = '\r';
              insert_newline = 1;
            }
          }
        }
      }
    }
    else { /* !istext */
      if ((c1 = fgetc(fin)) == EOF)
        break;
      c2 = fgetc(fin);
      c3 = fgetc(fin);
    }

    if (linelen + 4 >= 76) {
      fputc('\n', fout);
      linelen = 0;
    }

    ch = c1 >> 2;
    fputc(Base64_chars[ch], fout);

    if (c2 != EOF) {
      ch = ((c1 & 0x3) << 4) | (c2 >> 4);
      fputc(Base64_chars[ch], fout);
    }
    else {
      ch = (c1 & 0x3) << 4;
      fputc(Base64_chars[ch], fout);
      fputs("==", fout);
      break;
    }

    if (c3 != EOF) {
      ch = ((c2 & 0xf) << 2) | (c3 >> 6);
      fputc(Base64_chars[ch], fout);
    }
    else {
      ch = (c2 & 0xf) << 2;
      fputc(Base64_chars[ch], fout);
      fputc('=', fout);
      break;
    }

    ch = c3 & 0x3f;
    fputc(Base64_chars[ch], fout);

    linelen += 4;
  }

  fputc('\n', fout);
}

/* Don't print a Content-Type header if everything is the default. */
static int needs_content_type (BODY * a)
{
  char *p;

  if (a->type != TYPETEXT) return 1;
  if (strcasecmp(a->subtype, "plain") != 0) return 1;
  p = mutt_get_parameter("charset", a->parameter);
  if (!p) return 0;
  if (strcasecmp(p, "us-ascii") == 0) return 0;
  return 1;
}

int mutt_write_mime_header (BODY *a, FILE *f)
{
  PARAMETER *p;
  char buffer[STRING];
  char *t;

  if (needs_content_type (a))
  {
    sprintf(buffer, "Content-Type: %s/%s", TYPE(a->type), a->subtype);
    fputs(buffer, f);
    if (a->parameter) {
      int len = strlen(buffer);

      p = a->parameter;
      while (p) {
        sprintf(buffer, "%s=", p->attribute);
        rfc822_cat(buffer, p->value, tspecials);
        if (len + strlen(buffer) + 2 > 76)
	{
          fprintf(f, ";\n\t%s", buffer);
          len = strlen(buffer) + 8;
        }
        else
	{
          len += strlen(buffer);
          fprintf(f, "; %s", buffer);
        }
        p = p->next;
      }
    }
    fputc('\n', f);
  }
  if (a->encoding != ENC7BIT)
    fprintf(f, "Content-Transfer-Encoding: %s\n", ENCODING(a->encoding));
  if (a->description)
    fprintf(f, "Content-Description: %s\n", a->description);
  if (a->use_disp && (a->disposition == DISPATTACH || a->filename)) {
    fprintf(f, "Content-Disposition: %s", DISPOSITION(a->disposition));
    if (a->filename)
    {
      /* Strip off the leading path... */
      t = strrchr(a->filename, '/');
      if (t) t++;
      else t = a->filename;
      
      buffer[0] = 0;
      rfc822_cat(buffer, t, tspecials);
      fprintf(f, "; filename=%s", buffer);
    }
    fputc('\n', f);
  }
  /* Do NOT add the terminator here!!! */
  return 0;
}

int mutt_write_mime_body (BODY *a, FILE *f)
{
  char boundary[SHORT_STRING], buffer[LONG_STRING];
  FILE *fpin;
  BODY *t;
  PARAMETER *p;
  long len;

  if (a->type == TYPEMULTIPART)
  {
    /* First, find the boundary to use */
    p = a->parameter;
    while (p)
    {
      if (strcmp("boundary", p->attribute) == 0)
      {
        strcpy(boundary, p->value);
        break;
      }
      p = p->next;
    }

    t = a->parts;
    while (t)
    {
      fprintf(f, "\n--%s\n", boundary);
      mutt_write_mime_header(t, f);
      fputc('\n', f);
      if (mutt_write_mime_body(t, f) == -1) return -1;
      t = t->next;
    }
    fprintf(f, "\n--%s--\n", boundary);
    return 0;
  }

#ifdef _PGPPATH
  /* This is pretty gross, but it's the best solution for now... */
  if (a->type == TYPEAPPLICATION && strcmp (a->subtype, "pgp-encrypted") == 0)
  {
    fputs ("Version: 1\n", f);
    return 0;
  }
#endif /* _PGPPATH */

  if (Context && Context->folder && Context->folder->name &&
      strncmp(a->filename, Context->folder->name, strlen(Context->folder->name)) == 0)
  {
    /* Forwarded message. */
    short msgno;
    long bytes;
    char *c = strrchr(a->filename, ':');

    if (!c) goto recover; /* include the _whole_ file, not just a msg. */
    c++;
    fpin = Context->folder->fp;
    msgno = atoi(c) - 1;
    if (msgno < 0 || msgno >= Context->msgcount)
    {
      dprint(1,(debugfile,"write_mime_body(): found an invalid message number!\n"));
      return -1;
    }
    fseek(fpin, Context->hdrs[msgno]->offset, 0);
    bytes = Context->hdrs[msgno]->content->length + (Context->hdrs[msgno]->content->offset - Context->hdrs[msgno]->offset);
    /*
     * The first line of the message will be the mailbox separator, which
     * is _not_ an RFC822 header, so don't include it...
     */
    if (fgets(buffer, sizeof(buffer), fpin) == NULL) return -1;
    bytes -= strlen(buffer);
    while (bytes > 0) {
      if (fgets(buffer, sizeof(buffer), fpin) == NULL) return -1;
      len = strlen(buffer);
      bytes -= len;
      fputs(buffer, f);
    }
  }
  else {
  recover:
    if ((fpin = fopen(a->filename, "r")) == NULL)
    {
      dprint(1,(debugfile, "write_mime_body: %s no longer exists!\n",a->filename));
      return -1;
    }
    
    if (a->encoding == ENCQUOTEDPRINTABLE)
      encode_quoted(fpin, f, mutt_is_text_type(a->type, a->subtype));
    else if (a->encoding == ENCBASE64)
      encode_base64(fpin, f, mutt_is_text_type(a->type, a->subtype));
    else
    {
      while ((len = fread(buffer, 1, sizeof(buffer), fpin)) > 0)
        fwrite(buffer, 1, len, f);
    }
    fclose(fpin);
  }
  return 0;
}

#define BOUNDARYLEN 16
char *mutt_generate_boundary (void)
{
  char *rs = (char *)safe_malloc(BOUNDARYLEN+1);
  char *p = rs;
  const char bchars[] = {
    'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
    'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', 'A', 'B',
    'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P',
    'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '1', '2', '3', '4',
    '5', '6', '7', '8', '9', '0', '+', '='
  };
  int i;

  rs[BOUNDARYLEN] = 0;
  for (i=0;i<BOUNDARYLEN;i++)
    *p++ = bchars[LRAND() % sizeof(bchars)];
  *p = 0;
  return(rs);
}

/* analyze the contents of a file to determine which MIME encoding to use */
CONTENT *mutt_get_content_info (const char *fname)
{
  CONTENT *info;
  FILE *fp;
  int ch, from=0, whitespace=0, dot=0, linelen=0;

  if (!(fp = fopen(fname, "r")))
    return (0);
  info = (CONTENT *)safe_malloc(sizeof(CONTENT));
  memset(info, 0, sizeof(CONTENT));
  while ((ch = fgetc(fp)) != EOF)
  {
    linelen++;
    if (ch == '\n')
    {
      if (whitespace) info->space=1;
      if (dot) info->dot=1;
      if (linelen > info->linemax) info->linemax = linelen;
      whitespace=0;
      linelen=0;
      dot=0;
    }
    else if (ch == '\r')
    {
      if ((ch = fgetc (fp)) == EOF)
      {
        info->binary = 1;
        break;
      }
      else if (ch != '\n')
      {
        info->binary = 1;
	ungetc (ch, fp);
	continue;
      }
      else
      {
        if (whitespace) info->space = 1;
	if (dot) info->dot = 1;
        if (linelen > info->linemax) info->linemax = linelen;
        whitespace=0;
	dot=0;
        linelen=0;
      }
    }
    else if (ch & 0x80)
      info->hibin++;
    else if (ch == '\t')
    {
      info->ascii++;
      whitespace++;
    }
    else if (ch < 32 || ch == 127)
      info->lobin++;
    else
    {
      if (linelen == 1)
      {
        if (ch == 'F')
          from=1;
        else
          from=0;
        if (ch == '.')
          dot=1;
        else
          dot=0;
      }
      else if (from)
      {
        if (linelen == 2 && ch != 'r') from=0;
        else if (linelen == 3 && ch != 'o') from=0;
        else if (linelen == 4 && ch != 'm') from=0;
        else if (linelen == 5)
	{
          if (ch == ' ') info->from=1;
          from = 0;
        }
      }
      if (ch == ' ') whitespace++;
      info->ascii++;
    }
    if (linelen > 1) dot=0;
    if (ch != ' ' && ch != '\t') whitespace=0;
  }
  fclose (fp);
  return(info);
}

char *set_text_charset (CONTENT *info)
{
  if (!strcasecmp (Charset, "us-ascii"))
  {
    if (info->hibin != 0)
      return "unknown-8bit";
  }
  else if (info->hibin == 0)
  {
    if (mutt_compat_charset (Charset))
      return "us-ascii";
  }
  return (Charset);
}

/* 
 * given a file with extension ``s'', see if there is a registered MIME type.
 * returns the major MIME type, and copies the subtype to ``d''.
 * First look for ~/.mime.types, then look in a system mime.types if we can
 * find one.
 */
static int lookup_mime_type (char *d, const char *s)
{
  FILE *f = NULL;
  char *c, *p, *q, buf[LONG_STRING];
  int count=0;

  *d = 0;

read_file:

  switch (count)
  {
    case 0:

      sprintf (buf, "%s/.mime.types", Homedir);
      f = fopen (buf, "r");
      break;

    case 1:

      f = fopen (MIMETYPES, "r");
      break;

    default:

      return (-1);
  }

  count++;
  if (f == NULL) goto read_file;

  while (fgets(buf, sizeof(buf)-1, f) != NULL)
  {
    /* weed out any comments */
    if ((p = strchr(buf, '#'))) *p = 0;

    /* remove any leading space. */
    q = buf;
    while (*q && (*q == ' ' || *q == '\t')) q++;

    /* position on the next field in this line */
    if ((p = strpbrk(q, " \t")) == NULL) continue;
    *p = 0;
    p = skip_whitespace(p+1);

    /* cycle through the file extensions */
    while ((p = strtok(p, " \t\n"))) {
      if (strcasecmp(s, p) == 0) {
	/* get the content-type */

	if ((c = strchr(q, '/')) == NULL) {
	  /* malformed line, just skip it. */
	  break;
	}
	*c++ = 0;

	while (*c && *c != ' ' && *c != '\t' && *c != '\n' && *c != '\r')
	  *d++ = *c++;
	*d = 0;

	fclose(f);
	return(mutt_check_mime_type(q));
      }
      p = NULL;
    }
  }
  fclose(f);
  f = NULL;
  goto read_file;
}

BODY *mutt_make_attach (const char *path)
{
  char *p, buf[SHORT_STRING];
  int n;
  BODY *att = mutt_new_body();
  CONTENT *info;
  
  att->filename = safe_strdup(path);
  if ((p = strrchr (path, '.')))
  {
    p++;

    /* check to see if there is an entry in mime.types */
    n = lookup_mime_type (buf, p);

    if (n != TYPEOTHER)
    {
      att->type = n;
      att->subtype = safe_strdup (buf);
    }
  }
  if (!(info = mutt_get_content_info (path)))
  {
    mutt_free_body (&att);
    return NULL;
  }
  if (!att->subtype)
  {
    if (info->lobin == 0 || (info->lobin + info->hibin + info->ascii)/ info->lobin >= 10)
    {
      /*
       * Statistically speaking, there should be more than 10% "lobin" 
       * chars if this is really a binary file...
       */
      att->type = TYPETEXT;
      att->subtype = safe_strdup("plain");
      att->parameter = mutt_new_parameter();
      att->parameter->attribute = safe_strdup("charset");
      att->parameter->value = safe_strdup(set_text_charset(info));
    }
    else
    {
      att->type = TYPEAPPLICATION;
      att->subtype = safe_strdup ("octet-stream");
    }
  }

  /* now determine which Content-Transfer-Encoding to use */
  if (att->type != TYPETEXT || info->binary || info->linemax > 990)
  {
    /* Determine which encoding is smaller  */
    if (1.33 * (float)(info->lobin+info->hibin+info->ascii) < 3.0 * (float) (info->lobin + info->hibin) + (float)info->ascii)
      att->encoding = ENCBASE64;
    else
      att->encoding = ENCQUOTEDPRINTABLE;
  }
  else if (info->lobin)
    att->encoding = ENCQUOTEDPRINTABLE;
  else if (info->hibin)
    att->encoding = option(OPTALLOW8BIT) ? ENC8BIT : ENCQUOTEDPRINTABLE;
  else
    att->encoding = ENC7BIT;

#ifdef _PGPPATH
  /*
   * save the info in case this message is signed.  we will want to do Q-P
   * encoding if any lines begin with "From " so the signature won't be munged,
   * for example.
   */
  att->content = info;
  info = NULL;
#endif
  safe_free ((void **)&info);

  return (att);
}

static int get_toplevel_encoding (BODY *a)
{
  int e = ENC7BIT;

  while (a) {
    if (a->encoding == ENCBINARY) return(ENCBINARY);
    else if (a->encoding == ENC8BIT) {
      if (e == ENC7BIT) e = ENC8BIT;
    }
    a = a->next;
  }
  return(e);
}

BODY *mutt_make_multipart (BODY *b)
{
  BODY *new;

  new = mutt_new_body();
  new->type = TYPEMULTIPART;
  new->subtype = safe_strdup("mixed");
  new->encoding = get_toplevel_encoding(b);
  new->parameter = mutt_new_parameter();
  new->parameter->attribute = safe_strdup("boundary");
  new->parameter->value = mutt_generate_boundary();
  new->use_disp = 0;  
  new->parts = b;

  return new;
}

static char *mutt_make_date (char *s)
{
  time_t t = time(0);
  struct tm *l = gmtime(&t);
  int yday = l->tm_yday;
  int tz = l->tm_hour * 60 + l->tm_min;

  l = localtime(&t);
  tz = l->tm_hour * 60 + l->tm_min - tz;
  yday = l->tm_yday - yday;

  if (yday != 0)
    tz += yday * 24 * 60; /* GMT is next or previous day! */

  sprintf(s, "Date: %s, %d %s %d %02d:%02d:%02d %+03d%02d\n",
	  Weekdays[l->tm_wday], l->tm_mday, Months[l->tm_mon], l->tm_year+1900,
	  l->tm_hour, l->tm_min, l->tm_sec, tz/60, abs(tz) % 60);
  return s;
}

/*
 * wrapper around mutt_write_address() so we can handle very large
 * recipient lists without needing a huge temporary buffer in memory
 */
static void write_address_list (ADDRESS *adr, FILE *fp)
{
  ADDRESS *tmp;
  char buf[LONG_STRING];
  int count=0;
  int linelen=4; /* assume the line begins with "To: " or "Cc: " */
  int len;

  while (adr)
  {
    tmp = adr->next;
    adr->next = 0;
    buf[0] = 0;
    mutt_write_address(buf, sizeof(buf), adr);
    len = strlen(buf);
    if (count && linelen + len > 74)
    {
      if (count) {
	linelen = len + 8; /* tab is usually about 8 spaces... */
	fputs(",\n\t", fp);
      }
    }
    else
    {
      if (count)
      {
	fputs(", ", fp);
	linelen += 2;
      }
      linelen += len;
    }
    fputs(buf, fp);
    adr->next = tmp;
    adr = adr->next;
    count++;
  }
  fputc('\n', fp);
}

/*
 * need to write the list in reverse because they are stored in reverse order
 * when parsed to speed up threading
 */
static void write_references (LIST *r, FILE *f)
{
  if (r->next) write_references (r->next, f);
  fputc (' ', f);
  fputs (r->data, f);
}

/* Note: all RFC1522 encoding should be done outside of this routine, except
 * for the "real name."  This will allow this routine to be used more than
 * once, if necessary.
 *
 * mode == 1  => "lite" mode (used for edit_hdrs)
 * mode == 0  => normal mode.  write full header + MIME headers
 * mode == -1 => write just the envelope info (used for postponing messages)
 */

int mutt_write_rfc822_header (FILE *fp, ENVELOPE *env, BODY *attach, int mode)
{
  char buffer[LONG_STRING];
  LIST *tmp = env->userhdrs;

  if (mode == 0)
  {
    fprintf (fp, "Message-ID: %s\n", env->message_id);
    fputs (mutt_make_date (buffer), fp);
  }

#ifndef DONT_ADD_FROM
  if (env->from)
  {
    buffer[0] = 0;
    mutt_write_address(buffer, sizeof(buffer), env->from);
    fprintf(fp, "From: %s\n", buffer);
  }
#endif /* !DONT_ADD_FROM */

  if (env->to)
  {
    fputs("To: ", fp);
    write_address_list(env->to, fp);
  }
  else if (mode > 0)
    fputs("To: \n", fp);

  if (env->cc)
  {
    fputs("Cc: ", fp);
    write_address_list(env->cc, fp);
  }
  else if (mode > 0)
    fputs ("Cc: \n", fp);

  if (env->bcc)
  {
    fputs("Bcc: ", fp);
    write_address_list(env->bcc, fp);
  }
  else if (mode > 0)
    fputs ("Bcc: \n", fp);

  if (env->subject)
    fprintf(fp, "Subject: %s\n", env->subject);
  else if (mode == 1)
    fputs("Subject: \n", fp);

  if (mode <= 0)
  {
    if (env->references)
    {
      fputs ("References:", fp);
      write_references (env->references, fp);
      fputc('\n', fp);
    }
  }

  if (mode == 0)
  {
#ifndef NO_XMAILER
    /* Add a vanity header */
    fprintf(fp, "X-Mailer: %s\n", MuttVersion);
#endif

    /* Add the MIME headers */
    fputs("Mime-Version: 1.0\n", fp);
    mutt_write_mime_header(attach, fp);
  }

  /* Add any user defined headers */
  while (tmp)
  {
    fputs (tmp->data, fp);
    fputc ('\n', fp);
    tmp = tmp->next;
  }

  return (ferror (fp) == 0 ? 0 : -1);
}

static void encode_headers (LIST *h)
{
  char tmp[LONG_STRING];
  char *p;
  size_t len;

  while (h)
  {
    if ((p = strchr (h->data, ':')))
    {
      *p = 0;
      p = skip_whitespace (p + 1);
      snprintf (tmp, sizeof (tmp), "%s: ", h->data);
      len = strlen (tmp);
      rfc1522_encode_string (tmp + len, p, sizeof (tmp) - len);
      safe_free ((void **)&h->data);
      h->data = safe_strdup (tmp);
    }
    h = h->next;
  }
}

static char *gen_msgid (void)
{
  char buf[SHORT_STRING];
  time_t now = time(NULL);
  struct tm *tm = localtime(&now);

  snprintf(buf, sizeof(buf),
	   "<Mutt.%d%02d%02d%02d%02d%02d.%s@%s.%s>", tm->tm_year+1900,
	   tm->tm_mon+1, tm->tm_mday, tm->tm_hour, tm->tm_min,
	   tm->tm_sec, Username, Hostname, Domain);
  return(safe_strdup(buf));
}

static int invoke_sendmail (const char *cmd, const char *tempfile)
{
  char buf[LONG_STRING];

  /*
   * invoke sendmail in the background so the user doesn't have to wait for
   * it to complete.  An example would be for someone who has dial-on-demand
   * networking.
   */
  snprintf (buf, sizeof (buf), "(%s < %s; /bin/rm -f %s) &", cmd, tempfile, tempfile);
  return (mutt_system (buf));
}

int mutt_send_message (ENVELOPE *en, BODY *attach, const char *fcc)
{
  char tempfile[_POSIX_PATH_MAX], buffer[STRING], cmd[STRING];
  FILE *tempfp;

  /* Take care of 8-bit => 7-bit conversion. */
  rfc1522_encode_adrlist (en->to);
  rfc1522_encode_adrlist (en->cc);
  rfc1522_encode_adrlist (en->bcc);
  if (en->subject)
  {
    rfc1522_encode_string (buffer, (unsigned char *)en->subject, sizeof (buffer)-1);
    safe_free ((void **)&en->subject);
    en->subject = safe_strdup (buffer);
  }
  encode_headers (en->userhdrs);

  en->message_id = gen_msgid ();

  /* Write out the message in MIME form. */
  mutt_mktemp (tempfile);
  if ((tempfp = fopen (tempfile, "w")) == NULL)
    return (-1);

  mutt_write_rfc822_header (tempfp, en, attach, 0);
  fputc ('\n', tempfp); /* tie off the header. */

  mutt_write_mime_body (attach, tempfp);
  fclose (tempfp);

#ifdef USE_DSN
  sprintf (cmd, "%s -t -N %s -R %s", Sendmail, DsnNotify, DsnReturn);
#else
  sprintf (cmd, "%s -t", Sendmail);
#endif

#ifdef USE_8BITMIME
  if (attach->encoding == ENC8BIT)
    strcat (cmd, " -B8BITMIME");
#endif

  if (!option(OPTBATCHMODE))
    mutt_error ("Sending message...");

  invoke_sendmail (cmd, tempfile);

  if (!option(OPTBATCHMODE))
    mutt_error("Mail sent.");

  /* save a copy of the message, if necessary. */
  if (*fcc)
    mutt_folder_cc (en, attach, fcc);

  return 0;
}

void mutt_bounce_message (HEADER *h, char *to)
{
  int i;
  FILE *f;
  char cmd[STRING], date[SHORT_STRING], tempfile[_POSIX_PATH_MAX];

  if (!h) {
    for (i=0; i<Context->msgcount; i++)
      if (Context->hdrs[i]->tagged)
	mutt_bounce_message(Context->hdrs[i], to);
    return;
  }

  fseek(Context->folder->fp, h->offset, 0);
  snprintf(cmd, sizeof(cmd), "%s %s", Sendmail, to);

  mutt_mktemp(tempfile);
  if ((f = fopen(tempfile, "w")) != NULL) {
    mutt_copy_header(Context->folder->fp, h, f, CH_XMIT);
    fprintf(f, "Resent-From: %s", Username);
    if (Fqdn[0] != '@') fprintf(f, "@%s", Fqdn);
    fprintf(f, "\nResent-%s", mutt_make_date(date));
    fprintf(f, "Resent-To: %s\n\n", to);
    mutt_copy_bytes(Context->folder->fp, f, h->content->length);
    fclose(f);
    snprintf(cmd, sizeof(cmd), "%s %s", Sendmail, to);
    invoke_sendmail(cmd, tempfile);
  }
}

ADDRESS *mutt_default_from (void)
{
  ADDRESS *adr = mutt_new_address();

  if (Realname[0]) adr->personal = safe_strdup(Realname);
  adr->mailbox = safe_strdup(Username);
  adr->host = safe_strdup(Fqdn);
  return(adr);
}

/* given a list of addresses, return a list of unique addresses */
ADDRESS *mutt_remove_duplicates (ADDRESS *addr)
{
  ADDRESS *top = NULL;
  ADDRESS *tmp;
  
  if ((top = addr) == NULL) return(NULL);
  addr = addr->next;
  top->next = NULL;
  while (addr) {
    tmp = top;
    do {
      if (addr->mailbox && addr->host && tmp->mailbox && tmp->host &&
	  !strcasecmp(addr->mailbox, tmp->mailbox) &&
	  !strcasecmp(addr->host, tmp->host))
      {
	/* duplicate address, just ignore it */
	tmp = addr;
	addr = addr->next;
	tmp->next = NULL;
	mutt_free_address(&tmp);
      }
      else if (!tmp->next) {
	/* unique address.  add it to the list */
	tmp->next = addr;
	addr = addr->next;
	tmp = tmp->next;
	tmp->next = NULL;
	tmp = NULL; /* so we exit the loop */
      }
      else
	tmp = tmp->next;
    } while (tmp);
  }
  return(top);
}

void mutt_folder_cc (ENVELOPE *env, BODY *body, const char *path)
{
  FOLDER *f;
  time_t now;

  if (body->type == TYPEMULTIPART && strcmp(body->subtype, "signed") && !option(OPTFCCATTACH))
    body = body->parts;

  if (!(f = folder_open(path, "a")))
    return;
  mutt_block_signals();
  if (lock_file(f, TRUE) == -1)
  {
    folder_close(&f);
    mutt_unblock_signals();
    return;
  }
  fseek(f->fp, 0, 2);
  now = time(0);
  fprintf(f->fp, "From %s %s", Username, asctime(localtime(&now)));
  mutt_write_rfc822_header(f->fp, env, body, 0);
  fputs("Status: RO\n\n", f->fp);
  mutt_write_mime_body(body, f->fp);
  fputc('\n', f->fp);
  unlock_file(f);
  folder_close(&f);
  mutt_unblock_signals();
}
