/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "mutt.h"
#include "mutt_curses.h"
#include "keymap.h"

#include <string.h>

int mutt_check_menu (const char *s)
{
  if (strcmp (s, "main-menu") == 0)
    return (MENU_MAIN);
  else if (strcmp (s, "internal-pager") == 0)
    return (MENU_PAGER);
  else if (strcmp (s, "url-menu") == 0)
    return (MENU_URL);
  else if (strcmp (s, "send-attach-menu") == 0)
    return (MENU_SENDATTACH);
  else if (strcmp (s, "send-menu") == 0)
    return (MENU_SEND);
  else if (strcmp (s, "attach-menu") == 0)
    return (MENU_ATTACH);
  else if (strcmp (s, "folder-menu") == 0)
    return (MENU_FOLDER);
  else if (strcmp (s, "alias-menu") == 0)
    return (MENU_ALIAS);
  else
    return(-1);
}

/*
 * bind '<key_sequence>' function-name menu-name
 */
int ci_parse_bind (char *s, char *errmsg)
{
  int count = 0, menu = -1, i;
  char *op = 0;
  char *key = NULL;
  struct binding_t *bindings = 0;

  while ((s = strtokq (s, " \t")))
  {
    switch (count)
    {
      case 0:
	key = s;
	break;
      case 1:
	op = s;
	break;
      case 2:
	if ((menu = mutt_check_menu (s)) == -1)
	{
	  sprintf (errmsg, "%s: no such menu", s);
	  return (-1);
	}
	switch (menu)
	{
	  case MENU_MAIN:

	    bindings = OpMain;
	    break;

	  case MENU_SEND:

	    bindings = OpSend;
	    break;

	  case MENU_PAGER:

	    bindings = OpPager;
	    break;

	  case MENU_POST:

	    bindings = OpPost;
	    break;

	  case MENU_SENDATTACH:

	    bindings = OpSendAttach;
	    break;
	}
	break;
    }
    count++;
    s = NULL;
  }

  /*
   * Check to make sure that we have enough arguments to proceed.
   */
  if (!key || !op || menu == -1)
  {
    strcpy (errmsg, "too few arguments to bind");
    return (-1);
  }

  if (!strcasecmp ("noop", op))
  {
    bindkey (key, menu, OP_NULL, NULL);
    return 0;
  }
  else if (menu == MENU_PAGER || menu == MENU_MAIN || menu == MENU_SEND)
  {
    /*
     * These menus have keymaps of their own and do not refer to the
     * generic map.
     */
    for (i = 0; bindings[i].name; i++)
      if (strcmp (op, bindings[i].name) == 0)
      {
	bindkey (key, menu, bindings[i].op, NULL);
	return (0);
      }
  }
  else
  {
    /*
     * First check the "generic" list of commands.
     */
    for (i=0; OpGeneric[i].name; i++)
    {
      if (strcmp (op, OpGeneric[i].name) == 0)
      {
	bindkey (key, menu, OpGeneric[i].op, NULL);
	return (0);
      }
    }

    /*
     * Now check the menu-specific list of commands (if they exist).
     */
    if (bindings)
    {
      for (i=0; bindings[i].name; i++)
      {
	if (strcmp (op, bindings[i].name) == 0)
	{
	  bindkey (key, menu, bindings[i].op, NULL);
	  return (0);
	}
      }
    }
  }

  sprintf (errmsg, "%s: no such function in map", op);
  return (-1);
}

static int
expand_sequence (char *d, char *s, char *err, size_t errlen)
{
  while (*s)
  {
    if (*s == '^')
    {
      s++;
      if (*s == '^')
	*d++ = *s++;
      else if (*s == '[')
      {
	*d++ = '\033';
	s++;
      }
      else if (isalpha (*s))
	*d++ = toupper (*s++) - '@';
      else
      {
	snprintf (err, errlen, "^%c: unknown escape sequence", *s);
	return (-1);
      }
    }
    else
      *d++ = *s++;
  }
  *d = 0;
  return (0);
}

/* macro <menu> <key> <macro> */
int mutt_parse_macro (char *s, char *err, size_t errlen)
{
  int menu = -1, count = 0;
  char *key = NULL, *sequence = NULL;
  char buf[SHORT_STRING];

  *err = 0;
  while ((s = strtokq (s, " \t\n")) != NULL)
  {
    switch (count)
    {
      case 0: /* menu name */
	if ((menu = mutt_check_menu (s)) == -1)
	{
	  snprintf (err, errlen, "%s: no such menu", s);
	  return (-1);
	}
	break;
      case 1: /* key */
	key = s;
	break;
      case 2: /* key sequence */
	sequence = s;
	break;
      default:
	if (s)
	{
	  snprintf (err, errlen, "too many arguments to macro command");
	  return (-1);
	}
	break;
    }
    count++;
    s = NULL;
  }
  if (count < 3)
  {
    snprintf(err, errlen, "too few arguments to function macro");
    return(-1);
  }
  if (expand_sequence (buf, sequence, err, errlen))
    return (-1);
  bindkey (key, menu, OP_MACRO, buf);
  return 0;
}
