/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"
#include "mutt_curses.h"

#include <string.h>

int ColorDefs[MT_COLOR_MAX];

struct index_object {
  char *name;
  short value;
};

#ifdef HAVE_COLOR
static struct index_object colors[] = {
  { "black", COLOR_BLACK },
  { "blue", COLOR_BLUE },
  { "cyan", COLOR_CYAN },
  { "green", COLOR_GREEN },
  { "magenta", COLOR_MAGENTA },
  { "red", COLOR_RED },
  { "white", COLOR_WHITE },
  { "yellow", COLOR_YELLOW },
  { 0, 0 }
};
#endif /* HAVE_COLOR */

static struct index_object fields[] = {
  { "header",     MT_COLOR_HEADER },
  { "quoted",     MT_COLOR_QUOTED },
  { "signature",  MT_COLOR_SIGNATURE },
  { "subject",    MT_COLOR_SUBJECT },
  { "from",       MT_COLOR_FROM },
  { "indicator",  MT_COLOR_INDICATOR },
  { "attachment", MT_COLOR_ATTACHMENT },
  { "status",     MT_COLOR_STATUS },
  { "tree",       MT_COLOR_TREE },
  { "error",      MT_COLOR_ERROR },
  { "normal",     MT_COLOR_NORMAL },
  { "tilde",      MT_COLOR_TILDE },
  { 0, 0 }
};

void ci_start_color (void)
{
  memset (ColorDefs, A_NORMAL, sizeof (int) * MT_COLOR_MAX);

  /* set some defaults */
  ColorDefs[MT_COLOR_STATUS] = A_REVERSE;
  ColorDefs[MT_COLOR_INDICATOR] = A_REVERSE;

#ifdef HAVE_COLOR
  start_color ();
#endif
}

#ifdef HAVE_COLOR
static int lookup_color (const char *s)
{
  int i;

  for (i=0; colors[i].name; i++)
  {
    if (strcasecmp (s, colors[i].name) == 0)
      return (colors[i].value);
  }
  return(-1);
}

/* usage: color <object> <fg> <bg> */
int ci_parse_color (char *s, char *errmsg, size_t errlen)
{
  int count = 0, object = 0, bold = 0, fg = 0, bg = 0, i;

  /* ignore color commands if we're running on a mono terminal */
  if (!has_colors ())
    return 0;

  while ((s = strtok (s, " \t")) != NULL)
  {
    switch (count) {
    case 0:
      for (i=0; fields[i].name; i++)
      {
	if (strcasecmp (s, fields[i].name) == 0)
	  break;
      }
      if (i == MT_COLOR_MAX)
      {
	snprintf (errmsg, errlen, "%s: no such object", s);
	return (-1);
      }
      object = fields[i].value;
      break;

    case 1:
      if (strncasecmp (s, "bright", 6) == 0)
      {
	bold |= A_BOLD;
	s += 6;
      }
      if ((fg = lookup_color (s)) == -1)
      {
	snprintf (errmsg, errlen, "%s: no such color", s);
	return (-1);
      }
      break;

    case 2:
      if ((bg = lookup_color (s)) == -1)
      {
	snprintf (errmsg, errlen, "%s: no such color", s);
	return (-1);
      }
      break;
    }
    count++;
    s = NULL;
  }
  init_pair (object+1, fg, bg);
  ColorDefs[object] = bold | COLOR_PAIR ((object+1));
  return 0;
}
#endif /* HAVE_COLOR */

/*
 * command: mono <object> <attribute>
 *
 * set attribute for an object when using a terminal with no color support
 */
int ci_parse_mono (char *s, char *err, size_t errlen)
{
  int i;
  int object;
  int attr = 0;

#ifdef HAVE_COLOR
  /* if we have color, ignore the mono commands */
  if (has_colors ())
    return 0;
#endif

  if ((s = strtok(s, " \t")) == NULL)
    return (0); /* empty statement? */
  for (i=0; fields[i].name; i++)
  {
    if (strcasecmp (s, fields[i].name) == 0) break;
  }
  if (i == MT_COLOR_MAX)
  {
    snprintf(err, errlen, "%s: no such attribute", s);
    return(-1);
  }
  object = fields[i].value;
  while ((s = strtok (NULL, " \t")) != (char *)0)
  {
    if (strcasecmp (s, "bold") == 0)
      attr |= A_BOLD;
    else if (strcasecmp (s, "underline") == 0)
      attr |= A_UNDERLINE;
    else if (strcasecmp (s, "none") == 0)
      attr |= A_NORMAL;
    else if (strcasecmp (s, "reverse") == 0)
      attr |= A_REVERSE;
    else
    {
      snprintf (err, errlen, "%s: no such attribute", s);
      return (-1);
    }
    s = 0;
  }
  ColorDefs[object] = attr;
  return (0);
}
