/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"
#include "mutt_curses.h"
#include "keymap.h"
#include "rfc822.h"

#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/wait.h>
#include <string.h>
#include <sys/stat.h>
#include <errno.h>

#define REDRAW_FULL 1
#define REDRAW_INDEX 2
#define REDRAW_CURRENT 4
#define REDRAW_MOTION 8
#define REDRAW_STATUS 16

#define HDR_OFFSET 1

#define CHECK_MSGCOUNT if (!Context) \
	{ \
		beep (); \
		mutt_error ("No mailbox is open."); \
		break; \
	} \
	else if (!Context->msgcount) \
	{ \
		beep (); \
		mutt_error ("There are no messages."); \
		break; \
	}

#define CURHDR Context->hdrs[Context->v2r[current]]
#define OLDHDR Context->hdrs[Context->v2r[oldcurrent]]

/* The folder the user last saved to.  Used by ci_save_message() */
static char LastFolder[_POSIX_PATH_MAX] = "";

static char LastSearch[SHORT_STRING] = "";

/*
 * force_subj is used to force printing of the subject in threading mode
 * so that the subject is displayed when threads overlap a page boundary
 */
static void ci_generate_header (char *s, size_t l, HEADER *h, int force_subj)
{
  int len;
  int cols = option (OPTARROWCURSOR) ? COLS-2 : COLS;

  mutt_generate_header (s, l, h, force_subj);
  len = strlen (s);
  if (len < cols)
    sprintf (s+len, "%*.*s", cols-len, cols-len, "");
  else
    s[cols-1] = 0;
}

void ci_show_last_error (void)
{
  CLEARLINE (LINES-1);
  SETCOLOR (MT_COLOR_ERROR);
  addstr (Errorbuf);
  SETCOLOR (MT_COLOR_NORMAL);
}

#ifdef _PGPPATH
int mutt_check_pgp (HEADER *h)
{
  if (h->pgp)
  {
    if (!isendwin ()) endwin ();

    if (h->pgp == PGPENCRYPT)
    {
      /* make sure we have the passphrase available for decryption */
      if (!pgp_valid_passphrase ())
	return (-1);
    }
    else
      unset_option (OPTVERIFYSIG);
  }

  return 0;
}
#endif /* _PGPPATH */

static int ci_display_message (HEADER *cur)
{
  char tempfile[_POSIX_PATH_MAX], buffer[STRING];
  STATE *st;
  int rc = 0, builtin = 0;

  if (cur->mailcap)
  {
    if ((rc = trioption (M_USEMAILCAP, M_ASK)))
    {
      rc = mutt_yesorno ("Display message using mailcap?", 1);
      if (rc == -1) return 0;
    }
    else
      rc = trioption (M_USEMAILCAP, M_YES);

    if (rc)
    {
      mutt_display_attachment (cur->content, M_REGULAR);
      goto mark_it_read;
    }
  }

  mutt_parse_mime_message (cur);

#ifdef _PGPPATH
  /* see if PGP is needed for this message.  if so, we should exit curses */
  if (cur->pgp)
  {
    if (cur->pgp == PGPENCRYPT)
    {
      if (!pgp_valid_passphrase ())
	return 0;
    }
    else if (cur->pgp == PGPSIGN)
    {
      /* find out whether or not the verify signature */
      if (trioption (M_VERIFYSIG, M_YES) ||
	  (trioption (M_VERIFYSIG, M_ASK) && mutt_yesorno ("Verifiy PGP signature?", 0) == 1))
	set_option (OPTVERIFYSIG);
      else
	unset_option (OPTVERIFYSIG);
    }
    endwin();
  }
#endif

  st = mutt_new_state ();
  st->displaying = 1;

  if (strcmp (Pager, "builtin") == 0)
  {
    mutt_mktemp (tempfile);
    if ((st->fpout = fopen (tempfile, "w")) == NULL)
    {
      mutt_error ("Could not create temporary file!");
      mutt_free_state (&st);
      return (0);
    }
    builtin++;
  }
  else
  {
    if (!isendwin ()) endwin ();
    mutt_create_filter (Pager, &(st->fpout), 0);
  }

  mutt_make_string (buffer, sizeof (buffer), PagerFmt, cur);
  if (builtin)
    buffer[COLS] = 0;
  else
    fprintf (st->fpout, "%s\n\n", buffer);

  st->fpin = Context->folder->fp;
  mutt_copy_header (Context->folder->fp, cur, st->fpout, (Context->weed ? CH_WEED : 0)| CH_DECODE | CH_FROM);
  mutt_body_handler (cur->content, st);

  if (fclose (st->fpout) != 0 && errno != EPIPE)
  {
    mutt_perror ("fclose");
    mutt_unlink (tempfile);
    mutt_free_state (&st);
    return (0);
  }

  mutt_free_state (&st);

  if (builtin)
  {
    /* Invoke the builtin pager */
    rc = ci_simple_pager (buffer, tempfile, cur);
  }
  else
  {
    wait (&rc);
    keypad (stdscr, TRUE);
    if (option (OPTPROMPTAFTER))
    {
      rc = ci_any_key_to_continue ("Command: ");
      rc = dokey (MENU_PAGER);
    }
    else
      rc = 0;
  }

mark_it_read:

  /* mark the message as read and update the new count, if needed */
  if (!cur->read)
  {
    cur->read = 1;
    cur->changed = 1;
    if (!cur->old)
      Context->new--;
  }

  return (rc);
}

static int ci_next_undeleted (int msgno)
{
  int i;

  for (i=msgno+1; i < Context->vcount; i++)
    if (! Context->hdrs[Context->v2r[i]]->deleted)
      return (i);
  return (-1);
}

static int ci_previous_undeleted (int msgno)
{
  int i;

  for (i=msgno-1; i>=0; i--)
    if (! Context->hdrs[Context->v2r[i]]->deleted)
      return (i);
  return (-1);
}

/* returns 1 if OK to proceed, 0 to abort */
static int confirm_func (const char *s)
{
  char tmp[_POSIX_PATH_MAX];
  struct stat st;
  int ret=1;
  int is_mail_dir;
  int is_folder=0;
  FILE *f;

  strfcpy (tmp, Maildir, sizeof (tmp));
  mutt_expand_path (tmp, sizeof (tmp));
  is_mail_dir = (strncmp (tmp, s, strlen (tmp)) == 0);

  if (stat (s, &st) != -1)
  {
    if (is_mail_dir)
    {
      if ((f = fopen (s, "r")) == NULL)
      {
	mutt_perror ("fopen");
	ret = 0;
	goto finish;
      }
      fgets (tmp, sizeof (tmp), f);
      if (!strncmp ("From ", tmp, 5))
	is_folder = 1;
      fclose (f);
    }
    if (option (OPTCONFIRMAPPEND) || (!is_folder && option (OPTCONFIRMFILES)))
    {
      snprintf (tmp, sizeof (tmp), "Append messages to %s %s?",
		is_folder ? "mailbox" : "file", s);
      if (mutt_yesorno (tmp, 1) == 0)
	ret = 0;
    }
  }
  else if (errno == ENOENT)
  {
    if (option(OPTCONFIRMCREATE) || (is_mail_dir && option(OPTCONFIRMFOLDERS)))
    {
      snprintf (tmp, sizeof (tmp), "Create %s %s?", is_mail_dir ? "folder" : "file", s);
      if (mutt_yesorno (tmp, 1) < 1) ret=0;
    }
  }
  else
  {
    mutt_perror ("stat");
    ret = 0;
  }

finish:

  CLEARLINE (LINES-1);
  return (ret);
}

/* returns 0 if the copy/save was successful, or -1 on error/abort */
static int ci_save_message (HEADER *h, int delete, int *redraw)
{
  int i;
  char prompt[SHORT_STRING], buffer[_POSIX_PATH_MAX];

  *redraw = 0;

  snprintf (prompt, sizeof (prompt), "%s%s to mailbox: ",
	    delete ? "Save" : "Copy", h ? "" : " tagged");

  if (h)
    mutt_default_save (buffer, sizeof (buffer), h->env);
  else
  {
    /* look for the first tagged message */

    for (i = 0; i < Context->vcount; i++)
    {
      if (Context->hdrs[Context->v2r[i]]->tagged)
      {
	h = Context->hdrs[Context->v2r[i]];
	break;
      }
    }
    if (h)
    {
      mutt_default_save (buffer, sizeof (buffer), h->env);
      h = NULL;
    }
  }

  CLEARLINE (LINES-1);
  printw ("%s (? for list): %s", prompt, buffer);
  refresh ();

  if ((i = ci_getch ()) == ERR)
  {
    CLEARLINE (LINES-1);
    return (-1);
  }

  if (i == '?')
  {    
    mutt_select_file (buffer, sizeof (buffer));
    *redraw = REDRAW_FULL; /* need to redraw the screen */
    if (!buffer[0]) return (-1);
  }
  else if (!CI_is_return (i))
  {
    if (IsPrint (i)) buffer[0] = 0;
    ungetch (i);
    if (ci_get_field (prompt, buffer, sizeof (buffer), M_FILE) == -1 ||
	!buffer[0])
      return (-1);

    if (!h) *redraw = REDRAW_INDEX | REDRAW_STATUS;
  }
  
  /*
   * This is an undocumented feature of ELM pointed out to me by Felix von
   * Leitner <leitner@prz.fu-berlin.de>
   */
  if (!strcmp (buffer, "."))
    strfcpy (buffer, LastFolder, sizeof (buffer));
  else
    strfcpy (LastFolder, buffer, sizeof (LastFolder));

  mutt_expand_path (buffer, sizeof (buffer));

  /* check to make sure that this file is really the one the user wants */
  if (!confirm_func (buffer))
  {
    CLEARLINE (LINES-1);
    return (-1);
  }

  mutt_save_message (h, buffer, delete);

  return (0);
}

void ci_bounce_message (HEADER *h)
{
  char prompt[SHORT_STRING];
  char buf[LONG_STRING] = { 0 };
  ADDRESS *adr = 0, *padr;

  FOREVER
  {
    if (ci_get_field("Bounce message to: ", buf, sizeof(buf), 0) != 0 ||
	!buf[0])
      return;
    if (mutt_parse_adrlist (&adr, buf, "@") == 0)
      break;
    mutt_free_address (&adr);
    mutt_error (ParseError);
  }
  adr = mutt_expand_aliases (adr);

  buf[0] = 0;
  mutt_write_address (buf, sizeof (buf), adr);
  buf[(COLS+1)/2] = 0; /* so we have a nicer display */
  snprintf (prompt, sizeof (prompt), "Bounce message to %s...?", buf);
  if (mutt_yesorno (buf, 1) < 1)
  {
    CLEARLINE (LINES-1);
    return;
  }

  buf[0] = 0;
  padr = adr;
  while (padr)
  {
    strcat (buf, " ");
    strcat (buf, padr->mailbox);
    if (*padr->host != '@')
    {
      strcat (buf, "@");
      strcat (buf, padr->host);
    }
    padr = padr->next;
  }
  mutt_free_address (&adr);

  if (Context->tagged)
    mutt_bounce_message (NULL, buf);
  else
    mutt_bounce_message (h, buf);
  mutt_error ("Message%s bounced.", Context->tagged ? "s" : "");
}

void ci_pipe_message (HEADER *h)
{
  char buffer[LONG_STRING];
  FILE *f;

  buffer[0] = 0;
  if (ci_get_field ("Pipe to command: ", buffer, sizeof (buffer), 0) != 0 ||
      !buffer[0])
    return;

  mutt_expand_path (buffer, sizeof (buffer));
  endwin ();
  mutt_create_filter (buffer, &f, NULL);
  fseek (Context->folder->fp, h->offset, 0);
  mutt_copy_bytes (Context->folder->fp, f, h->content->length + (h->content->offset - h->offset));
  fclose (f);
  wait (NULL);
  ci_any_key_to_continue (NULL);
}

static int ci_select_sort_method (void)
{
  int method = Sort; /* save the current method in case of abort */
  int ch;

  Sort = 0;
  while (!Sort) {
    mvaddstr(LINES-1,0, "Sort by (d)ate/(f)rom/(r)eceived/(s)ubj/(t)hreads/(u)nsorted/si(z)e?: ");
    ch = ci_getch ();
    if (ch == ERR || CI_is_return (ch))
    {
      Sort = method;
      CLEARLINE (LINES-1);
      return (-1);
    }
    switch (ch)
    {
      case 'D':
	Sort = SORTREVERSE;
      case 'd':
	Sort |= SORTDATE;
	break;
      case 'F':
	Sort = SORTREVERSE;
      case 'f':
	Sort |= SORTFROM;
	break;
      case 'R':
	Sort = SORTREVERSE;
      case 'r':
	Sort |= SORTRECEIVED;
	break;
      case 'S':
	Sort = SORTREVERSE;
      case 's':
	Sort |= SORTSUBJECT;
	break;
      case 't':
      case 'T':
	Sort = SORTTHREADS;
	break;
      case 'U':
	Sort = SORTREVERSE;
      case 'u':
	Sort |= SORTORDER;
	break;
      case 'Z':
	Sort = SORTREVERSE;
      case 'z':
	Sort |= SORTSIZE;
	break;
    }
  }
  CLEARLINE (LINES-1);
  return (Sort != method ? 0 : -1); /* no need to resort if it's the same */
}

/*
 * Return the index of the first new message, or failing that, the first
 * unread message.
 */
static int ci_first_message (void)
{
  int old = -1, i;

  if (Context->msgcount)
  {
    if (option (OPTPOINTNEW))
    {
      for (i=0; i < Context->vcount; i++)
      {
	if (! Context->hdrs[Context->v2r[i]]->read &&
	    ! Context->hdrs[Context->v2r[i]]->deleted)
	{
	  if (! Context->hdrs[Context->v2r[i]]->old)
	    return (i);
	  else if (old == -1)
	    old = i;
	}
      }
      if (old != -1)
	return (old);
    }
    return ((Sort & SORTREVERSE) ? 0 : Context->vcount - 1);
  }
  return 0;
}

#define print_enriched_string(X) _print_enriched_string(X,1)
  
static void _print_enriched_string (unsigned char *s, int do_color)
{
  if (Sort != SORTTHREADS)
  {
    /* Don't need to do any work if we aren't using threads! */
    addstr ((char *)s);
    return;
  }
  
  while (*s)
  {
    if (*s < ctrl('G'))
    {
      if (do_color) SETCOLOR(MT_COLOR_TREE);
      while (*s && *s < ctrl('G'))
      {
	switch(*s)
	{
	  case 1:
	    addch(ACS_LLCORNER);
	    break;
	  case 2:
	    addch(ACS_LTEE);
	    break;
	  case 3:
	    addch(ACS_HLINE);
	    break;
	  case 4:
	    addch(ACS_VLINE);
	    break;
	  case 5:
	    addch(' ');
	    break;
	  case 6:
	    addch('>');
	    break;
	}
	s++;
      }
      if (do_color) SETCOLOR(MT_COLOR_NORMAL);
    }
    else
    {
      addch (*s);
      s++;
    }
  }
}

/* invoke a subshell */
void mutt_subshell (void)
{
  CLEARLINE (LINES-1);
  endwin ();
  puts ("Type \"exit\" to return to Mutt.");
  fflush (stdout);
  mutt_system (Shell);
}

/* enter a mutt command */
void mutt_enter_command (void)
{
  char buffer[LONG_STRING], errbuf[SHORT_STRING];

  buffer[0] = 0;
  if (ci_get_field (":", buffer, sizeof (buffer), 0) != 0 || !buffer[0])
    return;
  mutt_parse_rc_line (buffer, 0, errbuf, sizeof (errbuf));
  if (errbuf[0])
  {
    /*
     * since errbuf could potentially contain printf() sequences in it,
     * we must call mutt_error() in this fashion so that vsprintf()
     * doesn't expect more arguments that we passed
     */
    mutt_error ("%s", errbuf);
  }
}

void mutt_display_address (ADDRESS *adr)
{
  char buf[SHORT_STRING];

  buf[0] = 0;
  mutt_write_address (buf, sizeof (buf), adr);
  buf[COLS-1] = 0;
  mvaddstr (LINES-1, 0, buf);
}

#ifdef _PGPPATH
void mutt_forget_passphrase (void)
{
  pgp_void_passphrase ();
  mutt_error ("PGP passphrase forgotten.");
}
#endif /* _PGPPATH */

/* returns -1 on error so the caller knows that a redraw is not required */
int mutt_view_attachments (HEADER *hdr)
{
  if (hdr->content->type == TYPEMULTIPART)
  {
    mutt_parse_mime_message (hdr);	
    ci_attach (hdr->content);
    return 0;
  }
  else
  {
    beep ();
    mutt_error ("Not a multipart message.");
    return (-1);
  }
}

void mutt_set_flag (HEADER *h, int flag, int bf)
{
  switch (flag)
  {
    case M_DELETE:
      h->deleted = bf;
      break;

    case M_UNDELETE:
      h->deleted = !bf;
      break;

    case M_NEW:
      if (bf)
      {
	if (h->read || h->old)
	{
	  h->read = 0;
	  h->old = 0;
	  Context->new++;
	}
      }
      else
      {
	if (!h->read && !h->old)
	{
	  Context->new--;
	  h->read = 1;
	}
      }
      break;

    case M_OLD:
      if (bf)
      {
	h->old = 1;
	if (!h->read)
	  Context->new--;
      }
      else
      {
	h->old = 0;
	if (!h->read)
	  Context->new++;
      }
      break;

    case M_REPLIED:
      h->replied = bf;

      /* replied messages should be marked as read if not already */
      if (bf && !h->read)
      {
	h->read = 1;
	if (!h->old)
	  Context->new--;
      }
      break;

    case M_FLAG:
      h->flagged = bf;
      break;

    case M_TAG:
      if (bf)
      {
	if (!h->tagged)
	{
	  h->tagged = 1;
	  Context->tagged++;
	}
      }
      else
      {
	if (h->tagged)
	{
	  h->tagged = 0;
	  Context->tagged--;
	}
      }
      break;
  }
  CLEARLINE (LINES-1);
}

static void tag_set_flag (int flag, int bf)
{
  int j;

  for (j=0; j < Context->vcount; j++)
    if (Context->hdrs[Context->v2r[j]]->tagged)
      mutt_set_flag (Context->hdrs[Context->v2r[j]], flag, bf);
}

int mutt_change_flag (HEADER *h, int bf)
{
  int i, flag;

  CLEARLINE (LINES-1);
  printw ("%s flag? (D/N/O/r/*/!): ", bf ? "Set" : "Clear");
  if ((i = ci_getch ()) == ERR)
  {
    CLEARLINE (LINES-1);
    return (-1);
  }

  CLEARLINE (LINES-1);

  switch (i)
  {
    case 'd':
    case 'D':
      flag = M_DELETE;
      break;

    case 'N':
    case 'n':
      flag = M_NEW;
      break;

    case 'o':
    case 'O':
      flag = M_OLD;
      break;

    case 'r':
    case 'R':
      flag = M_REPLIED;
      break;

    case '*':
      flag = M_TAG;
      break;

    case '!':
      flag = M_FLAG;
      break;

    default:
      beep ();
      return (-1);
  }

  if (h)
    mutt_set_flag (h, flag, bf);
  else
    tag_set_flag (flag, bf);

  return 0;
}

/*
 * This function handles the message index window as well as commands returned
 * from the pager (MENU_PAGER).
 */
void ci_main (void)
{
  int top = 0;                 /* first message on the screen */
  int oldcurrent = 0;          /* the last message the user visited */
  int current;                 /* the current message */
  int op = -1;                 /* function to execute */
  int redraw = REDRAW_FULL;
  int menu = MENU_MAIN;        /* which menu are we currently in? */
  int done = 0;                /* controls when to exit the "event" loop */
  int i = 0, j;
  int tag = 0;                 /* has the tag-prefix command been pressed? */
  char buffer[LONG_STRING];

  current = ci_first_message ();
  Context->weed = 1; /* header weeding is on to begin with */
  strcpy (Errorbuf, "Type ? for help.");

  FOREVER
  {
    tag = 0; /* clear the tag-prefix */

    if (menu == MENU_MAIN)
    {
      int oldcount = Context ? Context->msgcount : 0;

      /*
       * check for new mail in the mailbox.  If nonzero, then something has
       * changed about the file (either we got new mail or the file was
       * modified from underneath us.
       */
      if (mutt_check_mailbox (Context) != 0)
      {
	/* take note of the current message */
	if (oldcount)
	{
	  if (current < Context->vcount)
	    oldcurrent = CURHDR->index;
	  else
	    oldcount = 0; /* invalid message number! */
	}

	mutt_sort_headers (Context);

	if (oldcount)
	{
	  current = -1;
	  /* restore the current message to the message it was pointing to */
	  for (i = 0; i < Context->vcount; i++)
	    if (Context->hdrs[Context->v2r[i]]->index == oldcurrent)
	    {
	      current = i;
	      break;
	    }

	  /* make sure we actually found it */
	  if (current < 0)
	    current = 0;
	}
	else
	  current = 0;

	redraw = REDRAW_FULL;
      }
      
      if (redraw == REDRAW_FULL)
      {
	clear ();
	ci_show_last_error ();

	redraw = REDRAW_INDEX | REDRAW_STATUS;
      }

      if (redraw & REDRAW_STATUS)
      {
	mutt_status_line (buffer, sizeof (buffer), Status);
	if (!option (OPTSTATUSONTOP))
	{
	  CLEARLINE (LINES-2);
	}
	else
	  CLEARLINE (0);
	SETCOLOR (MT_COLOR_STATUS);
	printw ("%-*.*s", COLS, COLS, buffer);
	SETCOLOR (MT_COLOR_NORMAL);
	redraw &= ~REDRAW_STATUS;
      }

      if (Context->folder)
      {
	if (current >= top + PAGELEN)
	{
	  while (top + PAGELEN  - 1 < current)
	    top += PAGELEN;
	  redraw = REDRAW_INDEX;
	}
	else if (current < top)
	{
	  while (top > current)
	    top -= PAGELEN;
	  if (top < 0) top = 0;
	  redraw = REDRAW_INDEX;
	}
	else if (oldcurrent != current)
	{
	  if (!redraw)
	    redraw = REDRAW_MOTION;
	}
      }

      if (redraw == REDRAW_INDEX)
      {
	for (i = top; i < top + PAGELEN; i++)
	  if (Context && i < Context->vcount)
	  {
	    ci_generate_header (buffer, sizeof (buffer), Context->hdrs[Context->v2r[i]], (i==top ? M_FORCESUBJ : 0));
	    CLEARLINE(i-top+HDR_OFFSET);
	    
	    if (option(OPTARROWCURSOR))
	      move(i-top+HDR_OFFSET, 2);
	    else if (i == current)
	      SETCOLOR(MT_COLOR_INDICATOR);
	    _print_enriched_string ((unsigned char *)buffer, i!=current || option(OPTARROWCURSOR));
	    if (!option (OPTARROWCURSOR) && i == current)
	      SETCOLOR (MT_COLOR_NORMAL);
	  }
	  else
	    CLEARLINE (i-top+HDR_OFFSET);
	redraw = 0;
      }
      else if (redraw == REDRAW_MOTION)
      {
	if (option (OPTARROWCURSOR))
	{
	  ci_generate_header (buffer, sizeof(buffer), Context->hdrs[Context->v2r[oldcurrent]], (oldcurrent==top ? M_FORCESUBJ : 0));
	  mvaddstr (oldcurrent-top+HDR_OFFSET, 0, "  ");
	  print_enriched_string ((unsigned char *)buffer);
	}
	else
	{
	  /* redraw the current line in case the flags changed. */
	  CLEARLINE (oldcurrent-top+HDR_OFFSET);
	  ci_generate_header (buffer, sizeof(buffer), OLDHDR, (oldcurrent == top ? M_FORCESUBJ : 0));
	  print_enriched_string ((unsigned char *)buffer);
		  
	  move (current-top+HDR_OFFSET, 0);
	  ci_generate_header (buffer, sizeof(buffer), CURHDR, (current == top ? M_FORCESUBJ : 0));
	  SETCOLOR (MT_COLOR_INDICATOR);
	  _print_enriched_string ((unsigned char *)buffer, 0);
	  SETCOLOR (MT_COLOR_NORMAL);
	}
	redraw = 0;
      }
      else if (redraw == REDRAW_CURRENT)
      {
	if (option (OPTARROWCURSOR))
	  move (current-top+HDR_OFFSET, 2);
	else
	{
	  move (current-top+HDR_OFFSET, 0);
	  SETCOLOR(MT_COLOR_INDICATOR);
	}	      
	ci_generate_header (buffer, sizeof(buffer), CURHDR, (current == top ? M_FORCESUBJ : 0));
	_print_enriched_string ((unsigned char *)buffer, option(OPTARROWCURSOR));
	if (!option(OPTARROWCURSOR))
	  SETCOLOR(MT_COLOR_NORMAL);
	redraw = 0;
      }
      
      if (Context)
      {
	oldcurrent = current;
	if (option (OPTARROWCURSOR))
	{
	  move (current-top+HDR_OFFSET, 0);
	  if (Context->msgcount)
	  {
	    SETCOLOR(MT_COLOR_INDICATOR);
	    addstr("->");
	    SETCOLOR(MT_COLOR_NORMAL);
	  }
	}
	else
	  move (current - top + HDR_OFFSET, COLS - 1);
      }
      else
	move (LINES-1, COLS-1);
      refresh ();

      if (Timeout)
	timeout (Timeout*1000); /* milliseconds */      
      op = dokey (MENU_MAIN);
      if (Timeout)
	timeout (-1); /* restore blocking operation */

      /* special handling for the tag-prefix function */
      if (op == OP_MAIN_TAG_PREFIX)
      {
	if (!Context)
	{
	  beep ();
	  mutt_error ("No mailbox is open.");
	  continue;
	}

	if (!Context->tagged)
	{
	  beep ();
	  mutt_error ("No tagged messages.");
	  continue;
	}
	tag = 1;

	/* give visual indication that the next command is a tag- command */
	move (LINES-1, 0);
	clrtoeol ();
	addstr ("tag-");

	/* get the real command */
	if ((op = dokey (MENU_MAIN)) == OP_MAIN_TAG_PREFIX)
	{
	  /* abort tag sequence */
	  CLEARLINE (LINES-1);
	  continue;
	}
      }

      ci_clear_error ();
    }

    if (op == -1) /* might have caught SIGWINCH */
    {
#ifdef USE_SLANG_CURSES
      if (Sigwinch)
      {
	mutt_resize_screen ();
	redraw = REDRAW_FULL;
	menu = MENU_MAIN;
	Sigwinch = 0;
	top = 0; /* so we scroll the right amount */
      }
#endif
      continue;
    }

    switch (op)
    {
      case OP_BOTTOM_OF_PAGE:

	CHECK_MSGCOUNT;

	current = top + PAGELEN - 1;
	if (current > Context->vcount - 1)
	  current = Context->vcount - 1;
	break;

      case OP_MAIN_DELETE_PATTERN:

	CHECK_MSGCOUNT;

	ci_pattern_function (M_DELETE, "Delete messages matching: ", CURHDR);
	redraw = REDRAW_INDEX;
	break;


#ifdef USE_POP
      case OP_MAIN_FETCH_MAIL:

	mutt_fetchPopMail ();
	redraw = REDRAW_FULL;
	break;
#endif /* USE_POP */

      case OP_MAIN_FIRST_MESSAGE:

	CHECK_MSGCOUNT;

	current = 0;
	break;

      case OP_HELP:

	ci_help (MENU_MAIN);
	redraw = REDRAW_FULL;
	break;

      case OP_MAIN_JUMP:

	CHECK_MSGCOUNT;

	ungetch (LastKey);
	buffer[0] = 0;
	if (ci_get_field ("Jump to message: ", buffer, sizeof (buffer), 0) != 0 ||
	    !buffer[0])
	  break;

	if (! isdigit (buffer[0]))
	{
	  beep ();
	  mutt_error ("Argument must be a message number.");
	  break;
	}

	i = atoi (buffer);
	if (i > 0 && i <= Context->msgcount)
	{
	  if (Context->hdrs[i-1]->virtual != -1)
	    current = Context->hdrs[i-1]->virtual;
	  else
	  {
	    beep ();
	    mutt_error ("That message is not visible.");
	  }
	}
	else
	{
	  beep();
	  mutt_error ("Invalid message number.");
	}

	break;

      case OP_MAIN_LAST_MESSAGE:

	CHECK_MSGCOUNT;

	current = Context->vcount - 1;
	while (top + PAGELEN < current) top += PAGELEN;
	redraw = REDRAW_INDEX;
	break;

      case OP_MAIN_LIMIT:

	CHECK_MSGCOUNT;

	oldcurrent = Context->vcount ? CURHDR->index : -1;

	if (ci_pattern_function (M_LIMIT, "Limit to messages matching: ", CURHDR) == 0)
	{
	  if (oldcurrent >= 0)
	  {
	    /* try to find what used to be the current message */
	    current = -1;
	    for (i = 0; i < Context->vcount; i++)
	      if (Context->hdrs[Context->v2r[i]]->index == oldcurrent)
	      {
		current = i;
		break;
	      }
	    if (current < 0)
	      current = 0;
	  }
	  else
	    current = 0;
	  redraw = REDRAW_INDEX | REDRAW_STATUS;
	}
	break;	  

      case OP_MIDDLE_OF_PAGE:

	CHECK_MSGCOUNT;

	i = top + PAGELEN;
	if (i > Context->vcount - 1)
	  i = Context->vcount - 1;
	current = top + (i - top) / 2;
	break;

      case OP_NEXT_PAGE:

	CHECK_MSGCOUNT;

	if (top + PAGELEN < Context->vcount)
	{
	  top += PAGELEN;
	  current = top;
	  redraw = REDRAW_INDEX;
	}
	else
	{
	  beep ();
	  mutt_error ("You are on the last page.");
	}
	break;

      case OP_PREVIOUS_PAGE:

	CHECK_MSGCOUNT;

	if (top > 0)
	{
	  top -= PAGELEN;
	  if (top < 0)
	    top  = 0;
	  current = top;
	  redraw = REDRAW_INDEX;
	}
	else
	{
	  beep ();
	  mutt_error ("You are on the first page.");
	}
	break;

      case OP_QUIT:

	if (mutt_close_mailbox (Context) == 0)
	  done = 1;
	else
	  redraw = REDRAW_FULL; /* new mail arrived? */
	break;

      case OP_REDRAW:

	clearok (stdscr, TRUE);
	redraw = REDRAW_FULL;
	break;

      case OP_MAIN_SEARCH:
      case OP_MAIN_SEARCH_REVERSE:

	CHECK_MSGCOUNT;

	if ((current = ci_search_command (current, (op == OP_MAIN_SEARCH_REVERSE))) == -1)
	  current = oldcurrent;
	break;

      case OP_MAIN_SORT_MESSAGES:

	if (ci_select_sort_method () == 0)
	{
	  if (Context && Context->msgcount)
	  {
	    oldcurrent = CURHDR->index;
	    mutt_sort_headers (Context);
	    /* try to restore the current message */
	    for (i = 0; i < Context->vcount; i++) {
	      if (Context->hdrs[Context->v2r[i]]->index == i)
		current = i;
	    }
	    redraw = REDRAW_INDEX | REDRAW_STATUS;
	  }
	}
	break;

      case OP_MAIN_SYNC_FOLDER:

	CHECK_MSGCOUNT;

	if (mutt_sync_mailbox (Context) == 0)
	  current = ci_first_message ();
	redraw = REDRAW_INDEX | REDRAW_STATUS;
	break;

      case OP_TAG_MESSAGE:

	CHECK_MSGCOUNT;

	if (CURHDR->tagged)
	{
	  CURHDR->tagged = 0;
	  Context->tagged--;
	}
	else
	{
	  CURHDR->tagged = 1;
	  Context->tagged++;
	}
	redraw = REDRAW_STATUS;
	if (option (OPTRESOLVE) && current < Context->vcount - 1)
	  current++;
	else
	  redraw |= REDRAW_CURRENT;

	break;

      case OP_MAIN_TAG_PATTERN:

	CHECK_MSGCOUNT;

	ci_pattern_function (M_TAG, "Tag messages matching: ", CURHDR);
	redraw = REDRAW_INDEX | REDRAW_STATUS;
	break;

      case OP_TOP_OF_PAGE:

	CHECK_MSGCOUNT;

	current = top;
	break;

      case OP_MAIN_UNDELETE_PATTERN:

	CHECK_MSGCOUNT;

	if (ci_pattern_function (M_UNDELETE, "Undelete messages matching: ", CURHDR) == 0)
	  redraw = REDRAW_INDEX;
	break;

      case OP_MAIN_UNTAG_PATTERN:

	CHECK_MSGCOUNT;

	if (ci_pattern_function (M_UNTAG, "Untag messages matching: ", CURHDR) == 0)
	  redraw = REDRAW_INDEX | REDRAW_STATUS;
	break;

      case OP_UNDELETE:

	CHECK_MSGCOUNT;

	if (tag)
	{
	  tag_set_flag (M_DELETE, 0);
	  redraw = REDRAW_INDEX;
	}
	else
	{
	  CURHDR->deleted = 0;
	  if (option (OPTRESOLVE) && current < Context->vcount - 1)
	    current++;
	  else
	    redraw = REDRAW_CURRENT;
	}
	break;

      case OP_MAIN_SEARCH_BODY:

	CHECK_MSGCOUNT;

	if (ci_get_field ("Search messages for: ", LastSearch, sizeof (LastSearch), 0) == -1
	    || !LastSearch[0])
	  break;

	if ((current = mutt_body_search (current, LastSearch)) == -1)
	  current = oldcurrent;
	break;

	/* --------------------------------------------------------------------
	 * The following operations can be performed inside of the pager.
	 */

      case OP_MAIN_CHANGE_FOLDER:

	mvaddstr (LINES-1, 0, "Open mailbox ('?' for list): ");
	refresh ();
	i = ci_getch ();
	if (i == ERR || CI_is_return(i))
	{
	  CLEARLINE (LINES-1);
	  break;
	}
	if (i != '?')
	{
	  ungetch (i);
	  buffer[0] = 0;
	  if (ci_get_field ("Open mailbox: ", buffer, sizeof (buffer), M_FILE) != 0 || !buffer[0])
	  {
	    CLEARLINE(LINES-1);
	    break;
	  }
	}
	else
	{
	  refresh ();
	  mutt_select_file (buffer, sizeof (buffer));
	  redraw = REDRAW_FULL;
	  if (!buffer[0]) break;
	}
	mutt_expand_path (buffer, sizeof (buffer));

	if (!mutt_is_valid_mailbox (buffer)) break;
	if (mutt_close_mailbox (Context) != 0)
	{
	  redraw = REDRAW_INDEX | REDRAW_STATUS;
	  break;
	}
	mutt_folder_hook (buffer);
	if ((Context = mutt_open_mailbox (buffer, Context, 0)) != NULL)
	  current = ci_first_message ();
	else
	  current = 0;
	ci_clear_error ();
	redraw = REDRAW_FULL;
	break;

      case OP_DELETE_THREAD:

	CHECK_MSGCOUNT;

	mutt_do_thread (CURHDR, M_DELETE);
	if (option (OPTRESOLVE))
	  if ((current = ci_next_undeleted (current)) == -1)
	    current = oldcurrent;
	redraw = REDRAW_INDEX;
	break;

      case OP_UNDELETE_THREAD:

	CHECK_MSGCOUNT;

	mutt_do_thread (CURHDR, M_UNDELETE);
	if (option(OPTRESOLVE))
	{
	  if ((current = mutt_next_thread (CURHDR)) == -1)
	    current = oldcurrent;
	}
	redraw = REDRAW_INDEX;
	break;

      case OP_DISPLAY_MESSAGE:
      case OP_DISPLAY_HEADERS: /* don't weed the headers */

	CHECK_MSGCOUNT;

	/*
	 * toggle the weeding of headers so that a user can press the key
	 * again while reading the message.
	 */
	if (op == OP_DISPLAY_HEADERS)
	  Context->weed = !Context->weed;

	op = ci_display_message (CURHDR);
	menu = MENU_PAGER;
	oldcurrent = current;
	continue;

      case OP_MAIN_EXIT:

	if (menu == MENU_MAIN)
	{
	  mutt_fastclose_mailbox (Context);
	  done = 1;
	}
	break;

      case OP_MAIN_NEXT_UNDELETED:

	CHECK_MSGCOUNT;

	if (current >= Context->vcount - 1)
	{
	  if (menu == MENU_MAIN)
	  {
	    beep ();
	    mutt_error ("You are on the last message.");
	  }
	  break;
	}
	if ((current = ci_next_undeleted (current)) == -1)
	{
	  current = oldcurrent;
	  if (menu == MENU_MAIN)
	  {
	    mutt_error ("No undeleted messages.");
	    beep ();
	  }
	}
	else if (menu == MENU_PAGER)
	{
	  op = OP_DISPLAY_MESSAGE;
	  continue;
	}
	break;

      case OP_MAIN_NEXT_MESSAGE:

	CHECK_MSGCOUNT;

	if (current >= Context->vcount - 1)
	{
	  if (menu == MENU_MAIN)
	  {
	    mutt_error ("You are on the last message.");
	    beep ();
	  }
	  break;
	}
	current++;
	if (menu == MENU_PAGER)
	{
	  op = OP_DISPLAY_MESSAGE;
	  continue;
	}
	break;

      case OP_MAIN_PREVIOUS_UNDELETED:

	CHECK_MSGCOUNT;

	if (current < 1)
	{
	  beep ();
	  mutt_error ("You are on the first message.");
	  break;
	}
	if ((current = ci_previous_undeleted (current)) == -1)
	{
	  current = oldcurrent;
	  if (menu == MENU_MAIN)
	  {
	    beep ();
	    mutt_error ("No undeleted messages.");
	  }
	}
	else if (menu == MENU_PAGER)
	{
	  op = OP_DISPLAY_MESSAGE;
	  continue;
	}
	break;

      case OP_MAIN_PREVIOUS_MESSAGE:

	CHECK_MSGCOUNT;

	if (current < 1)
	{
	  if (menu == MENU_MAIN)
	  {
	    beep ();
	    mutt_error ("You are on the first message.");
	  }
	  break;
	}
	current--;
	if (menu == MENU_PAGER)
	{
	  op = OP_DISPLAY_MESSAGE;
	  continue;
	}
	break;

      case OP_MAIN_COPY_MESSAGE:
      case OP_MAIN_SAVE_MESSAGE:

	CHECK_MSGCOUNT;

	if (ci_save_message (tag ? NULL : CURHDR,
			     (op == OP_MAIN_SAVE_MESSAGE), &redraw) != 0 ||
	    op != OP_MAIN_SAVE_MESSAGE)
	  break;

	if (tag) redraw = REDRAW_INDEX;
	else if (option (OPTRESOLVE) || menu == MENU_PAGER)
	{
	  if ((current = ci_next_undeleted (current)) == -1)
	  {
	    current = oldcurrent;
	    redraw |= REDRAW_CURRENT;
	  }
	  else if (menu == MENU_PAGER)
	  {
	    op = OP_DISPLAY_MESSAGE;
	    continue;
	  }
	}
	else
	  redraw |= REDRAW_CURRENT;
	break;

      case OP_DELETE:

	CHECK_MSGCOUNT;

	if (tag)
	{
	  tag_set_flag (M_DELETE, 1);
	  redraw = REDRAW_INDEX;
	}
	else
	{
	  CURHDR->deleted = 1;
	  if (option (OPTRESOLVE))
	  {
	    if ((current = ci_next_undeleted (current)) == -1)
	    {
	      current = oldcurrent;
	      redraw = REDRAW_CURRENT;
	    }
	    else if (menu == MENU_PAGER)
	    {
	      op = OP_DISPLAY_MESSAGE;
	      continue;
	    }
	  }
	  else
	    redraw = REDRAW_CURRENT;
	}

	break;

      case OP_MAIN_NEXT_NEW:
      case OP_MAIN_NEXT_UNREAD:
      case OP_MAIN_PREV_NEW:
      case OP_MAIN_PREV_UNREAD:

	CHECK_MSGCOUNT;

	i = current;
	current = -1;
	for (j=0; j!=Context->vcount; j++)
	{
	  if (op == OP_MAIN_NEXT_NEW || op == OP_MAIN_NEXT_UNREAD)
	  {
	    i++;
	    if (i > Context->vcount-1)
	    {
	      mutt_error ("Search wrapped to top.");
	      i = 0;
	    }
	  }
	  else
	  {
	    i--;
	    if (i < 0)
	    {
	      mutt_error ("Search wrapped to bottom.");
	      i = Context->vcount - 1;
	    }
	  }

	  if (! Context->hdrs[Context->v2r[i]]->deleted &&
	      ! Context->hdrs[Context->v2r[i]]->read)
	  {
	    if (op == OP_MAIN_NEXT_UNREAD ||
		! Context->hdrs[Context->v2r[i]]->old)
	    {
	      current = i;
	      break;
	    }
	  }
	}
	if (current == -1)
	{
	  current = oldcurrent;
	  beep ();
	  mutt_error ("No new messages.");
	}
	else if (menu == MENU_PAGER)
	{
	  op = OP_DISPLAY_MESSAGE;
	  continue;
	}
	break;

      case OP_FLAG_MESSAGE:
#define TOGGLE_FLAG(x) (x)->flagged = !(x)->flagged; (x)->changed = 1

	CHECK_MSGCOUNT;

	TOGGLE_FLAG (Context->hdrs[Context->v2r[current]]);
	if (option (OPTRESOLVE))
	{
	  if (current < Context->vcount - 1)
	    current++;
	  else
	    redraw = REDRAW_CURRENT;
	}
	else if (menu == MENU_MAIN)
	  redraw = REDRAW_CURRENT;
	break;

      case OP_TOGGLE_NEW:

	CHECK_MSGCOUNT;

	if (tag)
	{
	  tag_set_flag (M_NEW, 1);
	  redraw = REDRAW_STATUS | REDRAW_INDEX;
	}
	else
	{
	  if (CURHDR->read || CURHDR->old)
	  {
	    CURHDR->read = 0;
	    CURHDR->old = 0;
	    CURHDR->changed = 1;
	    Context->new++;
	  }
	  else
	  {
	    CURHDR->read = 1;
	    CURHDR->changed = 1;
	    Context->new--;
	  }

	  if (option (OPTRESOLVE))
	  {
	    if ((current = ci_next_undeleted (current)) == -1)
	    {
	      current = oldcurrent;
	      redraw = REDRAW_CURRENT | REDRAW_STATUS;
	    }
	    else
	      redraw = REDRAW_STATUS;
	  }
	  else
	    redraw = REDRAW_CURRENT | REDRAW_STATUS;
	}
	break;

      case OP_MAIN_NEXT_THREAD:

	CHECK_MSGCOUNT;

	if ((current = mutt_next_thread (CURHDR)) < 0)
	{
	  current = oldcurrent;
	  beep ();
	  mutt_error ("No more threads.");
	}
	else if (menu == MENU_PAGER)
	{
	  op = OP_DISPLAY_MESSAGE;
	  continue;
	}
	break;

      case OP_MAIN_PREVIOUS_THREAD:

	CHECK_MSGCOUNT;

	if ((current = mutt_previous_thread (CURHDR)) < 0)
	{
	  current = oldcurrent;
	  beep ();
	  mutt_error ("You are on the first thread.");
	}
	else if (menu == MENU_PAGER)
	{
	  op = OP_DISPLAY_MESSAGE;
	  continue;
	}
	break;

      case OP_MAIN_SET_FLAG:
      case OP_MAIN_CLEAR_FLAG:

	CHECK_MSGCOUNT;

	if (mutt_change_flag (tag ? NULL : CURHDR, (op == OP_MAIN_SET_FLAG)) == 0)
	{
	  redraw = REDRAW_STATUS;
	  if (!tag && option (OPTRESOLVE))
	  {
	    if ((current = ci_next_undeleted (current)) == -1)
	    {
	      current = oldcurrent;
	      redraw |= REDRAW_CURRENT;
	    }
	  }
	  else
	    redraw |= REDRAW_CURRENT;
	}
	break;

	/* --------------------------------------------------------------------
	 * These functions can be invoked directly in the internal-pager
	 */

      case OP_BOUNCE_MESSAGE:

	CHECK_MSGCOUNT;

	ci_bounce_message (CURHDR);
	break;

      case OP_BROWSE_URL:

	CHECK_MSGCOUNT;

	ci_browse_url (CURHDR);
	redraw = REDRAW_FULL;
	break;

      case OP_CREATE_ALIAS:

	CHECK_MSGCOUNT;

	ci_mkalias (CURHDR->env);
	break;

      case OP_DISPLAY_ADDRESS:

	CHECK_MSGCOUNT;

	mutt_display_address (CURHDR->env->from);
	break;

      case OP_ENTER_COMMAND:

	mutt_enter_command ();
	redraw = REDRAW_FULL;

	break;

      case OP_FORWARD_MESSAGE:

	CHECK_MSGCOUNT;

	ci_send_message (SENDFORWARD, NULL, NULL, NULL, tag ? NULL : CURHDR);
	redraw = REDRAW_FULL;
	break;

#ifdef _PGPPATH
      case OP_FORGET_PASSPHRASE:

	mutt_forget_passphrase ();
	break;
#endif /* _PGPPATH */

      case OP_GROUP_REPLY:

	CHECK_MSGCOUNT;

	ci_send_message (SENDREPLY|SENDGROUPREPLY, NULL, NULL, NULL, tag ? NULL : CURHDR);
	redraw = REDRAW_FULL;
	break;

      case OP_LIST_REPLY:

	CHECK_MSGCOUNT;

	ci_send_message (SENDREPLY|SENDLISTREPLY, NULL, NULL, NULL, tag ? NULL : CURHDR);
	redraw = REDRAW_FULL;
	break;

      case OP_MAIL:

	ci_send_message (0, NULL, NULL, NULL, NULL);
	redraw = REDRAW_FULL;
	break;

      case OP_PIPE_MESSAGE:

	CHECK_MSGCOUNT;

	ci_pipe_message (CURHDR);
	break;

      case OP_PRINT:

	CHECK_MSGCOUNT;

	mutt_print_message (tag ? NULL : CURHDR);
	break;

      case OP_RECALL_MESSAGE:

	ci_send_message (SENDPOSTPONED, NULL, NULL, NULL, NULL);
	redraw = REDRAW_FULL;
	break;

      case OP_REPLY:

	CHECK_MSGCOUNT;

	ci_send_message (SENDREPLY, NULL, NULL, NULL, tag ? NULL : CURHDR);
	redraw = REDRAW_FULL;
	break;

      case OP_SUBSHELL:

	mutt_subshell ();
	break;

      case OP_VIEW_ATTACHMENTS:

	CHECK_MSGCOUNT;

	if (mutt_view_attachments (CURHDR) == 0) redraw = REDRAW_FULL;
	break;

      default:

	if (menu == MENU_MAIN)
	{
	  beep ();
	  mutt_error ("Key is not bound.  Press '?' for help.");
	}
    }

    if (menu == MENU_PAGER)
    {
      menu = MENU_MAIN;
      Context->weed = 1; /* turn header weeding back on. */
      redraw = REDRAW_FULL;
    }

    if (done)
      break;
  }
}
