/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "muttlib.h"

#include <fcntl.h>
#include <dirent.h>
#include <sys/stat.h>
#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <utime.h>

#ifdef HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif

#define MAXLOCKATTEMPT 5

extern void ci_endwin (const char *);

/* struct used by mutt_sync_mailbox() to store new offsets */
struct m_update_t {
  long hdr;
  long body;
};

FOLDER *folder_open (const char *f, const char *mode)
{
  FOLDER *t;

  t = (FOLDER *)safe_malloc (sizeof (FOLDER));
  memset (t, 0, sizeof (FOLDER));
  if ((t->fp = fopen (f, mode)) == NULL)
  {
    mutt_perror (f);
    dprint (1, (debugfile, "folder_open: can't open %s (errno %d).\n", f, errno));
    safe_free ((void **)&t);
    return (0);
  }
  t->name = safe_strdup (f);
  return (t);
}

void folder_close (FOLDER **f)
{
  if (*f)
  {
    safe_free ((void **)&(*f)->name);
    fclose ((*f)->fp);
    safe_free ((void **)f);
  }
}

static int copy_message (FOLDER *in, HEADER *hdr, FOLDER *out)
{
  mutt_copy_header (in->fp, hdr, out->fp, CH_FROM | CH_UPDATE);

  /*
   * Copy the body of the message to the target destination.  We do an
   * explicit fseek() here to make sure that we are in the right place to
   * start copying data since mutt_copy_header() may not leave in->fp at the
   * correct location.
   */
  fseek (in->fp, hdr->content->offset, 0);
  if (mutt_copy_bytes (in->fp, out->fp, hdr->content->length) == -1)
    return (-1);

  /* add the message separator */
  if (fputc ('\n', out->fp) == EOF)
    return (-1);

  if (ferror (out->fp) != 0)
  {
    dprint (1, (debugfile, "copy_message(): ferror() returned nonzero!\n"));
    return (-1);
  }
  return (0);
}

int mutt_is_spool (const char *s)
{
  char path[_POSIX_PATH_MAX];

  strfcpy (path, Spoolfile, sizeof (path));
  mutt_expand_path (path, sizeof (path));
  return (strcmp (s, path) == 0);
}

int lock_file (FOLDER *f, int excl)
{
#ifdef USE_FCNTL
  struct flock lck;
#endif

#ifdef USE_DOTLOCK
  char lockfile[_POSIX_PATH_MAX], nfslockfile[_POSIX_PATH_MAX];
  int fd, count=0;
  struct stat sb;

  /* this assumes that setgid() has already been called */
#if !defined(WORLD_WRITABLE_SPOOL) && !defined(USE_SETGID) && !defined(HOMESPOOL)
  if (!mutt_is_spool (f->name))
#endif
  {
    snprintf (lockfile, sizeof (lockfile), "%s.lock", f->name);
  try_again:
    while ((fd = open (lockfile, O_WRONLY|O_EXCL|O_CREAT, 0644)) < 0)
    {
      if (errno != EEXIST && errno != EAGAIN)
	return (-1);
      if (++count == MAXLOCKATTEMPT)
      {
	if (mutt_yesorno ("Lock count exceeded, remove lock?", 1) == 1)
	{
	  unlink (lockfile);
	  count=0;
	  continue;
	}
	else
	{
	  dprint (1,(debugfile,"lock_file(): lock count exceded on %s.\n",
		    f->name));
	  return (-1);
	}
      }
      mutt_error ("Waiting for lock attempt #%d...", count);
      sleep (2);
    }
    close (fd);

    /*
     * locking does not work well over NFS on many systems, so do an extra
     * check.
     */
    sprintf (nfslockfile, "%s.%s.%d", lockfile, Hostname, (int)getpid());
    link (lockfile, nfslockfile);
    if (stat (nfslockfile, &sb) == -1)
    {
      dprint(1,(debugfile,"lock_file(): could not stat the NFS lockfile %s.\n",
		nfslockfile));
      mutt_perror ("stat");
      return (-1);
    }
    unlink (nfslockfile);
    if (sb.st_nlink != 2) goto try_again;
  }
#endif /* USE_DOTLOCK */

#ifdef USE_FCNTL
  memset (&lck, 0, sizeof(struct flock));
  lck.l_type = excl ? F_WRLCK : F_RDLCK;
  lck.l_whence = SEEK_SET;
  if (fcntl (fileno (f->fp), F_SETLKW, &lck) == -1)
  {
    dprint(1,(debugfile, "lock_file(): fcntl errno %d.\n", errno));
    return (-1);
  }
#endif

#ifdef USE_FLOCK
  if (flock (fileno (f->fp), excl ? LOCK_EX : LOCK_SH) == -1) return (-1);
#endif /* USE_FLOCK */

  return (0);
}

void unlock_file (FOLDER *f)
{
#ifdef USE_FCNTL
  struct flock unlockit = { F_UNLCK, 0, 0, 0 };
#endif
#ifdef USE_DOTLOCK
  char lockfile[_POSIX_PATH_MAX];
#endif

#ifdef USE_FLOCK
  flock(fileno (f->fp), LOCK_UN);
#endif

#ifdef USE_FCNTL
  memset (&unlockit, 0, sizeof(struct flock));
  unlockit.l_type = F_UNLCK;
  unlockit.l_whence = SEEK_SET;
  fcntl (fileno (f->fp), F_SETLKW, &unlockit);
#endif

#ifdef USE_DOTLOCK
  snprintf (lockfile, sizeof (lockfile), "%s.lock", f->name);
  unlink (lockfile);
#endif
}

/*
 * This routine checks the on-disk state of the currently opened mailbox.  If
 * the mailbox has been modified, a sanity check is performed to make sure
 * that the state in-memory is still valid, such as if new mail is appended
 * to the mailbox.
 */
int mutt_check_mailbox (CONTEXT *ctx)
{
  struct stat st;
  char buffer[LONG_STRING];
#ifdef USE_SETGID
  int isSetGid = 0;
#endif

  if (!ctx || !ctx->folder) return 0; /* no mailbox is open */

  if (stat (ctx->folder->name, &st) == -1)
  {
    mutt_perror ("stat");
    mutt_fastclose_mailbox (ctx);
    return (-1);
  }

  if (st.st_mtime == ctx->folder->mtime && st.st_size == ctx->folder->size)
    return (0); /* Not modified... */

  dprint (1, (debugfile, "check_mailbox(): folder was modified, performing sanity check...\n"));

  if (st.st_size < ctx->folder->size)
  {
    dprint (1, (debugfile, "check_mailbox(): folder was smaller than when previously read!\nprevious size = %ld\ncurrent size = %ld\n", ctx->folder->size, (long)st.st_size));

    if (ctx->locked) unlock_file (ctx->folder);
    mutt_fastclose_mailbox (ctx);
    mutt_error ("Mailbox was externally modified!");
    return (-1);
  }
  else if (st.st_size == ctx->folder->size)
  {
    /* the file was touched, but it is still the same length, so just exit */
    ctx->folder->mtime = st.st_mtime;
    return (0);
  }

  /* lock the file if it isn't already */
  if (!ctx->locked)
  {
    mutt_block_signals ();

#ifdef USE_SETGID
    if (mutt_is_spool (Context->folder->name))
    {
      if (SETEGID (MailGid) != 0)
      {
	mutt_perror ("setegid");
	mutt_unblock_signals ();
	mutt_fastclose_mailbox (ctx);
	return (-1);
      }
      isSetGid = 1;
    }
#endif

    if (lock_file (ctx->folder, FALSE) == -1)
    {
#ifdef USE_SETGID
      if (isSetGid) SETEGID (UserGid);
#endif
      mutt_unblock_signals ();
      mutt_fastclose_mailbox (ctx);
      mutt_error ("Couldn't lock mailbox %s!", ctx->folder->name);
      return (-1);
    }
  }

  /*
   * Check to make sure that the only change to the mailbox is that 
   * message(s) were appended to this file.  My heuristic is that we should
   * see the message separator at *exactly* what used to be the end of the
   * folder.
   */
  fseek (ctx->folder->fp, ctx->folder->size, 0);
  if (fgets (buffer, sizeof (buffer), ctx->folder->fp) == NULL)
  {
    dprint (1,(debugfile,"check_mailbox(): fgets() returned 0!\n"));

    mutt_error ("Error reading mailbox!");
    unlock_file (ctx->folder);
#ifdef USE_SETGID
    if (isSetGid) SETEGID (UserGid);
#endif
    mutt_unblock_signals ();
    mutt_fastclose_mailbox (ctx);
    return (-1);
  }

  if (strncmp ("From ", buffer, 5) != 0)
  {
    dprint (1, (debugfile, "check_mailbox(): folder was externally rewritten!\nHere's what I got:\n%s\n", buffer));

    mutt_error ("Mailbox was corrupted!");
    unlock_file (ctx->folder);
#ifdef USE_SETGID
    if (isSetGid) SETEGID (UserGid);
#endif /* USE_SETGID */
    mutt_unblock_signals ();
    mutt_fastclose_mailbox (ctx);
    return (-1);
  }

  fseek (ctx->folder->fp, ctx->folder->size, 0);
  mutt_parse_mailbox (ctx);

  /*
   * Only unlock the folder if it was locked inside of this routine.  It may
   * have been locked elsewhere, like in mutt_checkpoint_mailbox().
   */
  if (!ctx->locked)
  {
    unlock_file (ctx->folder);
#ifdef USE_SETGID
    if (isSetGid) SETEGID (UserGid);
#endif
    mutt_unblock_signals ();
  }

  return (-1); /* signal the fact that new mail has arrived. */
}

/* Close and abort any in-memory changes to the folder. */
void mutt_fastclose_mailbox (CONTEXT *ctx)
{
  int i;

  if (ctx)
  {
    folder_close (&ctx->folder);

    for (i = 0; i < ctx->msgcount; i++) mutt_free_header (&ctx->hdrs[i]);
    safe_free ((void **)&ctx->hdrs);
    safe_free ((void **)&ctx->v2r);
    memset (ctx, 0, sizeof(CONTEXT));
  }
}

/* return 1 if the folder has been modified */
int mutt_was_modified (CONTEXT *ctx)
{
  int i;

  for (i=0; i<ctx->msgcount; i++)
    if (ctx->hdrs[i]->changed || ctx->hdrs[i]->deleted) return 1;
  return 0;
}

/*
 * Args:
 *	update	--	If true, update memory to reflect the new offsets into
 *			the stream as a result of rewriting the mailbox.  This
 *			avoids having to reread the entire mailbox, thus
 *			speeding up the sync-mailbox operation.
 */
static int mutt_checkpoint_mailbox (CONTEXT *ctx, int update)
{
  FOLDER tmp;
  char tempfile[_POSIX_PATH_MAX];
  int i, j, saved = 0, save_sort = SORTORDER, is_spool;
  int need_sort = 0; /* flag to restort mailbox if new mail arrives */
  struct stat statbuf;
  struct utimbuf utimebuf;
  struct m_update_t *newOffset = NULL;

  if (access (ctx->folder->name, W_OK) == -1)
  {
    dprint(1, (debugfile,"mutt_checkpoint_mailbox(): access() return -1 (errno %d).\n",
	      errno));
    mutt_unblock_signals (); /* in case we came from mutt_close_mailbox() */
    if (mutt_yesorno ("Mailbox is read-only.  Abort changes?", 0) == 1)
    {
      mutt_fastclose_mailbox (ctx);
      return (1);
    }
    return (0);
  }

  is_spool = mutt_is_spool (ctx->folder->name);

  /*
   * If the messages were sorted in some way other than mailbox-order,
   * resort so that we save to disk in a correct way.
   */
  if (Sort != SORTORDER)
  {
    save_sort = Sort;
    Sort = SORTORDER;
    mutt_sort_headers (ctx);
  }

  /*
   * need to open the file for writing in such a way that it does not truncate
   * the file, so use read-write mode.
   */
  if ((ctx->folder->fp = freopen (ctx->folder->name, "r+", ctx->folder->fp)) == NULL)
  {
    mutt_unblock_signals ();
    mutt_error ("Could not reopen mailbox!");
    dprint (1,(debugfile,"mutt_checkpoint_mailbox(): could not reopen folder in read/write (r+) mode!\n"));
    return (0);
  }

  mutt_block_signals ();

#if USE_SETGID
  if (is_spool)
  {
    if (SETEGID (MailGid) != 0)
    {
      mutt_perror ("setegid");
      return 0;
    }
  }
#endif

  if (lock_file (ctx->folder, TRUE) == -1)
  {
    dprint(1,(debugfile,"mutt_checkpoint_mailbox(): unable to lock mailbox %s.\n",
	      ctx->folder->name));
    mutt_error ("Unable to lock mailbox!");
    goto bail;
  }

  /* indicate that we have this mailbox locked */
  ctx->locked = 1;

  /* Check to make sure that the file hasn't changed on disk */
  if (mutt_check_mailbox (ctx) != 0)
  {
    need_sort = 1;
    goto bail;
  }

  /*
   * Create a temporary file to write the new version of the mailbox in.
   */
  mutt_mktemp (tempfile);
  if ((i = open (tempfile, O_WRONLY | O_EXCL | O_CREAT, 0600)) == -1 ||
      (tmp.fp = fdopen (i, "w")) == NULL)
  {
    dprint(1, (debugfile, "mutt_checkpoint_mailbox(): could not create temp file! (errno=%d)\n", errno));

    mutt_error ("Could not create temporary file!");
    goto bail;
  }

  tmp.name = tempfile;
  
  if (!WriteMsginc) mutt_error ("Synchronizing %s...", ctx->folder->name);

  if (update)
  {
    /* allocate space for the new offsets */
    newOffset = (struct m_update_t *)safe_malloc (sizeof (struct m_update_t) * ctx->msgcount);
    memset (newOffset, 0, sizeof (struct m_update_t) * ctx->msgcount);
  }

  for (i=0; i < ctx->msgcount; i++)
  {
    if (! ctx->hdrs[i]->deleted)
    {

      saved++;
      if (WriteMsginc && (saved % WriteMsginc == 0 || saved == 1))
	mutt_error ("Writing message %d...", saved);

      /* save the new offset for this message */
      if (update) newOffset[i].hdr = ftell (tmp.fp);

      if (copy_message (ctx->folder, ctx->hdrs[i], &tmp) == -1)
      {
	fclose (tmp.fp);
	mutt_unlink (tempfile);
	mutt_error ("Error while writing mailbox!");
	goto bail;
      }

      /*
       * Since messages could have been deleted, the offsets stored in memory
       * will be wrong, so update what we can, which is the offset of this
       * message, and the offset of the body.  If this is a mulitipart message,
       * we just flush the in memory cache so that the message will be reparsed
       * if the user accesses it later.
       */
      if (update)
      {
	newOffset[i].body = ftell (tmp.fp) - ctx->hdrs[i]->content->length - 1;
	mutt_free_body (&ctx->hdrs[i]->content->parts);
      }
    }
  }

  if (fclose (tmp.fp) != 0)
  {
    dprint(1, (debugfile, "mutt_checkpoint_mailbox(): fclose() returned non-zero.\n"));
    unlink (tempfile);
    goto bail;
  }

  if (saved || option (OPTKEEPEMPTY) || is_spool)
  {
    FILE *src = fopen (tempfile, "r");
    long len;
    char buf[LONG_STRING];
    
    /* Save the state of this folder. */
    stat (ctx->folder->name, &statbuf);
    
    /*
     * Copy the folder back into place.  Note that since the stream is
     * read-write at this point, there is no need to worry about running
     * out of space since this just writes over the existing file.  However,
     * we must use ftruncate() to free up any extra space when done
     * copying the temporary file.
     */
    fseek (ctx->folder->fp, 0, 0);
    while ((len = fread(buf, 1, sizeof(buf), src)) > 0)
      fwrite (buf, 1, len, ctx->folder->fp);
    ftruncate (fileno (ctx->folder->fp), ftell (ctx->folder->fp));
    fclose (src);

    /* update the size of the mailbox accordingly */
    ctx->folder->size = ftell (ctx->folder->fp);

    unlock_file (ctx->folder);
    ctx->locked = 0;

    fclose (ctx->folder->fp);

    /* Restore the previous access/modification times */
    utimebuf.actime = statbuf.st_atime;
    utimebuf.modtime = statbuf.st_mtime;
    utime (ctx->folder->name, &utimebuf);

    /* reopen the mailbox in read-only mode */
    if ((ctx->folder->fp = fopen (ctx->folder->name, "r")) == NULL)
    {
      mutt_unblock_signals ();
      mutt_fastclose_mailbox (ctx);
      mutt_error ("Fatal error!  Could not reopen mailbox!");
      return (0);
    }

    /* free up memory and update the header for each saved message */
    if (update)
    {
      ctx->new = 0;
      ctx->tagged = 0;
      for (i=0, j=0; i < ctx->msgcount; i++)
      {
	if (!ctx->hdrs[i]->deleted)
	{
	  ctx->hdrs[i]->changed = 0;
	  ctx->hdrs[i]->offset = newOffset[i].hdr;
	  ctx->hdrs[i]->content->hdr_offset = newOffset[i].hdr;
	  ctx->hdrs[i]->content->offset = newOffset[i].body;
	  ctx->hdrs[i]->index = j; /* FOO -- WRONG! */
	  ctx->hdrs[i]->msgno = j + 1;
	  if (i != j)
	  {
	    ctx->hdrs[j] = ctx->hdrs[i];
	    ctx->hdrs[i] = NULL;
	  }
	  if (ctx->hdrs[j]->tagged) ctx->tagged++;
	  if (!ctx->hdrs[j]->read && !ctx->hdrs[j]->old) ctx->new++;
	  j++;
	}
	else
	{
	  /* free up memory from deleted message */
	  mutt_free_header (&ctx->hdrs[i]);
	}
      }
      safe_free ((void **)&newOffset);
      ctx->msgcount = j;
    }
    else
      ctx->msgcount = saved;
  }
  else
  {
    /* no messages left, so unlink and unlock the file and free memory */
    unlink (ctx->folder->name);
    unlock_file (ctx->folder);
    mutt_fastclose_mailbox (ctx);
  }

  unlink (tempfile);

#ifdef USE_SETGID
  if (is_spool) SETEGID (UserGid);
#endif

  mutt_unblock_signals ();
  Sort = save_sort; /* Restore the default value. */

  return (1); /* signal success */

  /* Come here in case of disaster */
bail:

  if (ctx->locked)
  {
    unlock_file (ctx->folder);
    ctx->locked = 0;
  }

#ifdef USE_SETGID
  if (is_spool) SETEGID (UserGid);
#endif

  mutt_unblock_signals ();

  safe_free ((void **)&newOffset);

  if ((ctx->folder->fp = freopen (ctx->folder->name, "r", ctx->folder->fp)) == NULL)
  {
    mutt_error ("Could not reopen mailbox!");
    sleep (1);
    mutt_exit (1);
  }

  if (need_sort || save_sort != Sort)
  {
    Sort = save_sort;
    mutt_sort_headers (ctx);
  }

  dprint(1,(debugfile,"mutt_sync_mailbox(): operation failed.\n"));

  return (0);
}

int mutt_sync_mailbox (CONTEXT *ctx)
{
  int msgcount = ctx->msgcount;

  if (!mutt_was_modified (ctx))
  {
    mutt_error ("Mailbox is unchanged.");
    return (-1);
  }
  if (mutt_checkpoint_mailbox (ctx, 1))
  {
    mutt_sort_headers (ctx);
    mutt_error ("%d messages kept, %d deleted.",
		ctx->msgcount, msgcount - ctx->msgcount);
    return (0);
  }
  return (-1);
}

/* return 0 if successful, otherwise return -1 on error */
int mutt_close_mailbox (CONTEXT *ctx)
{
  int i, modified=0, read=0, deleted=0, move_messages=0, keep_deleted=0;
  int isSpool = 0;
  char mbox[_POSIX_PATH_MAX];
  char buf[SHORT_STRING];
  FOLDER *f;
  
  if (!ctx->folder) return (0); /* empty context, so this is a null-op */

  for (i=0; i < ctx->msgcount; i++)
  {
    if (ctx->hdrs[i]->deleted)
    {
      deleted++;
      continue;
    }
    else if (! ctx->hdrs[i]->old && option (OPTMARKOLD))
    {
      /* Mark new messages as old. */
      ctx->hdrs[i]->old = 1;
      ctx->hdrs[i]->changed = 1;
      modified++;
    }
    else if (ctx->hdrs[i]->changed)
      modified++;
    
    /* Note that deleted messages are not counted. */
    if (ctx->hdrs[i]->read) read++;
  }

  if (read && !option (OPTHOLD))
  {
    char *p;

    if ((p = mutt_find_hook (M_MBOXHOOK, ctx->folder->name)))
    {
      isSpool = 1;
      strfcpy (mbox, p, sizeof (mbox));
    }
    else
    {
      strfcpy (mbox, Inbox, sizeof (mbox));
      isSpool = mutt_is_spool (ctx->folder->name) && !mutt_is_spool (mbox);
    }
    mutt_expand_path (mbox, sizeof (mbox));

    if (isSpool)
    {
      if (option (OPTASKMOVE))
      {
	snprintf (buf, sizeof (buf), "Move read messages to %s?", mbox);
	if ((move_messages = mutt_yesorno (buf, 1)) < 0)
	  return (-1);
      }
      else
	move_messages = 1;
    }
    else
      read = 0; /* no messages will be moved */
  }

  if (deleted && option (OPTASKDEL))
  {
    if ((keep_deleted = mutt_yesorno ("Keep deleted messages?", 0)) < 0)
      return (-1);
  }

  if (move_messages)
  {
    if (!(f = folder_open (mbox, "a"))) return (-1);

    mutt_block_signals ();
    if (lock_file (f, 1) != 0) goto cleanup;
    mutt_error ("Moving read messages to %s...", mbox);

    for (i = 0; i < ctx->msgcount; i++)
      if (ctx->hdrs[i]->read && !ctx->hdrs[i]->deleted)
      {
	if (copy_message (ctx->folder, ctx->hdrs[i], f) == -1)
	{
	  unlock_file (f);
	  mutt_error ("Error while moving messages!");
	  goto cleanup;
	}
	ctx->hdrs[i]->deleted = 1;
      }
    unlock_file (f);
    folder_close (&f);
    mutt_unblock_signals ();
  }
  else if (!modified && !deleted)
  {
    mutt_unblock_signals ();
    mutt_error ("Mailbox is unchanged.");
    mutt_fastclose_mailbox (ctx);
    return (0);
  }

  if (keep_deleted)
  {
    for (i = 0; i < ctx->msgcount; i++) ctx->hdrs[i]->deleted = 0;
    deleted = 0;
  }

  /*
   * if mutt_sync_mailbox() fails, the user will likely end up with two copies
   * of a message, because we have already appended the "read" messages to
   * the "mbox" mailbox by this point.  however, since nothing will have been
   * lost, this isn't really that big of a problem.
   */
  if (!mutt_checkpoint_mailbox (ctx, 0)) return (-1);

  /* print a status message */
  if (isSpool)
    mutt_error ("%d kept, %d moved, %d deleted.", ctx->msgcount, read, deleted);
  else
    mutt_error ("%d kept, %d deleted.", ctx->msgcount, deleted);
  mutt_fastclose_mailbox (ctx);
  sleep (1); /* pause to let the user see the message */
  return (0);

cleanup:

  folder_close (&f);
  mutt_unblock_signals ();
  return (-1);
}

CONTEXT *mutt_open_mailbox (char *fol, CONTEXT *ctx, int flags)
{
#ifdef USE_SETGID
  int is_setgid = 0;
#endif

  if (!ctx) ctx = (CONTEXT *)safe_malloc (sizeof (CONTEXT));
  memset (ctx, 0, sizeof (CONTEXT));
  ctx->weed = 1;
  if (!(ctx->folder = folder_open (fol, "r")))
  {
    safe_free ((void **)&ctx);
    return (NULL);
  }
  mutt_block_signals ();

#ifdef USE_SETGID
  if (mutt_is_spool (ctx->folder->name))
  {
    is_setgid = 1;
    if (SETEGID (MailGid) != 0)
    {
      mutt_perror ("setegid");
      return (NULL);
    }
  }
#endif

  if (lock_file (ctx->folder, 0) == -1)
  {
    folder_close (&ctx->folder);
#ifdef USE_SETGID
    if (is_setgid) SETEGID (UserGid);
#endif
    mutt_unblock_signals ();
    safe_free ((void **)&ctx);
    return (NULL);
  }
  mutt_parse_mailbox (ctx);
  unlock_file (ctx->folder);

#ifdef USE_SETGID
  if (is_setgid) SETEGID (UserGid);
#endif

  mutt_unblock_signals ();

  /* we don't always need to sort the mailbox (like the postponed mailbox) */
  if (flags & M_NOSORT)
  {
    int i;

    ctx->vcount = 0;
    for (i=0; i<ctx->msgcount; i++)
    {
      ctx->hdrs[i]->virtual = ctx->vcount;
      ctx->hdrs[i]->msgno = i + 1;
      ctx->v2r[ctx->vcount] = i;
      ctx->vcount++;
    }
  }
  else
    mutt_sort_headers (ctx);
  return (ctx);
}

int mutt_save_attachment (BODY *m, char *path)
{
  STATE s;

  if (access (path, F_OK) == 0)
  {
    mutt_error ("File already exists!");
    return (-1);
  }

  if ((s.fpout = fopen (path, "w")) == NULL)
  {
    mutt_perror ("fopen");
    return -1;
  }

  fseek ((s.fpin = Context->folder->fp), m->offset, 0);
  s.prefix = 0;

  switch (m->encoding)
  {
    case ENC7BIT:
    case ENC8BIT:
    case ENCBINARY:
      mutt_decode_xbit (&s, m->length);
      break;
    case ENCQUOTEDPRINTABLE:
      mutt_decode_quoted (&s, m->length, mutt_is_text_type(m->type, m->subtype));
      break;
    case ENCBASE64:
      mutt_decode_base64 (&s, m->length, mutt_is_text_type(m->type, m->subtype));
      break;
  }

  if (fclose (s.fpout) == 0) return 0;
  
  dprint(1,(debugfile, "save_attachment: fclose() returned non-zero! (errno=%d)\n", errno));

  return -1;
}

int mutt_save_message (HEADER *h, const char *path, int delete)
{
  FOLDER *fp = NULL;
  int rc = 0, i, count = 0;

  if ((fp = folder_open (path, "a")) == NULL)
  {
    rc = -1;
    goto recover;
  }

  if ((rc = lock_file (fp, TRUE)) == -1) goto recover;

  if (h)
  {
    if ((rc = copy_message (Context->folder, h, fp)) == -1)
    {
      unlock_file (fp);
      goto recover;
    }
    if (delete)
    {
      h->deleted = 1;
      if (h->tagged)
      {
	h->tagged = 0;
	Context->tagged--;
      }
    }
    count++;
  }
  else
  {
    for (i=0; i < Context->vcount; i++)
      if (Context->hdrs[Context->v2r[i]]->tagged)
      {
	if ((rc = copy_message (Context->folder, Context->hdrs[Context->v2r[i]], fp)) == -1)
	{
	  unlock_file (fp);
	  goto recover;
	}
	if (delete)
	{
	  Context->hdrs[Context->v2r[i]]->deleted = 1;
	  if (Context->hdrs[Context->v2r[i]]->tagged)
	    Context->hdrs[Context->v2r[i]]->tagged = 0;
	}
	count++;
      }
    if (delete) Context->tagged = 0;
  }
  fflush (fp->fp);
  unlock_file (fp);

 recover:

  folder_close (&fp);

  if (rc < 0)
    mutt_error ("Error while copying message!");
  else
    mutt_error ("%s %d message%s to %s.", delete?"Saved":"Copied", count,
		count > 1 ? "s" : "", path);
  return (rc);
}

/* wipe and remove a file */
void mutt_unlink (const char *s)
{
  struct stat stbuf;
  FILE *f;
  char b[LONG_STRING];
  int bytes = 0, chunk;

  if (stat (s, &stbuf) == -1) return;
  if ((f = fopen (s, "r+")) == NULL) return;
  memset (b, 0, sizeof (b));
  while (bytes < stbuf.st_size)
  {
    chunk = stbuf.st_size - bytes;
    if (chunk > LONG_STRING)
      chunk = LONG_STRING;
    fwrite (b, 1, chunk, f);
    bytes += chunk;
  }
  fclose (f);
  unlink (s);
}
