/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "muttlib.h"
#include "rfc1522.h"
#include "mutt_curses.h"

#include <ctype.h>
#include <stdlib.h>
#include <string.h>

int mutt_is_mail_list (ADDRESS *addr)
{
  LIST *p = Mailing_lists;

  if (!addr->mailbox) return 0;
  while (p) {
    if (strncasecmp(addr->mailbox, p->data, strlen(p->data)) == 0)
      return 1;
    p = p->next;
  }
  return 0;
}

static int
check_for_mailing_list (ADDRESS *adr, char *pfx, char *buf, int buflen)
{
  ADDRESS *list;
  int len = strlen(pfx);

  while (adr)
  {
    if (mutt_is_mail_list (adr))
    {
      strcpy (buf, pfx);
      len = strlen (pfx);
      buflen -= len;
      buf += len;

      if (option (OPTREVALIAS) && (list = alias_reverse_lookup(adr)) && list->personal)
	strfcpy (buf, list->personal, buflen);
      else
	strfcpy (buf, adr->mailbox, buflen);
      return 1;
    }
    adr = adr->next;
  }
  return 0;
}

static void make_from (ENVELOPE *hdr, char *buffer, size_t len, int do_lists)
{
  ADDRESS *tmpaddr;

  if (do_lists || mutt_addr_is_user (hdr->from))
  {
    if (check_for_mailing_list (hdr->to, "To ", buffer, len)) return;
    if (check_for_mailing_list (hdr->cc, "Cc ", buffer, len)) return;
  }

  /* check to see if this message is from me */
  if (mutt_addr_is_user (hdr->from) && hdr->to)
  {
    strcpy (buffer, "To ");
    if (option (OPTREVALIAS) && (tmpaddr = alias_reverse_lookup (hdr->to)) &&
        tmpaddr->personal)
      strfcpy (buffer+3, tmpaddr->personal, len-3);
    else if (hdr->to->personal)
      strfcpy (buffer + 3, hdr->to->personal, len - 3);
    else
    {
      tmpaddr = hdr->to->next;
      hdr->to->next = NULL;
      mutt_write_address (buffer+3, len-3, hdr->to);
      hdr->to->next = tmpaddr;
    }
    return;
  }

  if (option (OPTREVALIAS) && (tmpaddr = alias_reverse_lookup (hdr->from)) &&
      tmpaddr->personal)
    strfcpy (buffer, tmpaddr->personal, len);
  else if (hdr->from && hdr->from->personal)
    strfcpy (buffer, hdr->from->personal, len);
  else
  { 
    buffer[0] = 0;
    mutt_write_address (buffer, len, hdr->from);
  }
}

/*
 * Return values:
 * 0: user is not in list
 * 1: user is unique recipient
 * 2: user is in the TO list
 * 3: user is in the CC list
 * 4: user is originator
 */
static int user_is_recipient (ENVELOPE *hdr)
{
  ADDRESS *a;

  if (mutt_addr_is_user (hdr->from))
    return 4;

  a = hdr->to;
  while (a)
  {
    if (mutt_addr_is_user (a))
    {
      if (!a->next && !hdr->cc && hdr->to == a)
	return (1);
      else
	return (2);
    }
    a = a->next;
  }

  a = hdr->cc;
  while (a)
  {
    if (mutt_addr_is_user (a))
      return (3);
    a = a->next;
  }
  return (0);
}

/*
 * %a = address of author
 * %c = size of message in bytes
 * %C = current message number
 * %d = date and time of message
 * %D = date and time in ISO-8601 format
 * %f = entire from line
 * %F = entire from line, unless from self
 * %l = number of lines in the message
 * %L = list-from function.
 * %n = name of author
 * %i = message-id
 * %u = user (login) name of author
 * %M = month name of message
 * %m = number of month
 * %N = number of the day of the month
 * %s = subject
 * %S = message status (e.g., [New], [Old], [Deleted], [Flagged], ...)
 * %T = number of message in the mailbox
 * %Y = year
 * %y = last two digits of the year
 *
 * %>X = right justify rest of line and pad with character X
 * %|X = pad to end of line with character X
 *
 */

void
_mutt_make_string (char *dest, size_t destlen, char *s, HEADER *hdr, int flags)
{
  char buf[STRING], buf2[STRING], prefix[SHORT_STRING], fmt[SHORT_STRING];
  char *cp;
  int len, count;
  char *wptr = dest; /* pointer to current writing position */
  int wlen = 0;      /* how many characters written so far */
  int ch;            /* char to use as filler */

  destlen--; /* save room for the terminal null (\0) character */
  dest[destlen] = 0;
  while (*s && wlen < destlen)
  {
    if (*s == '%')
    {
      s++;
      if (*s == '%')
      {
	*wptr++ = '%';
	wlen++;
	continue;
      }
      /* strip off the formatting commands */
      cp=prefix;
      count=0;
      while (count < sizeof(prefix) && (*s == '-' || *s == '.' || isdigit(*s)))
      {
	*cp++ = *s++;
        count++;
      }
      *cp = 0;

      switch (*s) {
      case '>': /* right justify the rest of the line */
	s++;
	ch = *s++;
	_mutt_make_string(buf, sizeof(buf), s, hdr, flags);
	len = COLS - wlen - strlen(buf);
	while (len > 0 && wlen < destlen)
	{
	  *wptr++ = ch;
	  len--;
	  wlen++;
	}
	break;

      case '|': /* pad to end of line */
	s++;
	ch = *s++;
	if (destlen > COLS) destlen = COLS;
	while (wlen < destlen)
	{
	  *wptr++ = ch;
	  wlen++;
	}
	break;

      case 'a':
	mutt_simple_address (buf2, sizeof (buf2), hdr->env->from);
	snprintf (fmt, sizeof (fmt), "%%%ss", prefix);
	sprintf (buf, fmt, buf2);
	break;

      case 'c':
	snprintf (fmt, sizeof (fmt), "%%%sd", prefix);
	sprintf (buf, fmt, hdr->content->length);
	break;

      case 'C':
	snprintf (fmt, sizeof (fmt), "%%%sd", prefix);
	snprintf (buf, sizeof (buf), fmt, hdr->msgno);
	break;

      case 'd':
	snprintf (buf2, sizeof(buf2), "%s %d, %d %02d:%02d:%02d %c%02d%02d",
		  Months[hdr->month], hdr->day, hdr->year+1971, hdr->hours,
		  hdr->minutes, hdr->seconds, hdr->zoccident ? '-' : '+',
		  hdr->zhours, hdr->zminutes);
	snprintf (fmt, sizeof (fmt), "%%%ss", prefix);
	snprintf (buf, sizeof (buf), fmt, buf2);
	break;

      case 'D':
	snprintf (buf, sizeof (buf), "%04d-%02d-%02dT%02d:%02d:%02d", hdr->year+1971, hdr->month+1, hdr->day, hdr->hours, hdr->minutes, hdr->seconds);
	break;

      case 'f':
	buf2[0] = 0;
	mutt_write_address (buf2, sizeof(buf2), hdr->env->from);
	snprintf (fmt, sizeof (fmt), "%%%ss", prefix);
	sprintf (buf, fmt, buf2);
	break;

      case 'F':
	snprintf (fmt, sizeof(fmt), "%%%ss", prefix);
	make_from (hdr->env, buf2, sizeof(buf2), 0);
	snprintf (buf, sizeof (buf), fmt, buf2);
	break;

      case 'i':
	snprintf (fmt, sizeof (fmt), "%%%ss", prefix);
        sprintf (buf, fmt, hdr->env->message_id ? hdr->env->message_id : "<no.id>");
	break;

      case 'l':
	snprintf (fmt, sizeof (fmt), "%%%sd", prefix);
	sprintf (buf, fmt, hdr->lines);
	break;

      case 'L':
	make_from (hdr->env, buf2, sizeof(buf2), 1);
	snprintf (fmt, sizeof (fmt), "%%%ss", prefix);
	sprintf (buf, fmt, buf2);
	break;

      case 'M':
	snprintf (fmt, sizeof (fmt), "%%%ss", prefix);
	snprintf (buf, sizeof (buf), fmt, Months[hdr->month]);
	break;

      case 'm':
	snprintf (fmt, sizeof (fmt), "%%%sd", prefix);
	sprintf (buf, fmt, hdr->month+1);
	break;

      case 'N':
	snprintf (fmt, sizeof (fmt), "%%%sd", prefix);
	sprintf (buf, fmt, hdr->day);
	break;

      case 'n':
	if (hdr->env->from && hdr->env->from->personal)
	  strfcpy (buf2, hdr->env->from->personal, sizeof (buf2));
	else
	{
	  buf2[0] = 0;
	  mutt_write_address (buf2, sizeof (buf2), hdr->env->from);
	}
	snprintf (fmt, sizeof(fmt), "%%%ss", prefix);
	sprintf (buf, fmt, buf2);
	break;

      case 's':
	if ((flags != M_NOTREE) && Sort == SORTTHREADS)
	{
	  if (hdr->tree)
	  {
	    if (flags == M_FORCESUBJ || hdr->subject_changed)
	      snprintf (buf, sizeof(buf), "%s%s", hdr->tree, hdr->env->subject);
	    else
	      strfcpy (buf, hdr->tree, sizeof(buf));
	  }
	  else
	    strfcpy (buf, hdr->env->subject ? hdr->env->subject : "", sizeof (buf));
	}
	else
	{
	  snprintf (fmt, sizeof (fmt), "%%%ss", prefix);
	  snprintf (buf, sizeof (buf), fmt, hdr->env->subject ? hdr->env->subject : "");
	}
	break;

      case 'S':
	if (hdr->deleted)
	  ch = 'D';
	else if (hdr->read)
	{
	  if (hdr->tagged)
	    ch = '*';
	  else if (hdr->flagged)
	    ch = '!';
	  else if (hdr->replied)
	    ch = 'r';
	  else
	    ch = '-';
	}
	else if (hdr->old)
	  ch = 'O';
	else
	  ch = 'N';
	snprintf (buf, sizeof (buf), "%c", ch);
	break;

      case 't':
	snprintf (fmt, sizeof (fmt), "%%%sc", prefix);
	snprintf (buf, sizeof (buf), fmt, Tochars[user_is_recipient(hdr->env)]);
	break;

      case 'T':
	snprintf (fmt, sizeof(fmt), "%%%sd", prefix);
	snprintf (buf, sizeof(buf), fmt, Context->msgcount);
	break;

      case 'u':
	snprintf (fmt, sizeof(fmt), "%%%ss", prefix);
	sprintf (buf, fmt, hdr->env->from->mailbox);	
	break;

      case 'Y':
	snprintf (fmt, sizeof(fmt), "%%%sd", prefix);
	sprintf (buf, fmt, hdr->year + 1971);
	break;

      case 'y':
	snprintf (fmt, sizeof (fmt), "%%%sd", prefix);
	sprintf (buf, fmt, (hdr->year + 1971) % 100);
	break;
      }

      if ((len = strlen (buf)) + wlen > destlen)
	len = destlen - wlen;
      memcpy(wptr, buf, len);
      wptr += len;
      wlen += len;
    }
    else if (*s == '\\')
    {
      s++;
      if (!*s) break;
      if (*s == 'n')
      {
	*wptr++ = '\n';
	wlen++;
      }
      else if (*s == 't')
      {
	*wptr++ = '\t';
	wlen++;
      }
      else
      {
	*wptr++ = *s;
	wlen++;
      }
    }
    else
    {
      *wptr++ = *s;
      wlen++;
    }
    s++;
  }
  *wptr = 0;
}

void
mutt_generate_header (char *buf, size_t buflen, HEADER *cur, int force_subj)
{
  char flag;
  int len;

  if (cur->mailcap)
    flag = 'M';
#ifdef _PGPPATH
  else if (cur->pgp == PGPSIGN)
    flag = 'S';
  else if (cur->pgp == PGPENCRYPT)
    flag = 'P';
  else if (cur->pgp == PGPKEY)
    flag = 'K';
#endif /* _PGPPATH */
  else
    flag = ' ' ;
    
  /* first do the fixed part of the header */
  snprintf (buf, buflen, "%4d %c%c%c%c ",
	    cur->msgno,
	    cur->read ? (cur->replied ? 'r': ' ') : (cur->old ? 'O' : 'N'),
	    cur->deleted ? 'D' : ' ',
	    flag,
	    cur->tagged ? '*' : (cur->flagged ? '!' :
				 (Tochars[user_is_recipient(cur->env)])));

  len = strlen (buf);
  buf += len;
  buflen -= len;

  /* now process the user configurable part. */
  _mutt_make_string (buf, buflen, Hdr_format, cur, force_subj);
}
