/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "muttlib.h"
#include "rfc1522.h"

#include <ctype.h>
#include <string.h>

#define IsPrint(c) (((c) > 0xa0) && ((c) < 0xff))

typedef void encode_t (char *, unsigned char *, size_t);

extern char *tspecials;

static void q_encode_string (char *d, unsigned char *s, size_t len)
{
  char charset[SHORT_STRING];
  int cslen;
  int wordlen;
  char *wptr = d;

  snprintf (charset, sizeof (charset), "=?%s?Q?",
	    strcasecmp ("us-ascii", charset) == 0 ? "unknown-8bit" : Charset);
  cslen = strlen (charset);

  /*
   * Hack to pull the Re: and Fwd: out of the encoded word for better
   * handling by agents which do not support RFC1522.
   */

  if (strncasecmp ("re: ", s, 4) == 0)
  {
    strncpy (wptr, s, 4);
    wptr += 4;
    s += 5;
  }
  else if (strncasecmp ("fwd: ", s, 5) == 0)
  {
    strncpy (wptr, s, 5);
    wptr += 5;
    s += 5;
  }

  strcpy (wptr, charset);
  wptr += cslen;
  wordlen = cslen;

  while (*s)
  {
    if (wordlen >= 72)
    {
      strcpy(wptr, "?=\n ");
      wptr += 4;
      strcpy(wptr, charset);
      wptr += cslen;
      wordlen = cslen;
    }

    if (*s == ' ')
    {
      *wptr++ = '_';
      wordlen++;
    }
    else if (*s & 0x80 || (strchr(tspecials, *s) != NULL))
    {
      if (wordlen >= 70) {
	strcpy(wptr, "?=\n ");
	wptr += 4;
	strcpy(wptr, charset);
	wptr += cslen;
	wordlen = cslen;
      }
      sprintf(wptr, "=%2X", *s);
      wptr += 3;
      wordlen += 3;
    }
    else
    {
      *wptr++ = *s;
      wordlen++;
    }
    s++;
  }
  strcpy (wptr, "?=");
}

static void b_encode_string (char *d, unsigned char *s, size_t len)
{
  char charset[SHORT_STRING];
  char *wptr = d;
  int cslen;
  int wordlen;

  snprintf(charset, sizeof(charset), "=?%s?B?", Charset);
  cslen = strlen(charset);
  strcpy(wptr, charset);
  wptr += cslen;
  wordlen = cslen;

  while (*s) {

    if (wordlen >= 71) {
      strcpy(wptr, "?=\n ");
      wptr += 4;
      strcpy(wptr, charset);
      wptr += cslen;
      wordlen = cslen;
    }

    *wptr++ = Base64_chars[ (*s >> 2) & 0x3f ];
    *wptr++ = Base64_chars[ ((*s & 0x3) << 4) | ((*(s+1) >> 4) & 0xf) ];
    s++;
    if (*s) {
      *wptr++ = Base64_chars[ ((*s & 0xf) << 2) | ((*(s+1) >> 6) & 0x3) ];
      s++;
      if (*s) {
	*wptr++ = Base64_chars[ *s & 0x3f ];
	s++;
      }
      else
	*wptr++ = '=';
    }
    else {
      *wptr++ = '=';
      *wptr++ = '=';
    }

    wordlen += 4;
  }

  strcpy(wptr, "?=");
}

void rfc1522_encode_string (char *d, unsigned char *s, size_t l)
{
  int count = 0;
  int len;
  unsigned char *p = s;
  encode_t *encoder;

  /* First check to see if there are any 8-bit characters */
  while (*p) {
    if (*p & 0x80) count++;
    p++;
  }
  if (!count) {
    strfcpy(d, (char *)s, l);
    return;
  }

  if (strcasecmp("us-ascii", Charset) == 0 ||
      strncasecmp("iso-8859", Charset, 8) == 0)
    encoder = q_encode_string;
  else {
    /* figure out which encoding generates the most compact representation */
    len = strlen(s);
    if ((count * 2) + len <= (4 * len) / 3)
      encoder = q_encode_string;
    else
      encoder = b_encode_string;
  }

  (*encoder)(d, s, l);
}

void rfc1522_encode_address (ADDRESS *addr)
{
  ADDRESS *ptr = addr;
  char buffer[STRING];

  while (ptr) {
    if (ptr->personal) {
      rfc1522_encode_string (buffer, (unsigned char *)&ptr->personal, sizeof(buffer));
      safe_free((void **)&ptr->personal);
      ptr->personal = safe_strdup(buffer);
    }
    ptr = ptr->next;
  }
}

static int rfc1522_decode_word (char *d, const char *s, int len)
{
  char *p = safe_strdup(s);
  char *pp = p;
  char *pd = d;
  int enc = 0, filter = 0, count = 0, c1, c2, c3, c4;

  while ((pp = strtok (pp, "?")) != NULL) {
    count++;
    switch (count) {
    case 2:
      if (strcasecmp (pp, Charset) != 0)
      {
	if (mutt_compat_charset(Charset) && mutt_compat_charset(pp))
	{
	  /* Charset doesn't match, but it's a superset of US-ASCII */
	  filter = 1;
	}
	else
	{
	  strfcpy(d, s, len);
	  safe_free((void **)&p);
	  return(0);
	}
      }
      break;
    case 3:
      if (toupper(*pp) == 'Q')
        enc = ENCQUOTEDPRINTABLE;
      else if (toupper(*pp) == 'B')
        enc = ENCBASE64;
      else
        return (-1);
      break;
    case 4:
      if (enc == ENCQUOTEDPRINTABLE) {
        while (*pp && len > 0) {
          if (*pp == '_') {
            *pd++ = ' ';
            len--;
          }
          else if (*pp == '=') {
            *pd++ = (hexval(pp[1]) << 4) | hexval(pp[2]);
            len--;
            pp += 2;
          }
          else {
            *pd++ = *pp;
            len--;
          }
          pp++;
        }
        *pd = 0;
      }
      else if (enc == ENCBASE64) {
        while (*pp && len > 0) {
          c1 = Index_64[(int)pp[0]];
          c2 = Index_64[(int)pp[1]];
          c3 = Index_64[(int)pp[2]];
          c4 = Index_64[(int)pp[3]];   
          *pd++ = (c1 << 2) | ((c2 >> 4) & 0x3);
          if (--len == 0) break;
          *pd++ = ((c2 & 0xf) << 4) | ((c3 >> 2) & 0xf);
          if (--len == 0) break;
          *pd++ = ((c3 & 0x3) << 6) | c4;
          if (--len == 0) break;
          pp += 4;
        }
        *pd = 0;
      }
      break;
    }
    pp = 0;
  }
  safe_free ((void **)&p);
  if (filter) {
    pd = d;
    while (*pd) {
      if (!IsPrint((unsigned)*pd)) *pd = '?';
      pd++;
    }
  }
  return (0);
}

/* this routine decodes a non-structured RFC1522-encoded string */
void rfc1522_decode_string (char *d, const char *s, int len)
{
  char *q = safe_strdup (s);
  char *p = q;
  char *pp = p;
  char *pd = d;
  char savechar = 0;
  int toklen;

  while (*p && len > 0)
  {
    pp = strpbrk (p, " \t\n");
    if (pp)
    {
      savechar = *pp;
      *pp = 0;
      toklen = pp - p;
    }
    else
    {
      toklen = strlen (p);
      savechar = 0;
    }

    if (p[0] == '=' && p[1] == '?' && p[toklen-2] == '?' && p[toklen-1] == '=')
    {
      char *r;

      rfc1522_decode_word (pd, p, len);

      /*
       * Check for a continuation word.  RFC1522 states that all whitespace
       * between words should be discarded.
       */
      r = skip_whitespace (pp + 1);
      if (*r == '=' && *(r+1) == '?')
	pp = --r;
    }
    else
      strncpy (pd, p, len);

    len -= strlen (pd);
    pd += strlen (pd);

    if (savechar)
    {
      *pd++ = savechar;
      len--;
      savechar = 0;
    }

    if (pp)
      p = pp + 1;
    else
      break;
  }
  *pd = 0;
  safe_free ((void **)&q);
}

/* Gack!  Parsing RFC822 structured fields is a royal pain... */
static int rfc822_toklen (char *s)
{
  char *p = s;
  short depth;

  while (p) {
    if (*p == '(') {
      depth = 0;
      while (*p) {
        if (*p == '\\')
	  p++;
        else if (*p == '\"')
	  p += rfc822_toklen(p);
        else if (*p == '(')
	  depth++;
        else if (*p == ')')
	  depth--;
        p++;
        if (depth == 0)
	  break;
      }
      break;
    }
    else if (*p == '\"') {
      p++;
      while (*p) {
        if (*p == '\\')
	  p++;
        else if (*p == '\"')
	  break;
	p++;
      }
      break;
    }
    else if (strchr(".:@ \t", *p) != 0)
      break;
    p++;
  }
  return(p != s ? p-s : 1);
}

static int comment_toklen (char *s)
{
  char *p = strpbrk(s, " \r\n\t(");
  if (p) return (p != s ? p-s : 1);
  else return(strlen(s));
}

static int simple_toklen(char *s)
{
  char *p = strpbrk(s, " \r\n\t");
  if (p) return (p != s ? p-s : 1);
  else return(strlen(s));
}

#define IN_STRUCTURED 1
#define IN_COMMENT    2
#define IN_PHRASE     4

int rfc1522_decode_real (char *d, char *s, int len, int flag)
{
  int toklen, wordlen;
  char *ps, savechar, *pd = d;
  
  while (*s && (len > 0))
  {
    if (flag & IN_STRUCTURED)
      toklen = rfc822_toklen (s);
    else if (flag & IN_COMMENT)
      toklen = comment_toklen (s);
    else
      toklen = simple_toklen (s);
    
    ps = s + toklen;
    savechar = *ps;
    *ps = 0;
    
    if (((flag & IN_STRUCTURED) || (flag & IN_COMMENT)) && *s == '(')
    {
      *pd++ = '(';
      len--;
      s[toklen-1] = 0;
      if (rfc1522_decode_real (pd, s+1, len, IN_COMMENT) == -1)
	return (-1);
      wordlen = strlen(pd);
      pd += wordlen;
      len -= wordlen;
      *pd++ = ')';
      len--;
    }
    else if ((((flag & IN_STRUCTURED) && (flag & IN_PHRASE)) ||
        !(flag & IN_STRUCTURED)) &&
       (s[0] == '=' && s[1] == '?' && s[toklen-2] == '?' && s[toklen-1] == '='))
    {
      
      if (rfc1522_decode_word (pd, s, len) == -1)
	return (-1);
      wordlen = strlen (pd);
      pd += wordlen;
      len -= wordlen;
    }
    else
    {
      len -= toklen;
      if (len < 0)
	return (-1);
      strncpy (pd, s, toklen);
      pd += toklen;
    }

    *ps = savechar;
    s = ps;
  }
  *pd = 0;
  return (0);
}

int rfc1522_valid (const char *s)
{
  char *p = strstr (s, "=?");

  if (!p)
    return (0);
  p = strstr (p+2, "?=");
  if (!p)
    return (0);
  return (1);
}

#define HDR_STRUCTURED 1
#define HDR_PHRASE     2

void rfc1522_decode (char *s, int len, int flags)
{
  char buffer[LONG_STRING];
  int passflags = 0;

  if (!rfc1522_valid (s))
    return; /* nothing to do */
  strncpy (buffer, s, sizeof (buffer));

  if (flags & HDR_STRUCTURED)
  {
    passflags = IN_STRUCTURED;
    if (flags & HDR_PHRASE)
      passflags |= IN_PHRASE;
  }

  if (rfc1522_decode_real (s, buffer, len, passflags) == -1)
    strcpy (s, buffer);
}

/*
 * This piece of code, and the idea of using the HDR_* flags, is attributed
 * to Kari Hurtta <kari.hurtta@fmi.fi>, who helped me get this right.
 */
int rfc1522_classify (const char *s)
{
  if (strncasecmp ("from:", s, sizeof("from:")-1) == 0 ||
      strncasecmp ("to:", s, sizeof("to:")-1) == 0 ||
      strncasecmp ("cc:", s, sizeof("cc:")-1) == 0 ||
      strncasecmp ("bcc:", s, sizeof("bcc:")-1) == 0)
    return (HDR_STRUCTURED | HDR_PHRASE);
  return (0);
}

void rfc1522_decode_adrlist (ADDRESS *a)
{
  while (a)
  {
    if (a->personal && rfc1522_valid (a->personal))
    {
      rfc1522_decode (a->personal, strlen (a->personal), 0);
      safe_realloc ((void **)&a->personal, strlen (a->personal)+1);
    }
    a = a->next;
  }
}

void rfc1522_encode_adrlist (ADDRESS *p)
{
  char buffer[LONG_STRING];

  while (p)
  {
    if (p->personal)
    {
      rfc1522_encode_string (buffer, (unsigned char *)p->personal, sizeof (buffer)-1);
      safe_free ((void **)&p->personal);
      p->personal = safe_strdup (buffer);
    }
    p = p->next;
  }
}
