/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"
#include "mutt_curses.h"
#include "rfc822.h"
#include "rfc1522.h"
#include "keymap.h"

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/stat.h>

#define DISPOSITION(X) X==DISPBODY?"attachment":"inline"

#define HDR_TO 1
#define HDR_CC 2
#define HDR_BCC 3
#define HDR_SUBJECT 4
#define HDR_FCC 6
#define HDR_PGP 8
#define HDR_ATTACH 10

#define HDR_XOFFSET 6

extern char *tspecials;

#ifdef ENABLE_RANDOM_SIG
extern char *addRandomSig (char *);
#endif /* ENABLE_RANDOM_SIG */

#ifdef _PGPPATH
int pgp_send_menu (void)
{
  int c = -1;

  CLEARLINE (LINES-1);
  addstr ("(e)ncrypt, (s)ign, (b)oth or (f)orget it? ");
  while (c == -1)
  {
      refresh ();
      switch (c = ci_getch ())
      {
	case 'e':
	  c = PGPENCRYPT;
	  break;
	case 's':
	  c = PGPSIGN;
	  break;
	case 'b':
	  c = PGPENCRYPT | PGPSIGN;
	  break;
	case 'f':
	  c = 0;
	  break;
	default:
	  beep();
	  c = -1;
      }
  }
  CLEARLINE (LINES-1);
  refresh ();
  return (c);
}
#endif /* _PGPPATH */

FILE *mutt_open_read (const char *path)
{
  FILE *f;
  int len = strlen (path);

  if (path[len-1] == '|')
  {
    /* read from a pipe */

    char *s = safe_strdup (path);

    s[len-1] = 0;
    mutt_create_filter (s, NULL, &f);
    free (s);
  }
  else
    f = fopen (path, "r");
  return (f);
}

static void append_signature (ENVELOPE *env, FILE *f)
{
  char buffer[LONG_STRING];
  FILE *tmpfp;
  int len, isRemote = 0;

  if (Local_sig[0] || Remote_sig[0])
  {
    if (env->to && !is_local_site (env->to->host))
      isRemote++;
    if (env->cc && !is_local_site (env->cc->host))
      isRemote++;
    if (env->bcc && !is_local_site (env->bcc->host))
      isRemote++;

    /* if no addresses were given, default to remotesignature */
    if (isRemote || !(env->to || env->cc || env->bcc))
      strfcpy (buffer, Remote_sig, sizeof (buffer));
    else
      strfcpy (buffer, Local_sig, sizeof (buffer));
  }
  else
    strfcpy (buffer, Signature, sizeof (buffer));

  mutt_expand_path (buffer, sizeof (buffer));

  if ((tmpfp = mutt_open_read (buffer)))
  {
    if (option (OPTSIGDASHES))
      fputs("\n-- \n", f); /* trailing space is recommended! */
    while ((len = fread (buffer, 1, sizeof (buffer), tmpfp)) > 0)
      fwrite (buffer, 1, len, f);
    fclose (tmpfp);
  }
}

static int include_message (HEADER *cur, FILE *out)
{
  STATE *s;
  char buffer[STRING];
  
  s = mutt_new_state();
  s->fpin = Context->folder->fp;
  s->fpout = out;

  mutt_make_string (buffer, sizeof(buffer), Attribution, cur);
  state_puts (buffer, s);
  state_putc ('\n', s);

  mutt_make_string (buffer, sizeof (buffer), Prefix, cur);
  s->prefix = buffer;

  if (option (OPTHEADER))
  {
    fseek (s->fpin, cur->offset, 0);
    mutt_copy_header (s->fpin, cur, s->fpout, CH_DECODE|CH_PREFIX);
  }

#ifdef _PGPPATH
  if (cur->pgp)
  {
    if (cur->pgp == PGPENCRYPT)
    {
      /* make sure we have the user's passphrase before proceeding... */
      pgp_valid_passphrase ();
    }

    /* need to exit curses to see PGP output */
    endwin ();
  }
#endif /* _PGPPATH */
  mutt_body_handler (cur->content, s);
  
  if (PostIndentString[0])
  {
    mutt_make_string (buffer, sizeof (buffer), PostIndentString, cur);
    state_puts (buffer, s);
    state_putc ('\n', s);
  }

  mutt_free_state (&s);
  return 0;
}

static ADDRESS *remove_user (ADDRESS *a)
{
  ADDRESS *top = 0, *last = 0;

  while (a)
  {
    if (!mutt_addr_is_user (a))
    {
      if (top)
      {
        last->next = a;
        last = last->next;
      }
      else
        last = top = a;
      a = a->next;
      last->next = 0;
    }
    else
    {
      ADDRESS *tmp = a;
      
      a = a->next;
      tmp->next = 0;
      mutt_free_address (&tmp);
    }
  }
  return top;
}

static ADDRESS *find_mailing_lists (ADDRESS *t, ADDRESS *c)
{
  ADDRESS *top = 0, *ptr = t, *tmp;

  while (t) {
    if (mutt_is_mail_list (t))
    {
      /*
       * rfc822_cpy_adr() copies the whole list, not just one element, so
       * temporarily cut the list.
       */
      tmp = t->next;
      t->next = 0;
      if (top)
        ptr->next = rfc822_cpy_adr(t);
      else
        ptr = top = rfc822_cpy_adr(t);
      t->next = tmp;
      while (ptr->next) ptr = ptr->next;
    }
    t = t->next;
    if (!t && c)
    {
      /* now look at the CC list */
      t = c;
      c = 0;
    }
  }
  return top;
}

void mutt_include_forward (HEADER *h, FILE *f)
{
  char buffer[LONG_STRING] = { 0 };

  mutt_write_address(buffer, sizeof(buffer), h->env->from);
  fprintf(f, "-----Forwarded message from %s-----\n\n", buffer);
  fseek(Context->folder->fp, h->offset, 0);
  while (fgets(buffer, sizeof(buffer)-1, Context->folder->fp) != NULL)
  {
    if (strncmp("From ", buffer, 5) != 0 &&
	strncmp("Status:", buffer, 7) != 0 &&
	strncmp("X-Status:", buffer, 9) != 0 &&
	strncasecmp("Content-Length:", buffer, 15) != 0)
      fputs(buffer, f);
    if (buffer[0] == '\n')
      break;
  }
  /* if this message has a single TEXT/PLAIN part, try to decode it */
  if (h->content->type == TYPETEXT &&
      strcasecmp (h->content->subtype, "plain") == 0)
  {
    STATE *s = mutt_new_state ();
    
    s->fpin = Context->folder->fp;
    s->fpout = f;
    s->displaying = 0;
    mutt_body_handler (h->content, s);
    mutt_free_state (&s);
  }
  else
  {
    /* just copy the message as-is */
    mutt_copy_bytes (Context->folder->fp, f, h->content->length);
  }
  fputs ("\n-----End of forwarded message-----\n", f);
}

static void draw_envelope (ENVELOPE *e, BODY *b)
{
  char buf[SHORT_STRING];
  int offset=1;

  clear();
  SETCOLOR(MT_COLOR_STATUS);
  mvprintw(LINES-2, 0, "%-*.*s", COLS, COLS, "");
  mvaddstr(LINES-2, 0, "---Mutt: Send Mode");

  SETCOLOR(MT_COLOR_NORMAL);

  mvaddstr(HDR_TO, 0,      "  To: ");
  buf[0] = 0;
  mutt_write_address(buf, sizeof(buf), e->to);
  if (buf[0])
    printw("%-*.*s", COLS-HDR_XOFFSET, COLS-HDR_XOFFSET, buf);

  mvaddstr(HDR_CC, 0,      "  Cc: ");
  buf[0] = 0;
  mutt_write_address(buf, sizeof(buf), e->cc);
  if (buf[0])
    printw("%-*.*s", COLS-HDR_XOFFSET, COLS-HDR_XOFFSET, buf);

  mvaddstr(HDR_BCC, 0,     " Bcc: ");
  buf[0] = 0;
  mutt_write_address(buf, sizeof(buf), e->bcc);
  if (buf[0])
    printw("%-*.*s", COLS-HDR_XOFFSET, COLS-HDR_XOFFSET, buf);

  mvaddstr(HDR_SUBJECT, 0, "Subj: ");
  if (e->subject)
    printw("%-*.*s", COLS-HDR_XOFFSET, COLS-HDR_XOFFSET, e->subject);

  mvaddstr(HDR_ATTACH, 0, "===== Attachments =====");
  if (b)
  {
    while (b && (HDR_ATTACH+offset<LINES-5))
    {
      mvprintw(HDR_ATTACH+offset, 0, "%-2d %s", offset, b->filename);
      offset++;
      b = b->next;
    }
  }
  else
    mvaddstr (HDR_ATTACH+offset, 0, "(none)");
}

static int edit_address_list (ADDRESS **addr, int line)
{
  char buffer[HUGE_STRING] = ""; /* needs to be large for alias expansion */
  ADDRESS *tmpaddr = NULL;

  mutt_write_address (buffer, sizeof (buffer), *addr);

  FOREVER
  {
    if (ci_enter_string (buffer, sizeof (buffer), line, HDR_XOFFSET, M_ALIAS) != 0)
      return (-1);

    if (mutt_parse_adrlist (&tmpaddr, buffer, "@") == 0)
    {
      mutt_free_address (addr);
      *addr = mutt_expand_aliases (tmpaddr);
      /* redraw the expanded list so the user can see the result */
      buffer[0] = 0;
      mutt_write_address (buffer, sizeof (buffer), *addr);
      move (line, HDR_XOFFSET);
      clrtoeol ();
      printw ("%-*.*s", COLS-HDR_XOFFSET, COLS-HDR_XOFFSET, buffer);      
      return (0);
    }
    mutt_error (ParseError);
  }

  /* not reached */
}

static int edit_envelope (ENVELOPE *en)
{
  char buffer[LONG_STRING] = { 0 };

  draw_envelope (en, NULL);
  mutt_error ("Press TAB to list aliases.");
  if (edit_address_list (&en->to, HDR_TO) || !en->to)
    return (-1);
  if (option(OPTNEEDREDRAW))
  {
    draw_envelope (en, NULL);
    unset_option (OPTNEEDREDRAW);
  }
  if (option(OPTASKCC))
  {
    edit_address_list (&en->cc, HDR_CC);
    if (option(OPTNEEDREDRAW))
    {
      draw_envelope (en, NULL);
      unset_option (OPTNEEDREDRAW);
    }
  }
  if (option(OPTASKBCC))
  {
    edit_address_list(&en->bcc, HDR_BCC);
    if (option(OPTNEEDREDRAW))
    {
      draw_envelope(en, NULL);
      unset_option(OPTNEEDREDRAW);
    }
  }
  if (en->subject)
    strfcpy(buffer, en->subject, sizeof(buffer));
  else
    buffer[0] = 0;
  safe_free((void **)&en->subject);
  if ((ci_enter_string(buffer, sizeof(buffer), HDR_SUBJECT, HDR_XOFFSET, 0) != 0 || !buffer[0]) &&
      !mutt_yesorno("No subject, continue?", 0))
  {
    CLEARLINE(LINES-1);
    return(-1);
  }
  en->subject = safe_strdup(buffer);
  CLEARLINE(LINES-1);
  return 0;
}

static void process_user_defined_headers (ENVELOPE *env)
{
  USERHEADER *uh = User_defined_headers;
  LIST *last = NULL;
  char buf[LONG_STRING];

  while (uh)
  {
    if (strcasecmp ("to", uh->key) == 0)
      rfc822_parse_adrlist (&env->to, uh->val, "@");
    else if (strcasecmp ("cc", uh->key) == 0)
      rfc822_parse_adrlist (&env->cc, uh->val, "@");
    else if (strcasecmp ("bcc", uh->key) == 0)
      rfc822_parse_adrlist (&env->bcc, uh->val, "@");
    else if (strcasecmp ("from", uh->key) == 0)
    {
      /*
       * User has specified a default From: address.  Get rid of the
       * default address.
       */
      mutt_free_address (&env->from);
      strfcpy (buf, uh->val, sizeof(buf));
      rfc822_parse_adrlist (&env->from, buf, Fqdn);
    }
    else
    {
      snprintf(buf, sizeof(buf), "%s: %s", uh->key, uh->val);
      if (last)
      {
	last->next = mutt_new_list();
	last = last->next;
      }
      else
	last = env->userhdrs = mutt_new_list();
      last->data = safe_strdup(buf);
    }
    uh = uh->next;
  }
}

LIST *mutt_copy_list (LIST *p)
{
  LIST *t, *r=NULL, *l=NULL;

  while (p)
  {
    t = (LIST *)safe_malloc (sizeof (LIST));
    t->data = safe_strdup (p->data);
    t->next = NULL;
    if (l)
    {
      r->next = t;
      r = r->next;
    }
    else
      l = r = t;
    p = p->next;
  }
  return(l);
}

/*
 * compare two e-mail addresses and return 1 if they are equivalent
 */
static int mutt_addrcmp (ADDRESS *a, ADDRESS *b)
{
  if (!a->mailbox || !b->mailbox) return 0;
  if (strcasecmp(a->mailbox, b->mailbox)) return 0;
  if (!a->host || !b->host) return 0;
  if (strcasecmp(a->host, b->host)) return 0;
  return 1;
}

static BODY *make_forward (HEADER *hdr)
{
  BODY *body = mutt_new_body();
  char buffer[LONG_STRING];

  body->type = TYPEMESSAGE;
  body->subtype = safe_strdup("rfc822");
  body->encoding = hdr->content->encoding;
  snprintf(buffer, sizeof(buffer), "%s:%d", Context->folder->name, hdr->msgno);
  body->filename = safe_strdup(buffer);
  strcpy(buffer, "Forwarded message from ");
  mutt_write_address (buffer+23, sizeof (buffer)-23, hdr->env->from);
  body->description = safe_strdup(buffer);
  body->use_disp = 0;
  return body;
}

static int
envelope_defaults (int flags, HEADER *cur, ENVELOPE *en, BODY **at,
		   FILE *tempfp)
{
  ENVELOPE *curenv = NULL;
  ADDRESS *tmpaddr;
  LIST *tmp;
  char buffer[STRING], prompt[STRING];
  int i = 0, tag = 0;

  if (Context && flags && flags != SENDPOSTPONED)
  {
    if (!cur)
    {
      tag = 1;
      for (i = 0; i < Context->vcount; i++)
	if (Context->hdrs[Context->v2r[i]]->tagged)
	{
	  cur = Context->hdrs[Context->v2r[i]];
	  curenv = cur->env;
	  break;
	}

      if (!cur)
      {
	/*
	 * This could happen if the user tagged some messages and then did
	 * a limit such that none of the tagged message are visible.
	 */
	beep ();
	mutt_error ("No tagged messages are visible!");
	return (-1);
      }
    }
    else
      curenv = cur->env;
  }

  if (flags & SENDREPLY)
  {
    if (flags & SENDLISTREPLY)
    {
      if ((en->to = find_mailing_lists (curenv->to, curenv->cc)) == 0)
      {
	mutt_error ("Could not find any mailing lists in the message!");
	return (-1);
      }
    }
    else
    {
      if (curenv->reply_to)
      {
	if (option (OPTASKREPLYTO) &&
	    !mutt_addrcmp (curenv->from, curenv->reply_to))
	{
	  /*
	   * There are quite a few mailing lists which set the Reply-To:
	   * header field to the list address, which makes it quite impossible
	   * to send a message to only the sender of the message.  This
	   * provides a way to do that.
	   */
          buffer[0] = 0;
	  mutt_simple_address (buffer, sizeof (buffer), curenv->reply_to);
          snprintf (prompt, sizeof (prompt), "Reply to %s?", buffer);
	  if ((i = mutt_yesorno (buffer, 1)) == 1)
	    en->to = rfc822_cpy_adr (curenv->reply_to);
	  else if (i == 0)
	    en->to = rfc822_cpy_adr (curenv->from);
	  else
	    return (-1); /* abort */
	}
	else
	  en->to = rfc822_cpy_adr (curenv->reply_to);
      }
      else
	en->to = rfc822_cpy_adr (curenv->from);
    }

    if (curenv->subject)
    {
      if (strncasecmp ("re:", curenv->subject, 3) == 0)
      {
	en->subject = safe_strdup (curenv->subject);
	en->subject[0] = 'R';
	en->subject[1] = 'e';
      }
      else
      {
	en->subject = (char *)safe_malloc(strlen(curenv->subject)+5);
	sprintf (en->subject, "Re: %s", curenv->subject);
      }
    }

    if (flags & SENDGROUPREPLY)
    {
      if (tag)
      {
	ADDRESS *toRecips = NULL;

	/* clear the default TO list and get all originators */
	mutt_free_address (&en->to);

	/* add the recipients/originators from each message to the list */

	tmpaddr = NULL;
	for (i = 0; i < Context->vcount; i++)
	{
	  if (Context->hdrs[Context->v2r[i]]->tagged)
	  {
	    /* get the originators */

	    if (toRecips)
	      toRecips->next = rfc822_cpy_adr (Context->hdrs[Context->v2r[i]]->env->from);
	    else
	      toRecips = en->to = rfc822_cpy_adr (Context->hdrs[Context->v2r[i]]->env->from);

	    while (toRecips && toRecips->next)
	      toRecips = toRecips->next;

	    /* process TO and CC fields into the new CC list */

	    if (tmpaddr)
	    {
	      tmpaddr->next = rfc822_cpy_adr (Context->hdrs[Context->v2r[i]]->env->to);
	    }
	    else
	      tmpaddr = en->cc = rfc822_cpy_adr (Context->hdrs[Context->v2r[i]]->env->to);

	    while (tmpaddr && tmpaddr->next)
	      tmpaddr = tmpaddr->next;

	    if (tmpaddr)
	      tmpaddr->next = rfc822_cpy_adr (Context->hdrs[Context->v2r[i]]->env->cc);
	    else
	      tmpaddr = en->cc = rfc822_cpy_adr (Context->hdrs[Context->v2r[i]]->env->cc);

	    while (tmpaddr && tmpaddr->next)
	      tmpaddr = tmpaddr->next;
	  }
	}
      }
      else
      {
	en->cc = rfc822_cpy_adr (curenv->to);
	if ((tmpaddr = en->cc))
	{
	  while (tmpaddr->next)
	    tmpaddr = tmpaddr->next;
	  tmpaddr->next = rfc822_cpy_adr (curenv->cc);
	}
	else
	  en->cc = rfc822_cpy_adr (curenv->cc);
      }

      if (! option (OPTMETOO))
	en->cc = remove_user (en->cc);
    }

    /* add the In-Reply-To field */
    strcpy (buffer, "In-Reply-To: ");
    mutt_make_string (buffer + 13, sizeof (buffer) - 13, In_reply_to, cur);
    tmp = en->userhdrs;
    while (tmp && tmp->next)
      tmp = tmp->next;
    if (tmp)
    {
      tmp->next = mutt_new_list ();
      tmp = tmp->next;
    }
    else
      tmp = en->userhdrs = mutt_new_list ();
    tmp->data = safe_strdup (buffer);

    if (option (OPTAUTOINCLUDE))
      i = 1;
    else if ((i = mutt_yesorno ("Include message in reply?", 1)) == -1)
      return (-1);

    if (i)
    {
      if (tag)
      {
	for (i = 0; i < Context->vcount; i++)
	  if (Context->hdrs[Context->v2r[i]]->tagged)
	  {
	    mutt_parse_mime_message (Context->hdrs[Context->v2r[i]]);
	    if (include_message (Context->hdrs[Context->v2r[i]], tempfp) == -1)
	    {
	      mutt_error ("Could not include all requested messages!");
	      return (-1);
	    }
	    fputc ('\n', tempfp);
	  }
      }
      else
      {
	mutt_parse_mime_message (cur);
	include_message (cur, tempfp);
      }
    }

    en->references = mutt_copy_list (curenv->references);
    if (curenv->message_id)
      mutt_add_to_list (&en->references, curenv->message_id);
  }
  else if (flags & SENDFORWARD)
  {
    if (!tag)
    {
      if (option (OPTMIMEFWD))
	*at = make_forward (cur);
      else
	mutt_include_forward (cur, tempfp);
    }
    else
    {
      /* forward tagged messages */

      BODY *tmp, *last = *at;
      int i;

      for (i = 0; i < Context->vcount; i++)
	if (Context->hdrs[Context->v2r[i]]->tagged)
	{
	  if (!cur)
	  {
	    cur = Context->hdrs[Context->v2r[i]];
	    curenv = cur->env;
	  }
	  if (option (OPTMIMEFWD))
	  {
	    tmp = make_forward (Context->hdrs[Context->v2r[i]]);
	    if (last)
	    {
	      last->next = tmp;
	      last = tmp;
	    }
	    else
	      *at = last = tmp;
	  }
	  else
	    mutt_include_forward (Context->hdrs[Context->v2r[i]], tempfp);
	}
    }

    /* set the default subject for the message. */
    if (!curenv->subject)
      en->subject = 0;
    else
    {
      mutt_make_string (buffer, sizeof (buffer), ForwFmt, cur);
      en->subject = safe_strdup (buffer);
    }
  }

  /* the CC field can get cluttered, especially with lists */
  en->to = mutt_remove_duplicates (en->to);
  en->cc = mutt_remove_duplicates (en->cc);

  if (!option (OPTEDITHDRS) || (!option (OPTAUTOEDIT) && (flags == 0 || flags & SENDFORWARD)))
  {
    if (edit_envelope (en) == -1)
      return (-1);
  }
  
  return (0);
}

static void choose_fcc (char *s, size_t slen, ENVELOPE *env)
{
  char buf[_POSIX_PATH_MAX];

  if (!option (OPTBATCHMODE) && option (OPTASKCOPY) &&
      mutt_yesorno ("Save a copy of this message?", 0) != 1)
    return; /* nothing to do */

  if (option (OPTSAVENAME) || option (OPTFORCENAME))
  {
    /* check to see if the user has a save-hook defined for a recipient */
    *s = 0;
    if (mutt_save_hook (env->from, s, slen) != 0 &&
	mutt_save_hook (env->to, s, slen) != 0 &&
	mutt_save_hook (env->cc, s, slen) != 0)
    {
      /* no save-hook, so default from the message recipient list */

      if (env->to)
	snprintf (buf, sizeof (buf), "%s", env->to->mailbox);
      else if (env->cc)
	snprintf (buf, sizeof (buf), "%s", env->cc->mailbox);
      else
	snprintf (buf, sizeof (buf), "%s", env->bcc->mailbox);
      mutt_strlower (buf);
      snprintf (s, slen, "%s/%s", Maildir, buf);
      if (!option (OPTFORCENAME))
      {
	/* check to see if the file already exists */
	mutt_expand_path (s, slen);
	if (access (s, W_OK) == -1)
	  strfcpy (s, Outbox, slen); /* default to the $record mailbox */
      }
    }
  }
  else
    strfcpy (s, Outbox, slen);
  mutt_expand_path (s, slen);
}

void
ci_send_message (int flags, ENVELOPE *env, BODY *attach, char *inc, HEADER *cur)
{
  char buffer[LONG_STRING], tempfile[_POSIX_PATH_MAX];
  char fcc[_POSIX_PATH_MAX] = ""; /* where to copy this message */
  int i, redraw = 1, done = 0, postponed=0;
  BODY *new;
  FILE *tempfp;
#ifdef _PGPPATH
  int pgpaction = 0;
  char *pgpkeylist = 0;

  if (option (OPTPGPAUTOSIGN))
    pgpaction = PGPSIGN;
#endif

  mutt_mktemp (tempfile);

  if ((tempfp = fopen (tempfile, "w")) == NULL)
  {
    dprint(1,(debugfile, "newsend_message: can't create tempfile %s (errno=%d)\n", tempfile, errno));
    mutt_error ("Can't open temporary file!");
   return;
  }

  /* If the user specified a file to include on the command line, do so now */
  if (inc)
  {
    FILE *f = fopen (inc, "r");
    
    if (f)
    {
      while ((i = fread (buffer, 1, sizeof(buffer), f)) > 0)
	fwrite (buffer, 1, i, tempfp);
      fclose (f);
    }
  }

  if (env)
  {
    env->to = mutt_expand_aliases (env->to);
    env->cc = mutt_expand_aliases (env->cc);
#ifndef DONT_ADD_FROM
    env->from = mutt_default_from ();
#endif /* !DONT_ADD_FROM */
  }
  else
  {
    if (flags == SENDPOSTPONED)
    {
      if ((flags = ci_get_postponed_message (tempfp, &env, &cur, &attach)) < 0)
	goto cleanup;
      postponed = 1;
    }
    else if (!flags && option (OPTASKRECALL))
    {
      /*
       * If the user is composing a new message, check to see if there
       * are any postponed messages first.
       */
      struct stat st;

      strfcpy (buffer, Postponed, sizeof (buffer));
      mutt_expand_path (buffer, sizeof (buffer));
      if (stat (buffer, &st) == 0)
      {
	if (S_ISREG(st.st_mode))
	{
	  if (st.st_size > 0 && mutt_yesorno ("Recall postponed message?", 0))
	  {
	    if ((flags = ci_get_postponed_message (tempfp, &env, &cur, &attach)) < 0)
	      flags=0;
	    else
	      postponed = 1;
	  }
	}
	else
	  mutt_error ("Warning!  Mailbox specified by $postponed is not a regular file.");
      }
    }
    if (!env)
    {
      env = mutt_new_envelope ();
#ifndef DONT_ADD_FROM
      env->from = mutt_default_from ();
#endif /* !DONT_ADD_FROM */
    }
  }
  process_user_defined_headers (env);

  if (option (OPTBATCHMODE))
  {
    while ((i = fread (buffer, 1, sizeof (buffer), stdin)) > 0)
      fwrite (buffer, 1, i, tempfp);
  }
  else if (!postponed)
  {
    if (envelope_defaults (flags, cur, env, &attach, tempfp) == -1)
      goto cleanup;
  }

  if (!postponed)
    append_signature (env, tempfp);

  fclose (tempfp);

  if (!option (OPTBATCHMODE))
  {
    struct stat st;
    time_t mtime;

    stat(tempfile, &st);
    mtime = st.st_mtime;
    if (option(OPTEDITHDRS))
      ci_edit_headers(tempfile, &env);
    else
      ci_edit_file(tempfile);

    if (!postponed && (flags != SENDFORWARD))
    {
      stat (tempfile, &st);
      /* if the file was not modified, bail out now */
      if (mtime == st.st_mtime)
      {
	mutt_error ("Aborted unmodified message.");
	goto cleanup;
      }
    }
  }
  else
    done = 1; /* don't go into the loop. */

  if (! option (OPTBATCHMODE))
    ci_clear_error ();

main_loop:

  while (!done)
  {
    if (redraw)
    {
      draw_envelope (env, attach);

      mvprintw(HDR_FCC, 0, "Fcc:     %-*.*s", COLS-HDR_XOFFSET, COLS-HDR_XOFFSET, fcc);
#ifdef _PGPPATH
      CLEARLINE(HDR_PGP);
      addstr("This message will ");
      switch (pgpaction)
      {
	case PGPENCRYPT:
	  addstr("be PGP encrypted.");
	  break;
	case PGPSIGN:
	  addstr("be PGP signed.");
	  break;
	case PGPENCRYPT|PGPSIGN:
	  addstr("be PGP signed and encrypted.");
	  break;
	default:
	  addstr("NOT be PGP signed OR encrypted!");
	  break;
      }
#endif /* _PGPPATH */
      ci_show_last_error ();
      redraw = 0;
    }

    CLEARLINE (LINES-1);
    addstr ("Send message? ([y]/n/?): ");
    refresh ();

    i = dokey (MENU_SEND);

    ci_clear_error ();

    switch (i)
    {
      case OP_SEND_EDIT_TO:
	edit_address_list(&env->to, HDR_TO);
	if (option(OPTNEEDREDRAW))
	{
	  unset_option(OPTNEEDREDRAW);
	  redraw = 1;
	}
	break;

      case OP_SEND_EDIT_BCC:
	edit_address_list(&env->bcc, HDR_BCC);
	if (option(OPTNEEDREDRAW))
	{
	  unset_option(OPTNEEDREDRAW);
	  redraw = 1;
	}
	break;

      case OP_SEND_EDIT_CC:
	edit_address_list(&env->cc, HDR_CC);
	if (option(OPTNEEDREDRAW))
	{
	  unset_option(OPTNEEDREDRAW);
	  redraw = 1;
	}
	break;

      case OP_SEND_EDIT_SUBJECT:
	if (env->subject)
	  strfcpy(buffer, env->subject, sizeof(buffer));
	else
	  buffer[0] = 0;
	if (ci_enter_string(buffer, sizeof(buffer), HDR_SUBJECT, HDR_XOFFSET, 0) == 0) {
	  safe_free((void **)&env->subject);
	  env->subject = safe_strdup(buffer);
	}
	break;

      case OP_SEND_EDIT_FCC:
	buffer[0] = 0;
	strcpy(buffer, fcc);
	if (ci_enter_string(buffer, sizeof(buffer), HDR_FCC, HDR_XOFFSET, M_FILE) == 0) {
	  strfcpy(fcc, buffer, sizeof(fcc));
	  mutt_expand_path (fcc, sizeof (fcc));
	  move(HDR_FCC, HDR_XOFFSET);
	  clrtoeol();
	  printw("%-*.*s", COLS-HDR_XOFFSET, COLS-HDR_XOFFSET, fcc);
	}
	break;

      case OP_SEND_EDIT_MESSAGE:
	if (option (OPTEDITHDRS))
	{
	  ci_edit_headers(tempfile, &env);
	  redraw = 1; /* headers may have changed! */
	}
	else
	  ci_edit_file(tempfile);
	break;

      case OP_SEND_EDIT_HEADERS:
	ci_edit_headers(tempfile, &env);
	redraw = 1;
	break;

      case OP_SEND_ATTACH_FILE:
	attach = ci_attach_menu(attach);
	redraw = 1;
	break;

      case OP_SEND_SEND_MESSAGE:
	done = 1;
	break;

      case OP_SEND_ABORT_MESSAGE:
	if (!option(OPTASKPOSTPONE) || !mutt_yesorno("Postpone this message? ", FALSE))
	{
	  mutt_error("Mail not sent.");
	  goto cleanup;
	}

	/* fall through to postpone! */

      case OP_SEND_POSTPONE_MESSAGE:

	mutt_postpone_message (tempfile, env, (cur && (flags & SENDREPLY)) ? cur->env->message_id : NULL, attach);
	mutt_error ("Message postponed.");
	goto cleanup;

      case OP_REDRAW:

	clearok (stdscr, TRUE);
	redraw = 1;
	break;

      case OP_HELP:

	ci_help (MENU_SEND);
	redraw = 1;
	break;

      case OP_ENTER_COMMAND:

	if (ci_get_field (": ", buffer, sizeof (buffer), 0) == 0 && buffer[0])
	{
	  char error[SHORT_STRING] = "";

	  mutt_parse_rc_line (buffer, 0, error, sizeof (error));
	  if (error[0])
	    mutt_error ("%s", error);
	}
	break;

#ifdef ISPELL
      case OP_SEND_ISPELL:
	endwin ();
	snprintf (buffer, sizeof (buffer), "%s -x %s", ISPELL, tempfile);
	mutt_system (buffer);
	break;
#endif /* ISPELL */

#ifdef _PGPPATH
      case OP_SEND_PGP_MENU:
	pgpaction = pgp_send_menu ();
	redraw = 1;
	break;
#endif /* _PGPPATH */

      default:
	beep();
    }
  }

  if (!option (OPTBATCHMODE))
  {
    if (!env->to && !env->cc && !env->bcc)
    {
      mutt_error ("No recipients are specified!");
      done = 0;
      goto main_loop;
    }
    if (!env->subject && !mutt_yesorno ("Empty subject, send message anyway?", 0))
    {
      done = 0;
      goto main_loop;
    }
#ifdef _PGPPATH
    /*
     * Do a quick check to make sure that we can find all of the encryption
     * keys if the user has requested this service.
     */
    if (pgpaction & PGPENCRYPT)
    {
      if ((pgpkeylist = pgp_findKeys (env->to, env->cc, env->bcc)) == NULL)
      {
	done = 0;
	goto main_loop;      
      }
    }

    if (pgpaction & PGPSIGN)
    {
      /* make sure we have the user's passphrase before we go any furthur */
      while (!pgp_valid_passphrase ())
      {
	if (!mutt_yesorno ("Do you still want to sign this message?", 1))
	{
	  pgpaction &= ~PGPSIGN;
	  break;
	}
      }
    }
#endif
  }

  /* find where to save a copy of this message */
  if (!fcc[0]) choose_fcc (fcc, sizeof(fcc), env);

  /* Create a record for the main body of this message. */
  new = mutt_make_attach(tempfile);
  new->use_disp = 0; /* do not print a Content-Disposition field */
  new->unlink = 1; /* ok to remove this file after sending */

  /* now add any attachments. */
  if (attach)
  {
    new->next = attach;
    attach = mutt_make_multipart (new);
  }
  else
    attach = new;

#ifdef _PGPPATH
  if (pgpaction)
  {
    endwin();
    if (pgpaction & PGPENCRYPT)
    {
      attach = pgp_encryptMessage (attach, pgpkeylist, pgpaction & PGPSIGN);
      safe_free ((void **)&pgpkeylist);
    }
    else if (pgpaction == PGPSIGN)
      attach = pgp_signMessage (attach);
  }
#endif

  mutt_send_message (env, attach, fcc);

  /* now free up the memory used to generate this message. */
  mutt_free_body (&attach);
  mutt_free_envelope (&env);

  if (!option (OPTBATCHMODE) && (flags & SENDREPLY))
  {
    if (!postponed && Context->tagged)
    {
      for (i = 0; i < Context->msgcount; i++)
	if (Context->hdrs[i]->tagged && !Context->hdrs[i]->replied)
	{
	  Context->hdrs[i]->replied = 1;
	  Context->hdrs[i]->changed = 1;
	}
    }
    else if (cur && !cur->replied)
    {
      cur->replied = 1;
      cur->changed = 1;
    }
  }
  return; /* all done */

cleanup:

  if (tempfp) fclose (tempfp);
  mutt_unlink (tempfile);
  mutt_free_body (&attach);
  mutt_free_envelope (&env);
}
