/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"
#include "mutt_curses.h"
#include <signal.h>
#include <string.h>

static short IsEndwin = 0;
static sigset_t Signals;
static short SignalsBlocked = 0;

#ifdef USE_SLANG_CURSES
short Sigwinch = 0;
#endif

/* Attempt to catch "ordinary" signals and shut down gracefully. */
RETSIGTYPE ci_sig_exit (int sig)
{
  ci_endwin(NULL); /* just to be safe */
#if SYS_SIGLIST_DECLARED
  printf("Caught %s...  Exiting.\n", sys_siglist[sig]);
#else
#if (__sun__ && __svr4__)
  printf("Caught %s...  Exiting.\n", _sys_siglist[sig]);
#else
  printf("Caught signal %d...  Exiting.\n", sig);
#endif
#endif
  exit(0);
}

RETSIGTYPE sighandler (int sig)
{
  switch (sig) {
  case SIGTSTP: /* user requested a suspend */
    IsEndwin = isendwin();
    if (!IsEndwin) endwin();
    kill (0, SIGSTOP);
  case SIGCONT:
    if (!IsEndwin) refresh();
    break;
#ifdef USE_SLANG_CURSES
  case SIGWINCH:
    Sigwinch = 1;
    break;
#endif
  default:
    break;
  }
}

#ifdef USE_SLANG_CURSES
int mutt_intr_hook (void)
{
  return (-1);
}
#endif

void ci_signal_init (void)
{
  struct sigaction act;

  memset (&act, 0, sizeof(struct sigaction));

  act.sa_handler = SIG_IGN;
  sigaction (SIGPIPE, &act, 0);

  act.sa_handler = ci_sig_exit;
  sigaction (SIGTERM, &act, 0);
  sigaction (SIGHUP, &act, 0);
  sigaction (SIGINT, &act, 0);

  act.sa_handler = sighandler;
#ifdef SA_INTERRUPT
  /* POSIX.1 uses SA_RESTART, but SunOS 4.x uses this instead */
  act.sa_flags = SA_INTERRUPT;
#endif
  sigaction (SIGCONT, &act, 0);
#ifdef USE_SLANG_CURSES
  sigaction (SIGWINCH, &act, 0);
#endif

  /*
   * SIGTSTP is the one signal in which we want to restart a system call if it
   * was interrupted in progress.  This is especially important if we are in
   * the middle of a system() call, like if the user is editing a message.
   * Otherwise, the system() will exit when SIGCONT is received and Mutt will
   * resume even though the subprocess may not be finished.
   */
#ifdef SA_RESTART
  act.sa_flags = SA_RESTART;
#else
  act.sa_flags = 0;
#endif
  sigaction (SIGTSTP, &act, 0);

#ifdef USE_SLANG_CURSES
  /*
   * This bit of code is required because of the implementation of
   * SLcurses_wgetch().  If a signal is received (like SIGWINCH) when we
   * are in blocking mode, SLsys_getkey() will not return an error unless
   * a handler function is defined and it returns -1.  This is needed so
   * that if the user resizes the screen while at a prompt, it will just
   * abort and go back to the main-menu.
   */
  SLang_getkey_intr_hook = mutt_intr_hook;
#endif
}

/* signals which are important to block while doing critical ops */
void mutt_block_signals (void)
{
  if (!SignalsBlocked) {
    sigemptyset(&Signals);
    sigaddset(&Signals, SIGINT);
    sigaddset(&Signals, SIGWINCH);
    sigaddset(&Signals, SIGHUP);
    sigaddset(&Signals, SIGTERM);
    sigaddset(&Signals, SIGTSTP);
    sigprocmask(SIG_BLOCK, &Signals, 0);
    SignalsBlocked = 1;
  }
}

/* restore the previous signal mask */
void mutt_unblock_signals (void)
{
  if (SignalsBlocked) {
    sigprocmask(SIG_UNBLOCK, &Signals, 0);
    SignalsBlocked = 0;
  }
}
