#!/bin/sh

# James Galvin <galvin@tis.com>					4/94
# 
# This script expects to be invoked by a MIME user agent with arguments
# corresponding to data to be digitally signed with PEM.  A common
# configuration is for a MIME user agent to collect the data to be signed in
# a file and invoke this script with that file.
# 
# When using the Rand MH Version 6.8.3, the following addition to the user's
# .mh_profile file is recommended:
# 
# 	mhn-compose-application/pem-signature: pemsign.sh %s %f %a

# Exactly 3 arguments are required, in the order indicated, as follows:
#
# 	type		-- the MIME content type of the data to be signed;
#			   this information is currently unused
#			   text content types must be canonicalized
# 	subtype		-- the MIME subtype of the data to be signed;
#			   this information is currently unused
# 	data		-- the file with the data to be signed; this data is
#			   assumed to be 7bit ASCII
# 
# These 3 arguments may be preceded, followed, or separated by the
# following optional arguments:
# 
# 	orig=ALIAS	-- indicate digital signature originator of
#			   message; only one may be specified
# 	protocol=application/pem-signature
# 			-- indicates the PEM protocol will be used
# 	micalg=HASH	-- the hash algorithm to use to generate the hash
#			   value that is signed to create the digital
#			   signature
# 
# If there are multiple occurances of any optional argument the last
# occurance will override all other occurances.


status=99
trap 'rm -f /tmp/s$$; exit $status' 0 1 2 3 9 15

umask 077


# a helpful message is printed and then we exit; exit code argument required

usage() {
	echo Usage: $0: type subtype file micalg=hash \[ orig=alias \] 1>&2
	status=$1
	exit
}


# parses 3 of the required arguments, micalg has already been handled

doargs() {
	if [ $# -ne 3 ]; then
		usage 6
	fi

	TYPE=$1
	STYP=$2
	DATA=$3

	if [ ! -f $DATA ]; then
		echo $0: $DATA: not a file 1>&2
		usage 3
	fi

	if [ ! -r $DATA ]; then
		echo $0: $DATA: unable to open for reading 1>&2
		usage 4
	fi
}


ARGS=
ORIG=
HASH=

if [ $# -lt 4 ]; then
	usage 2
fi

# parse out the optional arguments

while [ $# -ge 1 ]; do
	case $1 in
	orig=*)
		ORIG="sig-alias `echo $1 | sed -n 's/orig=\(.*\)/\1/p'`"
		;;

	protocol=application/pem-signature)
		;;
	micalg=md5)
		HASH="mic-alg `echo $1 | sed -n 's/micalg=\(.*\)/\1/p'`"
		;;	

	micalg=*)
		echo $0: unknown option \"$1\" 1>&2
		usage 1 ;;
	protocol=*)
		echo $0: unknown option \"$1\" 1>&2
		usage 5 ;;

	*)
		ARGS="$ARGS $1" ;;
	esac
	shift
done

# only the mandatory arguments should be left

doargs $ARGS

# we create a temporary file so that we can give sign stdin/stdout so it can
# prompt for any password that it might need

cat $DATA | canon > /tmp/s$$

if [ -r /dev/tty -a -w /dev/tty ]; then
	sign $ORIG $HASH data-in /tmp/s$$ 1>/dev/tty
	status=$?
else
	cat /tmp/s$$ | sign $ORIG $HASH
fi

if [ $status -ne 0 ]; then
	exit
fi

cat $DATA

exit
