;
;  configuration options
;
(defvar pem-default-integrity t)           ; Integrity Enhancements on
(defvar pem-default-confidentiality nil)   ; Privacy Ehnancements off
(defvar pem-auto-incorporate t)            ; Asynchronous rescan of inbox
                                             ;  after rcc

;
; Global variables
;
(defvar pem-rcc-execution-buffer "Rcc Command Window")
(defvar pem-password-shell-map nil)

;
; Missing Elisp functions
;
;
; It is hard to believe that elisp doesn't have the member function
;
(defun member (item list)
  (cond
   ((not (listp list)) nil)
   ((null list) nil)
   ((eq item (car list)) list)
   (t (member item (cdr list)))))

;
;  The following two functions take the values of the variables 
;  representing pem options and turn them into command line strings
;  or mode line strings.
;

(defun pem-shell-command-options ()
   (cond 
     (pem-confidentiality "-privacy")
     (pem-integrity "-integrity")
     (t "")))

(defun pem-mode-line ()
  (cond
    (pem-confidentiality "mh-e letter (private) ")
    (pem-integrity "mh-e letter (integrity)")
    (t "mh-e letter")))
    
;
; This function is hooked into the pem-letter-mode to add the pem 
; enhancements.  All it does at present is add some local variables and
; touch up the mode line.
;
(defun pem-letter-mode-hook ()
  (make-local-variable 'pem-integrity)
  (make-local-variable 'pem-confidentiality)
  (setq pem-integrity pem-default-integrity)
  (setq pem-confidentiality pem-default-confidentiality)
  (mh-set-mode-name (pem-mode-line)))

;
;  Now I need a series of functions to implement a secret password.
;  The entry point to these functions is mh-exec-cmd-password.  The
;  technique is stolen from the terminal emulation (M-x terminal).
;

;;
;; Execute MH command COMMAND with DISPLAY flag and ARGS putting the output
;; into buffer after point.  Set mark after inserted text.
;; This function is different than those in mh-e.el in that it uses the 
;; pem-password-shell-map.  This map ensures that keystrokes destined for
;; the process do not get echoed (unless the process explicitly echos them.
;; I have also modified the mode line to reflect this status.  The buffer
;; is marked read-only so that commands like ^Y won't confuse the user.
;;
(defun mh-exec-cmd-password (command &rest args)
  (apply 'start-process
         "Mail Delivery Process"
          (current-buffer) (expand-file-name command mh-progs)
         (mh-list-to-string args))
  (cond ((not buffer-read-only) (toggle-read-only)))
  (auto-fill-mode 0)
  (use-local-map pem-password-shell-map)
  (setq mode-line-process '(": %s"))
  (mh-set-mode-name "Password Mode")
  (make-local-variable 'pem-process)
  (setq pem-process (get-buffer-process (current-buffer))))

;
;  For all keystrokes that would ordinarily be inserted into a buffer,
;  the pem-password-shell-map indicates that they should invoke 
;  pem-pass-through.  The function pem-pass-through takes the keystroke
;  and sends it to the process but does not echo it.  I have also made
;  the enter and linefeed keys point to pem-enter that tells the process to
;  execute the previous keystrokes.  Finally for those with control characters 
;  in their passwords, I have provided the ability to quote the characters.
;
;
(if pem-password-shell-map
    nil
  (let ((counter 0)
        (map (copy-keymap (current-global-map))))
    (define-key map "\C-q" 'pem-quoted-insert)
    (define-key map "\C-m" 'pem-enter)
    (define-key map "\C-j" 'pem-enter)
    (while (< counter 256)
       (cond
         ((eq (lookup-key map (char-to-string counter)) 
              'self-insert-command)
          (define-key map (char-to-string counter) 'pem-pass-through)))
       (setq counter (+ counter 1)))
    (setq pem-password-shell-map map)))

;
; This function sends the previous keystroke to the process but
; does not echo it.
;
(defun pem-pass-through ()
  "Send the last character typed through the terminal-emulator
without any interpretation"
  (interactive)
  (send-string pem-process (make-string 1 last-input-char)))

;
; This function sends a linefeed to the process.  Due to the way tty's 
; work, the process interprets this as an execute command.
;
(defun pem-enter ()
  "Send the line feed character to the process to emulate sending a line"
  (interactive)
  (end-of-buffer)
  (send-string pem-process "\C-j"))

;
; Finally a function for those maniacs who have control characters in their
; passwords.
;
(defun pem-quoted-insert ()
  (interactive)
  (send-string pem-process (make-string 1 (read-char))))

;
;    Process sentinels:  This code almost certainly needs some fixing up.
;                        It is not even clear that they are worth their 
;                        convenience.
;

(defun pem-send-letter-sentinel (process string)
  (cond ((not (eq (process-status process) 'run))
	 (beep)
	 (message "MH Send Letter Process completed"))))


;
;  If so configured by the user, this function will cause a rescan
;  of the +inbox right after the rcc command completes.  This is 
;  asynchronous, so it can be disturbing if the rcc command takes a 
;  long time.
;
(defun pem-rcc-sentinel (process string)
  (cond ((not (eq (process-status process) 'run))
	 (beep)
	 (cond (pem-auto-incorporate (mh-scan-folder "+inbox" "all")))
	 (message "MH Rcc Process completed"))))



;
; The following functions are the interactive emacs functions
;


;
; pem-toggle-integrity toggles the integrity option
;

(defun pem-toggle-integrity ()
  "Toggle the integrity enhancements"
  (interactive)
  (setq pem-integrity (not pem-integrity))
  (cond 
    (pem-confidentiality
     (message
        (cond 
          (pem-integrity
           "Integrity Enhancements On (but redundant due to privacy option)")
          (t
           "Integrity Option Off (but overridden by privacy option)")))))
  (mh-set-mode-name (pem-mode-line)))

(defun pem-toggle-confidentiality ()
  "Toggle the privacy enhancments"
  (interactive)
  (setq pem-confidentiality (not pem-confidentiality))
  (mh-set-mode-name (pem-mode-line)))


;
;  This function is a variation of mh-send-letter.  If integrity and 
;  confidentiality are not set then it just calls mh-send-letter.  
;  Otherwise I have modified the arg case of mh-send-letter to account
;  for the integrity and confidentility options and to use 
;  mh-exec-cmd-password.
;

(defun pem-mh-send-letter (&optional arg)
  "Send the draft letter in the current buffer.
If optional prefix argument is provided, monitor delivery.
C-ci and C-cp toggle the integrity and privacy enhancements
Run mh-before-send-letter-hook before doing anything."
  (interactive "P")
  (let
      ((file-name (buffer-file-name))
       (draft-buffer (current-buffer))
       (pem-options (pem-shell-command-options)))
    (cond
     ((or pem-confidentiality pem-integrity)
      (set-buffer-modified-p t)         ; Make sure buffer is written
      (save-buffer)
      (kill-buffer draft-buffer)
      (pop-to-buffer "MH mail delivery")
      (if mh-send-args
          (mh-exec-cmd-password "send" "-watch" "-nopush"
                                pem-options
                                "-nodraftfolder" mh-send-args file-name)
        (mh-exec-cmd-password "send" "-watch" "-nopush"
			      pem-options
                              "-nodraftfolder" file-name))
      (set-process-sentinel (get-buffer-process (current-buffer))
                            'pem-send-letter-sentinel))
     (t (mh-send-letter arg)))))

;
;  This function is an interactive emacs function that implements the
;  pem rcc command.
;

(defun mh-rcc ()
  "Calls rcc on the current message"
  (interactive)
  (let ((msgnum (mh-get-msg-num t)))
    (pop-to-buffer pem-rcc-execution-buffer)
    (mh-exec-cmd-password (concat mh-progs "rcc") msgnum)
    (set-process-sentinel (get-buffer-process (current-buffer)) 
                          'pem-rcc-sentinel)))


;--------------------------------------------------------------------;
; Variations of functions in mh-e.el                                 ;
;   The following functions are rewrites of functions appearing in   ;
;   mh-e.el.  They will need to be changed if mh-e.el changes.       ;
;--------------------------------------------------------------------;


;
;  CHANGES
;    Append the pem-letter-mode-hook to the letter-mode hooks
;
(cond 
 ((not (member 'pem-letter-mode-hook mh-letter-mode-hook))
  (setq mh-letter-mode-hook
        (append mh-letter-mode-hook (list 'pem-letter-mode-hook)))))

;
;  CHANGES
;    "\C-c\C-c" used to point to mh-send-letter.  pem-mh-send-letter
;    adds the capability for privacy enhanced mail.
;

(define-key mh-letter-mode-map "\C-c\C-c" 'pem-mh-send-letter)
;
;  CHANGES
;    the following keys used to be undefined.
;
(define-key mh-letter-mode-map "\C-ci" 'pem-toggle-integrity)
(define-key mh-letter-mode-map "\C-cp" 'pem-toggle-confidentiality)
(define-key mh-folder-mode-map "R" 'mh-rcc)
