/*********************************************************************
**
**     File name:               ssh_prepare.c
**
**                              Copyright 1997 Tadayoshi Kohno.
**				All rights reserved.
**                              See the LICENSE file.
**
**     Purpose:                 prepare the ssh connection (pty, exec shell)
**
**     Author/Date:             Tadayoshi Kohno, 21 February 1998
**
**     Notes:
**	These functions are to be called after authentication
**	but before switching to an interactive xxx process (via the request
**	for a shell).
**
**     Functions:
**	ssh_request_pty		request a pty
**	ssh_request_exec_shell	request a shell
**
*********************************************************************/

#ifndef lint
static char *RCSid="$Header: /home/cia/kohno/libssh/libssh/RCS/ssh_prepare.c,v 3.0 1998/02/24 00:10:44 kohno Exp $";
#endif

#include "ssh.h"
#include "ssh_types.h"

#include "ssh_prepare.h"

/******************************************************************************
**
**     File/Function name:	ssh_request_pty
**
**     Purpose:			request a pty
**
**     Preconditions:		connected, authenticated xxx
**
**     Parameters:		sockfd		socket to server
**				ssh_info	infor about current connection
**
**				terminal_type	terminal type ("vt100", ...)
**				height		terminal height in rows
**				width		terminal width in columns
**				x_pixels	width in x_pixels
**				y_pixels	height in y_pixels
**
**				tty_modes	tty modes in binary format
**				tty_mode_len	bytes for tty_modes
**
**     Exit (post) conditions:	S_GOOD
**
**     Error conditions:	S_BAD		error in request
**				ssh_errno set
**
**     Side effects:		pty requested and granted from server
**
**     Author/Date:		Tadayoshi Kohno, 21 February 1998
**
**     Notes:
**
******************************************************************************/

int ssh_request_pty
(
	socket_type sockfd,		/* socket to talk over */
	struct ssh_struct * ssh_info,	/* information about connection */

	char * terminal_type,		/* terminal type */
	uint32_t height,		/* height of terminal */
	uint32_t width,			/* width of terminal */
	uint32_t x_pixels,		/* width in pixels */
	uint32_t y_pixels,		/* height in pixels */

	uint8_t * tty_modes,		/* tty mode encoded in binary */
	int tty_mode_len		/* length of *tty_modes */
)
{
	uint8_t data[SSH_MAX_PACKET];	/* data for packet */
	uint32_t data_len;		/* length of data */

	int err;			/* error code */


	/*
	**	pack a pty request
	*/
	if (ssh_cmsg_request_pty_encode(data, &data_len, terminal_type,
		height, width, x_pixels, y_pixels, tty_modes, tty_mode_len))
	{
		return(S_BAD);
	}

	/*
	**	send the request off
	*/
	if (ssh_send(sockfd, ssh_info, data, data_len, SSH_CMSG_REQUEST_PTY))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"error sending request for pty", "ssh_request_pty");

		/* propagate ssh_errno */
		return(S_BAD);
	}

	/*
	**	try to recieve a reply
	*/
	if ((err = ssh_recv(sockfd, ssh_info, data, &data_len))
		!= SSH_SMSG_SUCCESS)
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"error receiving reply to pty request",
			"ssh_request_pty");

		if (err != SSH_MSG_NOTYPE)
		{
			ssh_errno_set(SSH_ERRNO_REQUEST_PTY);
		}

		fprintf(stderr, "error code: %d\n", err);

		/* propagate ssh_errno */
		return(S_BAD);
	}

	ssh_debugger_new(&(ssh_info->debug_info),
		"request for pty granted", "ssh_request_pty");

	return(S_GOOD);
}

/******************************************************************************
**
**     File/Function name:	ssh_request_exec_shell
**
**     Purpose:			request a shell from the server
**
**     Preconditions:		user authenticated
**
**     Parameters:		sockfd		socket to server
**				ssh_info	info about current connection
**
**     Exit (post) conditions:	S_GOOD
**
**     Error conditions:	S_BAD		error forming request
**						or sending request
**				ssh_errno set
**
**     Side effects:		request for shell sent to server.
**				client should now enter interactive mode xxx.
**
**     Author/Date:		Tadayoshi Kohno, 21 February 1998
**
**     Notes:
**
******************************************************************************/

int ssh_request_exec_shell
(
	socket_type sockfd,		/* socket to server */
	struct ssh_struct * ssh_info	/* info about current connection */
)
{
	uint8_t data[SSH_MAX_PACKET];	/* data for packet */
	uint32_t data_len;		/* length of data */

	/*
	**	compose request for a shell
	*/
	if (ssh_cmsg_exec_shell_encode(data, &data_len))
	{
		return(S_BAD);
	}
	
	/*
	**	send off request for shell
	*/
	if (ssh_send(sockfd, ssh_info, data, data_len,
		SSH_CMSG_EXEC_SHELL))
	{
		ssh_debugger_new(&(ssh_info->debug_info),
			"error sending request for a shell",
			"ssh_request_exec_shell");

		return(S_BAD);
	}

	/*
	**	request is off and now we can enter an interactive session
	*/
	ssh_debugger_new(&(ssh_info->debug_info),
		"shell requested", "ssh_request_exec_shell");

	return(S_GOOD);
}

