/*
 * S.F.S. v. 0.5alpha-linux
 *
 * nomefile : short description.
 *
 * Authors:	Giuseppe Cattaneo, <cattaneo@udsab.dia.unisa.it>
 *		Giuseppe Persiano, <giuper@udsab.dia.unisa.it>
 *		Andrea Cozzolino, <andcoz@mikonos.dia.unisa.it>
 *		Angelo Celentano, <angcel@mikonos.dia.unisa.it>
 *		Aniello Del Sorbo, <anidel@mikonos.dia.unisa.it>
 *		Ermelindo Mauriello, <ermmau@mikonos.dia.unisa.it>
 *		Raffaele Pisapia, <rafpis@mikonos.dia.unisa.it>
 *
 * Permission to use, copy, and modify this software without fee
 * is hereby granted, provided that this entire notice is included in
 * all copies of any software which is or includes a copy or
 * modification of this software and in all copies of the supporting
 * documentation for such software.
 *
 * This software maybe be used for any purpose provided
 * the above copyright notice is retained.  It is supplied
 * as is, with no warranty expressed or implied.
 *
 * This code derives from: 
 *		 The LINUX User-Space NFS Server 1.5
 *	    by Mark Shand, Donald J. Becker, Rick Sladkey,
 *		 Orest Zborowski, Fred N. van Kempen *
 */







#include <string.h>
#include <rpc/rpc.h>
#include "sundries.h"
#include "mount.h"
#include "tcfs_prot.h"
#include <linux/fs.h>

#define MS_SYNC MS_SYNCHRONOUS

/* True for fake mount (-f).  */
int fake = 0;

/* Don't write a entry in /etc/mtab (-n).  */
int nomtab = 0;

/* True for readonly (-r).  */
int readonly = 0;

/* Nonzero for chatty (-v).  */
int verbose = 0;

/* True for read/write (-w).  */
int readwrite = 0;

/* True if ruid != euid.  */
int suid = 0;

/* Map from -o and fstab option strings to the flag argument to mount(2).  */
struct opt_map
{
  const char *opt;		/* option name */
  int  inv;			/* true if flag value should be inverted */
  int  mask;			/* flag mask value */
};

/* Custom mount options for our own purposes.  */
#define MS_NOAUTO	0x80000000
#define MS_USER		0x40000000

/* Options that we keep the mount system call from seeing.  */
#define MS_NOSYS	(MS_NOAUTO|MS_USER)

/* Options that we keep from appearing in the options field in the mtab.  */
#define MS_NOMTAB	(MS_REMOUNT|MS_NOAUTO|MS_USER)

/* OPTIONS that we make ordinary users have by default.  */
#define MS_SECURE	(MS_NOEXEC|MS_NOSUID|MS_NODEV)

const struct opt_map opt_map[] =
{
  { "defaults",	0, 0		},	/* default options */
  { "ro",	0, MS_RDONLY	},	/* read-only */
  { "rw",	1, MS_RDONLY	},	/* read-write */
  { "exec",	1, MS_NOEXEC	},	/* permit execution of binaries */
  { "noexec",	0, MS_NOEXEC	},	/* don't execute binaries */
  { "suid",	1, MS_NOSUID	},	/* honor suid executables */
  { "nosuid",	0, MS_NOSUID	},	/* don't honor suid executables */
  { "dev",	1, MS_NODEV	},	/* interpret device files  */
  { "nodev",	0, MS_NODEV	},	/* don't interpret devices */
  { "sync",	0, MS_SYNC	},	/* synchronous I/O */
  { "async",	1, MS_SYNC	},	/* asynchronous I/O */
  { "remount",  0, MS_REMOUNT   },      /* Alter flags of mounted FS */
  { "auto",	1, MS_NOAUTO	},	/* Can be mounted using -a */
  { "noauto",	0, MS_NOAUTO	},	/* Can  only be mounted explicitly */
  { "user",	0, MS_USER	},	/* Allow ordinary user to mount */
  { "nouser",	1, MS_USER	},	/* Forbid ordinary user to mount */
  /* add new options here */
#ifdef MS_NOSUB
  { "sub",	1, MS_NOSUB	},	/* allow submounts */
  { "nosub",	0, MS_NOSUB	},	/* don't allow submounts */
#endif
  { NULL,	0, 0		}
};

static inline void
parse_opt (const char *opt, int *mask, char *extra_opts)
{
  const struct opt_map *om;

  for (om = opt_map; om->opt != NULL; om++)
    if (streq (opt, om->opt))
      {
	if (om->inv)
	  *mask &= ~om->mask;
	else
	  *mask |= om->mask;
	if (om->mask == MS_USER)
	  *mask |= MS_SECURE;
	return;
      }
  if (*extra_opts)
    strcat(extra_opts, ",");
  strcat(extra_opts, opt);
}
  
/* Take -o options list and compute 4th and 5th args to mount(2).  flags
   gets the standard options and extra_opts anything we don't recognize.  */
static void
parse_opts (char *opts, int *flags, char **extra_opts)
{
  char *opt;

  *flags = 0;
  *extra_opts = NULL;

  if (opts != NULL)
    {
      *extra_opts = xmalloc (strlen (opts) + 1); 
      **extra_opts = '\0';

      for (opt = strtok (opts, ",");
	   opt != NULL;
	   opt = strtok (NULL, ","))
	parse_opt (opt, flags, *extra_opts);
    }

  if (readonly)
    *flags |= MS_RDONLY;
  if (readwrite)
    *flags &= ~MS_RDONLY;
}

/* Try to build a canonical options string.  */
static char *
fix_opts_string (int flags, char *extra_opts)
{
  const struct opt_map *om;
  char *new_opts;
  char *tmp;

  new_opts = (flags & MS_RDONLY) ? "ro" : "rw";
  for (om = opt_map; om->opt != NULL; om++)
    {
      if (om->mask & MS_RDONLY)
	continue;
      if (om->inv || !om->mask || (flags & om->mask) != om->mask)
	continue;
      tmp = xmalloc(strlen(new_opts) + strlen(om->opt) + 2);
      sprintf(tmp, "%s,%s", new_opts, om->opt);
      new_opts = tmp;
      flags &= ~om->mask;
    }
  if (extra_opts && *extra_opts)
    {
      tmp = xmalloc(strlen(new_opts) + strlen(extra_opts) + 2);
      sprintf(tmp, "%s,%s", new_opts, extra_opts);
      new_opts = tmp;
    }
  return new_opts;
}

static int
try_mount5 (char *spec, char *node, char *type, int flags, char *mount_opts) {
   FILE *procfs_file;
   char line[100];
   char fsname[50];
   
   if (type) return mount5 (spec, node, type, flags & ~MS_NOSYS, mount_opts);
   if (( procfs_file = fopen("/proc/filesystems", "r")) == NULL) {
				/* If /proc/filesystems is not available,
				   preserve the old behavior of mount. */
      return mount5 (spec,
		     node,
		     "nfs",
		     flags & ~MS_NOSYS, mount_opts);
   }
   while (fgets(line, sizeof(line), procfs_file)) {
      if (sscanf (line, "nodev %[^\n]\n", fsname) == 1) continue;
      if (sscanf (line, " %[^ \n]\n", fsname) != 1) continue;
      if (mount5 (spec, node, fsname, flags & ~MS_NOSYS, mount_opts) == 0) {
      fclose(procfs_file);
      return 0;
      }
   }
   fclose(procfs_file);
   return -1;
}

static int
mount_one (char *spec, char *node, char *type, char *opts, int freq, int pass)
{
  struct mntent mnt;
  int mnt_err;
  int flags;
  char *extra_opts;
  char *mount_opts;

  if (type == NULL)
    {
      if (strchr (spec, ':') != NULL)
	type = "nfs";
    }

  parse_opts (xstrdup (opts), &flags, &extra_opts);

  /* root may allow certain types of mounts by ordinary users */
  if (suid && !(flags & MS_USER))
    die (3, "mount: only root can mount %s on %s", spec, node);

  /* quietly succeed for fstab entries that don't get mounted automatically */
  if (flags & MS_NOAUTO)
    return 0;

  mount_opts = extra_opts;

  if (!fake && type && streq (type, "nfs"))
    if (nfsmount (spec, node, &flags, &extra_opts, &mount_opts) != 0)
      return 1;
  
  block_signals (SIG_BLOCK);
  
  if (fake
      || (try_mount5 (spec, node, type, flags & ~MS_NOSYS, mount_opts)) == 0)
    /* Mount succeeded, write mtab entry.  */
    {
      if (!nomtab)
	{
	  mnt.mnt_fsname = canonicalize (spec);
	  mnt.mnt_dir = canonicalize (node);
	  mnt.mnt_type = type;
	  mnt.mnt_opts = fix_opts_string (flags & ~MS_NOMTAB, extra_opts);
	  mnt.mnt_freq = freq;
	  mnt.mnt_passno = pass;
      
	  if (flags & MS_REMOUNT)
	    {
	      close_mtab ();
	      update_mtab (mnt.mnt_dir, &mnt);
	      open_mtab ("a+");
	    }
	  else
	    if ((addmntent (F_mtab, &mnt)) == 1)
	      die (1, "mount: error writing %s: %s",
		   MOUNTED, strerror (errno));
	}

      block_signals (SIG_UNBLOCK);
      return 0;
    }

  mnt_err = errno; /* work around for errno bug in sigprocmask */

  block_signals (SIG_UNBLOCK);

  /* Mount failed, complain, but don't die.  */
  switch (mnt_err)
    {
    case EPERM:
      if (geteuid() == 0)
	error ("mount: mount point %s is not a directory", node);
      else
	error ("mount: must be superuser to use mount");
      break;
    case EBUSY:
      error ("mount: wrong fs type, %s already mounted, %s busy, "
	"or other error", spec, node);
      break;
    case ENOENT:
      error ("mount: mount point %s does not exist", node); break;
    case ENOTDIR:
      error ("mount: mount point %s is not a directory", node); break;
    case EINVAL:
      error ("mount: %s not a mount point", spec); break;
    case EMFILE:
      error ("mount table full"); break;
    case EIO:
      error ("mount: %s: can't read superblock", spec); break;
    case ENODEV:
      error ("mount: fs type %s not supported by kernel", type); break;
    case ENOTBLK:
      error ("mount: %s is not a block device", spec); break;
    case ENXIO:
      error ("mount: %s is not a valid block device", spec); break;
    case EACCES:
      error ("mount: block device %s is not permitted on its filesystem", spec);
      break;
    default:
      error ("mount: %s", strerror (mnt_err)); break;
    }
  return 1;
}

/* Create mtab with a root entry.  */
static void
create_mtab (void)
{
  struct mntent *fstab;
  struct mntent mnt;
  int flags;
  char *extra_opts;

  if ((F_mtab = setmntent (MOUNTED, "a+")) == NULL)
    die (1, "mount: can't open %s for writing: %s", MOUNTED, strerror (errno));

  /* Find the root entry by looking it up in fstab, which might be wrong.
     We could statfs "/" followed by a slew of stats on /dev/ but then
     we'd have to unparse the mount options as well....  */
  if ((fstab = getfsfile ("/")) || (fstab = getfsfile ("root")))
    {
      parse_opts (xstrdup (fstab->mnt_opts), &flags, &extra_opts);
      mnt = *fstab;
      mnt.mnt_fsname = canonicalize (fstab->mnt_fsname);
      mnt.mnt_dir = "/";
      mnt.mnt_opts = fix_opts_string (flags, extra_opts);

      if (addmntent (F_mtab, &mnt) == 1)
	die (1, "mount: error writing %s: %s", MOUNTED, strerror (errno));
    }
  if (fchmod (fileno (F_mtab), S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH) < 0)
    die (1, "mount: error changing mode of %s: %s", MOUNTED, strerror (errno));
  endmntent (F_mtab);
}

extern char version[];
static struct option longopts[] =
{
  { "fake", 0, 0, 'f' },
  { "help", 0, 0, 'h' },
  { "no-mtab", 0, 0, 'n' },
  { "read-only", 0, 0, 'r' },
  { "ro", 0, 0, 'r' },
  { "verbose", 0, 0, 'v' },
  { "version", 0, 0, 'V' },
  { "read-write", 0, 0, 'w' },
  { "rw", 0, 0, 'w' },
  { "options", 1, 0, 'o' },
  { NULL, 0, 0, 0 }
};

const char *usage_string = "\
usage: sfsmount [-hV]\n\
       sfsmount [-nfrvw]\n\
       sfsmount [-nfrvw] [-o options] localhost:path localdummy remotehost:path localmount\n\
See the man page for info.\n\
";

static void
usage (FILE *fp, int n)
{
  fprintf (fp, "%s", usage_string);
  exit (n);
}

int main (int argc, char *argv[])
{
  int c,flags=0;
  char *options = NULL,*extra_opts=NULL,*opts=NULL;
  struct mntent *fs;
  char *spec;
  int result = 0;
  struct stat statbuf;
  char *arg[4],*tmpargv,*tmpargv1,*tmpargv2;
	
  while ((c = getopt_long (argc, argv, "fhnrvVwo:", longopts, NULL)) != EOF)
    switch (c)
      {
      case 'f':			/* fake (don't actually do mount(2) call) */
	++fake;
	break;
      case 'h':			/* help */
	usage (stdout, 0);
	break;
      case 'n':			/* mount without writing in /etc/mtab */
	++nomtab;
	break;
      case 'r':			/* mount readonly */
	++readonly;
	readwrite = 0;
	break;
      case 'v':			/* be chatty */
	++verbose;
	break;
      case 'V':			/* version */
	printf ("%s\n", version);
	exit (0);
      case 'w':			/* mount read/write */
	++readwrite;
	readonly = 0;
	break;
      case 'o':			/* specify mount options */
	options = optarg;
	break;
      case 0:
	break;
      case '?':
      default:
	usage (stderr, 1);
	break;
      }

  argc -= optind;
  argv += optind;

  if (argc == 0)
    usage (stderr, 1);

  if (getuid () != geteuid ())
    {
      suid = 1;
      if (options || readwrite || nomtab || fake || argc != 1)
	die (2, "mount: only root can do that");
    }

  if (!nomtab)
    {
      lock_mtab ();
      if (stat(MOUNTED, &statbuf) < 0)
	create_mtab ();
      open_mtab ("a+");
    }
  else if (stat(MOUNTED, &statbuf) >= 0)
    open_mtab ("r");

  arg[0] = strstr(argv[0],":")+1;
  tmpargv = malloc(strlen(argv[3]));

  if (tmpargv==NULL)
    {
      perror ("Error: out of memory!");
      exit(1);
    }

  strcpy(tmpargv, argv[3]);
  tmpargv1 = malloc(strlen(argv[2]));

  if (tmpargv1==NULL)
    {
      perror ("Error: out of memory");
      exit(1);
    }

  strcpy(tmpargv1, argv[2]);
  tmpargv2 = malloc(strlen(argv[1]));

  if (tmpargv2==NULL)
    {
      perror ("Error: out of memory");
      exit(1);
    }

  strcpy(tmpargv2, argv[1]);

  arg[1] = tmpargv;
  arg[2] = tmpargv1;
  arg[3] = tmpargv2;
 
  if (stick(arg[0],arg[1],arg[2],arg[3])==1)
    {
      fprintf (stderr, "Stick error!\n");
      exit (1);
    }

  if (verbose)
    {
      printf ("Mounting\n");
      printf ("%s on %s\n", argv[0], argv[1]);
      printf ("%s on %s\n", argv[2], argv[3]);
    }
  
  if (mount_one (argv[0], argv[1], NULL, "port=4049", 0, 0)!=0) 
    {
      if (!nomtab)
	{
	  endmntent (F_mtab);
	  unlock_mtab ();
	}
      exit(1);
    }
  
  if (mount_one (argv[2], argv[3], NULL, NULL   , 0, 0)!=0) 
    {
      enum sfsstat statsfs;
      sfs_unstickargs unstickargs;
      
      unstickargs.name=(char*)malloc ((NFS_MAXPATHLEN+1)*sizeof(char));
	
      fprintf (stderr, "Error while mounting %s on %s\n", argv[2], argv[3]);

      if (umount (argv[1])<0)
	{
	  perror ("Error while umounting the first mount point!");
	  exit (1);
	}

      if (!nomtab)
	  update_mtab(argv[1], NULL);

      strcpy (unstickargs.name, argv[1]);
      callrpc ("localhost", ADM_PROGRAM, ADM_VERSION, ADMPROC_UNSTICK, (xdrproc_t)xdr_sfs_unstickargs, (char*)&unstickargs, (xdrproc_t)xdr_sfsstat, (char*)&statsfs);

      switch (statsfs)
	{
	case SFS_OK:
	  return 0;
	default:
	  fprintf (stderr, "Error while unsticking %s\n", unstickargs.name);
	  exit (1);
	}
      
      if (!nomtab)
	{
	  endmntent (F_mtab);
	  unlock_mtab ();
	}
      
      exit(1);
    }

  if (!nomtab)
    {
      endmntent (F_mtab);
      unlock_mtab ();
    }
  
  exit (0);
}
