/*
 * T.C.F.S. 2.0 Alpha 1 
 *
 *      	   This  program  handles  RPC  "NFS"  data  requests
 *              adopting a secure transfer protocol.
 *                 This   is  an  unsecure   and  unchecked  version,
 *              use at your own risk.
 *
 *              Please, report Bugs to: <tcfs@edu-gw.dia.unisa.it>
 *
 * Authors:	Giuseppe Cattaneo, <cattaneo@udsab.dia.unisa.it>
 *		Giuseppe Persiano, <giuper@udsab.dia.unisa.it>
 *		Andrea Cozzolino, <andcoz@edu-gw.dia.unisa.it>
 *		Angelo Celentano, <angcel@edu-gw.dia.unisa.it>
 *		Aniello Del Sorbo, <anidel@edu-gw.dia.unisa.it>
 *		Ermelindo Mauriello, <ermmau@edu-gw.dia.unisa.it>
 *		Raffaele Pisapia, <rafpis@edu-gw.dia.unisa.it>
 *
 *   Permission to  use, copy, and modify  this software  without fee
 * is hereby granted, provided that this entire notice is included in
 * all copies  of  any  software  which  is  or  includes a  copy  or
 * modification of this software and in all copies  of the supporting
 * documentation for such software.
 *
 *   This  software is  distribuited  under  the  GNU General  Public
 * License  (version  2, June  1991). Check  the  file  'COPYING'  for
 * more  infos. Some  parts of  this  software  derive  from the  NFS
 * implementation in the Linux kernel 2.0.x.
 *
 * This software  maybe be used  for any  purpose provided  the above
 * copyright  notice  is retained.  It  is  supplied  as is,  with no
 * warranty expressed or implied.
 *
 */

/* -+-_== */

#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/fcntl.h>
#include <linux/stat.h>
#include <linux/mm.h>
#include <linux/tcfs_fs.h>
#include <linux/malloc.h>
#include <linux/pagemap.h>

#include <asm/segment.h>
#include <asm/system.h>

#include "cypitfc.h"

/* These are the function needed to handle keys hash table */
extern struct hash_entry ** hash_table;
extern void init_hash(void);
extern struct hash_entry * hash_add(int uid,char *deskey, void *other_data);
extern struct hash_entry * hash_lookup(int uid);

static int tcfs_file_mmap(struct inode *, struct file *, struct vm_area_struct *);
static int tcfs_file_read(struct inode *, struct file *, char *, int);
static int tcfs_file_write(struct inode *, struct file *, const char *, int);
static int tcfs_fsync(struct inode *, struct file *);

static struct file_operations tcfs_file_operations = {
	NULL,			/* lseek - default */
	tcfs_file_read,		/* read */
	tcfs_file_write,	/* write */
	NULL,			/* readdir - bad */
	NULL,			/* select - default */
	tcfs_ioctl,		/* ioctl - default */
	tcfs_file_mmap,		/* mmap */
	NULL,			/* no special open is needed */
	NULL,			/* release */
	tcfs_fsync,		/* fsync */
};

struct inode_operations tcfs_file_inode_operations = {
	&tcfs_file_operations,	/* default file operations */
	NULL,			/* create */
	NULL,			/* lookup */
	NULL,			/* link */
	NULL,			/* unlink */
	NULL,			/* symlink */
	NULL,			/* mkdir */
	NULL,			/* rmdir */
	NULL,			/* mknod */
	NULL,			/* rename */
	NULL,			/* readlink */
	NULL,			/* follow_link */
	tcfs_readpage,		/* readpage */
	NULL,			/* writepage */
	NULL,			/* bmap */
	NULL			/* truncate */
};

static inline void revalidate_inode(struct tcfs_server * server, struct inode * inode)
{
	struct tcfs_fattr fattr;

#ifdef DEBUG_TCFS
	printk("Revalidate_inode %s\n",inode->u.tcfs_i.pathname);
#endif
	if (jiffies - TCFS_READTIME(inode) < TCFS_ATTRTIMEO(inode)) {
		return;
	}
	TCFS_READTIME(inode) = jiffies;
	if (tcfs_proc_getattr(server, TCFS_FH(inode), &fattr) == 0) {
		tcfs_refresh_inode(inode, &fattr);
		if (fattr.mtime.seconds == TCFS_OLDMTIME(inode)) {
			if ((TCFS_ATTRTIMEO(inode) <<= 1) > server->acregmax)
				TCFS_ATTRTIMEO(inode) = server->acregmax;
			if (S_ISREG(inode->i_mode) && TCFS_IS_SECURE(inode->u.tcfs_i.tcfs_fl.cflag) && 
			inode->i_size>0 && inode->u.tcfs_i.tcfs_fl.bf.spure>0)
				inode->i_size-=8-inode->u.tcfs_i.tcfs_fl.bf.spure;
			return;
		}
		TCFS_OLDMTIME(inode) = fattr.mtime.seconds;
	}
	if (tcfs_proc_geteattr(&(inode->i_sb->u.tcfs_sb.x_server),inode->u.tcfs_i.pathname,&(inode->u.tcfs_i.tcfs_fl.cflag))<0) {
#ifdef DEBUG_TCFS
	printk("TCFS: tcfs_proc_geteattr returned error\n");
#endif
		inode->u.tcfs_i.tcfs_fl.bf.mb=0;
		inode->u.tcfs_i.tcfs_fl.bf.spure=0;
		inode->u.tcfs_i.tcfs_fl.bf.unknow=0;
	}
	if (S_ISREG(inode->i_mode) && TCFS_IS_SECURE(inode->u.tcfs_i.tcfs_fl.cflag) && 
		inode->i_size>0 && inode->u.tcfs_i.tcfs_fl.bf.spure>0)
		inode->i_size-=8-inode->u.tcfs_i.tcfs_fl.bf.spure;
	invalidate_inode_pages(inode);
}

static int tcfs_file_read(struct inode * inode, struct file * file,
	char * buf, int count)
{
	int result;
	unsigned long old;
	struct hash_entry *htmp;

#ifdef DEBUG_TCFS
	printk("TCFS: tcfs_file_read %s pos=%d count=%d\n",inode->u.tcfs_i.pathname,(int)file->f_pos,(int)count);
#endif
	if (TCFS_IS_SECURE(inode->u.tcfs_i.tcfs_fl.cflag)) {
		htmp=hash_lookup(current->uid);
		if (htmp==NULL || current->uid!=inode->i_uid) {
			return -EACCES;
		}
	}

	TCFS_SEMR_DOWN(inode);
	
	/* DES job is done into readpage ... here we do a
	 * normal read ...  */
	revalidate_inode(TCFS_SERVER(inode), inode);
	
	if (inode==NULL || file==NULL || buf==NULL) {
		TCFS_SEMR_UP(inode);
		return -EIO;
	}
	old=file->f_pos;
	result=generic_file_read(inode, file, buf, count);
/*	if (result>0 && TCFS_IS_SECURE(inode->u.tcfs_i.tcfs_fl.cflag) && 
		inode->u.tcfs_i.tcfs_fl.bf.spure>0 && 
		old+result>inode->i_size - 8+inode->u.tcfs_i.tcfs_fl.bf.spure ) {
		result-=8-inode->u.tcfs_i.tcfs_fl.bf.spure;
	}*/
	TCFS_SEMR_UP(inode);
	return result;
}

static int tcfs_file_mmap(struct inode * inode, struct file * file, struct vm_area_struct * vma)
{
	int ret;
	struct hash_entry *htmp;

#ifdef DEBUG_TCFS
	printk("TCFS: tcfs_file_mmap %s\n",inode->u.tcfs_i.pathname);
#endif	
	if (TCFS_IS_SECURE(inode->u.tcfs_i.tcfs_fl.cflag)) {
		htmp=hash_lookup(current->uid);
		if (htmp==NULL || current->uid!=inode->i_uid) {
			return -EACCES;
		}
	}

	TCFS_SEMR_DOWN(inode);

	revalidate_inode(TCFS_SERVER(inode), inode);
	ret=generic_file_mmap(inode, file, vma);
	TCFS_SEMR_UP(inode);
	return ret;
}

static int tcfs_fsync(struct inode *inode, struct file *file)
{
#ifdef DEBUG_TCFS
	printk("TCFS: tcfs_fsync\n");
#endif
	return 0;
}

extern int tcfs_proc_kwrite(struct inode * inode, int offset,int count, const char *data, struct tcfs_fattr *fattr);

static int tcfs_crywrite(struct inode *inode, struct file *file,const char *buf,
	int count, struct hash_entry *htmp)
{
	loff_t newoff,diff,pos;
	struct tcfs_fattr fattr;
	int result,resw;
	char * tmp;
	int towrite=count;
	unsigned char *mbuf=NULL;
	
	while(mbuf==NULL) {
		mbuf=kmalloc(1024,GFP_KERNEL);
		schedule();
	}
	tmp=buf;
	if (file->f_flags & O_APPEND) {
		pos = inode->i_size;
	} else {
	  pos=file->f_pos;
	}
	do {
		schedule();
		newoff=pos & 0xfffffc00; /* Calculate offset of DES block */
		diff=pos-newoff;
		memset(mbuf,'\0',1024);
		result=tcfs_proc_read(&(inode->i_sb->u.tcfs_sb.s_server),TCFS_FH(inode),
					newoff,1024,mbuf,&fattr); /* Read old block */

/*		printk("spure=%d size=%d fpos=%d\n",(int)inode->u.tcfs_i.tcfs_fl.bf.spure,(int)inode->i_size,(int)pos);*/
		if (result <0 ) {/* Error reading file */
			kfree(mbuf);
			return result;
		}
		if (result > 0)
			mkdecrypt(mbuf,DES8(result),htmp->ks);
		if (result==0) { /* End of File */
			if (towrite<1024-diff) { /* Block end into this DES frame */
				int error;
		/*		printk("Caso 1 ... subcase 1 %ld diff=%d towrite=%d DES8=%d\n",(long)newoff,(int)diff,(int)towrite,DES8(towrite+diff));*/
				memcpy_fromfs(&mbuf[diff],tmp,towrite);
				mkencrypt(mbuf,DES8(towrite+diff),htmp->ks);
				resw=tcfs_proc_kwrite(inode,newoff,DES8(towrite+diff),mbuf,&fattr);
				if (resw<0) {
					kfree(mbuf);
					return resw;
				}
				pos+=towrite;
				tmp+=towrite;
				inode->u.tcfs_i.tcfs_fl.bf.spure=(int)(towrite+diff)%8;
				/*inode->i_size += towrite;*/
				towrite=0;
				error=tcfs_proc_seteattr(NULL,&(inode->i_sb->u.tcfs_sb.x_server),inode->u.tcfs_i.pathname,inode->u.tcfs_i.tcfs_fl.cflag,NULL);
				if (error<0) {
					kfree(mbuf);
					return error;
				}
			} else { /* Need another DES frame */
		/*		printk("Caso 1 ... subcase 2 %ld\n",(long)newoff);*/
				memcpy_fromfs(&mbuf[diff],tmp,1024);
				mkencrypt(mbuf,1024,htmp->ks);
				resw=tcfs_proc_kwrite(inode,newoff,1024,mbuf,&fattr);
				if (resw<0) {
					kfree(mbuf);
					return resw;
				}
				pos+=1024-diff;
				towrite-=1024-diff;
				tmp+=1024-diff;
			}
			continue;
		}
		if (result == 1024) { /* File not at the end */
			if (towrite<1024-diff) { /* Block end into this DES frame */
		/*		printk("Caso 2 ... subcase 1 %ld\n",(long)newoff);*/
				memcpy_fromfs(&mbuf[diff],tmp,towrite);
				mkencrypt(mbuf,1024,htmp->ks);
				resw=tcfs_proc_kwrite(inode,newoff,1024,mbuf,&fattr);
				if (resw<0) {
					kfree(mbuf);
					return resw;
				}
				tmp+=towrite;
				pos+=towrite;
				towrite=0;
			} else {
		/*		printk("Caso 2 ... subcase 2 %ld\n",(long)newoff);*/
				memcpy_fromfs(&mbuf[diff],tmp,1024-diff);
				mkencrypt(mbuf,1024,htmp->ks);
				resw=tcfs_proc_kwrite(inode,newoff,1024,mbuf,&fattr);
				if (resw<0) {
					kfree(mbuf);
					return resw;
				}
				towrite-=1024-diff;
				tmp+=1024-diff;
				pos+=1024-diff;
			}
		} else { /* File at the end */
			if (towrite<1024-diff) { /* Block end into this DES frame */
				int kk,error;
				if (inode->u.tcfs_i.tcfs_fl.bf.spure>0)
					result-=8-inode->u.tcfs_i.tcfs_fl.bf.spure;
		/*		printk("Caso 3 ... subcase 1 %ld %ld %d %d\n",(long)newoff,(long)diff,towrite,result);*/
				memcpy_fromfs(&mbuf[diff],tmp,towrite);
				kk=(towrite+diff>result)?towrite+diff:result;
				mkencrypt(mbuf,DES8(kk),htmp->ks);
				resw=tcfs_proc_kwrite(inode,newoff,DES8(kk),mbuf,&fattr);
				if (resw<0) {
					kfree(mbuf);
					return resw;
				}
				tmp+=towrite;
				pos+=towrite;
				inode->u.tcfs_i.tcfs_fl.bf.spure=(int)kk%8;
				/*inode->i_size = file->f_pos;*/
				towrite=0;
				error=tcfs_proc_seteattr(NULL,&(inode->i_sb->u.tcfs_sb.x_server),inode->u.tcfs_i.pathname,inode->u.tcfs_i.tcfs_fl.cflag,NULL);
				if (error<0) {
					kfree(mbuf);
					return error;
				}
			} else {
		/*		printk("Caso 3 ... subcase 2 %ld\n",(long)newoff);*/
				memcpy_fromfs(&mbuf[diff],tmp,1024-diff);
				mkencrypt(mbuf,1024,htmp->ks);
				resw=tcfs_proc_kwrite(inode,newoff,1024,mbuf,&fattr);
				if (resw<0) {
					kfree(mbuf);
					return resw;
				}
				towrite-=1024-diff;
				tmp+=1024-diff;
				pos+=1024-diff;
			}
		}
	} while(towrite>0);
	file->f_pos=pos;
	if(pos>inode->i_size)
	  inode->i_size=pos;
	/* Avoid possible Solaris 2.5 tcfsd bug */
	if (inode->i_ino == fattr.fileid) {
		tcfs_refresh_inode(inode, &fattr);
		if (inode->u.tcfs_i.tcfs_fl.bf.spure>0)
			inode->i_size-=8-inode->u.tcfs_i.tcfs_fl.bf.spure;
	}
	kfree(mbuf);
	invalidate_inode_pages(inode);
	return count;
}

static int tcfs_file_write(struct inode *inode, struct file *file, const char *buf,
			  int count)
{
	int result, written, wsize;
	struct tcfs_fattr fattr;
	unsigned long pos;
	struct hash_entry *htmp;
	
#ifdef DEBUG_TCFS
	printk("TCFS: tcfs_file_write %s\n",inode->u.tcfs_i.pathname);
#endif	
	TCFS_SEMR_DOWN(inode);

	if (!inode) {
		printk("tcfs_file_write: inode = NULL\n");
		TCFS_SEMR_UP(inode);
		return -EINVAL;
	}
	if (!S_ISREG(inode->i_mode)) {
		printk("tcfs_file_write: write to non-file, mode %07o\n",
			inode->i_mode);
		TCFS_SEMR_UP(inode);
		return -EINVAL;
	}
	if (count <= 0) {
		TCFS_SEMR_UP(inode);
		return 0;
	}
	
	if (TCFS_IS_SECURE(inode->u.tcfs_i.tcfs_fl.cflag)) {
		htmp=hash_lookup(current->uid);
		if (htmp==NULL || current->uid!=inode->i_uid) {
			TCFS_SEMR_UP(inode);
			return -EACCES;
		}
		result=tcfs_crywrite(inode,file,buf,count,htmp);
		TCFS_SEMR_UP(inode);
		return result;
	}
	pos = file->f_pos;
	if (file->f_flags & O_APPEND)
		pos = inode->i_size;
	wsize = TCFS_SERVER(inode)->wsize;
	result = 0;
	written = 0;
	while (written < count) {
		int hunk = count - written;
		if (hunk >= wsize)
			hunk = wsize;
		result = tcfs_proc_write(inode, pos, hunk, buf, &fattr);
		if (result < 0)
			break;
		pos += hunk;
		buf += hunk;
		written += hunk;
		if (hunk < wsize)
			break;
	}
	if (!written) {
		TCFS_SEMR_UP(inode);
		return result;
	}
	file->f_pos = pos;
	if (pos > inode->i_size)
		inode->i_size = pos;
	/* Avoid possible Solaris 2.5 tcfsd bug */
	if (inode->i_ino == fattr.fileid) {
		tcfs_refresh_inode(inode, &fattr);
	}
	TCFS_SEMR_UP(inode);
	return written;
}

