/*
 * Copyright (C) 1995, 1996 Systemics Ltd (http://www.systemics.com/)
 * All rights reserved.
 *
 */

package cryptix.pgp;

import java.io.*;

public final class PacketFactory
{
	public static final byte PublicKeyEncryptedType = 1;
	public static final byte SignatureType = 2;
	public static final byte SecretKeyCertificateType = 5;
	public static final byte PublicKeyCertificateType = 6;
	public static final byte CompressedDataType = 8;
	public static final byte ConvEncryptedDataType = 9;
	public static final byte LiteralDataType = 11;
	public static final byte KeyRingTrustType = 12;
	public static final byte UserIdType = 13;
	public static final byte CommentType = 14;

	private PacketFactory()
	{
	}

	public static final Packet
	read( DataInput in )
		throws IOException
	{
		int len;
		int type;
		try
		{
			PacketHeader header = new PacketHeader(in);
			len = header.len;
			type = header.type;
		}
		catch ( EOFException e )
		{
			return null;
		}

		switch (type)
		{
		case SecretKeyCertificateType:
			return new SecretKeyCertificate( in, len );
		case PublicKeyCertificateType:
			return new PublicKeyCertificate( in, len );
		case KeyRingTrustType:
			return new KeyRingTrust( in, len );	
		case PublicKeyEncryptedType:
			return new PublicKeyEncrypted( in, len );	
		case CompressedDataType:
			return new CompressedData( in, len );	
		case ConvEncryptedDataType:
			return new ConvEncryptedData( in, len );	
		case LiteralDataType:
			return new LiteralData( in, len );	
		case SignatureType:
			return new Signature( in, len );	
		case UserIdType:
			return new UserId( in, len );	
		case CommentType:
			return new Comment( in, len );
		default:
			throw new FormatException("Unknown packet type ("+type+")");
		}
	}

	public static final void
	write( DataOutput out, Packet pkt )
		throws IOException
	{
		byte buf[] = pkt.save();
		getHeaderFor( pkt, buf.length ).write( out );
		out.write( buf );
	}

	public static final byte[]
	save(Packet pkt)
		throws IOException
	{
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		write(new DataOutputStream(out), pkt);
		return out.toByteArray();
	}

	// I don't like the name 'load' - 'restore' would have been better
	public static final Packet
	load(byte buf[])
		throws IOException
	{
		return read(new DataInputStream(new ByteArrayInputStream(buf)));
	}


	private static final PacketHeader
	getHeaderFor( Packet pkt, int length )
		throws IOException
	{
		if ( pkt instanceof PublicKeyCertificate )
			return new PacketHeader( PublicKeyCertificateType, length, 2 );
		else if ( pkt instanceof SecretKeyCertificate )
			return new PacketHeader( SecretKeyCertificateType, length, 2 );
		else if ( pkt instanceof KeyRingTrust )
			return new PacketHeader( KeyRingTrustType, length, 1 );
		else if ( pkt instanceof PublicKeyEncrypted )
			return new PacketHeader( PublicKeyEncryptedType, length, 2 );
		else if ( pkt instanceof CompressedData )
			return new PacketHeader( CompressedDataType, length, 2 );
		else if ( pkt instanceof ConvEncryptedData )
			return new PacketHeader( ConvEncryptedDataType, length, 2 );
		else if ( pkt instanceof LiteralData )
			return new PacketHeader( LiteralDataType, length, 2 );
		else if ( pkt instanceof Signature )
			return new PacketHeader( SignatureType, length, 2 );
		else if ( pkt instanceof UserId )
			return new PacketHeader( UserIdType, length, 1 );
		else if ( pkt instanceof Comment )
			return new PacketHeader( CommentType, length, 1 );

		throw new IllegalArgumentException("PGP Packet type "+pkt.getClass().getName()+" not a valid PGP packet" );
	}
}
