/*____________________________________________________________________________
	Copyright (C) 1999 Network Associates, Inc.
	All rights reserved.

	$Id: PolicyManager.cp,v 1.17 1999/05/05 09:54:22 wprice Exp $
____________________________________________________________________________*/

#include "PolicyManager.h"
#include "pgpIPsecErrors.h"
#include "pgpEndianConversion.h"
#include "IPHandler.h"
#include "pgpIPsecESP.h"
#include "pgpIPsecAH.h"

#include <LowMem.h>


#define kAutomaticDataRekeyKB	2048

	EPMStatus
PMDoTransform(
	queue_t *				inQ,
	PGPIPsecBuffer *		inBuffers,
	PGPUInt32				inAddress,
	PGPBoolean				inIncoming,
	PGPUInt32				inSPI,				
	SSAList *				inSAList,
	SHostEntryList *		inHostEntryList,
	mblk_t **				outPacketBlock )
{
	EPMStatus				status = pmStatus_DropPacket;
	PGPError				err	= kPGPError_NoErr;
	PGPNetPrefHostEntry *	host = PMFindHost( inAddress, inHostEntryList );
	PGPnetIKESAUserData *	ud;
	PGPIPsecBuffer *		tunnelBuffers = NULL;
	mblk_t *				tunnelPacketMessage = NULL;

	DConLogNote("  PMDoTransform\n");
	if( inIncoming )
	{
		PGPikeSA	*	sa;
		PGPBoolean		tunnel = FALSE;
		
		if( !( sa = PMFindSAspi( inAddress, inSPI, inSAList ) ) )
		{
			err = kPGPIPsecError_NoSAFound;
			goto done;
		}
		if( host && ( host->hostType == kPGPnetSecureGateway ) )
			tunnel = TRUE;
		err = IPApplySAToPacket(inQ, TRUE, tunnel, sa, inBuffers, outPacketBlock);
		if( err )
			goto done;
		if( tunnel )
		{
			tunnelPacketMessage = *outPacketBlock;
			*outPacketBlock = NULL;
			tunnelBuffers = IPMessageToIPsecBuffers(tunnelPacketMessage);
			if (tunnelBuffers == NULL)
			{
				err = kPGPError_OutOfMemory;
				goto done;
			}
			IPGetSourceAddressAndSPI(tunnelBuffers, &inAddress, &inSPI);
			if( inSPI )
			{
				if( !( sa = PMFindSAspi( inAddress, inSPI, inSAList ) ) )
				{
					err = kPGPIPsecError_NoSAFound;
					goto done;
				}
				err = IPApplySAToPacket(inQ, TRUE, FALSE, sa, tunnelBuffers, outPacketBlock);
				if( err )
					goto done;
				
			}
			else
			{
				*outPacketBlock = tunnelPacketMessage;
				tunnelPacketMessage = NULL;
			}
		}
		status = pmStatus_ApplySA;	// successful
	}
	else
	{
		PGPikeSA	*	transSA = NULL,
					*	tunnelSA = NULL;
		SSAList		*	transSAL = NULL,
					*	tunnelSAL = NULL;
		
	retry:
		if( !host )
		{
			if( !( transSA = PMFindSA( inAddress, 0, FALSE, inSAList, &transSAL ) ) )
			{
				err = kPGPIPsecError_NoSAFound;
				goto done;
			}
		}
		else
		{
			if( host->hostType == kPGPnetSecureHost )
			{
				if( !( transSA = PMFindSA( inAddress, 0, FALSE, inSAList, &transSAL ) ) )
				{
					err = kPGPIPsecError_NoSAFound;
					goto done;
				}
			}
			else if( host->hostType == kPGPnetSecureGateway )
			{
				if( !(tunnelSA = PMFindSA( host->ipAddress, host->ipAddress, FALSE,
									inSAList, &tunnelSAL ) ) )
				{
					err = kPGPIPsecError_NoSAFound;
					goto done;
				}
			}
			if( host->childOf != -1 )
			{
				if( !( tunnelSA = PMFindSA(
								inHostEntryList->host[host->childOf].ipAddress,
								host->ipAddress, FALSE, inSAList, &tunnelSAL ) ) )
				{
					err = kPGPIPsecError_NoSAFound;
					goto done;
				}
			}
		}
		if( PMCheckExpiration( transSAL ) )
			goto retry;
		if( PMCheckExpiration( tunnelSAL ) )
			goto retry;
		
		PMCheckRekey( transSAL );
		PMCheckRekey( tunnelSAL );

		if( transSA )
		{
			err = IPApplySAToPacket(inQ, FALSE, FALSE, transSA, inBuffers, outPacketBlock);
			if( !err )
			{
				ud = (PGPnetIKESAUserData *)&transSA->userData[0];
				ud->packetsSent++;
				ud->bytesSent += (*outPacketBlock)->b_wptr - (*outPacketBlock)->b_rptr;
			}
			else
				goto done;
		}
		if( tunnelSA )
		{
			if (*outPacketBlock != NULL)
			{
				tunnelPacketMessage = *outPacketBlock;
				*outPacketBlock = NULL;
				tunnelBuffers = IPMessageToIPsecBuffers(tunnelPacketMessage);
				if (tunnelBuffers == NULL)
				{
					err = kPGPError_OutOfMemory;
					goto done;
				}
			}
			err = IPApplySAToPacket(inQ, FALSE, TRUE, tunnelSA,
					(tunnelBuffers == NULL) ? inBuffers : tunnelBuffers, outPacketBlock);
			if( !err )
			{
				ud = (PGPnetIKESAUserData *)&tunnelSA->userData[0];
				ud->packetsSent++;
				ud->bytesSent += (*outPacketBlock)->b_wptr - (*outPacketBlock)->b_rptr;
			}
			else
				goto done;
		}
		if( !tunnelSA && !transSA )
		{
			err = kPGPIPsecError_NoSAFound;
			goto done;
		}
		status = pmStatus_ApplySA;	// successful
	}
done:
	if ( tunnelBuffers != NULL )
		OTFreeMem(tunnelBuffers);
	if ( tunnelPacketMessage != NULL )
		freemsg(tunnelPacketMessage);
	if( err )
	{
		DConLogNote("  PMDoTransform: err=%ld\n", err);
		if (*outPacketBlock != NULL)
		{
			freemsg(*outPacketBlock);
			*outPacketBlock = NULL;
		}
		SendErrorToService( err, inAddress );
	}
	return status;
}

	EPMStatus
PMCheckPacketPolicy(
	PGPUInt32				inAddress,
	PGPBoolean				inIncoming,
	const mblk_t *			inPacket,
	PGPBoolean				inAllowUnconfigured,
	PGPBoolean				inRequireSecure,
	SSAList *				inSAList,
	SHostEntryList *		inHostEntryList,
	SPendingSAList *		inPendingSAList )
{
	EPMStatus				status = pmStatus_PassThrough;
	PGPNetPrefHostEntry *	host = PMFindHost( inAddress, inHostEntryList );
	PGPNetPrefHostEntry *	gateway;
	SSAList *				sal;
	
	if( host )
	{
		if( ( host->hostType == kPGPnetInsecureHost ) &&
			( host->childOf == -1 ) )
		{
			goto done;
		}
		else if( host->hostType == kPGPnetSecureGateway )
		{
			status = PMAcquireSA( host->ipAddress,
						host->ipAddress, host->ipMask, inIncoming, 
						inPacket, inSAList, inPendingSAList );
		}
		else if( ( host->hostType == kPGPnetSecureHost ) &&
					( host->childOf == -1 ) )
		{
			DConLogNote("  PMCheckPacketPolicy: SH-1\n");
			status = PMAcquireSA( inAddress, 0, 0, inIncoming,
								 	inPacket, inSAList, inPendingSAList );
		}
		else if( ( host->hostType == kPGPnetInsecureHost ) &&
					( host->childOf != -1 ) )
		{
			gateway = &inHostEntryList->host[host->childOf];
			status = PMAcquireSA(	gateway->ipAddress,
									host->ipAddress, host->ipMask, inIncoming,
									inPacket, inSAList, inPendingSAList );
		}
		else if( ( host->hostType == kPGPnetSecureHost ) &&
					( host->childOf != -1 ) )
		{
			EPMStatus	trStat,
						tuStat;
			
			trStat = PMAcquireSA( inAddress, 0, 0, inIncoming,
									inPacket, inSAList, inPendingSAList );
			gateway = &inHostEntryList->host[host->childOf];
			tuStat = PMAcquireSA(	gateway->ipAddress,
									host->ipAddress, host->ipMask, inIncoming,
									inPacket, inSAList, inPendingSAList );
			if( ( trStat == pmStatus_ApplySA ) && ( tuStat == pmStatus_ApplySA ) )
				status = pmStatus_ApplySA;
		}
	}
	else if( inAllowUnconfigured )
	{
		if( inRequireSecure )
			status = PMAcquireSA( inAddress, 0, 0, inIncoming,
									inPacket, inSAList, inPendingSAList );
		else
		{
			if( PMFindSA( inAddress, 0, inIncoming, inSAList, &sal ) )
				status = pmStatus_ApplySA;
		}
	}
	else
		status = pmStatus_DropPacket;
done:
	return status;
}

	PGPNetPrefHostEntry *
PMFindHost(
	PGPUInt32				inAddress,
	SHostEntryList *		inHostEntryList )
{
	PGPNetPrefHostEntry *	host = NULL,
						*	gwHost = NULL,
						*	hostList = NULL;
	PGPUInt32				hInx,
							numHosts = 0;
	
	DConLogNote("  PMFindHost: in=%lx\n", inAddress );
	if( inHostEntryList )
	{
		hostList = inHostEntryList->host;
		numHosts = inHostEntryList->numHosts;
	}
	for( hInx = 0; hInx < numHosts; hInx++ )
	{
		//DConLogNote("  PMFindHost: IP:%lx\n", hostList[hInx].ipAddress );
		if( inAddress == hostList[hInx].ipAddress )
		{
			host = &inHostEntryList->host[hInx];
			if( hostList[hInx].hostType == kPGPnetSecureGateway )
			{
				gwHost = host;
				continue;
			}
			DConLogNote("  PMFindHost: Found Host\n");
			return host;
		}
	}
	if( gwHost )
	{
		DConLogNote("  PMFindHost: Found Host(GW)\n");
		return gwHost;
	}
	for( hInx = 0; hInx < numHosts; hInx++ )
	{
		if( ( inAddress & hostList[hInx].ipMask ) ==
			  ( hostList[hInx].ipAddress & hostList[hInx].ipMask ) )
		{
			host = &inHostEntryList->host[hInx];
			DConLogNote("  PMFindHost: Found Subnet\n");
			return host;
		}
	}
	DConLogNote("  PMFindHost: None\n");
	return NULL;
}

	EPMStatus
PMAcquireSA(
	PGPUInt32			inAddress,
	PGPUInt32			ipAddrStart,
	PGPUInt32			ipMaskEnd,
	PGPBoolean			inIncoming,
	const mblk_t *		inPacket,
	SSAList *			inSAList,
	SPendingSAList *	inPendingSAList )
{
	PGPikeSA *			sa = NULL;
	PGPBoolean			pending = FALSE;
	EPMStatus			status;
	SPendingSAList *	pendList = inPendingSAList;
	SSAList *			sal;
	PGPUInt32			spi = 0;
	
	while( pendList )
	{
		if( ( pendList->ipAddress == inAddress ) &&
			( pendList->ipAddrStart == ipAddrStart ) &&
			( pendList->ipMaskEnd == ipMaskEnd ) &&
			( pendList->destIsRange == FALSE ) )
		{
			pending = TRUE;
			break;
		}
		pendList = pendList->next;
	}
	if( inIncoming )
	{
		PGPIPsecBuffer *	buffers = IPMessageToIPsecBuffers(inPacket);
		
		if (buffers != NULL)
		{
			pgpIPsecESPGetSPI(buffers, &spi);
			if (spi == 0) {
				pgpIPsecAHGetSPI(buffers, &spi);
			}
			OTFreeMem(buffers);
		}
	}
	if( spi )
	{
		sa = PMFindSAspi( inAddress, spi, inSAList );
		if( !sa )
		{
			if( PMFindSA( inAddress, ipAddrStart, inIncoming, inSAList, &sal ) )
				pending = TRUE;
		}
	}
	else
		sa = PMFindSA( inAddress, ipAddrStart, inIncoming, inSAList, &sal );
	if( sa )
	{
		DConLogNote("  PMAcquireSA: Apply\n");
		status	= pmStatus_ApplySA;
	}
	else
	{
		DConLogNote("  PMAcquireSA: Drop\n");
		status	= pmStatus_DropPacket;
		if( !pending )
		{
			if( AddIPAddressToPendingList( inAddress, FALSE,
					ipAddrStart, ipMaskEnd ) )
			{
				SendNeedSAToService( inAddress, FALSE, ipAddrStart, ipMaskEnd );
			}
		}
	}
	return status;
}

	PGPikeSA *
PMFindSA(
	PGPUInt32			inAddress,
	PGPUInt32			ipAddrStart,
	PGPBoolean			incoming,
	SSAList *			inSAList,
	SSAList **			outSAEntry )
{
	PGPikeSA *			sa = NULL;
	SSAList	 *			saWalk;
	
	*outSAEntry = NULL;
	for( saWalk = inSAList; saWalk; saWalk = saWalk->next )
	{
		if( ( !incoming && !saWalk->sa.activeOut ) || saWalk->killed )
			continue;
		if( saWalk->sa.ipAddress == inAddress )
		{
			if( ipAddrStart )
			{
				if( saWalk->sa.destIsRange )
				{
					if( ( ipAddrStart >= saWalk->sa.ipAddrStart ) &&
						( ipAddrStart <= saWalk->sa.ipMaskEnd ) )
					{
						sa = &saWalk->sa;
						*outSAEntry = saWalk;
						return sa;
					}
				}
				else
				{	
					if( ( ipAddrStart & saWalk->sa.ipMaskEnd ) ==
						( saWalk->sa.ipAddrStart & saWalk->sa.ipMaskEnd ) )
					{
						sa = &saWalk->sa;
						*outSAEntry = saWalk;
						return sa;
					}
				}
			}
			else
			{
				sa = &saWalk->sa;
				*outSAEntry = saWalk;
				return sa;
			}
		}
	}
	return NULL;
}

	PGPikeSA *
PMFindSAspi(
	PGPUInt32			inAddress,
	PGPUInt32			spi,
	SSAList *			inSAList )
{
	SSAList	 *			saWalk;
	
	for( saWalk = inSAList; saWalk; saWalk = saWalk->next )
	{
		if( ( saWalk->sa.ipAddress == inAddress ) && !saWalk->killed )
		{
			if( PMHasInSPI( &saWalk->sa, spi ) )
				return &saWalk->sa;
		}
	}
	return NULL;
}

	PGPBoolean
PMHasInSPI(
	PGPikeSA *			inSA,
	PGPUInt32			spi )
{
	PGPUInt32			trIndex;
	
	for( trIndex = 0; trIndex < inSA->numTransforms; trIndex++ )
	{
		PGPUInt32 saSPI = PGPEndianToUInt32( kPGPBigEndian,
							inSA->transform[trIndex].u.ipsec.inSPI );
		
		if( saSPI == spi )
			return TRUE;
	}
	return FALSE;
}

	void
PMCheckRekey(
	SSAList *				sal )
{
#define kPGPnetRekeyKBSeconds		30
	PGPikeSA *				sa;
	PGPnetIKESAUserData *	ud;
	PGPBoolean				bRekey = FALSE;
	PGPUInt32				kbSent = 1;
	PGPUInt32				avgKB = 0;

	if( !sal )
		return;
	sa = &sal->sa;
	ud = (PGPnetIKESAUserData *)&sa->userData[0];
	if( ud->reKeyInProgress )
		return;
	if( ud->packetsSent >= 0xFFFF7FFF )
	{
		bRekey = TRUE;
		goto rekey;
	}
	if( !sa->kbLifeTime || ( ud->bytesSent < 1024 ) )
		return;
	kbSent = ud->bytesSent / 1024;
	avgKB = kbSent / ( LMGetTime() - sa->birthTime );
	if( avgKB &&
		( kPGPnetRekeyKBSeconds >= ( sa->kbLifeTime - kbSent ) / avgKB ) )
	{
		bRekey = TRUE;
		goto rekey;
	}
	if( kbSent >= sa->kbLifeTime - kAutomaticDataRekeyKB )
	{
		bRekey = TRUE;
		goto rekey;
	}
rekey:
	if( bRekey )
	{
		ud->reKeyInProgress = TRUE;
		SendRekeySAToService( sal->index );
		DConLogNote("  PMCheckRekey: Rekey\n");
	}
}

	PGPBoolean
PMCheckExpiration(
	SSAList *				sal )
{
	PGPikeSA *				sa;
	PGPnetIKESAUserData *	ud;
	
	if( !sal )
		return FALSE;
	sa = &sal->sa;
	ud = (PGPnetIKESAUserData *)&sa->userData[0];
	if( ( ( sa->kbLifeTime != 0 ) &&
		( ud->bytesSent >= ( sa->kbLifeTime *1024 ) ) ) ||
		( ud->packetsSent >= MAX_PGPUInt32 - 1 ) )
	{
		sa->activeOut = FALSE;
		SendKillSAToService( sal->index );
		return TRUE;
	}
	return FALSE;
}

