/*____________________________________________________________________________
	Copyright (C) 1999 Network Associates, Inc. and it's affiliated companies
	All rights reserved.

	$Id: PGPsdaMacEncrypt.cp,v 1.7 1999/04/04 11:12:31 wprice Exp $
____________________________________________________________________________*/

#include "MacEnvirons.h"
#include "MacErrors.h"
#include "MacFiles.h"
#include "MacResources.h"
#include "MacStrings.h"
#include "pgpMem.h"
#include "pgpPubTypes.h"
#include "pgpUtilities.h"

#include "CipherContext.h"
#include "CipherProcGlue.h"
#include "lzss.h"
#include "pgpClientLib.h"
#include "PGPsdaMacCommon.h"
#include "PGPsdaMacEncrypt.h"
#include "PGPsdaMacFileFormat.h"
#include "SHA.h"

#define	kInputBufferSize	(128L * 1024L)
#define kCompressBufferSize	kPGPsdaCompressionBlockSize
#define kEncryptBufferSize	(2 * kPGPsdaCompressionBlockSize)
#define	kOutputBufferSize	(128L * 1024L)

#define kResourceCopyListResType	((OSType) 'SDAR')
#define kSDAResCopyListResID		128
#define kSEAResCopyListResID		129
#define kCommonResCopyListResID		130

#pragma options align=mac68k

typedef struct SDAResourceCopySpec
{
	OSType	sourceResType;
	short	sourceResID;
	OSType	destResType;
	short	destResID;

} SDAResourceCopySpec;

#pragma options align=reset

typedef struct PGPsdaEncryptionContext
{
	PGPContextRef	pgpContext;
	PGPUInt32 		numSourceItems;
	const FSSpec 	*sourceItems;
	FSSpec 			outFileSpec;
	CASTKey			encryptionKey;
	const char		*passphrase;
	PGPUInt32		hashReps;
	CipherProcRef	cipherProcRef;
	CipherContext	cipherContext;
	SDABuffer		inputBuffer;
	SDABuffer		compressBuffer;
	SDABuffer		encryptBuffer;
	SDABuffer		outputBuffer;
	PGPUserValue	userValue;
	PGPUInt32		totalBytes;
	PGPUInt32		totalItems;
	short			outFileRef;
	PGPUInt32		encryptedBlockIV;
	SHA				*hash;
	Boolean			isEncryptedArchive;
	
	PGPsdaEncryptProgressProc	progressProc;
	PGPsdaEncryptProgressInfo	progressInfo;
	
} PGPsdaEncryptionContext;

	static void
EndSDAEncryption(PGPsdaEncryptionContext *context)
{
	if( context->cipherProcRef != NULL )
	{
		CipherProc_Dispose( context->cipherProcRef );
		context->cipherProcRef = NULL;
	}
	
	FreeSDABuffer( &context->inputBuffer );
	FreeSDABuffer( &context->outputBuffer );
	FreeSDABuffer( &context->compressBuffer );
	FreeSDABuffer( &context->encryptBuffer );
	
	delete context->hash;
}

	static OSStatus
BeginSDAEncryption(PGPsdaEncryptionContext *context)
{
	OSStatus	err = noErr;
	
	context->hash = new( SHA );
	
	if( context->passphrase[0] != 0 )
	{
		PGPError	pgpErr;

		context->isEncryptedArchive = TRUE;
		
		pgpErr = PGPGuaranteeMinimumEntropy( context->pgpContext );
		if( IsntPGPError( pgpErr ) )
		{
			pgpErr = PGPContextGetRandomBytes( context->pgpContext,
							&context->cipherContext.salt,
							sizeof( context->cipherContext.salt ) );
		}

		/* Initialize cipher proc */
		
		err = PGPErrorToMacError( pgpErr );
		if( IsntErr( err ) )
		{
			HashSaltPassphrase( context->passphrase,
					&context->cipherContext.salt, &context->hashReps,
					&context->encryptionKey );
		
			err = CipherProc_Load( &context->cipherProcRef );
			if( IsntErr( err ) )
			{
				err = CipherProc_CallInitContext( context->cipherProcRef,
							&context->cipherContext,
							(uchar *) &context->encryptionKey );
			}
		}
	}
	
	/* Setup buffers */
	
	if( IsntErr( err ) )
	{
		err = AllocateSDABuffer( kInputBufferSize, &context->inputBuffer );
		if( IsntErr( err ) )
		{
			err = AllocateSDABuffer( kOutputBufferSize, &context->outputBuffer );
			if( IsntErr( err ) )
			{
				err = AllocateSDABuffer( kCompressBufferSize,
							&context->compressBuffer );
				if( IsntErr( err ) )
				{
					err = AllocateSDABuffer( kEncryptBufferSize,
								&context->encryptBuffer );
				}
			}
		}
	}
	
	if( IsErr( err ) )
	{
		EndSDAEncryption( context );
	}
		
	return( err );
}

	static OSStatus
CallProgressProc(PGPsdaEncryptionContext *context)
{
	OSStatus	err = noErr;
	
	if( IsntNull( context->progressProc ) )
	{
		pgpAssert( context->progressInfo.bytesProcessed <=
				context->progressInfo.totalBytesToProcess );
		
		if( context->progressInfo.bytesProcessed > context->progressInfo.totalBytesToProcess )
			context->progressInfo.bytesProcessed = context->progressInfo.totalBytesToProcess;
			
		err = (*context->progressProc)( &context->progressInfo,
					context->userValue );
	}
	
	return( err );
}

	static OSStatus
AnalyzeSourceFile(
	PGPsdaEncryptionContext *context,
	const FSSpec			*fileSpec,
	const CInfoPBRec		*fileCPB)
{
	OSStatus	err;
	
	context->totalBytes += sizeof( PGPsdaHeaderType ) +
								sizeof( PGPsdaFileHeader );
	context->totalBytes += cpbDataForkSize( fileCPB );
	context->totalBytes += cpbResForkSize( fileCPB );
	context->totalItems	+= 1;
	
	context->progressInfo.curItem 				= fileSpec;
	context->progressInfo.totalBytesToProcess 	= context->totalBytes;
	
	err = CallProgressProc( context );
	
	return( err );
}

	static OSStatus
AnalyzeSourceFolder(
	PGPsdaEncryptionContext *context,
	const FSSpec			*folderSpec,
	const CInfoPBRec		*folderCPB)
{
	OSStatus	err;
	CInfoPBRec	cpb;
	FSSpec		itemSpec;
	
	itemSpec.vRefNum	= folderSpec->vRefNum;
	itemSpec.parID		= folderCPB->dirInfo.ioDrDirID;
	
	pgpClearMemory( &cpb, sizeof( cpb ) );
	
	cpb.dirInfo.ioVRefNum	= itemSpec.vRefNum;
	cpb.dirInfo.ioDrDirID	= itemSpec.parID;
	cpb.dirInfo.ioNamePtr	= itemSpec.name;
	cpb.dirInfo.ioFDirIndex	= 1;
	
	while( TRUE )
	{
		err = PBGetCatInfoSync( &cpb );
		if( IsntErr( err ) )
		{
			if( cpbIsFile( &cpb ) )
			{
				err = AnalyzeSourceFile( context, &itemSpec, &cpb );
			}
			else
			{
				err = AnalyzeSourceFolder( context, &itemSpec, &cpb );
			}
		
			cpb.dirInfo.ioDrDirID = itemSpec.parID;
			++cpb.dirInfo.ioFDirIndex;
		}
		else
		{
			if( err == fnfErr )
				err = noErr;
				
			break;
		}
	}
	
	if( IsntErr( err ) )
	{
		context->totalItems	+= 1;
		context->totalBytes += sizeof( PGPsdaHeaderType ) +
									sizeof( PGPsdaBeginFolderHeader ) +
									sizeof( PGPsdaHeaderType );

		context->progressInfo.curItem 				= folderSpec;
		context->progressInfo.totalBytesToProcess 	= context->totalBytes;
	
		err = CallProgressProc( context );
	}
	
	return( err );
}

	static OSStatus
AnalyzeSourceItems(PGPsdaEncryptionContext *context)
{
	OSStatus	err = noErr;
	PGPUInt32	sourceIndex;
	
	context->progressInfo.state = kPGPsdaEncryptState_Analyzing;

	for( sourceIndex = 0; sourceIndex < context->numSourceItems; sourceIndex++ )
	{
		CInfoPBRec		cpb;
		const FSSpec	*sourceItem = &context->sourceItems[sourceIndex];
		
		err = FSpGetCatInfo( sourceItem, &cpb );
		if( IsntErr( err ) )
		{
			if( cpbIsFile( &cpb ) )
			{
				err = AnalyzeSourceFile( context, sourceItem, &cpb );
			}
			else
			{
				err = AnalyzeSourceFolder( context, sourceItem, &cpb );
			}
		}
		
		if( IsErr( err ) )
			break;
	}
	
	if( IsntErr( err ) )
	{
		/* Account of end-of-archive marker */
		context->totalBytes += sizeof( PGPsdaHeaderType );

		/*
		** Simple reject case. Check to see if there are enough free bytes on
		** the destination volume before proceeding. This check does nothing for 
		** allocation block issues.
		*/
		
		XVolumeParam	xpb;

		xpb.ioNamePtr	= nil;
		xpb.ioVRefNum	= context->outFileSpec.vRefNum;
		xpb.ioVolIndex	= 0;
		
		err = pgpPBXGetVolInfo( &xpb );
		if( IsntErr( err ) )
		{
			PGPUInt64	freeBytes = *((PGPUInt64 *) &xpb.ioVFreeBytes);
			
			if( context->totalBytes > freeBytes )
			{
				err = dskFulErr;
			}
		}
	}
	
	return( err );
}

	static OSStatus
Copy1Resource(
	short	srcResFile,
	OSType	srcResType,
	short	srcResID,
	short	destResFile,
	OSType	destResType,
	short	destResID)
{
	short		saveResFile = CurResFile();
	OSStatus	err			= noErr;
	Handle		theResource;
	
	UseResFile( srcResFile );
	
	theResource = Get1Resource( srcResType, srcResID );
	if( IsntNull( theResource ) )
	{
		Str255	resName;
		short	resID;
		OSType	resType;
		short	resAttrs;
		
		HNoPurge( theResource );
		
		GetResInfo( theResource, &resID, &resType, resName );
		resAttrs = GetResAttrs( theResource );
		
		DetachResource( theResource );
		
		UseResFile( destResFile );
		
		err = AddWriteResource( theResource, destResType, destResID, resName );
		if( IsntErr( err ) )
		{
			SetResAttrs( theResource, resAttrs );
			ChangedResource( theResource );
		}
	}
	else
	{
		err = ForceResError( resNotFound );
	}
	
	UseResFile( saveResFile );
	
	return( err );
}

	static OSStatus
CopyResourceList(
	short 			listResID,
	const FSSpec	*outFileSpec)
{
	OSStatus			err 		= noErr;
	short				saveResFile	= CurResFile();
	SDAResourceCopySpec	**copyList;
	
	copyList = (SDAResourceCopySpec **) Get1Resource(
						kResourceCopyListResType, listResID );
	if( IsntNull( copyList ) )
	{
		PGPUInt32			numResources;
		SDAResourceCopySpec	*curResInfo;
		short				outResFile;
		short				appResFile = CurResFile();
		
		HLock( (Handle) copyList );
		
		numResources 	= GetHandleSize( (Handle) copyList ) /
								sizeof( **copyList );
		curResInfo		= *copyList;
		
		outResFile = FSpOpenResFile( outFileSpec, fsRdWrPerm );
		if( outResFile > 0 )
		{
			PGPUInt32	resIndex;

			for( resIndex = 0; resIndex < numResources; resIndex++ )
			{
				err = Copy1Resource( appResFile, curResInfo->sourceResType,
							curResInfo->sourceResID, outResFile,
							curResInfo->destResType, curResInfo->destResID );
				if( IsErr( err ) )
					break;
					
				++curResInfo;
			}

			UseResFile( outResFile );
			UpdateResFile( outResFile );
			CloseResFile( outResFile );
			FlushVol( NULL, outFileSpec->vRefNum );
		}
		else
		{
			err = ForceResError( fnfErr );
		}
	
		ReleaseResource( (Handle) copyList );
	}
	else
	{
		err = ForceResError( resNotFound );
	}
	
	return( err );

	UseResFile( saveResFile );
	
	return( err );
}

	static OSStatus
CopySDAResources(PGPsdaEncryptionContext *context)
{
	OSStatus	err;
	OSType		fileCreator;
	
	if( context->isEncryptedArchive )
	{
		fileCreator = kPGPsdaCreator;			
	}
	else
	{
		fileCreator = kPGPseaCreator;			
	}		
	
	FSpCreateResFile( &context->outFileSpec, fileCreator, 'APPL',
			smSystemScript );

	err = CopyResourceList( kCommonResCopyListResID, &context->outFileSpec );
	if( IsntErr( err ) )
	{
		if( context->isEncryptedArchive )
		{
			err = CopyResourceList( kSDAResCopyListResID, &context->outFileSpec );
		}
		else
		{
			err = CopyResourceList( kSEAResCopyListResID, &context->outFileSpec );
		}
	}
	
	return( err );
}

	static OSStatus
FlushOutputBuffer(PGPsdaEncryptionContext *context)
{
	OSStatus	err = noErr;
	
	if( context->outputBuffer.used != 0 )
	{
		long	count = context->outputBuffer.used;
		
		err = FSWrite( context->outFileRef, &count, context->outputBuffer.data );
		if( IsntErr( err ) )
		{
			context->hash->Update( context->outputBuffer.data, count );
		}
		
		context->outputBuffer.used 	= 0;
		context->outputBuffer.pos 	= 0;
	}
	
	return( err );
}

	static OSStatus
FlushEncryptBuffer(
	PGPsdaEncryptionContext *context,
	PGPBoolean				encryptPartialLastBlock)
{
	OSStatus	err = noErr;
	
	if( context->encryptBuffer.used != 0 )
	{
		PGPUInt32	remainingCompBlocks;
		
		/* Encrypt data into the output buffer in 512 byte blocks */
	
		remainingCompBlocks 		= context->encryptBuffer.used / 512L;
		context->encryptBuffer.pos	= 0;

		while( remainingCompBlocks != 0 && IsntErr( err ) )
		{
			PGPUInt32	availOutBlocks;
			
			availOutBlocks = ( context->outputBuffer.size -
								context->outputBuffer.pos ) / 512L;
			if( availOutBlocks != 0 )
			{
				PGPUInt32	blocksToEncrypt;
				PGPUInt32	bytesToEncrypt;
				
				blocksToEncrypt = remainingCompBlocks;
				if( blocksToEncrypt > availOutBlocks )
					blocksToEncrypt = availOutBlocks;
				
				bytesToEncrypt = blocksToEncrypt * 512L;
				
				if( context->isEncryptedArchive )
				{
					err = CipherProc_CallEncrypt( context->cipherProcRef, 
							&context->cipherContext, context->encryptedBlockIV,
							blocksToEncrypt, context->encryptBuffer.data +
							context->encryptBuffer.pos, context->outputBuffer.data +
							context->outputBuffer.pos );
				}
				else
				{
					pgpCopyMemory( context->encryptBuffer.data +
							context->encryptBuffer.pos, context->outputBuffer.data +
							context->outputBuffer.pos, bytesToEncrypt );
				}
				
				context->outputBuffer.used 	+= bytesToEncrypt;
				context->outputBuffer.pos 	+= bytesToEncrypt;
				context->encryptBuffer.pos	+= bytesToEncrypt;
				remainingCompBlocks			-= blocksToEncrypt;
				availOutBlocks				-= blocksToEncrypt;
				context->encryptedBlockIV	+= blocksToEncrypt;
			}
		
			if( IsntErr( err ) && availOutBlocks == 0 )
			{
				err = FlushOutputBuffer( context );
			}
		}
		
		/* Handle remaining bytes, if any */
			
		if( IsntErr( err ) )
		{
			PGPUInt32	extraBytes;
			
			extraBytes = context->encryptBuffer.used % 512L;
			if( extraBytes != 0 )
			{
				if( encryptPartialLastBlock )
				{
					PGPByte		lastBlock[512];
					PGPUInt32	bytesToEncrypt = sizeof( lastBlock );
					
					pgpClearMemory( lastBlock, sizeof( lastBlock ) );
					pgpCopyMemory( context->encryptBuffer.data + context->encryptBuffer.pos,
							lastBlock, extraBytes );
					
					if( context->isEncryptedArchive )
					{	
						err = CipherProc_CallEncrypt( context->cipherProcRef, 
								&context->cipherContext, context->encryptedBlockIV++,
								1, lastBlock, context->outputBuffer.data +
								context->outputBuffer.pos );
					}
					else
					{
						pgpCopyMemory( lastBlock, context->outputBuffer.data +
								context->outputBuffer.pos, bytesToEncrypt );
					}
					
					context->outputBuffer.used 	+= bytesToEncrypt;
					context->outputBuffer.pos 	+= bytesToEncrypt;
							
					extraBytes = 0;
				}
				else
				{
					/* Move extra bytes to the beginning of the buffer */
					
					pgpCopyMemory( context->encryptBuffer.data + context->encryptBuffer.pos,
							context->encryptBuffer.data, extraBytes );
				}
			}

			context->encryptBuffer.pos 	= extraBytes;
			context->encryptBuffer.used	= extraBytes;
		}
	}
	
	return( err );
}

	static OSStatus
FlushCompressionBuffer(PGPsdaEncryptionContext *context)
{
	OSStatus	err = noErr;
	
	if( context->compressBuffer.used != 0 )
	{
		PGPUInt32				bytesUsed;
		PGPsdaCompBlockHeader	*header;
		PGPUInt32				compressedSize;
		
		/* We have a data to compress. Due to the fact that we're compressing
		** the data in chunks rather than as a single stream, we must write
		** out to the data stream the size of the compressed data, which is
		** not known until the compression is done. Thus, we reserve a 4-byte
		** value in the output stream for this value, compress the data, and
		** then overwrite this value. Is is assume that the buffer containing
		** the output of the compressed data is larg enough to hold the entire
		** compressed data stream for a full input buffer
		*/
		
		/* Encrypt buffer should be empty i.e. less than a block of data remaining */
		pgpAssert( context->encryptBuffer.used < 512L );
		
		header = (PGPsdaCompBlockHeader *) ( context->encryptBuffer.data +
						context->encryptBuffer.used );
		context->encryptBuffer.used	+= sizeof( *header );
		context->encryptBuffer.pos	= context->encryptBuffer.used;
		
		context->compressBuffer.pos = 0;
		
		bytesUsed = context->encryptBuffer.used;
		
		lzss_compress( context );
		
		header->magic 		= 'dave';
		
		compressedSize = context->encryptBuffer.used - bytesUsed;
		if( compressedSize >= context->compressBuffer.used )
		{
			/* Data actually grew. Just copy the bytes */
			
			header->blockSize 		= context->compressBuffer.used;
			header->notCompressed	= TRUE;
			
			pgpCopyMemory( context->compressBuffer.data, context->encryptBuffer.data +
						bytesUsed, header->blockSize );
						
			context->encryptBuffer.used = bytesUsed + header->blockSize;
			context->encryptBuffer.pos	= context->encryptBuffer.used;
		}
		else
		{
			header->blockSize 		= compressedSize;
			header->notCompressed	= FALSE;
		}
		
		err = FlushEncryptBuffer( context, FALSE );
		
		context->compressBuffer.used 	= 0;
		context->compressBuffer.pos		= 0;
	}
	
	return( err );
}


	static OSStatus
WriteToCompressor(
	PGPsdaEncryptionContext *context,
	const void				*data,
	PGPSize					dataSize)
{
	PGPUInt32	bytesRemaining 	= dataSize;
	PGPByte		*curData		= (PGPByte *) data;
	OSStatus	err 			= noErr;
	
	while( bytesRemaining != 0 && IsntErr( err ) )
	{
		PGPUInt32	bytesAvail;
		
		/* Add bytes to buffer, which may or may not fill the buffer */
		
		bytesAvail = context->compressBuffer.size -
							context->compressBuffer.used;
		if( bytesAvail != 0 )
		{
			PGPUInt32	bytesToCopy = bytesRemaining;
			
			if( bytesToCopy > bytesAvail )
				bytesToCopy = bytesAvail;
				
			pgpCopyMemory( curData, context->compressBuffer.data +
							context->compressBuffer.pos, bytesToCopy );
					
			context->compressBuffer.used 	+= bytesToCopy;
			context->compressBuffer.pos 	+= bytesToCopy;
			curData							+= bytesToCopy;
			bytesRemaining 					-= bytesToCopy;
		}
		
		if( context->compressBuffer.used == context->compressBuffer.size )
		{
			err = FlushCompressionBuffer( context );
		}
	}
	
	if( IsntErr( err ) )
	{
		context->progressInfo.state 			= kPGPsdaEncryptState_Encrypting;
		context->progressInfo.bytesProcessed 	+= dataSize;
	
		err = CallProgressProc( context );
	}
	
	return( err );
}

	int
compress_getc_buffer(void *pUserValue)
{
	PGPsdaEncryptionContext	*context = (PGPsdaEncryptionContext *) pUserValue;
	int						result;
	
	pgpAssert( context->compressBuffer.used <= context->compressBuffer.size );
	pgpAssert( context->compressBuffer.pos <= context->compressBuffer.used );
	
	if( context->compressBuffer.pos < context->compressBuffer.used )
	{
		result = context->compressBuffer.data[context->compressBuffer.pos++];
	}
	else
	{
		result = EOF;
	}
	
	return( result );
}

	int
compress_putc_buffer(
	int 	invoer,
	void 	*pUserValue)
{
	PGPsdaEncryptionContext	*context = (PGPsdaEncryptionContext *) pUserValue;
	
	pgpAssert( context->encryptBuffer.pos == context->encryptBuffer.used );
	pgpAssert( context->encryptBuffer.used < context->encryptBuffer.size );
	
	context->encryptBuffer.data[context->encryptBuffer.pos] = invoer;
	
	context->encryptBuffer.pos++;
	context->encryptBuffer.used++;

	return( 0 );
}

	static OSStatus
FlushBuffers(PGPsdaEncryptionContext *context)
{
	OSStatus	err = noErr;
	
	err = FlushCompressionBuffer( context );
	if( IsntErr( err ) )
	{
		err = FlushEncryptBuffer( context, TRUE );
		if( IsntErr( err ) )
		{
			err = FlushOutputBuffer( context );
		}
	}

	return( err );
}

	static OSStatus
EncryptFileFork(
	PGPsdaEncryptionContext *context,
	short					srcFileRef,
	PGPUInt32				forkLength)
{
	OSStatus	err 			= noErr;
	PGPUInt32	bytesRemaining	= forkLength;
	
	while( bytesRemaining != 0 && IsntErr( err ) )
	{
		PGPUInt32	bytesToRead;
		long		bytesRead;
		
		bytesToRead = bytesRemaining;
		if( bytesToRead > context->inputBuffer.size )
			bytesToRead = context->inputBuffer.size;
			
		bytesRead = bytesToRead;
		err = FSRead( srcFileRef, &bytesRead, context->inputBuffer.data );
		if( IsntErr( err ) )
		{
			err = WriteToCompressor( context, context->inputBuffer.data,
							bytesRead );
		}
	
		bytesRemaining -= bytesRead;
	}
	
	return( err );
}

	static OSStatus
EncryptFile(
	PGPsdaEncryptionContext *context,
	const FSSpec			*fileSpec,
	const CInfoPBRec		*fileCPB)
{
	OSStatus			err;
	PGPsdaFileHeader	header;
	PGPsdaHeaderType	headerType = kPGPsdaHeaderType_File;
	
	context->progressInfo.curItem = fileSpec;
	
	err = WriteToCompressor( context, &headerType, sizeof( headerType ) );
	if( IsntErr( err ) )
	{
		pgpClearMemory( &header, sizeof( header ) );
	
		CopyPString( fileSpec->name, header.name );
		
		header.fileType				= cpbFileType( fileCPB );
		header.fileCreator			= cpbFileCreator( fileCPB );
		header.finderFlags			= cpbFileFlags( fileCPB );
		header.modificationDate		= cpbModificationDate( fileCPB );
		header.creationDate			= cpbCreationDate( fileCPB );
		header.resForkLength		= cpbResForkSize( fileCPB );
		header.dataForkLength		= cpbDataForkSize( fileCPB );
		header.windowIconPos		= cpbFInfo( fileCPB ).fdLocation;
		header.windowID				= cpbFInfo( fileCPB ).fdFldr;
		header.scriptCode			= fileCPB->hFileInfo.ioFlXFndrInfo.fdScript;
		header.extendedFinderFlags	= fileCPB->hFileInfo.ioFlXFndrInfo.fdXFlags;
		
		if( cpbIsLocked( fileCPB ) )
			header.locked = TRUE;
	
		err = WriteToCompressor( context, &header, sizeof( header ) );
	}
	
	if( IsntErr( err ) )
	{
		short	fileRef;
		
		if( header.dataForkLength != 0 )
		{
			err = FSpOpenDF( fileSpec, fsRdPerm, &fileRef );
			if( IsntErr( err ) )
			{
				err = EncryptFileFork( context, fileRef, header.dataForkLength );
				
				FSClose( fileRef );
			}
		}
		
		if( IsntErr( err ) && header.resForkLength != 0 )
		{
			err = FSpOpenRF( fileSpec, fsRdPerm, &fileRef );
			if( IsntErr( err ) )
			{
				err = EncryptFileFork( context, fileRef,
						header.resForkLength );
				
				FSClose( fileRef );
			}
		}
	}
	
	return( err );
}

	static OSStatus
EncryptFolder(
	PGPsdaEncryptionContext *context,
	const FSSpec			*folderSpec,
	const CInfoPBRec		*folderCPB)
{
	OSStatus			err;
	PGPsdaHeaderType	headerType = kPGPsdaHeaderType_BeginFolder;

	err = WriteToCompressor( context, &headerType, sizeof( headerType ) );
	if( IsntErr( err ) )
	{
		PGPsdaBeginFolderHeader	beginHeader;
		
		pgpClearMemory( &beginHeader, sizeof( beginHeader ) );
		
		CopyPString( folderSpec->name, beginHeader.name );
		
		beginHeader.creationDate 		= folderCPB->dirInfo.ioDrCrDat;
		beginHeader.folderInfo 			= folderCPB->dirInfo.ioDrUsrWds;
		beginHeader.scriptCode 			= folderCPB->dirInfo.ioDrFndrInfo.frScript;
		beginHeader.extendedFlags 		= folderCPB->dirInfo.ioDrFndrInfo.frXFlags;
		
		err = WriteToCompressor( context, &beginHeader, sizeof( beginHeader ) );
	}
	
	if( IsntErr( err ) )
	{
		CInfoPBRec	cpb;
		FSSpec		itemSpec;
		
		itemSpec.vRefNum	= folderSpec->vRefNum;
		itemSpec.parID		= folderCPB->dirInfo.ioDrDirID;
		
		pgpClearMemory( &cpb, sizeof( cpb ) );
		
		cpb.dirInfo.ioVRefNum	= itemSpec.vRefNum;
		cpb.dirInfo.ioDrDirID	= itemSpec.parID;
		cpb.dirInfo.ioNamePtr	= itemSpec.name;
		cpb.dirInfo.ioFDirIndex	= 1;
		
		while( TRUE )
		{
			err = PBGetCatInfoSync( &cpb );
			if( IsntErr( err ) )
			{
				if( cpbIsFile( &cpb ) )
				{
					err = EncryptFile( context, &itemSpec, &cpb );
				}
				else
				{
					err = EncryptFolder( context, &itemSpec, &cpb );
				}
			
				cpb.dirInfo.ioDrDirID = itemSpec.parID;
				++cpb.dirInfo.ioFDirIndex;
			}
			else
			{
				if( err == fnfErr )
					err = noErr;
					
				break;
			}
		}
	}
	
	if( IsntErr( err ) )
	{
		headerType = kPGPsdaHeaderType_EndFolder;
		
		err = WriteToCompressor( context, &headerType, sizeof( headerType ) );
		if( IsntErr( err ) )
		{
			PGPsdaEndFolderHeader	endHeader;
			
			endHeader.modificationDate 	= folderCPB->dirInfo.ioDrMdDat;
			endHeader.windowScroll		= folderCPB->dirInfo.ioDrFndrInfo.frScroll;
			
			err = WriteToCompressor( context, &endHeader, sizeof( endHeader ) );
		}
	}
	
	return( err );
}

	static OSStatus
EncryptSourceItems(PGPsdaEncryptionContext *context)
{
	OSStatus	err = noErr;
	
	context->progressInfo.state 				= kPGPsdaEncryptState_DoneAnalyzing;
	context->progressInfo.totalBytesToProcess	= context->totalBytes;
	
	err = CallProgressProc( context );
	if( err != noErr )
		return( err );
		
	context->progressInfo.state = kPGPsdaEncryptState_Encrypting;
	
	err = FSpOpenDF( &context->outFileSpec, fsRdWrPerm, &context->outFileRef );
	if( IsntErr( err ) )
	{
		PGPUInt32			sourceIndex;
		PGPsdaArchiveHeader	header;
		long				count;
		
		pgpClearMemory( &header, sizeof( header ) );
		
		SetEOF( context->outFileRef, 0 );
		
		/* Write placeholder header */
		count = sizeof( header );
		err = FSWrite( context->outFileRef, &count, &header );
		
		for( sourceIndex = 0; sourceIndex < context->numSourceItems;
				sourceIndex++ )
		{
			CInfoPBRec		cpb;
			const FSSpec	*sourceItem = &context->sourceItems[sourceIndex];
			
			err = FSpGetCatInfo( sourceItem, &cpb );
			if( IsntErr( err ) )
			{
				if( cpbIsFile( &cpb ) )
				{
					err = EncryptFile( context, sourceItem, &cpb );
				}
				else
				{
					err = EncryptFolder( context, sourceItem, &cpb );
				}
			}
			
			if( IsErr( err ) )
				break;
		}
		
		if( IsntErr( err ) )
		{
			PGPsdaHeaderType	headerType = kPGPsdaHeaderType_EndArchive;

			err = WriteToCompressor( context, &headerType, sizeof( headerType ) );
			if( IsntErr( err ) )
			{
				/* Flush our "pipeline" */
			
				err = FlushBuffers( context );
			}
		}
		
		if( IsntErr( err ) && context->isEncryptedArchive )
		{
			err = ComputeCheckBytes( context->cipherProcRef,
						&context->cipherContext, &context->encryptionKey,
						&header.checkBytes );
		}
		
		if( IsntErr( err ) )
		{
			long		fileSize;
			SHA::Digest	digest;
			
			/* Write real header */
			
			SetFPos( context->outFileRef, fsFromStart, 0 );
			GetEOF( context->outFileRef, &fileSize );
			
			header.magic				= kPGPsdaArchiveMagic;
			header.fileSize				= fileSize;
			header.compressedBlockSize	= kPGPsdaCompressionBlockSize;
			header.isEncryptedArchive	= context->isEncryptedArchive;
			
			if( context->isEncryptedArchive )
			{
				header.hashReps	= context->hashReps;
			
				pgpAssert( sizeof( context->cipherContext.salt.saltBytes ) ==
						sizeof( header.passphraseSalt ) );
						
				pgpCopyMemory( context->cipherContext.salt.saltBytes,
						header.passphraseSalt, sizeof( header.passphraseSalt ) );
			}
					
			context->hash->Final( &digest );
			pgpCopyMemory( &digest, header.encryptedDataHash,
					sizeof( header.encryptedDataHash ) );
			
			count = sizeof( header );
			err = FSWrite( context->outFileRef, &count, &header );
		}
		
		FSClose( context->outFileRef );
		context->outFileRef = 0;
	}
	
	context->progressInfo.state = kPGPsdaEncryptState_DoneEncrypting;
	(void) CallProgressProc( context );
	
	return( err );
}

	static OSStatus
CreateArchive(PGPsdaEncryptionContext *context)
{
	OSStatus	err = noErr;
	CInfoPBRec	cpb;
	
	/* Delete output file if it exists */
	if( FSpGetCatInfo( &context->outFileSpec, &cpb ) == noErr )
	{
		err = FSpDelete( &context->outFileSpec );
	}
	
	/* Create our archive in the Temporary Items folder */
	
	if( IsntErr( err ) )
	{
		FSSpec		realOutFileSpec = context->outFileSpec;

		err = FindFolder( realOutFileSpec.vRefNum, kTemporaryFolderType,
					kCreateFolder, &context->outFileSpec.vRefNum,
					&context->outFileSpec.parID );
		if( IsntErr( err ) )
		{
			OSType	fileCreator;
			
			(void) FSpDelete( &context->outFileSpec );
			
			if( context->isEncryptedArchive )
			{
				fileCreator = kPGPsdaCreator;			
			}
			else
			{
				fileCreator = kPGPseaCreator;			
			}
			
			err = FSpCreate( &context->outFileSpec, fileCreator, 'APPL',
						smSystemScript );
			if( IsntErr( err ) )
			{
				err = CopySDAResources( context );
				if( IsntErr( err ) )
				{
					err = EncryptSourceItems( context );
				}
			
				if( IsntErr( err ) )
				{
					FInfo	fInfo;
					
					/* Set appropriate Finder flags */
					
					(void) FSpGetFInfo( &context->outFileSpec, &fInfo );
						fInfo.fdLocation.h 	= -1;	// Magic icon placement
						fInfo.fdLocation.v 	= -1;	// Magic icon placement
						fInfo.fdFlags		&= ~kHasBeenInited;
						fInfo.fdFlags		|= kHasBundle;
					(void) FSpSetFInfo( &context->outFileSpec, &fInfo );

					/* Move our temporary item to the destination */
					err = CatMove( context->outFileSpec.vRefNum, context->outFileSpec.parID,
								context->outFileSpec.name, realOutFileSpec.parID, NULL );
				}
				
				if( IsErr( err ) )
				{
					FSpDelete( &context->outFileSpec );
				}
			}
		}
	}
	
	return( err );
}

	OSStatus
PGPsdaEncrypt(
	PGPContextRef	context,
	PGPUInt32 		numSourceItems,
	const FSSpec 	*sourceItems,
	const FSSpec 	*outFileSpec,
	const char		*passphrase,
	PGPsdaEncryptProgressProc progressProc,
	PGPUserValue 	userValue)
{
	OSStatus					err;
	PGPsdaEncryptionContext		sdaContext;
	
	pgpAssert( numSourceItems != 0 );
	pgpAssert( IsntNull( sourceItems ) );
	pgpAssert( IsntNull( outFileSpec ) );
	pgpAssert( IsntNull( passphrase ) );
	
	/* Lock down the context because it contains sensitive items */
	if ( VirtualMemoryIsOn() )
	{
		HoldMemory( &sdaContext, sizeof( sdaContext ) );
	}
	
	pgpClearMemory( &sdaContext, sizeof( sdaContext ) );

	sdaContext.pgpContext		= context;
	sdaContext.numSourceItems 	= numSourceItems;
	sdaContext.sourceItems		= sourceItems;
	sdaContext.outFileSpec		= *outFileSpec;
	sdaContext.passphrase		= passphrase;
	sdaContext.progressProc		= progressProc;
	sdaContext.userValue		= userValue;
	sdaContext.encryptedBlockIV	= 1;
	
	err = BeginSDAEncryption( &sdaContext );
	if( IsntErr( err ) )
	{
		sdaContext.progressInfo.state = kPGPsdaEncryptState_Starting;

		err = CallProgressProc( &sdaContext );
		if( err == noErr )
		{
			err = AnalyzeSourceItems( &sdaContext );
			if( err == noErr )
			{
				err = CreateArchive( &sdaContext );
			}
		}
		
		EndSDAEncryption( &sdaContext );
	}

	pgpClearMemory( &sdaContext, sizeof( sdaContext ) );
	
	if ( VirtualMemoryIsOn() )
	{
		UnholdMemory( &sdaContext, sizeof( sdaContext ) );
	}

	pgpAssert( IsntErr( err ) );
	
	return( err );
}