/*
     Filter - Packet filtering software for the Drawbridge package
     Copyright (C) 1993 David K. Hess, Douglas Lee Schales, David R. Safford

     Please see the file `COPYING' for the complete copyright notice.

     FILTER.C - Version 1.0 - 4/21/93
*/


// Heart of the filter program.
//
#include <dos.h>
#include <io.h>
#include <alloc.h>
#include <process.h>
#include <stdlib.h>
#include <stdio.h>
#include <conio.h>
#include <mem.h>
#include <string.h>
#include <fcntl.h>
#include <sys\stat.h>
#include <dir.h>
#include "filter.h"
#include "wd.h"

// Filter data structures.
AddrTableEntry addrTable[MAX_NUM_NETWORKS];
AccessListTableEntry *in = (AccessListTableEntry *) NULL;
AccessListTableEntry *out = (AccessListTableEntry *) NULL;
AccessListTableEntry *source = (AccessListTableEntry *) NULL;
AccessListTableEntry *udp = (AccessListTableEntry *) NULL;
RejectTableEntry rejectTable[MAX_NUM_REJECT_ENTRIES];
AllowTableEntry allowTable[MAX_NUM_ALLOW_ENTRIES];

// Temp data structures for things that are loaded in more
//   than one packet.
AddrTableEntry newAddrTable[MAX_NUM_NEW_NETWORKS];
AccessListTableEntry *newIn = (AccessListTableEntry *) NULL;
AccessListTableEntry *newOut = (AccessListTableEntry *) NULL;
AccessListTableEntry *newSource = (AccessListTableEntry *) NULL;
AccessListTableEntry *newUdp = (AccessListTableEntry *) NULL;

// Boolean variables to tell if the data structures are dirty
//   and need to be written to disk.
int accessTableDirty = NO;
int rejectTableDirty = NO;
int allowTableDirty = NO;

// Ethernet table for bridging.
EtherHashEntry *etherHashTable = (EtherHashEntry *) NULL;

// Pointers to the structures defining the two interfaces.
WDInterface *campus = (WDInterface *) NULL;
WDInterface *internet = (WDInterface *) NULL;

// The kind of listening that the filter is set to.
int listenMode = 0;

// A buffer for constructing a linear version of an incoming
//   packet. (Needed since the buffers on the cards can wrap
//   around.)
unsigned char currPacket[1536];

// If a reboot command comes in, this variable will be set and
//   a reboot will occur once the reboot ack has been delivered.
unsigned char rebootRequested = NO;

// Buffer for the des key and status variable.
unsigned char desKey[128];
int desKeyLoaded = NO;

int checkIncomingTcp(in_addr srcAddr,in_addr dstAddr,
		unsigned short srcPort,unsigned short dstPort)
{
    int i;
    int result;
    int curr;
    int accessIndex;
    unsigned long hash;
    unsigned long host;
    in_addr network;
    AccessListTableEntry *accessList;

    //fprintf(stdout,"incoming SYN\n");

    result = NO;

    // fprintf(stdout,"%08lX ",srcAddr.S_addr);

    SWAP_LONG(srcAddr.S_addr);

    // fprintf(stdout,"%08lX ",srcAddr.S_addr);
    // fprintf(stdout,"%08lX ",dstAddr.S_addr);

    SWAP_LONG(dstAddr.S_addr);

    // fprintf(stdout,"%08lX\n",dstAddr.S_addr);

    SWAP_WORD(srcPort);
    SWAP_WORD(dstPort);

    // Do the heavy duty checking.

    // Get the network and host from the address.
    if (IN_CLASSB(dstAddr.S_addr)) {
	 // Class B address.
	 network.S_addr = dstAddr.S_addr & CLASSB_NETWORK;
	 host = dstAddr.S_addr & CLASSB_HOST;
    }
    else if (IN_CLASSC(dstAddr.S_addr))
    {
	 // Class C address.
	 network.S_addr = dstAddr.S_addr & CLASSC_NETWORK;
	 host = dstAddr.S_addr & CLASSC_HOST;
    }
    else {
	// We don't handle class A addresses.
	result = YES;

	// fprintf(stdout,"Class screwup\n");
	return result;
    }

    // fprintf(stdout,"host = %lX,%lu\n",host,host);

    // Hash into the table and see if the network has been defined.
    hash = (network.S_addr & NETWORK_HASH_MASK) >> 19;
    curr = (int) hash;

    while (addrTable[curr].network.S_addr != network.S_addr) {
	if (addrTable[curr].network.S_addr == 0) {
	    curr = -1;
	    break;
	}

	curr = (curr + 1) % MAX_NUM_NETWORKS;

	// Check if we have wrapped around.
	if (curr == hash) {
	    curr = -1;
	    break;
	}
    }

    // If not defined then default to access list 0.
    if (curr == -1) {
	accessIndex = 0;
	// fprintf(stdout,"Network not in hash table\n");
    }
    else {
	// Else use the specified access list index in the host
	//   table.

	accessIndex = addrTable[curr].hostTable[(unsigned short) host];
	// fprintf(stdout,"Access index = %d\n",accessIndex);
    }

    accessList = in + accessIndex * MAX_NUM_ACCESS_RANGES;

    // See if the destination port is allowed. Search the in access
    //   list.
    i = 0;
    while (dstPort > accessList[i].end) {

	if (accessList[i].begin == 0) {
	    i = -1;
	    break;
	}
	++i;
    }

    if (i != -1 && dstPort >= accessList[i].begin) {
	// fprintf(stdout,"permission allowed\n");
	result = YES;
    }
    else if (dstPort > 900) {

	//fprintf(stdout,"checking source\n");

	// If the destination port is not allowed then check the source
	//   port.
	accessList = source + accessIndex * MAX_NUM_ACCESS_RANGES;

	i = 0;
	while (srcPort > accessList[i].end) {
	    if (accessList[i].begin == 0) {
		i = -1;
		break;
	    }
	    ++i;
	}

	if (i != -1 && srcPort >= accessList[i].begin) {
	    // fprintf(stdout,"allowed\n");
	    result = YES;
	}
	else {
	    // fprintf(stdout,"not allowed\n");
	}
    }

    //fprintf(stdout,">");
    //fflush(stdout);
    //fprintf(stdout,"result = %d\n",result);

    return result;
}

int checkOutgoingTcp(in_addr srcAddr,in_addr dstAddr,
		unsigned short srcPort,unsigned short dstPort)
{
    int i;
    int j;
    int result;
    int curr;
    int accessIndex;
    unsigned long host;
    unsigned long hash;
    in_addr network;
    AccessListTableEntry *accessList;

    //printf("Checking Outgoing\n");
    result = NO;
    //printf("%08lX ", srcAddr.S_addr);
    SWAP_LONG(srcAddr.S_addr);
    //printf("%08lX ", srcAddr.S_addr);
    SWAP_LONG(dstAddr.S_addr);
    SWAP_WORD(srcPort);
    SWAP_WORD(dstPort);

    //fprintf(stdout,"dstPort = %d\n",dstPort);

    // Do the heavy duty checking.
    if (IN_CLASSB(srcAddr.S_addr)) {
	 // Class B address.
	 //fprintf(stdout,"Class B\n");
	 network.S_addr = srcAddr.S_addr & CLASSB_NETWORK;
	 host = srcAddr.S_addr & CLASSB_HOST;
    }
    else if (IN_CLASSC(srcAddr.S_addr))
    {
	 // Class C address.
	 //fprintf(stdout,"Class C\n");
	 network.S_addr = srcAddr.S_addr & CLASSC_NETWORK;
	 host = srcAddr.S_addr & CLASSC_HOST;
    }
    else {
	// We don't handle class A addresses.
	//fprintf(stdout,"Class ARGH!\n");
	result = YES;
	return result;
    }

    // Hash into the table and see if the network has been defined.
    hash = (network.S_addr & NETWORK_HASH_MASK) >> 19;
    curr = (int) hash;

    while (addrTable[curr].network.S_addr != network.S_addr) {
	if (addrTable[curr].network.S_addr == 0) {
	    curr = -1;
	    break;
	}

	curr = (curr + 1) & (MAX_NUM_NETWORKS - 1);

	if (curr == hash) {
	    curr = -1;
	    break;
	}
    }

    // If not defined then default to access list 0.
    if (curr == -1) {
	//fprintf(stdout,"Network not present in hash table\n");
	accessIndex = 0;
    }
    else {
	// Else use the specified access list index in the host
	//   table.
	//fprintf(stdout,"table at %04X:%04X - host = %d\n",
	//      FP_SEG(addrTable[curr].hostTable),
	//      FP_OFF(addrTable[curr].hostTable),
	//      host);
	accessIndex = addrTable[curr].hostTable[(unsigned short) host];
	//fprintf(stdout,"Network found - access list = %d\n",accessIndex);
    }

    // See if the destination port is allowed. Search the in access
    //   list.
    accessList = out + accessIndex * MAX_NUM_ACCESS_RANGES;
    i = 0;
    while (dstPort > accessList[i].end) {
	//fprintf(stdout,"end = %d\n",accessList[i].end);
	if (accessList[i].begin == 0) {
	    i = -1;
	    break;
	}
	++i;
    }

    if (i != -1 && dstPort >= accessList[i].begin) {
	//fprintf(stdout,"Attempt is permitted\n");
	result = YES;
    }
    else {
	//fprintf(stdout,"Attempt is not permitted\n");
    }

    // Now if still not allowed check the allow list.
    if (result == NO) {
	//fprintf(stdout,"Checking allow\n");

	// Check the outgoing packet to see if the destination is on the
	//   allow list.
	i = 0;

	while (allowTable[i].network.S_addr != 0 &&
	       i < MAX_NUM_ALLOW_ENTRIES) {

	    if ((allowTable[i].network.S_addr & allowTable[i].mask) ==
		(dstAddr.S_addr & allowTable[i].mask))
		break;

	    ++i;
	}

	//fprintf(stdout,"finished while i = %d\n",i);

	if (i < MAX_NUM_ALLOW_ENTRIES && allowTable[i].network.S_addr != 0) {
	    //fprintf(stdout,"going in if\n");

	    // Now check to see if the destination port is allowed
	    //   in the access list.
	    j = 0;

	    while (dstPort > allowTable[i].access[j].end) {
		//fprintf(stdout,"end = %d\n",allowTable[i].access[j].end);
		if (allowTable[i].access[j].begin == 0) {
		    j = -1;
		    break;
		}

		++j;
	    }

	    if (j != -1 && dstPort >= allowTable[i].access[j].begin) {
		//fprintf(stdout,"Attempt permitted via allow\n");
		result = YES;
	    }
	    else {
		//fprintf(stdout,"Attempt not permitted via allow\n");
	    }
	}
    }

    //fprintf(stdout,"<");
    //fflush(stdout);
    //fprintf(stdout,"result = %d\n",result);

    return result;
}

int checkIncomingUdp(in_addr srcAddr,in_addr dstAddr,
		unsigned short srcPort,unsigned short dstPort)
{
    int i;
    int result;
    int curr;
    int accessIndex;
    unsigned long hash;
    unsigned long host;
    in_addr network;
    AccessListTableEntry *accessList;

    result = NO;

    SWAP_LONG(srcAddr.S_addr);
    SWAP_LONG(dstAddr.S_addr);
    SWAP_WORD(srcPort);
    SWAP_WORD(dstPort);

    // Get the network and host from the address.
    if (IN_CLASSB(dstAddr.S_addr)) {
	 // Class B address.
	 network.S_addr = dstAddr.S_addr & CLASSB_NETWORK;
	 host = dstAddr.S_addr & CLASSB_HOST;
    }
    else if (IN_CLASSC(dstAddr.S_addr))
    {
	 // Class C address.
	 network.S_addr = dstAddr.S_addr & CLASSC_NETWORK;
	 host = dstAddr.S_addr & CLASSC_HOST;
    }
    else {
	// We don't handle class A or D addresses.
	result = YES;

	// fprintf(stdout,"Class screwup\n");
	return result;
    }

    // fprintf(stdout,"host = %lX,%lu\n",host,host);

    // Hash into the table and see if the network has been defined.
    hash = (network.S_addr & NETWORK_HASH_MASK) >> 19;
    curr = (int) hash;

    while (addrTable[curr].network.S_addr != network.S_addr) {
	if (addrTable[curr].network.S_addr == 0) {
	    curr = -1;
	    break;
	}

	curr = (curr + 1) % MAX_NUM_NETWORKS;

	// Check if we have wrapped around.
	if (curr == hash) {
	    curr = -1;
	    break;
	}
    }

    // If not defined then default to access list 0.
    if (curr == -1) {
	accessIndex = 0;
	// fprintf(stdout,"Network not in hash table\n");
    }
    else {
	// Else use the specified access list index in the host
	//   table.

	accessIndex = addrTable[curr].hostTable[(unsigned short) host];
	// fprintf(stdout,"Access index = %d\n",accessIndex);
    }

    accessList = udp + accessIndex * MAX_NUM_ACCESS_RANGES;

    // See if the destination port is allowed. Search the in access
    //   list.
    i = 0;
    while (dstPort > accessList[i].end) {

	if (accessList[i].begin == 0) {
	    i = -1;
	    break;
	}
	++i;
    }

    if (i != -1 && dstPort >= accessList[i].begin) {
	// fprintf(stdout,"permission allowed\n");
	result = YES;
    }

    return result;
}

int checkOutgoingUdp(in_addr srcAddr,in_addr dstAddr,
		unsigned short srcPort,unsigned short dstPort)
{
    int result;

    SWAP_LONG(srcAddr.S_addr);
    SWAP_LONG(dstAddr.S_addr);
    SWAP_WORD(srcPort);
    SWAP_WORD(dstPort);

    // Allow all UDP out for now.
    result = YES;

    return result;
}

int checkIncomingPacket(unsigned char *packet,int length,WDInterface *interface)
{
    int i;
    int result;
    EtherHeader *etherHead;
    IpHeader *ipHeader;
    TcpHeader *tcpHeader;
    UdpHeader *udpHeader;
    in_addr srcAddr;


    //fprintf(stdout,"check in\n");

    result = YES;

    etherHead = (EtherHeader *) packet;

    switch (etherHead->etherType) {
	case IPPROT:
	    ipHeader = (IpHeader *) (((unsigned char *) etherHead) +
				     sizeof(EtherHeader));


/*          for (i = 0;i < 20;++i) {
		fprintf(stdout," %02X",packet[i]);
	    }
	    fprintf(stdout,"\n");

	    fprintf(stdout,"size = %d\n",sizeof(IpHeader));
	    fprintf(stdout,"off = %d\n",ipHeader->ip_off);
	    fprintf(stdout,"hl = %d\n",ipHeader->ip_hl);
	    fprintf(stdout,"version = %d\n",ipHeader->ip_v);
	    fprintf(stdout,"proto = %d\n",(int) ipHeader->ip_p);
	    fprintf(stdout,"ttl = %d\n",(int) ipHeader->ip_ttl);
	    fprintf(stdout,"len = %d\n",(int) ipHeader->ip_len);

*/

	    // Pass all IP fragments that do not have offset 0 (beginning
	    //   of the packet) without checking since the TCP/UDP
	    //   headers are not in this packet.
	    if ((ipHeader->ip_off & IP_OFF_MASK) != 0) {
		//fprintf(stdout,"offset fragment\n");
		return YES;
	    }

	    srcAddr = ipHeader->ip_src;
	    SWAP_LONG(srcAddr.S_addr);

	    // Check the incoming packet to see if the src is on the
	    //   reject list.
	    i = 0;

	    while (rejectTable[i].network.S_addr != 0 &&
		   i < MAX_NUM_REJECT_ENTRIES) {

		if ((rejectTable[i].network.S_addr & rejectTable[i].mask) ==
		    (srcAddr.S_addr & rejectTable[i].mask))
		    break;

		++i;
	    }

	    if (i < MAX_NUM_REJECT_ENTRIES && rejectTable[i].network.S_addr != 0) {
		return NO;
	    }

	    switch (ipHeader->ip_p) {
		case TCPPROT:
		    tcpHeader = (TcpHeader *) (((unsigned char *) ipHeader) +
					       ipHeader->ip_hl * 4);

		    //fprintf(stdout,"ip = %08lX tcp = %08lX ",ipHeader,tcpHeader);

		    //for (i = 0;i < 20;++i) {
		    //  fprintf(stdout,"%02X ",((unsigned char *) tcpHeader)[i]);
		    //}
		    //fprintf(stdout,"\n");

		    // Check for "ACKless SYN".
		    if ((tcpHeader->th_flags & (TH_SYN | TH_ACK)) == TH_SYN)
			result = checkIncomingTcp(ipHeader->ip_src,ipHeader->ip_dst,
					tcpHeader->th_sport,tcpHeader->th_dport);

		    break;
		case UDPPROT:
		    // Check UDP protocols.
		    udpHeader = (UdpHeader *) (((unsigned char *) ipHeader) +
					       ipHeader->ip_hl * 4);

		    result = checkIncomingUdp(ipHeader->ip_src,ipHeader->ip_dst,
				udpHeader->uh_sport,udpHeader->uh_dport);

		    break;
		default:
		    // Everything else is allowed so far.
		    break;
	    }

	    break;

	case FILTERPROT:

	    if (listenMode & OUTSIDE_MASK) {
		//fprintf(stdout,"filter message from outside\n");

		// Put the message into the buffer so that it is
		//   contiguous.
		WDMove(length,packet,interface,currPacket);

		// Handle the message.
		filtMessage((EtherHeader *)currPacket,OUTSIDE_MASK);
	    }

	    // Don't forward these packets.
	    result = NO;

	    break;

	default:
	    // Everything else is allowed so far.
	    break;
    }

    return result;
}

int checkOutgoingPacket(unsigned char *packet,int length,WDInterface *interface)
{
    int result;
    EtherHeader *etherHead;
    IpHeader *ipHeader;
    TcpHeader *tcpHeader;
    UdpHeader *udpHeader;

    // fprintf(stdout,"check out\n");

    // exit(0);

    result = YES;

    etherHead = (EtherHeader *) packet;

    switch (etherHead->etherType) {
	case IPPROT:
	    ipHeader = (IpHeader *) (((unsigned char *) etherHead) +
				     sizeof(EtherHeader));

/*
	    fprintf(stdout,"sizeof ipheader = %d\n",sizeof(IpHeader));

	    for (i = 0;i < 32;++i) {
		fprintf(stdout," %02X",packet[i]);
	    }
	    fprintf(stdout,"\n");

	    fprintf(stdout,"offset (going out) = %d\n",ipHeader->ip_off);

	    fprintf(stdout,"IP - 0x%08lX 0x%08lX ",
		    ipHeader->ip_src.S_addr,
		    ipHeader->ip_dst.S_addr);
*/

	    // Pass all IP fragments that do not have offset 0 (beginning
	    //   of the packet) without checking since the TCP/UDP
	    //   headers are not in this packet. An area for improvement
	    //   would be to cache session info so we could drop all
	    //   disallowed fragments also instead of just the first one.
	    if ((ipHeader->ip_off & IP_OFF_MASK) != 0) {
		return YES;
	    }

	    //for (i = 0;i < 34;++i) {
	    //  fprintf(stdout,"%02X ",packet[i]);
	    //}
	    //fprintf(stdout,"\n");

	    switch (ipHeader->ip_p) {
		case TCPPROT:

		    //fprintf(stdout,"size = %d\n",sizeof(TcpHeader));
		    //fprintf(stdout,"TCP\n");
		    //fprintf(stdout,"hl = %d\n",ipHeader->ip_hl);

		    tcpHeader = (TcpHeader *) (((unsigned char *) ipHeader) +
					       ipHeader->ip_hl * 4);

		    //fprintf(stdout," flags = %02X",tcpHeader->th_flags);

		    if ((tcpHeader->th_flags & (TH_SYN | TH_ACK)) == TH_SYN) {
			// fprintf(stdout,"Outgoing SYN\n");
			result = checkOutgoingTcp(ipHeader->ip_src,ipHeader->ip_dst,
					tcpHeader->th_sport,tcpHeader->th_dport);
		    }

		    break;
		case UDPPROT:

		    //fprintf(stdout," UDP -");
		    // Check UDP packets also.
		    udpHeader = (UdpHeader *) (((unsigned char *) ipHeader) +
					       ipHeader->ip_hl * 4);

		    result = checkOutgoingUdp(ipHeader->ip_src,ipHeader->ip_dst,
				udpHeader->uh_sport,udpHeader->uh_dport);

		    break;
		default:
		    // Everything else is allowed so far.
		    break;
	    }

	    break;

	case FILTERPROT:

	    if (listenMode & INSIDE_MASK) {
		//fprintf(stdout,"filter message from inside\n");

		// Put the message into a contiguous buffer.
		WDMove(length,packet,interface,currPacket);

		// Handle the message.
		filtMessage((EtherHeader *)currPacket,INSIDE_MASK);
	    }

	    // Don't forward these packets.
	    result = NO;

	    break;

	default:
	    // Everything else is allowed so far.
	    break;
    }

    //fprintf(stdout,"\n");

    //fprintf(stdout,"result = %d\n",result);

    return result;
}

int bridge(WDInterface *fromInterface,
	   EtherHeader *etherHead)
{
    int hash;
    int curr;
    int pass;

    pass = YES;

    // Check if the destination address is a multicast or broadcast
    //   address and always forward if so.
    if (!(etherHead->etherDestHost.bytes[0] & 0x01)) {

	  // Check the hash table.
	  hash = (etherHead->etherDestHost.words[0] ^
		  etherHead->etherDestHost.words[1] ^
		  etherHead->etherDestHost.words[2]) & ETHERNET_HASH_MASK;

	  curr = hash;

	  while (etherHashTable[curr].address.words[0] !=
		 etherHead->etherDestHost.words[0] ||
		 etherHashTable[curr].address.words[1] !=
		 etherHead->etherDestHost.words[1] ||
		 etherHashTable[curr].address.words[2] !=
		 etherHead->etherDestHost.words[2]) {

	      if (etherHashTable[curr].interface == 0) {
		  curr = -1;
		  break;
	      }

	      curr = (curr + 1) & (MAX_NUM_ETHERS - 1);

	      // Check if we have wrapped around.
	      if (curr == hash) {
		  curr = -1;
		  break;
	      }
	  }

	  // If we found the address then we know what interface the
	  //   address is on. If this packet came from that interface then
	  //   don't forward the packet.
	  if (curr != -1 &&
	      fromInterface->cardID == etherHashTable[curr].interface)
	      pass = NO;
    }

    // Note that this check is somewhat superfluous. But if some broken
    //   implementation sources a broadcast or multicast address we don't
    //   want to add them to the bridge table!
    if (!(etherHead->etherSrcHost.bytes[0] & 0x01)) {

	// Check the source address. If not found then add into hash table.
	hash = (etherHead->etherSrcHost.words[0] ^
		etherHead->etherSrcHost.words[1] ^
		etherHead->etherSrcHost.words[2]) & ETHERNET_HASH_MASK;

	curr = hash;

	while (etherHashTable[curr].address.words[0] !=
	       etherHead->etherSrcHost.words[0] ||
	       etherHashTable[curr].address.words[1] !=
	       etherHead->etherSrcHost.words[1] ||
	       etherHashTable[curr].address.words[2] !=
	       etherHead->etherSrcHost.words[2]) {

	   if (etherHashTable[curr].interface == 0) {
	       // Insert the source ethernet address in the table.
	       etherHashTable[curr].address.addr =
		    etherHead->etherSrcHost.addr;
	       break;
	   }

	   curr = (curr + 1) & (MAX_NUM_ETHERS - 1);

	   // Check if we have wrapped around.
	   if (curr == hash) {
	       curr = -1;
	       break;
	   }
	}

	// If the address was entered OR found then update the interface.
	//   This is necessary if a device ever switches from being on
	//   one side of the filter to the other.
	if (curr != -1)
	    etherHashTable[curr].interface = fromInterface->cardID;
    }

    return pass;
}

void interfaceUpdate(WDInterface *interface)
{
    int length;

    if (!WDIsBusy(interface)) {

	// If the interface just finished transmitting a packet then
	//   mark that packet's buffer as empty and change the status
	//   of the card.
	if (interface->transmitting == YES) {

	    // Set the length for the buffer that finished to 0.
	    if (interface->currBuffer)
		interface->lengthBuf2 = 0;
	    else
		interface->lengthBuf1 = 0;

	    // Update the status of the interface and switch to the other
	    //   buffer.
	    interface->transmitting = NO;
	    interface->currBuffer   = 6 - interface->currBuffer;
	}

	// Start the interface transmitting any packet waiting in the
	//   current buffer. Check if the card is not busy and was not
	//   transmitting a packet.
	if (interface->transmitting == NO) {

	    length = interface->currBuffer ? interface->lengthBuf2 :
					     interface->lengthBuf1;

	    if (length) {
		// There is a packet waiting in the current buffer so start
		//   it.
		WDWrite(interface,length);
		interface->transmitting = YES;
	    }

	    // Else just leave the card idling.
	}
    }
}

void interfaceLoad(WDInterface *toInterface,
		   WDInterface *fromInterface,
		   unsigned char *packet,
		   unsigned short length,
		   unsigned char next)
{
    int currLength;
    int destBuffer;

    // Check if the current buffer is empty. If so put the packet there.
    currLength = toInterface->currBuffer ? toInterface->lengthBuf2 :
					   toInterface->lengthBuf1;

    if (currLength == 0) {
	destBuffer = toInterface->currBuffer;
    }
    else {
	// The current buffer has a packet in it. Check the other buffer.
	currLength = toInterface->currBuffer ? toInterface->lengthBuf1 :
					       toInterface->lengthBuf2;

	if (currLength == 0) {
//          fprintf(stdout,"d");
//          fflush(stdout);
	    destBuffer = 6 - toInterface->currBuffer;
	}
	else {
	    // Both buffers are full. Gonna have to wait until next time.
	    return;
	}
    }

    // The destination buffer is empty. Copy the packet to it.
    WDCopy(length,packet,fromInterface,toInterface,destBuffer);
    WDFree(fromInterface,next);

    if (destBuffer)
	toInterface->lengthBuf2 = length;
    else
	toInterface->lengthBuf1 = length;
}

void bufferLoad(WDInterface *toInterface)
{
    int currLength;
    int destBuffer;
    int length;
    unsigned char *toBuffer;

    // Check if the current buffer is empty. If so put the packet there.
    currLength = toInterface->currBuffer ? toInterface->lengthBuf2 :
					   toInterface->lengthBuf1;

    if (currLength == 0) {
	destBuffer = toInterface->currBuffer;
    }
    else {
	// The current buffer has a packet in it. Check the other buffer.
	currLength = toInterface->currBuffer ? toInterface->lengthBuf1 :
					       toInterface->lengthBuf2;

	if (currLength == 0) {
	    destBuffer = 6 - toInterface->currBuffer;
	}
	else {
	    // Both buffers are full. Gonna have to wait until next time.
	    return;
	}
    }

    // The destination buffer is empty. Copy the packet to it.
    toBuffer = toInterface->bufferAddress + (destBuffer << 8);

    //fprintf(stdout,"address = %08lX\n",toBuffer);

    // Copy the packet to the buffer.
    length = toInterface->packetLength;
    length = (length + 1) >> 1;

    moveWords(length,toBuffer,toInterface->packet);

    //for (i = 0;i < 20;++i) {
    //  fprintf(stdout,"%02X ",toBuffer[i]);
    //}
    //fprintf(stdout,"\n");

    toInterface->packetWaiting = NO;

    if (destBuffer)
	toInterface->lengthBuf2 = toInterface->packetLength;
    else
	toInterface->lengthBuf1 = toInterface->packetLength;

    //fprintf(stdout,"length = %d\n",toInterface->packetLength);
}

void forward(WDInterface *fromInterface,
	     WDInterface *toInterface,
	     int (*checkFunction)(unsigned char *buffer,int length,WDInterface *interface))
{
    register WDPacketHeader *header;
    register unsigned char *packet;
    register unsigned char next;
    register unsigned short length;

    // Check if the interface has finished sending a packet and clean the
    //   status if so.
    interfaceUpdate(toInterface);

    // If we have a packet of our own to send then handle it.
    if (toInterface->packetWaiting == YES) {
	//fprintf(stdout,"sending back a filter packet\n");
	// Copy our own packet over to the interface.
	bufferLoad(toInterface);
    }
    else {
	// Get a packet from fromInterface.
	header = (WDPacketHeader *) WDGetReadAccess(fromInterface);

	// If got a packet.
	if (header != (WDPacketHeader *) NULL) {

	    length = header->length - 4;
	    next   = header->next;
	    packet = ((unsigned char *) header) + 4;

	    // Check if it should be allowed out. This should work since
	    //   my ethernet address will never be "heard" so it will always
	    //   be passed by the bridging, but the checkFunction will kill
	    //   it after the packet is processed.
	    if (bridge(fromInterface,(EtherHeader *)packet) == YES &&
		checkFunction(packet,length,fromInterface) == YES ) {

		// Copy the packet to the card if there is an empty buffer
		//   spot. Note that if there is no room, the packet is
		//   not freed. It stays in the ring buffer on the from
		//   card.
		interfaceLoad(toInterface,fromInterface,
			      packet,length,next);
	    }
	    else {
		// Free the packet.
		WDFree(fromInterface,next);
	    }
	}
    }

    // Check if the interface is idle and if so start it sending any
    //   waiting packets.
    interfaceUpdate(toInterface);
}


void initMemory(void)
{
    unsigned long size;

    // Allocate the access list tables.
    in = (AccessListTableEntry *) farmalloc(MAX_NUM_ACCESS_LISTS *
					 MAX_NUM_ACCESS_RANGES *
					 sizeof(AccessListTableEntry));

    //fprintf(stdout,"in = %04X:%04X\n",FP_SEG(in),FP_OFF(in));

    out = (AccessListTableEntry *) farmalloc(MAX_NUM_ACCESS_LISTS *
					  MAX_NUM_ACCESS_RANGES *
					  sizeof(AccessListTableEntry));

    //fprintf(stdout,"out = %04X:%04X\n",FP_SEG(out),FP_OFF(out));

    source = (AccessListTableEntry *) farmalloc(MAX_NUM_ACCESS_LISTS *
					     MAX_NUM_ACCESS_RANGES *
					     sizeof(AccessListTableEntry));

    udp = (AccessListTableEntry *) farmalloc(MAX_NUM_ACCESS_LISTS *
					     MAX_NUM_ACCESS_RANGES *
					     sizeof(AccessListTableEntry));

    etherHashTable = (EtherHashEntry *) farmalloc(8192UL * sizeof(EtherHashEntry) + 16);
    etherHashTable = (EtherHashEntry *) MK_FP(FP_SEG(etherHashTable) + 1,0);

    // Clean out the address table.
    memset((void *) addrTable,0,sizeof(addrTable));

    // Clear out and initialize the reject table.
    memset((void *) rejectTable,0,sizeof(rejectTable));

    // Clear out and initialize the allow table.
    memset((void *) allowTable,0,sizeof(allowTable));

    memset((unsigned char *) newAddrTable,0,sizeof(newAddrTable));

    // Clean out the access lists.
    memset((void *) in,0,MAX_NUM_ACCESS_LISTS *
			   MAX_NUM_ACCESS_RANGES *
			   sizeof(AccessListTableEntry));

    memset((void *) out,0,MAX_NUM_ACCESS_LISTS *
			   MAX_NUM_ACCESS_RANGES *
			   sizeof(AccessListTableEntry));

    memset((void *) source,0,MAX_NUM_ACCESS_LISTS *
			   MAX_NUM_ACCESS_RANGES *
			   sizeof(AccessListTableEntry));

    memset((void *) udp,0,MAX_NUM_ACCESS_LISTS *
			   MAX_NUM_ACCESS_RANGES *
			   sizeof(AccessListTableEntry));

    // Clean out the ethernet address table. Do it in two steps since
    //   the size argument is an unsigned and our data structure takes
    //   64K.
    size = 8192UL * sizeof(EtherHashEntry) / 2UL;

    memset((void *) etherHashTable,0,(unsigned short) size);
    memset((void *) (((unsigned char *) etherHashTable) + (unsigned short) size),0,(unsigned short) size);

    // Set up the default tables these are all allow lists.
    in[0].begin     = 25;            // Mail
    in[0].end       = 25;
    in[1].begin     = 53;            // Name service
    in[1].end       = 53;
    out[0].begin    = 1;             // Everything
    out[0].end      = 0xFFFF;
    source[0].begin = 20;            // FTP data connections
    source[0].end   = 20;
    udp[0].begin    = 1;             // Disallow TFTP (69) and Portmapper (111).
    udp[0].end      = 68;
    udp[1].begin    = 70;
    udp[1].end      = 110;
    udp[2].begin    = 112;
    udp[2].end      = 0xFFFF;
}

void initTables(void)
{
    int fd;

#ifdef DES
    // Load in the default DES key.
    fd = open(DES_KEY_FILE,O_RDONLY | O_BINARY);

    if (fd != -1) {

	if (read(fd,(char *)desKey,sizeof(desKey)) == -1) {
	    fprintf(stdout,"Error reading in DES key\n");
	    exit(1);
	}

	desKeyLoaded = YES;
	fprintf(stdout,"Loaded DES key\n");

	close(fd);
    }
    else {
	fprintf(stdout,"No DES key found - starting in insecure mode\n");
    }
#else
    fprintf(stdout,"No DES support in this installation of Filter\n");
#endif

    // Load in the reject table.
    fd = open(REJECT_LIST_FILE,O_RDONLY | O_BINARY);

    if (fd != -1) {

	if (read(fd,(char *) rejectTable,sizeof(RejectTableEntry) *
		  MAX_NUM_REJECT_ENTRIES) == -1) {
	    fprintf(stdout,"Error reading in reject table\n");
	    exit(1);
	}

	fprintf(stdout,"Loaded reject table\n");

	close(fd);
    }
    else {
	fprintf(stdout,"No reject table found\n");
    }

    // Load in the allow table.
    fd = open(ALLOW_LIST_FILE,O_RDONLY | O_BINARY);

    if (fd != -1) {

	if (read(fd,(char *) allowTable,sizeof(AllowTableEntry) *
		  MAX_NUM_ALLOW_ENTRIES) == -1) {
	    fprintf(stdout,"Error reading in allow table\n");
	    exit(1);
	}

	fprintf(stdout,"Loaded allow table\n");

	close(fd);
    }
    else {
	fprintf(stdout,"No allow table found\n");
    }

    //fprintf(stdout,"source = %04X:%04X\n",FP_SEG(source),FP_OFF(source));
    //fprintf(stdout,"size = %ld etherHashTale = %04X:%04X\n",
    //               8192UL * sizeof(EtherHashEntry),
    //               FP_SEG(etherHashTable),
    //               FP_OFF(etherHashTable));

    // Load the default configuration. Begin with the access lists.
    fd = open(ACCESS_LIST_FILE,O_RDONLY | O_BINARY);

    if (fd == -1) {

	// If the access tables don't exist then they may end up being
	//   loaded from the net.
	fprintf(stdout,"No class table found\n");
    }
    else {
	if (read(fd,(char *)in,sizeof(AccessListTableEntry) *
		MAX_NUM_ACCESS_LISTS * MAX_NUM_ACCESS_RANGES) == -1) {
	    fprintf(stdout,"Error reading in access table\n");
	    exit(1);
	}

	// fprintf(stdout,"result = %u\n",result);

	if (read(fd,(char *)out,sizeof(AccessListTableEntry) *
			MAX_NUM_ACCESS_LISTS * MAX_NUM_ACCESS_RANGES) == -1) {
	    fprintf(stdout,"Error reading out access table\n");
	    exit(1);
	}

	// fprintf(stdout,"result = %u\n",result);

	if (read(fd,(char *)source,sizeof(AccessListTableEntry) *
		    MAX_NUM_ACCESS_LISTS * MAX_NUM_ACCESS_RANGES) == -1) {
	    fprintf(stdout,"Error reading source access table\n");
	    exit(1);
	}

	if (read(fd,(char *)udp,sizeof(AccessListTableEntry) *
		    MAX_NUM_ACCESS_LISTS * MAX_NUM_ACCESS_RANGES) == -1) {
	    fprintf(stdout,"Error reading udp access table\n");
	    exit(1);
	}

	// fprintf(stdout,"result = %u\n",result);

	fprintf(stdout,"Loaded class table\n");

	close(fd);
    }
}

void initNetworks(void)
{
    char wildcard[14];
    int done;
    int fd;
    int curr;
    unsigned long size;
    unsigned long currSize;
    unsigned long hash;
    in_addr network;
    struct ffblk ffblk;

    // Loop and read in every network. Each network is in a file named
    //   <network>.net where <network> is the IP network in hexadecimal.
    strcpy(wildcard,"*.");
    strcat(wildcard,NETWORK_EXTENSION);

    done = findfirst(wildcard,&ffblk,0);

    while (!done) {

	// WATCH OUT! File must be opened in binary mode or else
	//   text translation of carriage returns will occur.
	fd = open(ffblk.ff_name,O_RDONLY | O_BINARY);

	if (fd == -1) {
	    fprintf(stdout,"Can't open %s\n",ffblk.ff_name);
	    exit(1);
	}

	if (read(fd,(char *) &network,sizeof(network)) == -1) {
	    fprintf(stdout,"Error reading network for %s\n",ffblk.ff_name);
	    exit(1);
	}

	// Do the heavy duty checking.
	if (IN_CLASSB(network.S_addr)) {
	     // Class B address.
	     network.S_addr = network.S_addr & CLASSB_NETWORK;
	     size = 0x10000UL;
	}
	else if (IN_CLASSC(network.S_addr))
	{
	     // Class C address.
	     network.S_addr = network.S_addr & CLASSC_NETWORK;
	     size = 0x100UL;
	}
	else {
	    // We don't handle class A addresses.
	    fprintf(stdout,
		"Version %s does not handle class A or D addresses - %s skipped\n",
		VERSION,
		ffblk.ff_name);
	    close(fd);
	    done = findnext(&ffblk);

	    continue;
	}

	// Insert the network into the hash table.
	hash = (network.S_addr & NETWORK_HASH_MASK) >> 19;
	//fprintf(stdout,"hash = %d\n",hash);
	curr = (int) hash;

	while (addrTable[curr].network.S_addr != network.S_addr &&
	       addrTable[curr].network.S_addr != 0UL) {

	    curr = (curr + 1) & (MAX_NUM_NETWORKS - 1);

	    if (curr == hash) {
		fprintf(stdout,"Network hash table is full\n");
		exit(1);
	    }
	}

	// If not defined then insert it and allocate the memory.
	if (addrTable[curr].network.S_addr == 0UL) {

	    // fprintf(stdout,"inserted network at %d\n",curr);

	    // Set the address for the table.
	    addrTable[curr].network.S_addr = network.S_addr;

	    //fprintf(stdout, "Free core: %lu\n", farcoreleft());

	    // Allocate the host table. Mung up the pointer so that the block
	    //   of memory is aligned on a paragraph and the full 64K can
	    //   be accessed.
	    addrTable[curr].realHostTable = (unsigned char *) farmalloc(size + 16);
	    addrTable[curr].hostTable = (unsigned char *)
				 MK_FP(FP_SEG(addrTable[curr].realHostTable) + 1,0);

	    //fprintf(stdout,"hosttanle = %04X:%04X\n",
	    //  FP_SEG(addrTable[curr].hostTable),
	    //  FP_OFF(addrTable[curr].hostTable));
	}
	else {
	    // This case should not happen.
	    fprintf(stdout,"Overriding previous network entry\n");
	}

	// Since read() won't take longs.....
	currSize = size;
	while (currSize) {

	    // Read the host table in. Be careful with the pointer
	    //   math in calculating the read address.
	    if (read(fd,addrTable[curr].hostTable + (unsigned short) (size - currSize),
		     (unsigned short) (currSize < 32768UL ? currSize : 32768UL)) == -1) {

		fprintf(stdout,"Error reading in host table for %s\n",ffblk.ff_name);
		exit(1);
	    }

	    currSize -= currSize < 32768UL ? currSize : 32768UL;
	}

	// Initialize the other parameters.
	addrTable[curr].dirty = NO;

	fprintf(stdout,"Loaded network %d.%d.%d.%d\n",
		network.S_un_b.s_b4,
		network.S_un_b.s_b3,
		network.S_un_b.s_b2,
		network.S_un_b.s_b1);

	close(fd);

	done = findnext(&ffblk);
    }
}

void init(void)
{
    initMemory();
    initTables();
    initNetworks();
}

void usage(void)
{
    fprintf(stdout,"usage: bridge [-l (both|inside|outside)]\n");
    exit(1);
}

int main(int argc,char *argv[])
{
    int i;

    // Print out a start up message.
    fprintf(stdout,"Filter Version %s starting...\n",VERSION);

    // Handle the command line arguments.
    for (i = 1;i < argc;++i) {
	if (argv[i][0] == '-') {
	    switch (argv[i][1]) {
		case 'l':
		    ++i;
		    if (strcmp("inside",argv[i]) == 0) {
			listenMode = INSIDE_MASK;
			fprintf(stdout,"Listening for messages on inside\n");
		    }
		    else if (strcmp("outside",argv[i]) == 0) {
			listenMode = OUTSIDE_MASK;
			fprintf(stdout,"Listening for messages on outside\n");
		    }
		    else if (strcmp("both",argv[i]) == 0) {
			listenMode = OUTSIDE_MASK | INSIDE_MASK;
			fprintf(stdout,"Listening for messages on both sides\n");
		    }
		    else
			usage();
		    break;
		default:
		    fprintf(stdout,"unrecognized switch %s\n",argv[i]);
		    usage();
		    break;
	    }
	}
	else {
	    usage();
	}
    }

    if (listenMode == 0)
	fprintf(stdout,"Not listening for messages\n");

    // Initialize everything.
    init();

    // Define the two cards.
    fprintf(stdout,"Defining inside card...\n");
    campus   = WDDefine(CAMPUSPORT,(unsigned char *) MK_FP(CAMPUSMEM,0),DATAPATH);

    fprintf(stdout,"Defining outside card...\n");
    internet = WDDefine(INTERNETPORT,(unsigned char *) MK_FP(INTERNETMEM,0),DATAPATH);

    fprintf(stdout,"Beginning filtering\n");

    for (;;) {

	// Forward from internet to campus.
	forward(internet,campus,checkIncomingPacket);

	// Forward from campus to internet.
	forward(campus,internet,checkOutgoingPacket);

	if (kbhit()) {

	    // Exit if we get the proper key stroke.
	    if (getch() == '$')
		exit(0);
	}

	// Cold boot the PC. Pretty nasty incantation, huh?
	//  Just calls the BIOS where Intel chips go to on
	//  power up. It first checks to make sure that no packets
	//  are waiting to be delivered.
	if (rebootRequested         == YES &&
	    campus->packetWaiting   == NO &&
	    internet->packetWaiting == NO)
	    ((void (far *)(void)) MK_FP(0xF000,0xFFF0))();
    }
}
