/*
 * This program is an implementation of the ISAKMP Internet Standard.
 * Copyright (C) 1997 Angelos D. Keromytis.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * This code was written while the author was in Greece, in May/June
 * 1997.
 *
 * You may contact the author by:
 *   e-mail: angelos@dsl.cis.upenn.edu
 *  US-mail: Angelos D. Keromytis
 *           Distributed Systems Lab
 *           Computer and Information Science Department
 *           University of Pennsylvania
 *           Moore Building
 *           200 South 33rd Street
 *           Philadelphia, PA 19104	   
 */

#include <stdio.h>
#include <sys/socket.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/mman.h>
#include "constants.h"
#include "state.h"
#include "defs.h"

/*
 * This file has the functions that handle the
 * state hash table.
 */

#define STATE_TABLE_SIZE 32

static struct state *statetable[STATE_TABLE_SIZE];

/*
 * Had to go somewhere, might as well be this file. Initialize
 * global variables.
 */
void
init_vars(void)
{
    int i;

    our_port = PORT;

    /* 
     * Generate the secret value for responder cookies, and
     * schedule an event for refresh.
     */
    get_rnd_bytes(secret_of_the_day, SECRET_VALUE_LENGTH);
    event_schedule(EVENT_REINIT_SECRET, EVENT_REINIT_SECRET_DELAY, NULL, 0);
    
#ifndef linux
    i = open(ZERO_FILE, O_RDWR, 0);
    if (i == -1)
      exit_log("open() failed in init_vars()", 0, 0, 0);

    nullbuffer = (u_char *) mmap(0, UDP_SIZE, PROT_READ | PROT_WRITE, 
				 MAP_FILE | MAP_PRIVATE, i, 0);
    if (nullbuffer == (u_char *) -1)
      exit_log("mmap() failed in init_vars()", 0, 0, 0);
#endif
}

static u_int
state_hash(u_char *icookie, u_char *rcookie, struct sockaddr sa)
{
    u_int i, j;
    
    for (i = 0, j = 0; j < COOKIE_SIZE; j++)
      i += icookie[j] + rcookie[j];

    i += sa.sa_family;

    for (j = 0; j < sizeof(sa); j++)
      i += sa.sa_data[j];
    
    return i % STATE_TABLE_SIZE;
}

/* 
 * Get a state object
 */
struct state *
get_state(void)
{
    struct state *st;
    
    st = (struct state *) calloc(1, sizeof(struct state));
    if (st == (struct state *) NULL)
      exit_log("calloc() failed in get_state()", 0, 0, 0);
   
    return st;
}

/*
 * Initialize the state table
 */
void
init_state(void)
{
    int i;
    
    for (i = 0; i < STATE_TABLE_SIZE; i++)
      statetable[i] = (struct state *) NULL;
}

/*
 * Insert a state object in the hash table
 */
void
insert_state(struct state *st)
{
    int i;
    
    i = state_hash(st->st_icookie, st->st_rcookie, st->st_peer);
    
    st->st_prev = (struct state *) NULL;
    st->st_next = statetable[i];
    statetable[i] = st;
    if (st->st_next != (struct state *) NULL)
      st->st_next->st_prev = st;
}

/*
 * Delete a state object from the hash table, but don't free it
 */
void
delete_state(struct state *st)
{
    int i;

    /* If it's not the first in the list */

    if (st->st_prev != (struct state *) NULL)
    {
	st->st_prev->st_next = st->st_next;

	if (st->st_next != (struct state *) NULL)
	  st->st_next->st_prev = st->st_prev;

	return;
    }
    
    i = state_hash(st->st_icookie, st->st_rcookie, st->st_peer);
    statetable[i] = st->st_next;
    if (st->st_next != (struct state *) NULL)
      st->st_next->st_prev = (struct state *) NULL;
}

/*
 * Free a state object
 */
void
free_state(struct state *st)
{
    if ((st->st_packet != (u_char *) NULL) &&
	(st->st_packet_len != 0))
      free(st->st_packet);
    
    free(st);
}

/*
 * Find a state object.
 */
struct state *
find_full_state(u_char *icookie, u_char *rcookie, struct
		sockaddr sa)
{
#define SA_EQUAL(x, y)   ((x.sa_family == y.sa_family) &&\
                          (!bcmp(x.sa_data, y.sa_data, sizeof(x))))
    struct state *st;
    
    st = statetable[state_hash(icookie, rcookie, sa)];

    while (st != (struct state *) NULL)
      if (SA_EQUAL(sa, st->st_peer) &&
	  (!bcmp(icookie, st->st_icookie, COOKIE_SIZE)) &&
	  (!bcmp(rcookie, st->st_rcookie, COOKIE_SIZE)))
	break;
      else
	st = st->st_next;
    
    return st;
#undef SA_EQUAL
}

/*
 * Find a state object in the hash table, without using
 * the rcookie (to find state objects where we haven't yet
 * received a message from the responder).
 */
struct state *
find_half_state(u_char *icookie, struct sockaddr sa)
{
#define SA_EQUAL(x, y)   ((x.sa_family == y.sa_family) &&\
                          (!bcmp(x.sa_data, y.sa_data, sizeof(x))))
    struct state *st;
    u_char rcookie[COOKIE_SIZE];
    
    bzero(rcookie, COOKIE_SIZE);
    st = statetable[state_hash(icookie, rcookie, sa)];

    while (st != (struct state *) NULL)
      if (SA_EQUAL(sa, st->st_peer) &&
	  (!bcmp(icookie, st->st_icookie, COOKIE_SIZE)))
	break;
      else
	st = st->st_next;
    
    return st;
#undef SA_EQUAL
}
