/*
 * This program is an implementation of the ISAKMP Internet Standard.
 * Copyright (C) 1997 Angelos D. Keromytis.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * 
 * This code was written while the author was in Greece, in May/June
 * 1997.
 *
 * You may contact the author by:
 *   e-mail: angelos@dsl.cis.upenn.edu
 *  US-mail: Angelos D. Keromytis
 *           Distributed Systems Lab
 *           Computer and Information Science Department
 *           University of Pennsylvania
 *           Moore Building
 *           200 South 33rd Street
 *           Philadelphia, PA 19104	   
 */

#include <stdio.h>
#include <sys/types.h>
#include <time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "constants.h"
#include "state.h"
#include "defs.h"

/*
 * This file has the event handling routines. Events are
 * kept as a linked list of event structures. These structures
 * have information like event type, expiration time and a pointer
 * to event specific data (for example, to a state structure).
 */

static struct event *evlist = (struct event *) NULL;

/*
 * This routine returns a user readable form of an address contained in a
 * sockaddr structure. The return value CAN be a statically allocated
 * object (as is the case with inet_ntoa()).
 */
char *
get_address(struct sockaddr sa)
{
    struct sockaddr_in sin;
    
    switch (sa.sa_family)
    {
	case AF_INET:
	    bcopy(&sa, &sin, sizeof(sa));
	    return inet_ntoa(sin.sin_addr);
	    
	default:
	    return (char *) NULL;
    }
}

/*
 * Return a port (if applicable), from a struct sockaddr. If not applicable,
 * return -1.
 */
int
get_port(struct sockaddr sa)
{
    struct sockaddr_in sin;
    
    switch (sa.sa_family)
    {
	case AF_INET:
	    bcopy(&sa, &sin, sizeof(sa));
	    return ntohs(sin.sin_port);
	    
	default:
	    return -1;
    }
}

/*
 * This routine places an event in the event list.
 */
void
event_schedule(int type, time_t tm, void *ptr, int ptrlen)
{
    struct event *ev, *evt;
    struct state *st;
    
    ev = (struct event *) calloc(1, sizeof(struct event));
    if (ev == (struct event *) NULL)
      exit_log("calloc() failed in event_schedule()", 0, 0, 0);

    ev->ev_type = type;
    ev->ev_time = tm + time((time_t *) NULL);
    ev->ev_ptr = ptr;
    ev->ev_ptr_len = ptrlen;

    /* 
     * If the event is a retransmission, put a backpointer to the event
     * in the state object, so we can delete the event if we receive
     * a reply.
     */
    if (type == EVENT_RETRANSMIT)
    {
	st = (struct state *) ptr;
	st->st_event = ev;
    }
    
#ifdef DEBUG
    log(0, "inserting event %d, timeout in %u seconds", type, tm, 0);
#endif

    if (evlist == (struct event *) NULL) /* No other events */
    {
	evlist = ev;
	return;
    }

    /* We're installing the most close-to-happen event */
    if (evlist->ev_time >= ev->ev_time)
    {
	ev->ev_next = evlist;
	evlist = ev;
	return;
    }
    
    for (evt = evlist; evt->ev_next != (struct event *) NULL;
	 evt = evt->ev_next)
      if (evt->ev_next->ev_time >= ev->ev_time)
	break;

#ifdef DEBUG
    log(0, "event added after event %d (%p/%d)", evt->ev_type,
	evt->ev_ptr, evt->ev_ptr_len);
#endif

    ev->ev_next = evt->ev_next;
    evt->ev_next = ev;
}

/*
 * Handle the first event on the list.
 */
void
event_handle(int kernelfd, int sock)
{
    time_t tm;
    struct event *ev;
    struct state *st;
    struct sockaddr sa;
    struct sockaddr_in sin;
    
    ev = evlist;

    if (ev == (struct event *) NULL)    /* Just paranoid */
#ifdef DEBUG
    {
	log(0, "empty event list, yet we're called", 0, 0, 0);
#endif
      return;
#ifdef DEBUG
    }
#endif

    tm = time((time_t *) NULL);
    
    if (tm < ev->ev_time)
#ifdef DEBUG
    {
	log(0, "called while no event expired (%u/%u, %d)", tm, ev->ev_time,
	    ev->ev_type);
#endif
      return;
#ifdef DEBUG
    }
#endif

    evlist = evlist->ev_next;		/* Ok, we'll handle this event */

#ifdef DEBUG
    log(0, "next event is %d (%p/%d)", evlist->ev_type, evlist->ev_ptr,
	evlist->ev_ptr_len);
#endif

    switch (ev->ev_type)
    {
	case EVENT_REINIT_SECRET:
#ifdef DEBUG
	    log(0, "event EVENT_REINIT_SECRET handled", 0, 0, 0);
#endif
	    get_rnd_bytes(secret_of_the_day, SECRET_VALUE_LENGTH);
	    event_schedule(EVENT_REINIT_SECRET, EVENT_REINIT_SECRET_DELAY,
			   NULL, 0);
	    break;
	    
	case EVENT_RETRANSMIT:
	    st = (struct state *) ev->ev_ptr;
	    sa = st->st_peer;
#ifdef DEBUG
		    log(0, 
			"event EVENT_RETRANSMIT for %s, port %d, handled (%p)",
			(get_address(sa) != (char *) NULL ? get_address(sa) :
			 "(unknown address family)"),
			get_port(sa), st);
#endif
	    if (st->st_retransmit >= MAXIMUM_RETRANSMISSIONS)
	    {
		log(0, 
		    "max number of retransmissions(%d) reached for %s, port %d",
		    st->st_retransmit,
		    (get_address(sa) != (char *) NULL ? get_address(sa) :
		     "(unknown address family)"), get_port(sa));

		delete_state(st);
		free_state(st);

		/* XXX Send some message to the kernel */
		break;
	    }
	    else
	      st->st_retransmit++;
	    
	    if (sendto(sock, st->st_packet, st->st_packet_len, 0,
		       (struct sockaddr *) &(st->st_peer), 
		       sizeof(st->st_peer)) != st->st_packet_len)
	      log(1, "sendto() failed in event_handle, for %s, port %d",
		  (get_address(sa) != (char *) NULL ? get_address(sa) :
		   "(unknown address family)"), get_port(sa), 0);

	    event_schedule(EVENT_RETRANSMIT, EVENT_RETRANSMIT_DELAY, st, 0);
	    break;
	    
	default:
	    log(0, "unknown event %d expired, ignoring", ev->ev_type, 0, 0);
    }

    free(ev);
}

/*
 * Return the time until the next event in the queue
 * expires, -1 if no jobs in queue.
 */
int32_t
next_event(void)
{
    time_t tm;
  
    if (evlist == (struct event *) NULL)
      return -1;

#ifdef DEBUG
    log(0, "next event in %d seconds (%p/%d)", 
	evlist->ev_time - time((time_t *) NULL),
	evlist->ev_ptr, evlist->ev_ptr_len);
#endif

    tm = time((time_t*) NULL);

    if (evlist->ev_time - tm <= 0)
      return 0;
    else
      return (evlist->ev_time - tm);
}
