/*
 *	slip-attach.c
 *
 *	Slip-attach opens and initalises the named device at the
 *	requested speed, attaches the slip streams encodeing module and 
 *	and streams IP network module and initaliases them. 
 *
 *	Copyright CSIRO Division of Mathematics and Statistics 21 June 1990
 *
 *	Portions of this code may be held by, Doug Kingston,Rayan Zachariassen,
 *	and Greg Earle (see below).
 *
 *	Permission is hereby granted for this code to be distributed
 *	free of charge with this copyright intact and does not interfere
 *	with any copyrights held by the above. Derived works should
 *	be marked as so.
 */

/*
 * sliplogin.c
 *
 * This program initializes its own tty port to be an async TCP/IP interface.
 * It merely sets up the SLIP module all by its lonesome on the STREAMS stack,
 * initializes the network interface, and pauses forever waiting for hangup.
 *
 * It is a remote descendant of several similar programs with incestuous ties:
 * - Kirk Smith's slipconf, modified by Richard Johnsson @ DEC WRL.
 * - slattach, probably by Rick Adams but touched by countless hordes.
 * - the original sliplogin for 4.2bsd, Doug Kingston the mover behind it.
 * - a simple slattach-like program used to test the STREAMS SLIP code.
 *
 * Doug Kingston 8810??		- logging + first pass at adding I_STR ioctl's
 * Rayan Zachariassen 881011	- version for SunOS STREAMS SLIP
 * Greg Earle 890331		- added tip/UUCP style LCK lock file
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stropts.h>
#include <sys/termios.h>
#include <sys/ttold.h>
#include <sys/sockio.h>
#include <sys/file.h>
#include <sys/syslog.h>

#include <sys/slip.h>

#include <netinet/in.h>
#include <net/if.h>

#include <stdio.h>
#include <errno.h>
#include <ctype.h>
#include <netdb.h>

#include <signal.h>
#include <strings.h>
#include <pwd.h>
#ifdef BSD >= 43
#include <ttyent.h>
#endif

void exit();

main(argc, argv)
	int argc;
	char *argv[];
{
	int		fd, s, unit;
	int speed,i;
	int		sfd = 0;
	char	*device;
	char	*netmask;
	char	*localaddr;
	char	*dstaddr;
	struct termios	tios;
	struct ifreq	ifr;
	extern void	findid();

	if (getuid() == 0) {
		if (argc != 6) {
			(void) fprintf(stderr,
				       "Usage: %s device speed localaddr destaddr netmask\n", argv[0]);
			exit(1);
		}
		device = argv[1];
		speed = atoi(argv[2]);
		localaddr = argv[3];
		dstaddr = argv[4];
		netmask = argv[5];
	} else
		exit(1);

	switch (speed){
	case 50: speed = B50; break;
	case 75: speed = B75; break;
	case 110: speed = B110; break;
	case 134: speed = B134; break;
	case 150: speed = B150; break;
	case 200: speed = B200; break;
	case 300: speed = B300; break;
	case 600: speed = B600; break;
	case 1200: speed = B1200; break;
	case 1800: speed = B1800; break;
	case 2400: speed = B2400; break;
	case 4800: speed = B4800; break;
	case 9600: speed = B9600; break;
	case 19200: speed = B19200; break;
	case 38400: speed = B38400; break;
	default:
		(void)fprintf(stderr,"bad speed %d\n",speed);
		exit(1);
	}

	for ( i = getdtablesize(); i--; )
		(void)close(i);

	if (fork())
		exit(0);

	openlog("slip-attach", LOG_PID, LOG_DAEMON);

	if ((fd = open("/dev/tty", O_RDONLY, 0)) >= 0) {
		(void) ioctl(fd, TIOCNOTTY, 0);
		(void) close(fd);
	}


	if (( sfd = open(device,O_RDWR,0)) < 0 ) {
		syslog(LOG_ERR, "open: %s: %m",device);
		exit(1);
	}

	(void) fchmod(sfd, 0600);

	/* pop all streams modules */
	while (ioctl(sfd, I_POP, 0) == 0)
		continue;

	/* set up the line parameters */
	if (ioctl(sfd, TCGETS, (caddr_t)&tios) < 0) {
		syslog(LOG_ERR, "ioctl (TCGETS): %m");
		exit(1);
	}
	tios.c_cflag = CS8|CREAD|HUPCL|speed;
	tios.c_iflag = IGNBRK;

	if (ioctl(sfd, TCSETS, (caddr_t)&tios) < 0) {
		syslog(LOG_ERR, "ioctl (TCSETS): %m");
		exit(1);
	}

	/* push the SLIP module */
	if (ioctl(sfd, I_PUSH, "slipe") < 0) {
		syslog(LOG_ERR, "ioctl (I_PUSH) slipen: %m");
		exit(1);
	}
	if (ioctl(sfd, I_PUSH, "str_ip") < 0) {
		syslog(LOG_ERR, "ioctl (I_PUSH) str_ip: %m");
		exit(1);
	}

	/* find out what unit number we were assigned */
	if (ioctl(sfd, SLIOGUNIT, (caddr_t)&unit) < 0) {
		syslog(LOG_ERR, "ioctl (SLIOGUNIT): %m");
		exit(1);
	}

	syslog(LOG_NOTICE, "attaching stream%d: local %s remote %s mask %s\n",
		unit, localaddr, dstaddr, netmask);

	/* set the local and remote interface addresses */
	s = socket(AF_INET, SOCK_DGRAM, 0);

	(void) sprintf(ifr.ifr_name, "%s%d", "stream", unit);
	in_getaddr(netmask, &ifr.ifr_addr);
	if (ioctl(s, SIOCSIFNETMASK, (caddr_t)&ifr) < 0) {
		syslog(LOG_ERR, "ioctl (SIOCSIFNETMASK): %m");
		exit(1);
	}

	(void) sprintf(ifr.ifr_name, "%s%d", "stream", unit);
	in_getaddr(dstaddr, &ifr.ifr_addr);
	if (ioctl(s, SIOCSIFDSTADDR, (caddr_t)&ifr) < 0) {
		syslog(LOG_ERR, "ioctl (SIOCSIFDSTADDR): %m");
		exit(1);
	}

	(void) sprintf(ifr.ifr_name, "%s%d", "stream", unit);
	in_getaddr(localaddr, &ifr.ifr_addr);
	/* this has the side-effect of marking the interface up */
	if (ioctl(s, SIOCSIFADDR, (caddr_t)&ifr) < 0) {
		syslog(LOG_ERR, "ioctl (SIOCSIFADDR): %m");
		exit(1);
	}
	(void)pause();
	exit(0);
	/*NOTREACHED*/
}


in_getaddr(s, saddr)
	char *s;
	struct sockaddr *saddr;
{
	register struct sockaddr_in *sin = (struct sockaddr_in *)saddr;
	struct hostent *hp;
	struct netent *np;
	u_long val;
	extern struct in_addr inet_makeaddr();
	extern u_long inet_addr();
 
	bzero((caddr_t)saddr, sizeof *saddr);
	sin->sin_family = AF_INET;
	val = inet_addr(s);
	if (val != (u_long)-1) {
		sin->sin_addr.s_addr = val;
		return;
	}
	hp = gethostbyname(s);
	if (hp) {
		sin->sin_family = hp->h_addrtype;
		bcopy(hp->h_addr, (char *)&sin->sin_addr, hp->h_length);
		return;
	}
	np = getnetbyname(s);
	if (np) {
		sin->sin_family = np->n_addrtype;
		sin->sin_addr = inet_makeaddr((int)np->n_net, (int)INADDR_ANY);
		return;
	}
	(void) fprintf(stderr, "slip_attach: %s: bad value\n", s);
	syslog(LOG_ERR, "%s: bad value\n", s);
	exit(1);
	/* NOTREACHED */
}
