/* ao_elflib.c - ELF shared library support */

/*  Copyright 1995 Mark Russell, University of Kent at Canterbury.
 *
 *  You can do what you like with this source code as long as
 *  you don't try to make money out of it and you include an
 *  unaltered copy of this message (including the copyright).
 */

char ups_ao_elflib_c_sccsid[] = "@(#)ao_elflib.c	1.1 24/5/95 (UKC)";

#include <mtrprog/ifdefs.h>
#include "ao_ifdefs.h"
	
#ifdef AO_ELF

#include <sys/types.h>
#include <sys/stat.h>
#include <elf.h>
#include <link.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include <local/ukcprog.h>
#include <mtrprog/utils.h>
#include <mtrprog/io.h>

#include "ups.h"
#include "symtab.h"
#include "target.h"
#include "st.h"
#include "ao_syms.h"
#include "ao_core.h"
#include "ao_target.h"
#include "ao_elfpriv.h"
#include "ao_elfread.h"
#include "ao_elflib.h"
#include "ao_elfsym.h"
#include "ao_symscan.h"

typedef struct Solib_addr Solib_addr;

struct Solib_addr {
	dev_t dev;
	ino_t ino;
	off_t offset;
	taddr_t vaddr;
	size_t pagesize;
	Solib_addr *next;
};

struct Solib {
	symtab_t *symtab;
	
	const char *soname;
	const char *rpath;
	
	taddr_t min_file_vaddr;
	taddr_t debug_vaddr;
	
	dev_t dev;
	ino_t ino;
	
	Libdep *latest_libdep;

	bool is_last_symtab;
	Solib *next;
};

struct Libdep {
	const char *name;
	Solib *so;
	Libdep *par;
	Libdep *deps;
	Libdep *next;
};

static void
dump_libdep(Libdep *par, int level)
{
	Libdep *ld;
	
	printf("%*s%p: %s", level * 8, "", par, par->name);

	if (par->so != NULL)
		printf(" so=%s", par->so->symtab->st_path);

	fputc('\n', stdout);

	for (ld = par->deps; ld != NULL; ld = ld->next)
		dump_libdep(ld, level + 1);
}

void
dump_elf_libs(target_t *xp)
{
	Solib *so;
	iproc_t *ip;

	ip = GET_IPROC(xp);

	for (so = ip->ip_solibs; so != NULL; so = so->next) {
		printf("%s: soname=%s latest_libdep=%p\n",
		       so->symtab->st_path,
		       (so->soname != NULL) ? so->soname : "NULL",
		       so->latest_libdep);
	}

	fputc('\n', stdout);
	
	dump_libdep(ip->ip_solibs->latest_libdep, 0);
}

static Libdep *
make_libdep(alloc_pool_t *ap, const char *name, Libdep *par)
{
	Libdep *ld;

	ld = (Libdep *)alloc(ap, sizeof(Libdep));
	ld->name = name;
	ld->so = NULL;
	ld->par = par;
	ld->deps = NULL;
	ld->next = NULL;
	
	return ld;
}

static bool
search_lib_path(const char *dirs, const char *seps, const char *name,
		char **p_path)
{
	char **dirvec, **dp;

	if (dirs == NULL)
		return FALSE;
	
	dirvec = ssplit(dirs, seps);

	for (dp = dirvec; *dp != NULL; ++dp) {
		char *path;
		
		path = strf("%s/%s", *dp, name);

		if (access(path, F_OK) == 0) {
			*p_path = path;
			return TRUE;
		}

		free(path);
	}

	return FALSE;
}

/*  Apply the rules on page 5-20 of the System V ABI book to find the
 *  path (relative or absolute) of the shared library.
 */
static char *
get_ld_path(Libdep *ld)
{
	char *path;
	
	if (strchr(ld->name, '/') != NULL)
		return strsave(ld->name);

	if (search_lib_path(ld->par->so->rpath, ":", ld->name, &path) ||
	    search_lib_path(getenv("LD_LIBRARY_PATH"), ":;", ld->name, &path))
		return path;
	
	return strf("/usr/lib/%s", ld->name);
}

static void
set_base_address(Solib *so, off_t offset, taddr_t vaddr, size_t pagesize)
{
	taddr_t base_address;

	if (pagesize == 0) {
		base_address = vaddr;
	}
	else {
		taddr_t min_mem_vaddr, pagemask;
		
		pagemask = ~(pagesize - 1);
		
		min_mem_vaddr = (vaddr - offset) & pagemask;
		base_address = (min_mem_vaddr & pagemask) -
			(so->min_file_vaddr & pagemask);
	}
		
	change_base_address(so->symtab, base_address);
}

static bool
load_elf_shlib(target_t *xp, Libdep *ld)
{
	iproc_t *ip;
	char *path;
	int fd;
	bool ok;
	Solib *so;

	ip = GET_IPROC(xp);
	
	path = get_ld_path(ld);

	for (so = ip->ip_solibs; so != NULL; so = so->next) {
		if (strcmp(so->symtab->st_path, path) == 0) {
			ld->so = so;
			so->latest_libdep = ld;

			/*  We leave ld->deps as NULL, as they will
			 *  always be visited first under the earlier
			 *  instance of the symbol table.
			 */
			
			return TRUE;
		}
	}

	ok = (open_for_reading(path, "shared library", &fd) &&
	      check_elf_file_type(fd, path, ET_DYN, "load symbols from") &&
	      scan_elf_symtab(xp->xp_apool, path, fd, ld, (func_t **)NULL,
			      &ip->ip_solibs));

	free(path);

	if (ok) {
		Solib_addr *sa;
		
		so = ld->so;
		
		/*  TODO: should pass load address to scan_elf_symtab().
		 */
		for (sa = ip->ip_solib_addrs; sa != NULL; sa = sa->next) {
			if (sa->dev == so->dev && sa->ino == so->ino)
				break;
		}

		if (sa != NULL) {
			set_base_address(so,
					 sa->offset, sa->vaddr, sa->pagesize);
		}
				
	}

	return ok;
}

void
elf_zap_solib_addrs(xp)
target_t *xp;
{
	Solib_addr *sa, *next;
	iproc_t *ip;

	ip = GET_IPROC(xp);
	
	for (sa = ip->ip_solib_addrs; sa != NULL; sa = next) {
		next = sa->next;
		free(sa);
	}

	ip->ip_solib_addrs = NULL;
}

bool
elf_note_shlib_addr(iproc_t *ip, dev_t dev, ino_t ino,
		    off_t offset, taddr_t vaddr, size_t pagesize)
{
	Solib *so;
	
	for (so = ip->ip_solibs; so != NULL; so = so->next) {
		if (so->dev == dev && so->ino == ino)
			break;
	}

	if (so != NULL) {
		set_base_address(so, offset, vaddr, pagesize);
	}
	else {
		Solib_addr *sa;
		
		sa = e_malloc(sizeof(Solib_addr));
		sa->dev = dev;
		sa->ino = ino;
		sa->offset = offset;
		sa->vaddr = vaddr;
		sa->pagesize = pagesize;
		sa->next = ip->ip_solib_addrs;
		ip->ip_solib_addrs = sa;
	}
	
	return TRUE;
}

bool
elf_get_core_shlib_info(alloc_pool_t *ap, iproc_t *ip)
{
	struct r_debug debug;
	Solib *so;
	const char *path;
	taddr_t dbaddr, lmaddr;
	Coredesc *co;
	struct link_map lmap;
	
	so = ip->ip_solibs;
	co = ip->ip_core;
	path = so->symtab->st_path;

	if (so->debug_vaddr == 0) {
		errf("No DT_DEBUG entry in .dynamic section of %s", path);
		return FALSE;
	}

	if (!core_dread(co, so->debug_vaddr, &dbaddr, sizeof(dbaddr)) ||
	    !core_dread(co, dbaddr, (char *)&debug, sizeof(debug))) {
		errf("Can't read r_debug structure from %s", path);
		return FALSE;
	}

	for (lmaddr = (taddr_t)debug.r_map;
	     lmaddr != TADDR_NULL;
	     lmaddr = (taddr_t)lmap.l_next) {
		char libpath[1024]; /* TODO: use MAXPATHLEN or similar */
		struct stat stbuf;

		if (!core_dread(co, lmaddr, &lmap, sizeof(lmap)) ||
		    !core_readstr(co, (taddr_t)lmap.l_name,
				  libpath, sizeof(libpath))) {
			errf("Error reading lmap at address %lx in %s",
			     lmaddr, path);
			return FALSE;
		}

		/*  Skip the first entry - it refers to the executable file
		 *  rather than a shared library.
		 */
		if (lmap.l_prev == NULL)
			continue;
		
		if (stat(libpath, &stbuf) != 0) {
			errf("Can't stat shared library %s: %s (ignored)",
			     libpath, get_errno_str());
			continue;
		}

		elf_note_shlib_addr(ip, stbuf.st_dev, stbuf.st_ino,
				    0, lmap.l_addr, 0);
	}

	return TRUE;
}

bool
scan_main_elf_symtab(alloc_pool_t *target_ap, const char *path, int fd,
		     Solib **p_solibs, struct func_s **p_mainfunc)
{
	Libdep *ld;
	
	*p_solibs = NULL;
	ld = make_libdep(target_ap, path, (Libdep *)NULL);

	return scan_elf_symtab(target_ap, path, fd, ld, p_mainfunc, p_solibs);
}

static Libdep *
find_first_descendent(Libdep *par, int level)
{
	Libdep *ld;

	if (par == NULL)
		return NULL;
	
	if (level == 0)
		return par->deps;

	for (ld = par->deps; ld != NULL; ld = ld->next) {
		Libdep *res;

		res = find_first_descendent(ld, level - 1);

		if (res != NULL)
			return res;
	}

	return NULL;
}

static Libdep *
next_ld(Libdep *cur)
{
	Libdep *ld;
	int level;

	if (cur->next != NULL)
		return cur->next;

	if (cur->par == NULL)
		return cur->deps;

	level = 0;
	for (ld = cur->par; ld != NULL; ld = ld->par) {
		Libdep *res;
		
		res = find_first_descendent(ld->next, level);

		if (res != NULL)
			return res;
		
		++level;
	}

	return find_first_descendent(cur->par, 1);
}

bool
elf_next_symtab(target_t *xp, symtab_t *st, bool load_new, symtab_t **p_next_st)
{
	Solib *so;
	Libdep *ld;
	
	if (st == NULL) {
		*p_next_st = GET_IPROC(xp)->ip_solibs->symtab;
		return TRUE;
	}

	so = AO_STDATA(st)->st_solib;

	if (so->next != NULL) {
		*p_next_st = so->next->symtab;
		return TRUE;
	}

	if (!load_new || so->is_last_symtab)
		return FALSE;

	for (ld = next_ld(so->latest_libdep); ld != NULL; ld = next_ld(ld)) {
		if (!load_elf_shlib(xp, ld))
			continue;

		if (so->next != NULL) {
			/*  Loaded a new symbol table.
			 */
			*p_next_st = so->next->symtab;
			return TRUE;
		}
	}

	so->is_last_symtab = TRUE;
	return FALSE;
}

static bool
get_elf_shlib_info(alloc_pool_t *ap, Elfinfo *el,
		   const char **p_soname, const char **p_rpath,
		   taddr_t *p_debug_vaddr)
{
	Elf32_Shdr *dynsh, *strsh;
	Elf32_Dyn *dyntab;
	int i, count;
	char *strings;
	Libdep first_child, *last_child;
	
	if (!elf_find_section(el, ".dynamic", "shared library information",
			      &dynsh))
		return FALSE;
	
	if (!check_elf_entsize(dynsh, ".dynamic", sizeof(Elf32_Dyn)))
		return FALSE;
	count = dynsh->sh_size / sizeof(Elf32_Dyn);
	
	if (dynsh->sh_link >= el->num_sections) {
		errf("String table index (%ld) for .dynamic section "
		     "not in range 0..%d in %s %s",
		     dynsh->sh_link, el->num_sections - 1, el->what, el->path);
		return FALSE;
	}
	
	strsh = &el->sections[dynsh->sh_link];
	
	dyntab = e_malloc(dynsh->sh_size);
	strings = e_malloc(strsh->sh_size);

	if (!read_elf_section(el, ".dynamic section", dynsh, dyntab) ||
	    !read_elf_section(el, ".dynamic section strings", strsh, strings)) {
		free(dyntab);
		free(strings);
		return FALSE;
	}

	last_child = &first_child;

	*p_soname = *p_rpath = NULL;
	*p_debug_vaddr = 0;
	
	for (i = 0; i < count; ++i) {
		Elf32_Dyn *d;
		const char **p_str, *depname;
		off_t offset;

		d = &dyntab[i];
		
		switch (d->d_tag) {
		case DT_SONAME:
			p_str = p_soname;
			break;
		case DT_RPATH:
			p_str = p_rpath;
			break;
		case DT_NEEDED:
			p_str = &depname;
			break;
		case DT_DEBUG:
			*p_debug_vaddr = dynsh->sh_addr +
				((char *)&d->d_un.d_ptr - (char *)dyntab);
			continue;
		default:
			continue;
		}

		offset = d->d_un.d_val;

		if (offset >= strsh->sh_size) {
			errf("String offset for .dynamic entry %d in %s %s "
			     "not in range 0..%ld - ignored",
			     i, el->what, el->path, strsh->sh_size - 1);
			continue;
		}
		
		*p_str = alloc_strdup(ap, &strings[offset]);

		if (d->d_tag == DT_NEEDED) {
			last_child->next = make_libdep(ap, depname, el->libdep);
			last_child = last_child->next;
		}
	}

	free(dyntab);
	free(strings);

	last_child->next = NULL;
	el->libdep->deps = first_child.next;
	
	return i == count;
}

bool
add_solib_entry(alloc_pool_t *ap, symtab_t *st, Elfinfo *el, Solib **p_solibs)
{
	Solib *so, *so2, *last;
	const char *soname, *rpath;
	ao_stdata_t *ast;
	taddr_t debug_vaddr;
	struct stat stbuf;
	
	ast = AO_STDATA(st);
	
	if (fstat(ast->st_textfd, &stbuf) != 0) {
		errf("Can't fstat %s (fd=%d): %s",
		     st->st_path, ast->st_textfd, get_errno_str());
		return FALSE;
	}
	
	if (!get_elf_shlib_info(ap, el, &soname, &rpath, &debug_vaddr))
		return FALSE;

	so = alloc(ap, sizeof(Solib));
	so->symtab = st;
	so->soname = soname;
	so->rpath = rpath;
	so->min_file_vaddr = el->min_file_vaddr;
	so->debug_vaddr = debug_vaddr;
	so->dev = stbuf.st_dev;
	so->ino = stbuf.st_ino;
	so->latest_libdep = el->libdep;
	so->is_last_symtab = FALSE;
	so->next = NULL;

	ast->st_solib = so;

	last = NULL;
	for (so2 = *p_solibs; so2 != NULL; so2 = so2->next)
		last = so2;

	if (last != NULL)
		last->next = so;
	else
		*p_solibs = so;

	el->libdep->so = so;

	return TRUE;
}
		
#endif /* AO_ELF */
