/* ao_symscan.c - do initial symbol table scan */

/*  Copyright 1995 Mark Russell, University of Kent at Canterbury.
 *
 *  You can do what you like with this source code as long as
 *  you don't try to make money out of it and you include an
 *  unaltered copy of this message (including the copyright).
 */

char ups_ao_symscan_c_sccsid[] = "@(#)ao_symscan.c	1.3 20 Jun 1995 (UKC)";

#include <mtrprog/ifdefs.h>
#include "ao_ifdefs.h"

#ifdef AO_TARGET

#include <sys/types.h>
#include <time.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#ifdef AO_ELF
#	include "elfstab.h"
#else
#	include <a.out.h>
#	ifndef OS_RISCOS
#		include <stab.h>
#	endif
#endif

#include <local/ukcprog.h>

#include "ups.h"
#include "symtab.h"
#include "ci.h"
#include "st.h"
#include "ao_syms.h"
#include "ao_symscan.h"
#include "ao_symread.h"
#include "ao_elfsym.h"
#include "ao_symcb.h"
#include "util.h"

static void wrapup_stf PROTO((stf_t *stf, hf_t **fmap, int mapsize));
static hf_t *lookup_hf PROTO((hf_t *headers, int id));
static void build_symset PROTO((char *aout_symset, char *ofile_symset));
static bool is_fortran_void PROTO((const char *cptr));
static void reverse_stf_funclist PROTO((stf_t *stf));
static void adjust_fil_vars_addr_base PROTO((fil_t *flist, long delta));
static fil_t *add_sol_fil PROTO((alloc_pool_t *ap, fil_t *sfiles, fil_t *sofil,
				 const char *name));
static void add_to_fil_funclist PROTO((alloc_pool_t *ap, fil_t *fil, 
                                       func_t *f));
#ifdef AO_ELF
static func_t *note_ofile_function PROTO((alloc_pool_t *ap,
					  alloc_pool_t *tmp_ap, stf_t *stf,
					  const char *path,
					  fil_t *solfil, unsigned sline_offset,
					  const char *namestr, int symno));
#endif

#ifndef ST_TE
/*  Deduce language of source file from name.
 *  *.c is C, *.f is f77, *.f90 is f90, anything else is unknown.
 *
 *  BUG: dubious way of getting language.
 */
language_t
srctype(name)
const char *name;
{
	char *suf;

	if ((suf = strrchr(name, '.')) == NULL)
		return LANG_UNKNOWN;
	else
		++suf;
	
	if (strcmp(suf, "c") == 0)
		return LANG_C;
	
	if (strcmp(suf, "f") == 0)
		return LANG_F77;

	if (strcmp(suf, "f90") == 0)
		return LANG_F90;
	
	return LANG_UNKNOWN;
}

/*  Called when we have finished with an stf structure.  Create space
 *  for the map and copy the stuff to it.
 */
static void
wrapup_stf(stf, orig_fmap, mapsize)
stf_t *stf;
hf_t **orig_fmap;
int mapsize;
{
	hf_t **fmap;
	int i;
	
	if (mapsize == 0)
		panic("mapsize 0 in wrapup_stf");
	
	fmap = (hf_t **)alloc(stf->stf_symtab->st_apool,
			      mapsize * sizeof(hf_t *));
	memcpy((char *)fmap, (char *)orig_fmap, mapsize * sizeof(hf_t *));

	for (i = 0; i < mapsize; ++i) {
		stf_t *hstf;

		hstf = fmap[i]->hf_stf;

		if (hstf->stf_fmap == NULL) {
			hstf->stf_fmap = fmap;
			hstf->stf_mapsize = mapsize;
		}
	}
}

static void
reverse_stf_funclist(stf)
stf_t *stf;
{
	funclist_t *fl, *next, *newlist;

	newlist = NULL;
	for (fl = stf->stf_fil->fi_funclist; fl != NULL; fl = next) {
		next = fl->fl_next;
		fl->fl_next = newlist;
		newlist = fl;
	}
	
	stf->stf_fil->fi_funclist = newlist;
}

#ifdef OS_SUNOS
/*  Look up the header file entry with id id in the headers list headers.
 *
 *  Used when we find a N_EXCL symbol meaning a header file excluded
 *  because it has already been encountered.
 */
static hf_t *
lookup_hf(headers, id)
hf_t *headers;
int id;
{
	hf_t *hf;

	for (hf = headers; hf != NULL; hf = hf->hf_next)
		if (hf->hf_id == id)
			return hf;
	panic("id not found in lookup_hf");
	/* NOTREACHED */
	return NULL;	/* to keep gcc happy */

}
#endif /* OS_SUNOS */

/*  Parse a name (NAME in the grammar).
 *  If save is non zero, return a pointer to a saved copy of the name,
 *  otherwise return NULL.  Names are saved via an alloc() on ap.
 */
const char *
parse_name(p_s, ap)
const char **p_s;
alloc_pool_t *ap;
{
	const char *s;
	char *name;
	size_t len;

	s = *p_s;

	 
	/*  Gcc 2.3.3 seems to put out spaces in names, so skip them.
	 */
	while (isspace(*s))
		++s;

	/*  The test for ':' is needed because gcc 2.3.1 puts out symbol
	 *  table entries containing empty names (i.e. nothing before the
	 *  ':') for unnamed enums.  SunOS f77 emits `block.data' as a
	 *  symbol, so we allow dots in names.  I don't remember why `$'
	 *  is allowed.
	 */
	if (*s != ':' && !isalpha(*s) && *s != '_' && *s != '$' && *s != '.')
		panic("bad name in parse_name");

	while (isalnum(*s) || *s == '_' || *s == '$' || *s == '.')
		++s;
	
	if (ap != NULL) {
		len = s - *p_s;
		name = alloc(ap, len + 1);
		(void) strncpy(name, *p_s, len);
		name[len] = '\0';
	}
	else {
		name = NULL;
	}
	
	*p_s = s;
	return name;
}

/*  Return a pointer to a munged saved copy of fname.  Ext is true
 *  if name is an external symbol generated by the linker.  These
 *  symbols are expected to have an underscore prepended - if it
 *  is there it is stripped, otherwise the returned name is
 *  enclosed in brackets (e.g. "[name]").
 *
 *  If modsepc is not NUL, and it occurs in the name, we split the name
 *  into a module name and function name at the seperator character.
 *
 *  We use alloc() to get the space.
 *
 *  NOTE: this routine is not used on the DECstation 3100.
 */
void
parse_fname(ap, name, modsepc, ext, p_modname, p_fname)
alloc_pool_t *ap;
const char *name;
int modsepc;
bool ext;
const char **p_modname, **p_fname;
{
	const char *modname;
	char *fname;
	
#ifdef COFF_SUN386
	/*  On the 386i, external symbols don't seem to
	 *  have underscores prepended, so we force ext
	 *  to false.
	 */
	ext = FALSE;
#endif
	
	if (ext && *name != '_') {
		modname = NULL;
		fname = alloc(ap, strlen(name) + 3);
		sprintf(fname, "[%s]", name);
	}
	else {
		const char *cptr;
		char *pos;
		size_t len;

		if (ext)
			name++;
		
		for (cptr = name; *cptr != '\0' && *cptr != ':'; cptr++)
			;
		
		len = cptr - name;
		
		fname = alloc(ap, len + 1);
		memcpy(fname, name, len);
		fname[len] = '\0';

		if (modsepc == '\0' || (pos = strchr(fname, modsepc)) == NULL) {
			modname = NULL;
		}
		else {
			modname = fname;
			*pos++ = '\0';
			fname = pos;
		}
	}

	*p_modname = modname;
	*p_fname = fname;
}
#endif /* !ST_TE */

fsyminfo_t *
make_fsyminfo(ap, symno)
alloc_pool_t *ap;
int symno;
{
	fsyminfo_t *fs;

	fs = (fsyminfo_t *)alloc(ap, sizeof(fsyminfo_t));
	fs->fs_initial_lno_fil = NULL;
	fs->fs_initial_sline_offset = 0;
	fs->fs_symno = symno;
	fs->fs_cblist = 0;
#if defined(ARCH_SUN386) && defined(OS_SUNOS)
	fs->fs_coff_lno_start = 0;
	fs->fs_coff_lno_lim = 0;
#endif
	return fs;
}

/*  Allocate a new stf_t structure with alloc() and fill in the fields.
 */
stf_t *
make_stf(ap, name, st, symno, language, addr)
alloc_pool_t *ap;
const char *name;
symtab_t *st;
int symno;
language_t language;
taddr_t addr;
{
#ifdef AO_ELF
	int ar;
#endif
	stf_t *stf;

	stf = (stf_t *)alloc(ap, sizeof(stf_t));
	stf->stf_name = name;
	stf->stf_language = language;
	stf->stf_compiler_type = CT_UNKNOWN;
	stf->stf_symtab = st;
	stf->stf_symno = symno;
	stf->stf_addr = addr;
	stf->stf_flags = 0;
	stf->stf_fmap = NULL;
	stf->stf_mapsize = 0;
#ifdef ARCH_CLIPPER
	stf->stf_addrlist = NULL;
#endif
#ifndef ST_TE
	stf->stf_snlist = NULL;
	stf->stf_ftypes = NULL;
#endif
#ifdef AO_ELF
	stf->stf_objpath_hint = NULL;
	stf->stf_objname = NULL;
	stf->stf_obj_mtime = 0;
	stf->stf_global_prefix = NULL;
	stf->stf_symio = NULL;

	for (ar = 0; ar < AR_NTYPES; ++ar) {
		stf->stf_range[ar].base = 0;
		stf->stf_range[ar].lim = 0;
	}
#endif
	return stf;
}

/*  Allocate a new fil_t structure with alloc() and fill in the fields.
 */
fil_t *
make_fil(stf, parblock, path_hint, next)
stf_t *stf;
block_t *parblock;
const char *path_hint;
fil_t *next;
{
	fil_t *fil;
	alloc_pool_t *ap;

	ap = stf->stf_symtab->st_apool;

	fil = ci_make_fil(ap, stf->stf_name, (char *)stf,
			  ci_make_block(ap, parblock), next);
	
	fil->fi_path_hint = path_hint;
	fil->fi_language = stf->stf_language;
	fil->fi_symtab = stf->stf_symtab;

	return fil;
}

bool
find_sol_fil(sfiles, path_hint, name, p_fil)
fil_t *sfiles;
const char *path_hint, *name;
fil_t **p_fil;
{
	fil_t *fil;
	bool abspath;

	abspath = *name == '/';
	
	for (fil = sfiles; fil != NULL; fil = fil->fi_next) {
		if ((abspath || same_string(fil->fi_path_hint, path_hint)) &&
		    strcmp(fil->fi_name, name) == 0) {
			*p_fil = fil;
			return TRUE;
		}
	}

	return FALSE;
}

static fil_t *
add_sol_fil(ap, sfiles, sofil, name)
alloc_pool_t *ap;
fil_t *sfiles, *sofil;
const char *name;
{
	fil_t *fil;
	const char *path_hint;

	path_hint = (*name != '/') ? sofil->fi_path_hint : NULL;
	
	if (find_sol_fil(sfiles, path_hint, name, &fil))
		return fil;
	
	fil = ci_make_fil(ap, alloc_strdup(ap, name), (char *)NULL,
			  ci_make_block(ap, (block_t *)NULL), sofil->fi_next);
		
	fil->fi_path_hint = path_hint;

	/*  The only thing these entries are used for is displaying
	 *  source code, but we need fi_symtab because things like
	 *  open_source_file() go via it to get an alloc pool.
	 */
	fil->fi_symtab = sofil->fi_symtab;

	fil->fi_flags |= FI_DONE_VARS | FI_DONE_TYPES;

	sofil->fi_next = fil;

	return fil;
}

#ifndef ST_TE
#define SYMSET_SIZE	256

/*  Build the map of interesting symbols for scan_symtab.  We could do this
 *  at compile time with some effort, but it's much less hassle to do it at
 *  run time like this.
 */
static void
build_symset(aout_symset, ofile_symset)
char *aout_symset, *ofile_symset;
{
	static int common_syms[] = {
		N_BCOMM, N_STSYM, N_GSYM, N_LCSYM, N_FUN, N_SOL,
#ifdef OS_SUNOS
		N_BINCL, N_EXCL, N_EINCL,
#endif
#ifdef N_XLINE
		N_XLINE,
#endif
#ifdef AO_ELF
		N_UNDF,
#endif
	};
	static int aout_only_syms[] = {
		N_SO,
		N_TEXT, N_TEXT | N_EXT, N_BSS | N_EXT, N_DATA | N_EXT,
#if defined(OS_ULTRIX) || defined(ARCH_CLIPPER)
		N_DATA,
#endif
#ifdef AO_ELF
		N_OPT,
#endif
#ifdef N_MAIN
		N_MAIN
#endif
	};
	int i;

	memset(ofile_symset, '\0', SYMSET_SIZE);
	for (i = 0; i < sizeof(common_syms) / sizeof(common_syms[0]); ++i)
		ofile_symset[common_syms[i]] = TRUE;

	memcpy(aout_symset, ofile_symset, SYMSET_SIZE);
	for (i = 0; i < sizeof(aout_only_syms) / sizeof(aout_only_syms[0]); ++i)
		aout_symset[aout_only_syms[i]] = TRUE;
}

bool
parse_number(p_s, p_val)
const char **p_s;
int *p_val;
{
	const char *s;
	int res;

	s = *p_s;
	
	if (!isdigit(*s))
		return FALSE;
	
	res = 0;
	while (*s != '\0' && isdigit(*s))
		res = res * 10 + *s++ - '0';
	
	*p_s = s;
	*p_val = res;
	return TRUE;
}

bool
parse_typenum(p_s, p_fnum, p_tnum)
const char **p_s;
int *p_fnum, *p_tnum;
{
	const char *s;
	bool ok;

	s = *p_s;
	if (*s == '(') {
		++s;
		ok = (parse_number(&s, p_fnum) &&
		      *s++ == ',' &&
		      parse_number(&s, p_tnum) &&
		      *s++ == ')');
	}
	else {
		*p_fnum = 0;
		ok = parse_number(&s, p_tnum);
	}

	if (ok)
		*p_s = s;
	return ok;
}

bool
char_to_utypecode(c, p_typecode)
int c;
typecode_t *p_typecode;
{
	switch (c) {
	case 's':
		*p_typecode = TY_U_STRUCT;
		break;
	case 'u':
		*p_typecode = TY_U_UNION;
		break;
	case 'e':
		*p_typecode = TY_U_ENUM;
		break;
	default:
		return FALSE;
	}

	return TRUE;
}

/*  BUG: this is gross, and wrong to boot.  The number of basic types
 *       varies between f77 compilers.  See the comment about this
 *       in get_fi_vars().
 */
#define N_FIXED_FORTRAN_TYPES	9

bool
symtab_name_to_sn(snlist, name, p_sn)
snlist_t *snlist;
const char *name;
snlist_t **p_sn;
{
	snlist_t *sn;

	for (sn = snlist; sn != NULL; sn = sn->sn_next) {
		if (strcmp(sn->sn_symtab_name, name) == 0) {
			*p_sn = sn;
			return TRUE;
		}
	}

	return FALSE;
}

/*  Do a prescan of a symbol table.  We don't load in all of the symbol
 *  table on startup as that costs a lot in startup time for big symbol
 *  tables.  Instead we load enough to get us going (basically function
 *  and global variable names and addresses) and pull other bits in as
 *  needed.  This is a big win in normal use because the average debugger
 *  run touches only a small number of functions and global variables.
 *  Most of the symbol table is never read in at all.
 *
 *  The Sun C compiler has a system for reducing symbol table size by
 *  only including header file symbol table information once.  We have
 *  to do a fair amount of bookkeeping to resolve the header file
 *  references.
 *
 *  For most things that we load in this pass (like functions, global
 *  variables etc) we record the symbol number range of the object.
 *  This is so we can pull in more information when needed (e.g. the
 *  local variables of a function, the globals of a source file).
 */
void
scan_symtab(st, path, stf, p_flist, p_mainfunc_name)
symtab_t *st;
const char *path;
stf_t *stf;
func_t **p_flist;
const char **p_mainfunc_name;
{
	static int first_call = TRUE;
	static char aout_symset[SYMSET_SIZE], ofile_symset[SYMSET_SIZE];
	char *symset;
	snlist_t *sn;
	block_t *rootblock;
	fil_t *solfil;
	symio_t *symio;
#ifdef OS_SUNOS
	hf_t *headers, *hf;
#endif
#ifdef AO_ELF
	off_t file_offset, next_file_offset;
#endif
	hf_t **fmap, **istack;
	ao_stdata_t *ast;
	func_t *curfunc, *flist;
	const char *name, *mainfunc_name, *path_hint, *cptr;
	int symno;
	unsigned sline_offset;
	int isp, istack_size, mapsize, max_mapsize, nsyms;
	alloc_pool_t *ap, *tmp_ap;
	bool seen_func, doing_ofile;
#ifdef OS_ULTRIX
	int last_data_symno, have_unused_datasym;
	nlist_t data_nm;
	const char *lastname;
#endif

	doing_ofile = stf != NULL;
	
	if (first_call) {
		build_symset(aout_symset, ofile_symset);
		first_call = FALSE;
	}

	ast = AO_STDATA(st);
	flist = NULL;
	curfunc = NULL;
	rootblock = get_rootblock();

	
#ifdef OS_SUNOS
	headers = NULL;
#endif
	
#ifdef OS_ULTRIX
	/*  FRAGILE CODE
	 *
	 *  The Ultrix C compiler has a charming habit of correcting
	 *  itself over symbol addresses.  A symbol table frequently
	 *  has an N_DATA symbol with one address, followed soon
	 *  after by an N_STSYM for the same symbol name with a slightly
	 *  different address.  The N_DATA address is the correct one.
	 *
	 *  To cope with this, we remember the symbol number of N_DATA
	 *  symbols, and correct the N_STSYM address if necessary.
	 *
	 *  The compiler also tends to put out N_DATA symbols immediately
	 *  after N_STSYM symbols, but in these cases the addresses
	 *  seem to always be the same.
	 */
	have_unused_datasym = FALSE;
	last_data_symno = 0; /* to satisfy gcc */
#endif

#ifdef AO_ELF
	file_offset = next_file_offset = 0;
#endif

	max_mapsize = 32;
	fmap = (hf_t **) e_malloc(max_mapsize * sizeof(hf_t *));

	isp = 0;
	istack_size = 8;
	istack = (hf_t **) e_malloc(istack_size * sizeof(hf_t *));

	symno = -1;
	path_hint = NULL;
	seen_func = 0; /* to satisfy gcc */
	ap = st->st_apool;
	tmp_ap = alloc_create_pool();
	mainfunc_name = NULL;
	solfil = NULL;
	sline_offset = 0;

	if (doing_ofile) {
#ifdef AO_ELF
		symio = stf->stf_symio;
#else
		panic("doing_ofile set for non-ELF file in ss");
		symio = NULL;	/* to satisfy gcc */
#endif
		
		symset = ofile_symset;
		stf->stf_fnum = 0;
		fmap[0] = (hf_t *) alloc(ap, sizeof(hf_t));
		fmap[0]->hf_stf = stf;
		fmap[0]->hf_id = -1;
		fmap[0]->hf_next = NULL;
		mapsize = 1;
	       
	}
	else {
		symset = aout_symset;
		symio = ast->st_text_symio;
		
		mapsize = 0; /* for lint */
	}
	
	nsyms = get_symio_nsyms(symio);
	
	for (;;) {
		nlist_t nm;
		
		symno = findsym(symio, symno + 1, &nm, symset);

		if (symno == nsyms)
			break;
		
		switch(nm.n_type) {
			const char *symtab_name;
			language_t lang;
#ifdef AO_ELF
			bool has_debug_syms;
#endif
			
#ifdef AO_ELF
		case N_UNDF:
			file_offset = next_file_offset;
			next_file_offset = file_offset + nm.n_value;

			add_extra_string_offset(symio, symno, file_offset);
			
			break;
#endif

#ifdef N_MAIN
		case N_MAIN:
			mainfunc_name = symstring(symio, symno);
			break;
#endif
			
		case N_SO:
			solfil = NULL;
			sline_offset = 0;
			
#ifdef COFF_SUN386
			/*  The Sun 386i C compiler seems to put out
			 *  spurious N_SO symbols for functions.
			 *  The bad ones all seem to have a non zero
			 *  n_dbx_desc field, so skip these.
			 */
			if (nm.n_dbx_desc != 0)
				break;
#endif /* COFF_SUN386 */
			if (curfunc != NULL) {
				AO_FSYMDATA(curfunc)->fs_symlim = symno;
				curfunc = NULL;
			}
			if (stf != NULL)
				stf->stf_symlim = symno;

			name = symstring(symio, symno);

			/*  4.0 cc puts paths ending in '/' just before
			 *  the source files that follow.
			 */
			if (name[strlen(name) - 1] == '/') {
				path_hint = name;
				break;
			}

			if (stf != NULL) {
				wrapup_stf(stf, fmap, mapsize);
				reverse_stf_funclist(stf);
			}

#ifndef AO_ELF
			if (strcmp(name, "libg.s") == 0) {
				stf = NULL;
				break;
			}
#endif
			
			lang = srctype(name);
#ifdef OS_SUNOS_4
			/*  F77 files can be compiled with the f90 compiler,
			 *  so try and work out from the symbol table which
			 *  was used.
			 */
			if (IS_FORTRAN(lang)) {
				const char *s;

				s = symstring(symio, symno + 1);

				if (strncmp(s, "byte:", 5) == 0) {
					lang = LANG_F90;
				}
				else if (strncmp(s, "integer*2:", 10) == 0) {
					lang = LANG_F77;
				}
			}
#endif
			
			stf = make_stf(ap, name, st, symno, lang, nm.n_value);
#ifdef AO_ELF
			stf->stf_symio = symio;
#endif
			st->st_sfiles = make_fil(stf, rootblock, path_hint,
						 st->st_sfiles);
			stf->stf_fil = st->st_sfiles;

			if (isp != 0)
				panic("unmatched N_BINCL");

			stf->stf_fnum = 0;
			fmap[0] = (hf_t *) alloc(ap, sizeof(hf_t));
			fmap[0]->hf_stf = stf;
			fmap[0]->hf_id = -1;
			fmap[0]->hf_next = NULL;
			mapsize = 1;
			path_hint = NULL;
			seen_func = FALSE;
			symset[N_SLINE] = TRUE;
#ifdef OS_ULTRIX
			have_unused_datasym = FALSE;
#endif
			break;

		case N_SOL:
			/*  Names are relative to the last N_SO, so we need
			 *  one.
			 */
			if (stf != NULL) {
				solfil = add_sol_fil(ap, st->st_sfiles,
						     stf->stf_fil,
						     symstring(symio, symno));

				if (solfil == stf->stf_fil)
					solfil = NULL;

				if (curfunc != NULL && solfil != NULL &&
				    (solfil->fi_funclist == NULL ||
				     solfil->fi_funclist->fl_func != curfunc)) {
					add_to_fil_funclist(ap,
							    solfil, curfunc);
				}
				    
			}
			break;

#ifdef AO_ELF
		case N_OPT:
			/*  We shouldn't get N_OPT before the first N_SO,
			 *  but lets not core dump if we do.
			 */
			if (stf == NULL)
				break;
			
			cptr = symstring(symio, symno);
			elf_handle_optsym(ap, cptr, stf->stf_language,
					  &stf->stf_global_prefix,
					  &stf->stf_compiler_type,
					  &has_debug_syms);
			stf->stf_obj_mtime = nm.n_value;
			break;
#endif
			
		case N_SLINE:
			if (!seen_func && stf != NULL)
				stf->stf_flags |= STF_LNOS_PRECEDE_FUNCS;
			symset[N_SLINE] = FALSE;
			break;

#ifdef N_XLINE
		case N_XLINE:
			sline_offset = nm.n_desc << 16;
			break;
#endif
		
#ifdef OS_SUNOS
		case N_BINCL:
		case N_EXCL:
			if (stf == NULL)
				panic("header outside source file in ss");
			
			if (mapsize == max_mapsize) {
				max_mapsize *= 2;
				fmap = (hf_t **) e_realloc((char *)fmap,
						max_mapsize * sizeof(hf_t *));
			}
			
			if (nm.n_type == N_EXCL) {
				hf = lookup_hf(headers, (int)nm.n_value);
				fmap[mapsize++] = hf;
				break;
			}
			
			if (isp == istack_size) {
				istack_size *= 2;
				istack = (hf_t **) e_realloc((char *)istack,
						istack_size * sizeof(hf_t *));
			}
			
			istack[isp] = (hf_t *) alloc(ap, sizeof(hf_t));
			
			istack[isp]->hf_next = headers;
			headers = istack[isp++];

			name = alloc_strdup(ap, symstring(symio, symno));
			headers->hf_stf = make_stf(ap, name, st, symno,
						   st->st_sfiles->fi_language,
						   nm.n_value);

			if (nm.n_type == N_EXCL) {
				headers->hf_stf->stf_fil = NULL;
			}
			else {
				headers->hf_stf->stf_fil =
						make_fil(headers->hf_stf,
							 (block_t *)NULL,
							 (const char *)NULL,
							 (fil_t *)NULL);
			}

			headers->hf_stf->stf_fnum = mapsize;
			headers->hf_id = nm.n_value;
			fmap[mapsize++] = headers;
			break;

		case N_EINCL:
			if (isp == 0)
				panic("unmatched N_EINCL");
			
			--isp;
			istack[isp]->hf_stf->stf_symlim = symno;
			
			if (!doing_ofile)
				reverse_stf_funclist(stf);
			
			break;
#endif /* OS_SUNOS */
		case N_STSYM:
		case N_GSYM:
		case N_LCSYM:
			if (IS_FORTRAN(stf->stf_language) &&
			    symno - stf->stf_symno <= N_FIXED_FORTRAN_TYPES)
				break;

			cptr = symstring(symio, symno);
			name = parse_name(&cptr, tmp_ap);

#ifdef AO_ELF
			if (doing_ofile) {
				if (symtab_name_to_sn(stf->stf_snlist,
						      name, &sn)) {
					sn->sn_symno = symno;
					break;
				}

				/*  The Sun cc (SPARCompiler 3.0) leaves a
				 *  few symbols out of the .stab.index
				 *  section in the linked file, so don't
				 *  complain about extra symbols - just
				 *  silently add them.  These symbols always
				 *  seem to be local static variables in an
				 *  inner block scope, but I can't reproduce
				 *  this with a small test case.
				 */
			}

			/*  SPARCompiler 3.0 sometimes emit duplicate
			 *  symbols, of which the first seems to be bogus.
			 *  For example, there are two entries for
			 *  file_string in the symbol table for
			 *  test/C/core/f.c.
			 *
			 *  Just ignore the first one.
			 */
			if (stf->stf_snlist != NULL &&
			    strcmp(stf->stf_snlist->sn_symtab_name, name) == 0){
				stf->stf_snlist->sn_symno = symno;
				break;
			}

			symtab_name = name;
			name = elf_name_from_symtab_name(stf->stf_global_prefix,
							 symtab_name);
#else
			symtab_name = name;
#endif

			sn = push_symname(ap, &stf->stf_snlist,
					  symtab_name, name, symno);
			
#ifdef OS_ULTRIX
			sn->sn_addr = 0;
			
			if (have_unused_datasym) {
				lastname = symstring(symio, last_data_symno);
				
				if (*lastname == '_' &&
				    strcmp(lastname + 1, sn->sn_name) == 0) {
					getsym(symio, last_data_symno,
					       &data_nm);
					sn->sn_addr = data_nm.n_value;
				}
				
				have_unused_datasym = FALSE;
						
			}
#endif
			
			/*  The epcf90 emits continuation symbol table lines
			 *  with N_GSYM as the symbol, so skip them.
			 */
			for (;;) {
				size_t len;

				len = strlen(name);
				if (len == 0 || name[len - 1] != '\\')
					break;

				++symno;
				getsym(symio, symno, &nm);
				name = symstring(symio, symno);
			}
			
			break;
#ifdef OS_ULTRIX
		case N_DATA:
			last_data_symno = symno;
			have_unused_datasym = TRUE;
			break;
#endif
		case N_FUN:
			name = symstring(symio, symno);

			/*  Some compilers (e.g. gcc) put read only strings
			 *  in the text segment, and generate N_FUN symbols
			 *  for them.  We're not interested in these here.
			 */
			if ((cptr = strchr(name, ':')) == NULL ||
			    (cptr[1] != 'F' && cptr[1] != 'f'))
				break;

			if (curfunc != NULL) {
				AO_FSYMDATA(curfunc)->fs_symlim = symno;
				curfunc = NULL;
			}

			seen_func = TRUE;

#ifdef AO_ELF
			if (doing_ofile) {
				curfunc = note_ofile_function(ap, tmp_ap, stf,
							      path, solfil,
							      sline_offset,
							      name, symno);
				break;
			}
#endif
			add_function_to_symtab(st, &flist, name,
					       st->st_sfiles, solfil,
					       cptr[1] == 'f', FALSE,
					       symno, nm.n_value);
			AO_FSYMDATA(flist)->fs_initial_sline_offset =
								sline_offset;
			
			curfunc = flist;
			
			if (is_fortran_void(cptr))
				curfunc->fu_type = ci_code_to_type(TY_VOID);
			
			break;
		
		case N_TEXT:
		case N_TEXT | N_EXT:
			name = symstring(symio, symno);
			
			/*  Some compilers put N_TEXT symbols out with object
			 *  file names, often with the same addresses as real
			 *  functions.  We don't want these.
			 *
			 *  We also don't want N_TEXT symbols which immediately
			 *  follow an N_FUN symbol with the same address.
			 */
			if (nm.n_type == N_TEXT) {
				if (*name != '_') {
					/*  A .o file symbol is definitely
					 *  the end of the current function,
					 *  if any.
					 */
					if (curfunc != NULL) {
						AO_FSYMDATA(curfunc)->fs_symlim = symno;
						curfunc = NULL;
					}
					break;
				}
				if (curfunc != NULL && curfunc->fu_addr == nm.n_value)
					break;
			}
			
			add_function_to_symtab(st, &flist, name,
					       (fil_t *)NULL, (fil_t *)NULL,
					       (nm.n_type & N_EXT) == 0, TRUE, 
					       symno, nm.n_value);
			
			/*  You'd expect that we'd close down the current
			 *  function here, but some C compilers put out
			 *  N_TEXT symbols in the middle of the N_SLINE
			 *  symbols for a function.
			 *
			 *  Thus we leave curfunc alone, and rely on an N_SO or
			 *  a .o file N_TEXT to terminate the current
			 *  function.
			 */
			
			break;
#ifdef ARCH_CLIPPER
		case N_DATA:
			/*  The Clipper C compiler puts out an N_LCSYM symbol
			 *  with the wrong address for static globals, then
			 *  later puts out an N_DATA with the right address.
			 *  This we keep a list of N_DATA symbols for each
			 *  file so we can check the addresses later.
			 *
			 *  Note that we can't just stick the symbol in the
			 *  global addresses list, as we may have static
			 *  globals with the same name in different files.
			 */
			name = symstring(symio, symno);
			if (*name != '_')
				break;
			insert_global_addr(ap, &stf->stf_addrlist,
					   name + 1, (taddr_t)nm.n_value);
			break;
#endif
		case N_BSS | N_EXT:
		case N_DATA | N_EXT:
			name = symstring(symio, symno);
#ifndef COFF_SUN386
			if (*name != '_')
				break;
			++name;
#endif
			insert_global_addr(ap, &st->st_addrlist, name,
					   ast->st_base_address +
						          (taddr_t)nm.n_value);
			break;
		case N_BCOMM:
			symno = push_common_block(st, stf, curfunc,
						  symio, symno);
			break;
		default:
			panic("unexpected symbol in init_syms");
		}
	}

	if (curfunc != NULL) {
		AO_FSYMDATA(curfunc)->fs_symlim = symno;
		curfunc = NULL;
	}
	
	if (stf != NULL) {
		wrapup_stf(stf, fmap, mapsize);

		if (!doing_ofile)
			reverse_stf_funclist(stf);
		
		stf->stf_symlim = symno;
	}

	free((char *)fmap);
	free((char *)istack);
	alloc_free_pool(tmp_ap);

	if (!doing_ofile)
		*p_flist = flist;

	if (p_mainfunc_name != NULL)
		*p_mainfunc_name = mainfunc_name;
}

snlist_t *
push_symname(ap, p_snlist, symtab_name, name, symno)
alloc_pool_t *ap;
snlist_t **p_snlist;
const char *symtab_name, *name;
int symno;
{
	snlist_t *sn;
	
	sn = (snlist_t *)alloc(ap, sizeof(snlist_t));
	sn->sn_symno = symno;
	sn->sn_symtab_name = alloc_strdup(ap, symtab_name);
	sn->sn_name = sn->sn_symtab_name + (name - symtab_name);
	sn->sn_next = *p_snlist;
	*p_snlist = sn;

	return sn;

}
#endif /* !ST_TE */

#ifdef AO_ELF
static func_t *
note_ofile_function(ap, tmp_ap, stf, path, solfil, sline_offset, namestr, symno)
alloc_pool_t *ap, *tmp_ap;
stf_t *stf;
const char *path;
fil_t *solfil;
unsigned sline_offset;
const char *namestr;
int symno;
{
	func_t *f;
	fil_t *fil;
	const char *modname, *fname;
	int modsepc;

	fil = stf->stf_fil;
	modsepc = (fil->fi_language == LANG_F90) ? '$' : '\0';
			
	parse_fname(tmp_ap, namestr, modsepc, FALSE, &modname, &fname);

	f = name_and_fil_to_func(fname, fil);

	if (f == NULL) {
		errf("Unexpected function `%s' (symbol #%d) found in %s - %s",
		     fname, symno, path, "ignored");
	}
	else {
		fsyminfo_t *fs;

		fs = AO_FSYMDATA(f);
		fs->fs_symno = symno;
		fs->fs_initial_lno_fil = (solfil != NULL) ? solfil : fil;
		fs->fs_initial_sline_offset = sline_offset;
		
		if (solfil != NULL)
			add_to_fil_funclist(ap, solfil, f);
	}

	return f;
}
#endif /* AO_ELF */

static void
add_to_fil_funclist(ap, fil, f)
alloc_pool_t *ap;
fil_t *fil;
func_t *f;
{
	funclist_t *fl;
		
	fl = (funclist_t *)alloc(ap, sizeof(funclist_t));
	fl->fl_func = f;
	fl->fl_next = fil->fi_funclist;
	fil->fi_funclist = fl;
}
	
void
add_function_to_symtab(st, p_flist, namestr, fil, solfil,
		       is_static, is_textsym, symno, addr)
symtab_t *st;
func_t **p_flist;
const char *namestr;
fil_t *fil, *solfil;
bool is_static, is_textsym;
int symno;
taddr_t addr;
{
	fsyminfo_t *fs;
	func_t *f;
	const char *modname, *fname;
	int modsepc;
		
	modsepc = (fil != NULL && fil->fi_language == LANG_F90) ? '$' : '\0';
			
	parse_fname(st->st_apool, namestr, modsepc,
		    is_textsym && !is_static, &modname, &fname);

	fs = make_fsyminfo(st->st_apool, symno);

	/*  I put this in as a workaround for a bug (with a fixme
	 *  comment.  Now I can't remember why it was needed.
	 *
	 *  TODO: find out why it was needed, and maybe do something
	 *        better.
	 */
	fs->fs_symlim = 1;
	
	fs->fs_initial_lno_fil = (solfil != NULL) ? solfil : fil;
	
	f = ci_make_func(st->st_apool, fname, addr, st, fil, *p_flist);
	f->fu_symdata = (char *)fs;
	f->fu_predata = NULL;

	if (fil == NULL) {
		f->fu_flags |= FU_NOSYM | FU_DONE_BLOCKS | FU_DONE_LNOS;
	}
	else {
		add_to_fil_funclist(st->st_apool, fil, f);

		if (solfil != NULL)
			add_to_fil_funclist(st->st_apool, solfil, f);
	}

	if (is_static)
		f->fu_flags |= FU_STATIC;

	if (modname != NULL)
		add_module_function(st, modname, f);

	*p_flist = f;
}
			
			
static bool
is_fortran_void(cptr)
const char *cptr;
{
	/*  BUG: should look at the type properly.
	 *       There is no guarantee that type 11 is always void.
	 */
	return strcmp(cptr, ":F11") == 0;
}

/*  Adjust the addresses of all the global variables associated
 *  with source files in flist.  Called when a shared library
 *  mapping address changes across runs of the target.
 */
static void
adjust_fil_vars_addr_base(flist, delta)
fil_t *flist;
long delta;
{
	fil_t *fil;
	var_t *v;

	for (fil = flist; fil != NULL; fil = fil->fi_next) {
		if (AO_FIDATA(fil) == NULL)
			continue;
		
		AO_FIDATA(fil)->stf_addr += delta;
		
		if (fil->fi_flags & FI_DONE_VARS) {
			for (v = fil->fi_block->bl_vars;
			     v != NULL;
			     v = v->va_next)
				v->va_addr += delta;
		}
	}
}

/*  Deal with a change in the text offset of a symbol table.  This may
 *  be necessary when re-running the target as shared libraries may be
 *  mapped at different addresses.  It's also necessary when we have
 *  preloaded symbol tables with a nominal offset of zero.
 *
 *  We adjust the following:
 *
 *	function and line number addresses
 *	symbol table address to text file offset
 *	addresses of global variables
 *
 *  We don't change breakpoint addresses here - we do that by removing
 *  and recreating all breakpoints just after starting the target.
 */
void
change_base_address(st, new_addr)
symtab_t *st;
taddr_t new_addr;
{
	long delta;
	ao_stdata_t *ast;

	ast = AO_STDATA(st);
	delta = new_addr - ast->st_base_address;

	if (delta != 0) {
		adjust_addrlist_addr_offset(st->st_addrlist, delta);
		ast->st_addr_to_fpos_offset += delta;
		adjust_functab_text_addr_base(st->st_functab,
					      st->st_funclist, delta);
		adjust_fil_vars_addr_base(st->st_sfiles, delta);
		ast->st_base_address = new_addr;
	}
}


#endif /* AO_TARGET */
