/* data.c - higher level routines to read and write target data */

/*  Copyright 1991 Mark Russell, University of Kent at Canterbury.
 *
 *  You can do what you like with this source code as long as
 *  you don't try to make money out of it and you include an
 *  unaltered copy of this message (including the copyright).
 */

char ups_data_c_sccsid[] = "@(#)data.c	1.17 04 Jun 1995 (UKC)";

#include <mtrprog/ifdefs.h>

#include <sys/types.h>
#include <signal.h>
#include <setjmp.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>

#include <local/ukcprog.h>
#include <mtrprog/utils.h>

#include "ups.h"
#include "target.h"
#include "data.h"
#include "obj_stack.h"
#include "obj_target.h"
#include "state.h"
#include "util.h"

/*  BUG: we assume here that a register is the same size as
 *       a pointer, and that address and data registers are
 *       the same.
 */
#define REGBYTES	sizeof(taddr_t)

static volatile int Got_sigill;

static jmp_buf Sigill_env;

static void catch_sigill PROTO((int sig));
static int charval PROTO((int n));

static void
catch_sigill(unused_sig)
int unused_sig;
{
	Got_sigill = 1;
	longjmp(Sigill_env, 1);
}

/*  Extract a floating point value from the value_t union.
 *
 *  On the VAX (and possibly other machines) certain bit patterns cause
 *  exceptions when treated as a floating point value, so we bracket
 *  the extraction with a handler for these exceptions.
 */
const char *
get_real(words_big_endian, vl, want_hex, is_float)
bool words_big_endian;
value_t vl;
bool want_hex, is_float;
{
	static char buf[60];
	double d;
	void (*old_sigill_func)PROTO((int sig));

	if (want_hex) {
		if (is_float) {
			sprintf(buf, "<0x%08x>", vl.vl_int);
		}
		else {
			unsigned msword, lsword;

			if (words_big_endian) {
				msword = vl.vl_ints[0];
				lsword = vl.vl_ints[1];
			}
			else {
				msword = vl.vl_ints[1];
				lsword = vl.vl_ints[0];
			}
			
			sprintf(buf, "<0x%08x%08x>", msword, lsword);
		}
		return buf;
	}

	old_sigill_func = signal(SIGILL, catch_sigill);
	Got_sigill = FALSE;
	
	if (setjmp(Sigill_env) == 0)
		d = is_float ? vl.vl_float : vl.vl_double;
	else
		d = 0;

	(void) signal(SIGILL, old_sigill_func);

	if (Got_sigill) {
		/*  Note: get_real checks for this output when reading
		 *        a value so the exact format of the output
		 *	  is important.
		 */
		if (is_float) {
			sprintf(buf, "<illegal float 0x%08x>", vl.vl_int);
		}
		else {
			unsigned msword, lsword;

			if (words_big_endian) {
				msword = vl.vl_ints[0];
				lsword = vl.vl_ints[1];
			}
			else {
				msword = vl.vl_ints[1];
				lsword = vl.vl_ints[0];
			}
			
			sprintf(buf, "<illegal double 0x%x%08x>",
				msword, lsword);
		}
	}
	else {
		char *s;

		if (is_float)
			sprintf(buf, "%.6g", d);
		else
			sprintf(buf, "%.12g", d);
		
		/*  We always want a decimal point even if the value
		 *  is integral.
		 */
		s = buf;
		if (*s == '-')
			++s;
		while (isdigit(*s))
			++s;
		if (*s == '\0')
			strcpy(s, ".0");
	}

	return buf;
}

/*  Read a string from the data area.
 */
int
dgets(xp, addr, optr, max_nbytes)
target_t *xp;
taddr_t addr;
register char *optr;
int max_nbytes;
{
	char ibuf[4], *olim;
	register char *iptr;

	olim = optr + max_nbytes - 1;
	iptr = ibuf + sizeof(ibuf);
	
	while (optr < olim) {
		if (iptr == ibuf + sizeof(ibuf)) {
			if (dread(xp, addr, ibuf, sizeof(ibuf)) != 0)
				return -1;
			iptr = ibuf;
			addr += sizeof(ibuf);
		}
		if (*iptr == '\0')
			break;
		*optr++ = *iptr++;
	}
	
	*optr++ = '\0';
	
	return 0;
}

/*  Nominal address of the registers. Recognised by dread()
 *
 *  Just pick a value that can't be confused with a stack or data address.
 *
 *  BUG: this is unsafe now that we have mmap() and shared libraries -
 *  something could actually be mapped at this address.
 */
#define REG_ADDR	((taddr_t)(0xf0000000 - -REG_MIN_REGNO))

/*  Maximum number of registers.  Used only for limiting the range of
 *  pseudo addresses that will be interpreted as register numbers.
 */
#define MAX_REGS	256

/*  Is addr an encoded register address?
 */
#define IS_REG_ADDR(addr)  (addr >= (REG_ADDR - -REG_MIN_REGNO) && \
			    addr < REG_ADDR + MAX_REGS)

/*  Convert a register number to an "address" that will be recognised by
 *  dread().
 */
taddr_t
regno_to_addr(regno)
int regno;
{
	if (regno > MAX_REGS)
		panic("bad reg addr n rta");
	return REG_ADDR + regno;
}

int
dread_fpval(xp, addr, is_reg, is_double, buf)
target_t *xp;
taddr_t addr;
bool is_reg, is_double;
char *buf;
{
	int res;
	fpval_t fpval;

	if (IS_REG_ADDR(addr)) {
		res = xp_read_fpreg(xp, (int)(addr - REG_ADDR),
							is_double, &fpval);
	}
	else if (is_reg) {
		/*  A register that's been saved on the stack.
		 */
		res = xp_read_fpval(xp, addr, is_double, &fpval);
	}
	else
		return dread(xp, addr, buf,
				is_double ? sizeof(double) : sizeof(float));

	if (res != 0)
		return -1;

	if (is_double)
		*(double *)buf = fpval.d;
	else
		*(float *)buf = fpval.f;

	return 0;
}

/*  Read n bytes into buf from the data or stack area of the target process.
 *  We deduce from addr whether we are supposed to read the stack or
 *  the data area.
 *
 *  Return 0 for success, -1 for failure.
 */
int
dread(xp, addr, buf, nbytes)
target_t *xp;
taddr_t addr;
voidptr buf;
size_t nbytes;
{
	if (IS_REG_ADDR(addr)) {
		taddr_t regval;
		int regno;
		char *rptr;

		rptr = (char *)&regval;
		regno = addr - REG_ADDR;

		if (xp_readreg(xp, regno, &regval) != 0)
			return -1;
		
		if (nbytes == REGBYTES) {
			memcpy(buf, rptr, REGBYTES);
		}
		else if (nbytes > 0 && nbytes < REGBYTES) {
			if (xp->xp_words_big_endian)
				rptr += REGBYTES - nbytes;

			memcpy(buf, rptr, nbytes);
		}
		else if (nbytes == 2 * REGBYTES) {
			/*  We assume this is a pair of registers.
			 *  BUG: why?
			 */
			memcpy(buf, rptr, REGBYTES);
			
			if (xp_readreg(xp, regno + 1, &regval) != 0)
				return -1;
			
			memcpy((char *)buf + REGBYTES, rptr, REGBYTES);
		}
		else {
			/*  BUG: we can get here if we got an address from
			 *       the target in the range we use for register
			 *       addresses, so don't panic.
			 *
			 *  We need to dump the magic address scheme for
			 *  registers.
			 */
			return -1;
		}
		return 0;
	}

	return xp_read_data(xp, addr, buf, nbytes);
}

int
dwrite(xp, addr, buf, nbytes)
target_t *xp;
taddr_t addr;
constvoidptr buf;
size_t nbytes;
{
	if (!target_process_exists(xp))
		panic("dwrite called with no target process");

	if (IS_REG_ADDR(addr)) {
		taddr_t regval;
		int regno;
		char *rptr;

		rptr = (char *)&regval;
		regno = addr - REG_ADDR;

		if (nbytes == REGBYTES) {
			memcpy(rptr, buf, REGBYTES);
		}
		else if (nbytes > 0 && nbytes < REGBYTES) {
			if (xp->xp_words_big_endian)
				rptr += REGBYTES - nbytes;

			memcpy(rptr, buf, nbytes);
		}
		else if (nbytes == 2 * REGBYTES) {
			/*  We assume this is a pair of registers.
			 *  BUG: why?
			 */
			memcpy(rptr, (char *)buf + REGBYTES, REGBYTES);
			
			if (xp_setreg(xp, regno + 1, regval) != 0)
				return -1;
			
			memcpy(rptr, buf, REGBYTES);
		}
		else {
			panic("nbytes botch in dwrite");
		}
		
		return xp_setreg(xp, regno, regval);
	}

	return xp_write_data(xp, addr, buf, nbytes);
}

taddr_t
adjust_saved_reg_addr(xp, addr, size)
target_t *xp;
taddr_t addr;
size_t size;
{
	if (size < REGBYTES && xp->xp_words_big_endian)
		addr += REGBYTES - size;

	return addr;
}

static int
charval(n)
int n;
{
	 n = (unsigned char)n;
	 return isprint(n) ? n : '?';
}

void
dump_stack_to_file(xp, name)
target_t *xp;
const char *name;
{
	FILE *ofp;
	taddr_t sp, fp, ap, pc, addr;
	int val, nzeroes;
	const char *arg, *label;
	bool overwrite;
	bool is_labelled;

	if (xp_get_state(xp) == TS_NOTR) {
		errf("Target not running");
		return;
	}

	pc = xp_getreg(xp, UPSREG_PC);
	sp = xp_getreg(xp, UPSREG_SP);
	fp = xp_getreg(xp, UPSREG_FP);
	ap = xp_getreg(xp, UPSREG_AP);

	if (sp % sizeof(val) != 0) {
		errf("SP (0x%lx) not a multiple of %ld", sp, (long)sizeof(val));
		return;
	}

	if (!get_debug_output_path(name, "ups-stack", &arg, &name, &overwrite))
		return;

	if (*arg != '\0') {
		errf("Usage: dumpstack [> file]");
		return;
	}
		     
	if (!fopen_new_file("output file", name, overwrite, &ofp))
		return;
	
	errf("Dumping stack to file %s", name);

	fprintf(ofp, "sp:%08lx fp:%08lx ap:%08lx pc:%08lx\n", sp, fp, ap, pc);

	nzeroes = 0;
	for (addr = sp; xp_read_data(xp, addr, (char *)&val, 4)==0; addr += 4) {

		/*  On the BSDi box, dread() goes on and on succeeding,
		 *  generating huge stack dumps, hence this boredom theshold.
		 */
		if (val != 0) {
			nzeroes = 0;
		}
		else {
			if (++nzeroes == 1000) {
				fputs(
				   "Giving up after 1000 consecutive zeroes\n", 
									ofp);
				break;
			}
		}

		label = stack_addr_label(addr, &is_labelled);
		(void) fprintf(ofp, is_labelled ? "%32s " : "%-32s ", label);
		(void) fprintf(ofp, "%8lx |%8x|  |%12d|  |%c%c%c%c|\n",
				addr, val, val,
				charval(val),
				charval(val >> 8),
				charval(val >> 16),
				charval(val >> 24));
	}

	if (fclose_new_file("output file", name, TRUE, ofp))
		errf("Stack dump complete");
}
