/* exec.c - code for starting and managing target execution */

/*  Copyright 1991 Mark Russell, University of Kent at Canterbury.
 *
 *  You can do what you like with this source code as long as
 *  you don't try to make money out of it and you include an
 *  unaltered copy of this message (including the copyright).
 */

char ups_exec_c_sccsid[] = "@(#)exec.c	1.43 04 Jun 1995 (UKC)";

#include <mtrprog/ifdefs.h>

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/errno.h>

#include <local/ukcprog.h>
#include <mtrprog/utils.h>
#include <local/wn.h>
#include <local/arg.h>
#include <local/obj/obj.h>

#include "ups.h"
#include "symtab.h"
#include "va.h"
#include "ci.h"
#include "cursors.h"
#include "target.h"
#include "st.h"
#include "exec.h"
#include "dx.h"
#include "breakpoint.h"
#include "obj_stack.h"
#include "obj_target.h"
#include "obj_signal.h"
#include "obj_buildf.h"
#include "obj_bpt.h"
#include "obj_env.h"
#include "trun.h"
#include "printf.h"
#include "ui.h"
#include "menudata.h"
#include "tdr.h"
#include "state.h"

static stopres_t run_target PROTO((target_t *xp, rtype_t rtype));

void
do_menu_target_command(command)
int command;
{
	target_t *xp;
	bool want_refresh;
	stopres_t stopres;

	xp = get_current_target();

	want_refresh = TRUE;
	
	switch(command) {
	case MR_TGT_START:
		stopres = xp_start(xp);
		break;
	case MR_TGT_CONT:
		stopres = xp_cont(xp);
		break;
	case MR_TGT_STEP:
		stopres = xp_step(xp);
		break;
	case MR_TGT_NEXT:
		stopres = xp_next(xp);
		break;
	case MR_TGT_STOP:
		errf("The target is not currently running");
		stopres = SR_FAILED;	/* no redisplay */
		break;
	case MR_TGT_KILL:
		xp_kill(xp);
		stopres = SR_DIED;
		break;
	case MR_TGT_DONT_KILL:
		return;
	default:
		panic("bad cmd in dmtc");
		stopres = SR_FAILED;	/* to satisfy gcc */
	}

	refresh_target_display(xp, stopres, command == MR_TGT_START);
}

stopres_t
dx_next(xp)
target_t *xp;
{
	return run_target(xp, RT_NEXT);
}

stopres_t
dx_step(xp)
target_t *xp;
{
	return run_target(xp, RT_STEP);
}

stopres_t
dx_cont(xp)
target_t *xp;
{
	return run_target(xp, RT_CONT);
}

static stopres_t
run_target(xp, rtype)
target_t *xp;
rtype_t rtype;
{
	stopres_t stopres;
	
	indicate_target_running();
	stopres = dx_run_target(xp, rtype);
	indicate_target_stopped(xp_get_state(xp));

	return stopres;
}

void
exec_to_lnum(f, fil, lnum)
func_t *f;
fil_t *fil;
int lnum;
{
	target_t *xp;
	taddr_t addr;
	breakpoint_t *bp, *tmp_bp;
	bool just_started, have_process;
	tstate_t tstate;
	stopres_t stopres;

	xp = get_current_target();

	tstate = xp_get_state(xp);

	if (tstate == TS_HALTED) {
		errf("Can't continue target");
		return;
	}

	have_process = target_process_exists(xp);
	just_started = tstate != TS_STOPPED;

	if (map_lnum_to_addr(f, fil, lnum, &addr) != 0)
		return;

	bp = xp_addr_to_breakpoint(xp, addr);
	if (bp != NULL)
		tmp_bp = NULL;
	else
		tmp_bp = bp = xp_add_breakpoint(xp, addr);

	if (xp_disable_breakpoint(xp, (breakpoint_t *)NULL) != 0 ||
	    xp_enable_breakpoint(xp, bp) != 0) {
		stopres = SR_FAILED;
	}
	else {
		taddr_t fp;
		bool same_func;

		if (have_process) {
			func_t *cur_f;
			taddr_t pc, adjpc;
			
			get_current_func(&cur_f, &fp, &pc, &adjpc);
			same_func = f == cur_f;
		}
		else {
			fp = 0;
			same_func = FALSE;
		}
			
		for (;;) {
			stopres = have_process ? xp_cont(xp) : xp_start(xp);
			have_process = target_process_exists(xp);

			if (stopres != SR_BPT)
				break;

			if (!same_func || xp_getreg(xp, UPSREG_FP) >= fp)
				break;
		} 
	}

	if (tmp_bp != NULL)
		xp_remove_breakpoint(xp, tmp_bp);

	xp_enable_breakpoint(xp, (breakpoint_t *)NULL);

	refresh_target_display(xp, stopres, just_started);
}

void
refresh_target_display(xp, stopres, just_started)
target_t *xp;
stopres_t stopres;
bool just_started;
{
	bool old;
	objid_t obj_to_make_visible;

	if (stopres == SR_FAILED)
		return;

	save_var_display_state();
	
	if (td_have_window())
		wn_updating_off(WN_STDWIN);
	old = td_set_obj_updating(OBJ_UPDATING_OFF);

	if (stopres == SR_DIED) {
		close_target_display();
		obj_to_make_visible = NULL;
	}
	else {
		if (just_started)
			restore_file_displays();
		
		obj_to_make_visible = rebuild_display(xp);
	}

	td_set_obj_updating(old);

	if (obj_to_make_visible != NULL)
		ensure_visible(obj_to_make_visible);

	if (td_have_window())
		wn_updating_on(WN_STDWIN);
}

/*  Start the target process running.
 */
stopres_t
dx_start(xp)
target_t *xp;
{
	const char **argv;
	long rdlist;
	stopres_t stopres;

	if (setup_args(&argv, &rdlist) != 0)
		return SR_DIED;

	indicate_target_running();
	
	if (xp_create_child(xp, argv, get_environment(), rdlist,
			    &stopres) != 0)
		stopres = SR_DIED;

	indicate_target_stopped(xp_get_state(xp));

	if (stopres == SR_DIED)
		return stopres;
	if (stopres != SR_BPT)
		panic("xp not started properly in start_target");
			
	xp_set_base_sp(xp, xp_getreg(xp, UPSREG_SP));

	reinitialise_bpt_code_data();

	if (install_all_breakpoints(xp) != 0) {
		xp_kill(xp);
		return SR_DIED;
	}

	/*  If we have a breakpoint where the target is stopped, we
	 *  don't need to continue it.
	 */
	if (get_breakpoint_at_addr(xp, xp_getreg(xp, UPSREG_PC)) == NULL)
		stopres = xp_cont(xp);

	return stopres;
}

int
get_startup_stop_addrs(xp, p_main_addr, p_main_min_bpt_addr)
target_t *xp;
taddr_t *p_main_addr, *p_main_min_bpt_addr;
{
	func_t *mainfunc;

	mainfunc = xp_get_mainfunc(xp);
	*p_main_addr = mainfunc->fu_addr;
	return get_min_bpt_addr(mainfunc, p_main_min_bpt_addr);
}

stopres_t
xp_execto(xp, addr)
target_t *xp;
taddr_t addr;
{
	breakpoint_t *bp;
	int lastsig;
	const char *path;

	path = xp->xp_textpath;

	bp = xp_add_breakpoint(xp, addr);

	if (install_breakpoint(bp, xp) != 0) {
		failmesg("Can't install initial breakpoint in", "object file",
			 path);

		if (xp_remove_breakpoint(xp, bp) != 0)
			panic("can't back out uninstalled bpt");

		if (!xp_is_attached(xp))
			xp_kill(xp);

		return SR_DIED;
	}

	lastsig = 0;
	do {
		xp_restart_child(xp, lastsig, CT_CONT);
	} while (xp_get_stopres(xp) == SR_SIG);

	if (xp_remove_breakpoint(xp, bp) != 0)
		panic("can't uninstall initial breakpoint");
	
	return xp_get_stopres(xp);
}

ci_exec_result_t
call_target_function(ma, addr, args, nwords, p_res)
machine_t *ma;
taddr_t addr;
taddr_t *args;
int nwords;
taddr_t *p_res;
{
	target_t *xp;
	int argno, retval, argslots, type_index;
	const char *mesg;
	size_t argsizes_buf[40], *argsizes;
#define MAX_ARGSIZES	(sizeof argsizes_buf / sizeof *argsizes_buf)

	xp = get_current_target();

	if (addr == STOP_ADDR)
		return STOP;
	if (addr == PRINTF_ADDR)
		return ups_printf(xp, ma, args, nwords);

	if (nwords > MAX_ARGSIZES)
		panic("nwords too large in ctf");

	type_index = nwords;
	argsizes = argsizes_buf;

	for (argno = 0; argno < nwords; argno += argslots, ++type_index) {
		type_t *type;
		int val, i;

		type = (type_t *)args[type_index];
		switch (type->ty_code) {
		case DT_PTR_TO:
			switch (type->ty_base->ty_code) {
			case TY_U_STRUCT:
			case TY_U_UNION:
				val = 0;
				break;
			default:
				val = ci_typesize((lexinfo_t *)NULL, type->ty_base);
				break;
			}
			argslots = 1;
			break;
		case DT_ARRAY_OF:
			val = ci_typesize((lexinfo_t *)NULL, type);
			argslots = 1;
			break;
		case TY_FLOAT:
			val = 0;
			argslots = sizeof(float) / sizeof(long);
			break;
		case TY_DOUBLE:
			val = 0;
			argslots = sizeof(double) / sizeof(long);
			break;
		default:
			val = 0;
			argslots = 1;
			break;
		}
		for (i = 0; i < argslots; i++)
			*argsizes++ = val;
	}

	retval = xp_call_func(xp, (char *)ma, addr,
					args, argsizes_buf, nwords, p_res, &mesg);
	if (mesg != NULL)
		errf("Call of function %s failed: %s: %s",
		     addr_to_func(addr)->fu_name, mesg, get_errno_str());

	return (retval == 0) ? CI_ER_CONTINUE : CI_ER_INDIRECT_CALL_FAILED;
}

int
get_restart_sig(xp)
target_t *xp;
{
	int sig;
	
	sig = xp_get_lastsig(xp);
	return (sig != 0 && accept_signal(sig)) ? sig : 0;
}
