/* st_lookup.c - look up things by name or address in the symbol tables */

/*  Copyright 1992 Mark Russell, University of Kent at Canterbury.
 *
 *  You can do what you like with this source code as long as
 *  you don't try to make money out of it and you include an
 *  unaltered copy of this message (including the copyright).
 */

char ups_st_lookup_c_sccsid[] = "@(#)st_lookup.c	1.8 04 Jun 1995 (UKC)";

#include <mtrprog/ifdefs.h>

#include <sys/types.h>
#include <string.h>
#include <stdlib.h>

#include <local/ukcprog.h>
#include <local/arg.h>
#include <mtrprog/strcache.h>
#include <mtrprog/alloc.h>
#include <mtrprog/genmergesort.h>

#include "ups.h"
#include "symtab.h"
#include "ci.h"
#include "st.h"
#include "target.h"
#include "state.h"

/*  List of globals added to by find_matching_funcs() and
 *  find_matching_globals().
 */
typedef struct global_list_s {
	const char *gl_name;
	Module *gl_module;
	fil_t *gl_fil;
	func_t *gl_func;
	var_t *gl_var;
	taddr_t gl_addr;
	struct global_list_s *gl_next;
} global_list_t;

static bool filename_matches PROTO((fil_t *fil, fil_t *matchfil,
					const char *name, size_t namelen));
static void find_matching_funcs PROTO((symtab_t *st,
				       fil_t *matchfil, const char *filename,
				       size_t flen, const char *funcname,
				       global_list_t **p_glhead,
				       bool *p_found_file));
static void find_matching_globals PROTO((target_t *xp, fil_t *matchfil,
					 const char *filename,
					 size_t filename_len,
					 const char *varpat,
					 const char *varname,
					 global_list_t **p_glhead));
static void addvar PROTO((char *arg, var_t *v));
static void scan_symtabs PROTO((fil_t *fil, const char *filename,
				size_t filename_len, const char *funcpat,
				const char *funcname,
				bool want_vars, global_list_t **p_glhead,
				bool *p_found_file));
static void add_var_to_global_list PROTO((global_list_t **p_glhead,
					  var_t *v, fil_t *fil));
static int glcmp PROTO((global_list_t *f1, global_list_t *f2));
static void add_common_block_globals PROTO((const char *name, func_t *f,
					    common_block_t **p_cblock,
					    global_list_t **p_glhead));
static void dupf_mesg PROTO((const char *name,
			     global_list_t *gl1, global_list_t *gl2));
static bool modname_matches PROTO((Module *m, const char *name, 
                                   size_t namelen));
static void add_func_to_global_list PROTO((global_list_t **p_glhead, func_t *f, 
                                           Module *module));
static void add_to_global_list PROTO((global_list_t **p_glhead, 
                                      const char *name, taddr_t addr, 
                                      fil_t *fil, Module *module));
static bool have_exact_match PROTO((global_list_t *glhead, const char *name));


GENERIC_MERGE_SORT(static,sort_global_list,global_list_t,gl_next)

ALLOC_NEW_FREELIST(static,global_list_t,global_list,gl_next)

static bool
filename_matches(fil, matchfil, name, namelen)
fil_t *fil, *matchfil;
const char *name;
size_t namelen;
{
	size_t len;

	if (matchfil != NULL)
		return fil == matchfil;

	if (name == NULL)
		return TRUE;
	if (fil == NULL)
		return FALSE;
	if (namelen == 0)
		return TRUE;

	len = strlen(fil->fi_name);

	if (len < namelen ||
	    memcmp(&fil->fi_name[len - namelen], name, namelen) != 0)
		return FALSE;
	
	return len == namelen || fil->fi_name[len - namelen - 1] == '/';
}

static bool
modname_matches(m, name, namelen)
Module *m;
const char *name;
size_t namelen;
{
	size_t len;
	const char *modname;

	if (name == NULL)
		return TRUE;
	
	modname = get_module_name(m);
	len = strlen(modname);

	return len == namelen && memcmp(modname, name, namelen) == 0;
}

static void
find_matching_funcs(st, matchfil, filename, flen, funcpat,
							p_glhead, p_found_file)
symtab_t *st;
fil_t *matchfil;
const char *filename;
size_t flen;
const char *funcpat;
global_list_t **p_glhead;
bool *p_found_file;
{
	func_t *f;
	Module *m;

	for (f = st->st_funclist; f != NULL; f = f->fu_next) {
		if (filename_matches(f->fu_fil, matchfil, filename, flen)) {
			*p_found_file = TRUE;

			if (arg_match(f->fu_name, funcpat))
				add_func_to_global_list(p_glhead, f,
							(Module *)NULL);
		}
	}

	for (m = st->st_modules; m != NULL; m = get_next_module(m)) {
		if (modname_matches(m, filename, flen)) {
			funclist_t *fl;

			*p_found_file = TRUE;
			
			for (fl = get_module_funclist(m);
			     fl != NULL;
			     fl = fl->fl_next) {
				if (arg_match(fl->fl_func->fu_name, funcpat)) {
					add_func_to_global_list(p_glhead,
								fl->fl_func, m);
				}
			}
		}
	}
}

fil_t *
name_to_fil(name)
const char *name;
{
	target_t *xp;
	symtab_t *st;

	xp = get_current_target();
	st = NULL;
	
	while (xp_next_symtab(xp, st, TRUE, &st)) {
		fil_t *fil;

		for (fil = st->st_sfiles; fil != NULL; fil = fil->fi_next) {
			if (strcmp(fil->fi_name, name) == 0)
				return fil;
		}
	}

	return NULL;
}

func_t *
name_and_fil_to_func(name, fil)
const char *name;
fil_t *fil;
{
	funclist_t *fl;

	for (fl = fil->fi_funclist; fl != NULL; fl = fl->fl_next) {
		if (strcmp(fl->fl_func->fu_name, name) == 0)
			return fl->fl_func;
	}

	return NULL;
}

int
resolve_untyped_name(name, p_v)
const char *name;
var_t **p_v;
{
	global_list_t *glhead;

	glhead = NULL;
	find_matching_globals(get_current_target(), (fil_t *)NULL,
			      (const char *)NULL, 0, name, name, &glhead);

	/*  If there is more than one match, just pick the first one.
	 *
	 *  TODO: Pick best one.
	 */
	*p_v = (glhead != NULL) ? glhead->gl_var : NULL;
	
	free_global_list_list(glhead);

	return (*p_v != NULL) ? 0 : -1;
}

static void
find_matching_globals(xp, matchfil, filename, filename_len, varpat, varname,
		      p_glhead)
target_t *xp;
fil_t *matchfil;
const char *filename;
size_t filename_len;
const char *varpat, *varname;
global_list_t **p_glhead;
{
	symtab_t *st;

	/*  First search the source files of all the symbol tables.
	 */
	st = NULL;
	while (xp_next_symtab(xp, st,
			      !have_exact_match(*p_glhead, varname), &st)) {
		fil_t *fil;
		
		for (fil = st->st_sfiles; fil != NULL; fil = fil->fi_next) {
			var_t *v;

			if (!filename_matches(fil, matchfil,
					      filename, filename_len))
				continue;

			if (!st_fil_may_have_matching_globals(fil, varpat,
							      arg_match))
				continue;

			for (v = FI_VARS(fil); v != NULL; v = v->va_next) {
				if (arg_match(v->va_name, varpat)) {
					add_var_to_global_list(p_glhead,
							       v, fil);
				}
			}
		}
	}
	
	if (matchfil == NULL && filename == NULL &&
	    !have_exact_match(*p_glhead, varname)) {
		/*  Not found under any source file, so search the
		 *  untyped variables.
		 */
		st = NULL;
		while (xp_next_symtab(xp, st,
				      !have_exact_match(*p_glhead, varname),
				      &st)) {
			scan_addrlist(st, arg_match, varpat, addvar,
				      (char *)p_glhead);
		}
	}
}

static void
addvar(arg, v)
char *arg;
var_t *v;
{
	add_var_to_global_list((global_list_t **)arg, v, (fil_t *)NULL);
}

static void
add_var_to_global_list(p_glhead, v, fil)
global_list_t **p_glhead;
var_t *v;
fil_t *fil;
{
	add_to_global_list(p_glhead, v->va_name, v->va_addr,
			   fil, (Module *)NULL);
	(*p_glhead)->gl_var = v;
}

static void
add_func_to_global_list(p_glhead, f, module)
global_list_t **p_glhead;
func_t *f;
Module *module;
{
	add_to_global_list(p_glhead, f->fu_name, f->fu_addr,
			   f->fu_fil, module);
	(*p_glhead)->gl_func = f;
}

static void
add_to_global_list(p_glhead, name, addr, fil, module)
global_list_t **p_glhead;
const char *name;
taddr_t addr;
fil_t *fil;
Module *module;
{
	global_list_t *gl;

	gl = new_global_list();
	gl->gl_name = name;
	gl->gl_fil = fil;
	gl->gl_module = module;
	gl->gl_func = NULL;
	gl->gl_var = NULL;
	gl->gl_addr = addr;
	gl->gl_next = *p_glhead;
	*p_glhead = gl;
}

/*  Search for the common block where the variable called name
 *  is defined.  We loop over all the symbol tables, searching
 *  the common block lists of each.  (At present there are no
 *  shared library FORTRAN executables, so the only symbol
 *  table with a common block list will be the main one).
 */
static void
add_common_block_globals(name, f, p_cblock, p_glhead)
const char *name;
func_t *f;
common_block_t **p_cblock;
global_list_t **p_glhead;
{
	target_t *xp;
	symtab_t *st;

	xp = get_current_target();
	st = NULL;

	while (xp_next_symtab(xp, st, TRUE, &st)) {
		var_t *v;
		fil_t *fil;

		global_and_cblist_to_var(st->st_cblist, name, f,
					 p_cblock, &fil, &v);
		
		if (v != NULL) {
			add_var_to_global_list(p_glhead, v, fil);
			return;
		}
	}
}

static int
glcmp(gl1, gl2)
global_list_t *gl1, *gl2;
{
	int res;

	if ((res = strcmp(gl1->gl_name,  gl2->gl_name)) != 0)
		return res;
	
	if ((gl1->gl_func != NULL) != (gl2->gl_func != NULL))
		return (gl1->gl_func == NULL) ? -1 : 1;
	
	if (gl1->gl_func == NULL)
		return 0;

	if (gl1->gl_func == gl2->gl_func) {
		if ((gl1->gl_module != NULL) == (gl2->gl_module != NULL))
			return 0;
		
		return (gl1->gl_module != NULL) ? -1 : 1;
	}

	return addrcmp(gl1->gl_func, gl2->gl_func);
}

static bool
have_exact_match(glhead, name)
global_list_t *glhead;
const char *name;
{
	global_list_t *gl;
	
	for (gl = glhead; gl != NULL; gl = gl->gl_next) {
		if (strcmp(gl->gl_name, name) == 0)
			return TRUE;
	}

	return FALSE;
}

static void
scan_symtabs(matchfil, filename, filename_len, funcpat, funcname, want_vars,
							p_glhead, p_found_file)
fil_t *matchfil;
const char *filename;
size_t filename_len;
const char *funcpat, *funcname;
bool want_vars;
global_list_t **p_glhead;
bool *p_found_file;
{
	target_t *xp;
	symtab_t *st;

	xp = get_current_target();

	st = NULL;
	while (xp_next_symtab(xp, st,
			      !have_exact_match(*p_glhead, funcname), &st)) {
		find_matching_funcs(st, matchfil, filename, filename_len,
				    funcpat, p_glhead, p_found_file);
	}

	if (want_vars) {
		find_matching_globals(xp, matchfil, filename,
				      filename_len, funcpat, funcname,
				      p_glhead);
	}
}
					
int
find_func_by_name(name, p_f)
const char *name;
func_t **p_f;
{
	return find_global_by_name(name, (fil_t *)NULL, (func_t *)NULL,
				   (common_block_t *)NULL, FALSE,
				   p_f, (var_t **)NULL,
				   (common_block_t **)NULL, (fil_t **)NULL);
				
}

/*  We search all the symbol tables including those in the cache.
 *  This is so we can match names in shared libraries (like printf)
 *  before the target has been started.
 */
int
find_global_by_name(name, fil, f, cblock, exact, p_f, p_v, p_cblock, p_fil)
const char *name;
fil_t *fil;
func_t *f;
common_block_t *cblock;
bool exact;
func_t **p_f;
var_t **p_v;
common_block_t **p_cblock;
fil_t **p_fil;
{
	global_list_t *gl, *glhead, *prev;
	global_list_t *exact_gls[2], *partial_gls[2], *exact_extern_gl;
	int matches, exact_matches, exact_extern_matches, partial_matches;
	size_t filename_len;
	const char *filename, *funcname, *funcpat;
	char *infile;
	bool found_file;

	if ((funcname = strchr(name, ':')) != NULL && funcname[1] != ':') {
		if (funcname[1] == '\0') {
			errf("Missing function name after source file");
			return -1;
		}
		
		filename = name;
		filename_len = funcname - name; 
		++funcname;
	}
	else {
		filename = NULL;
		filename_len = 0;	/* to satisfy gcc */
		funcname = name;
	}

	if (exact) {
		funcpat = funcname;
	}
	else {
		static const char globchars[] = "*?[]";
		const char *s, *globname;
		char *namecopy;

		for (s = funcname; s != '\0'; ++s) {
			if (strchr(globchars, *s) != NULL)
				break;
		}
		
		if (*s == '\0') {
			namecopy = strf("%s*", funcname);
			globname = namecopy;
		}
		else {
			namecopy = NULL;
			globname = funcname;
		}

		funcpat = arg_do_quoting(&globname, globchars);

		if (namecopy != NULL)
			free(namecopy);

		if (funcpat == NULL)
			return -1;
	}

	glhead = NULL;
	found_file = FALSE;

	if (p_v != NULL)
		*p_cblock = NULL;

	if (cblock != NULL) {
		var_t *v;
		
		for (v = get_cblock_vars(cblock); v != NULL; v = v->va_next) {
			if (arg_match(v->va_name, funcpat)) {
				add_var_to_global_list(&glhead, v, fil);
				*p_cblock = cblock;
			}
		}
	}
	
	if (glhead == NULL && fil != NULL && filename == NULL) {
		scan_symtabs(fil, (const char *)NULL, 0, funcpat, funcname,
					p_v != NULL, &glhead, &found_file);
	}

	if (glhead == NULL) {
		scan_symtabs((fil_t *)NULL, filename, filename_len,
			     funcpat, funcname,
			     p_v != NULL, &glhead, &found_file);

		if (p_v != NULL)
			add_common_block_globals(name, f, p_cblock, &glhead);
	}

	if (filename != NULL && !found_file) {
		errf("No source file name `%.*s'", (int)filename_len, filename);
		return -1;
	}

	if (glhead == NULL) {
		const char *orvar;

		orvar = (p_v != NULL) ? " or variable" : "";

		if (filename != NULL)
			errf("No function%s matching `%s' in %.*s",
				     orvar, funcname,
				     (int)filename_len, filename);
		else
			errf("No function%s `%s'", orvar, funcname);

		return -1;
	}

	matches = exact_matches = exact_extern_matches = partial_matches = 0;
	exact_extern_gl = NULL;		/* to satisfy gcc */

	for (gl = glhead; gl != NULL; gl = gl->gl_next)
		++matches;
	glhead = sort_global_list(glhead, matches, glcmp);

	prev = NULL;

	for (gl = glhead; gl != NULL; gl = gl->gl_next) {

		/*  We can get duplicate symbol table entries for a
		 *  function or variable - ignore these.
		 */
		if (prev != NULL && gl->gl_addr == prev->gl_addr &&
		    strcmp(prev->gl_name, gl->gl_name) == 0 &&
		    (prev->gl_func != NULL) == (gl->gl_func != NULL))
			continue;
		
		if (strcmp(gl->gl_name, funcname) == 0) {
			if ((gl->gl_func != NULL) ?
				      (gl->gl_func->fu_flags & FU_STATIC) == 0 :
				      gl->gl_var->va_class == CL_EXT) {
				++exact_extern_matches;
				exact_extern_gl = gl;
			}

			if (exact_matches < 2)
				exact_gls[exact_matches] = gl;
			++exact_matches;
		}
		else {
			if (partial_matches < 2)
				partial_gls[partial_matches] = gl;
			++partial_matches;
		}

		prev = gl;
	}
	free_global_list_list(glhead);

	if (exact_matches == 1) {
		gl = exact_gls[0];
	}
	else if (partial_matches == 1 && exact_matches == 0) {
		gl = partial_gls[0];
	}
	else if (exact_extern_matches == 1) {
		gl = exact_extern_gl;
	}
	else if (exact_matches > 1) {
		dupf_mesg(funcname, exact_gls[0], exact_gls[1]);
		gl = exact_gls[0];
	}
	else {
		gl = NULL;
	}

	if (gl != NULL) {
		*p_f = gl->gl_func;
		if (p_v != NULL) {
			*p_v = gl->gl_var;
			*p_fil = gl->gl_fil;
		}
		return 0;
	}

	if (partial_matches <= 1)
		panic("pmatch botch in ffbn");

	if (strcmp(partial_gls[0]->gl_name, partial_gls[1]->gl_name) == 0) {
		dupf_mesg(partial_gls[0]->gl_name,
				partial_gls[0], partial_gls[1]);
		return -1;
	}

	infile = (filename != NULL && *filename != '\0')
				? strf(" in %.*s", (int)filename_len, filename)
				: strsave("");

	if (partial_matches == 2) {
		errf("`%s' matches both `%s' and `%s'%s",
				funcname, partial_gls[0]->gl_name,
				partial_gls[1]->gl_name, infile);
	}
	else {
		errf("`%s' matches `%s', `%s' (and %d more)%s",
				funcname, partial_gls[0]->gl_name,
				partial_gls[1]->gl_name, 
				partial_matches - 2, infile);
	}
	
	free(infile);
	
	return -1;
}

static void
dupf_mesg(funcname, gl1, gl2)
const char *funcname;
global_list_t *gl1, *gl2;
{
	const char *what;
	fil_t *fil1, *fil2;

	if ((gl1->gl_func != NULL) != (gl2->gl_func != NULL))
		what = "Name";
	else
		what = (gl1->gl_func != NULL) ? "Function" : "Variable";

	fil1 = gl1->gl_fil;
	fil2 = gl2->gl_fil;

	if (gl1->gl_module != NULL && gl2->gl_module != NULL) {
		errf("\bWarning: %s `%s' appears in modules %s and %s",
		     what, funcname,
		     get_module_name(gl1->gl_module),
		     get_module_name(gl2->gl_module));
	}
	else if (fil1 != NULL && fil2 != NULL) {
		errf("\bWarning: %s `%s' appears in both %s and %s",
				what, funcname, fil1->fi_name, fil2->fi_name);
	}
	else if (fil1 != NULL && fil2 == NULL) {
		errf("\bWarning: %s `%s' appears in both %s and elsewhere",
				what, funcname, fil1->fi_name);
	}
	else if (fil2 != NULL && fil1 == NULL) {
		errf("\bWarning: %s `%s' appears in both %s and elsewhere",
		     what, funcname, fil2->fi_name);
	}
	else
		errf("\bWarning: %s `%s' appears more than once!",
		     what, funcname);
}

/*  Return a pointer to the func_t block of the function wherein
 *  lies address addr, or NULL if addr does not lie within any
 *  known function.  We search all the symbol tables.
 */
func_t *
addr_to_func(addr)
taddr_t addr;
{
	symtab_t *st;
	func_t *f;
	target_t *xp;

	xp = get_current_target();
	st = NULL;
	
	while (xp_next_symtab(xp, st, TRUE, &st)) {
		if (addr_and_functab_to_func(st->st_functab, addr, &f))
			return f;
	}
	
	return NULL;
}

/*  Call (*func)() for each source file with the source file as an argument.
 *
 *  Do the source files in the cache as well.
 */
void
iterate_over_source_files(func)
void (*func)PROTO((fil_t *fil));
{
	target_t *xp;
	symtab_t *st;
	fil_t *fil;

	xp = get_current_target();
	st = NULL;

	while (xp_next_symtab(xp, st, TRUE, &st)) {
		for (fil = st->st_sfiles; fil != NULL; fil = fil->fi_next) {
			if ((fil->fi_flags & FI_HIDE_ENTRY) == 0)
				(*func)(fil);
		}
	}
}

/*  Call (*func)(cblock) for each common block.
 */
void
iterate_over_common_blocks(func)
void (*func)PROTO((common_block_t *cblock));
{
	target_t *xp;
	symtab_t *st;

	xp = get_current_target();
	st = NULL;

	while (xp_next_symtab(xp, st, TRUE, &st))
		iterate_over_cblist(st->st_cblist, func);
}

void
iterate_over_all_modules(func)
void (*func)PROTO((Module *module));
{
	target_t *xp;
	symtab_t *st;

	xp = get_current_target();
	st = NULL;

	while (xp_next_symtab(xp, st, TRUE, &st))
		iterate_over_modules(st->st_modules, func);
}
