;;;; $Id: req-mode.el,v 1.12 1994/11/15 18:24:55 mikee Exp $
;; req-mode commands for Emacs
;; Copyright (C) 1994 Walker Financial Corp.
;;
;; Permission is granted to use, modify, and give for any purpose
;; as long as this notification remains and credit is given to the
;; author.
;;
;; Written by Mike Eggleston, mikee@mail.wfc.com
;; Walker Financial Corp.

;;; $Log: req-mode.el,v $
; Revision 1.12  1994/11/15  18:24:55  mikee
; forgot to add the code to handle scrolling up and down within a
; request
;
; Revision 1.11  1994/11/07  16:04:52  mikee
; minor changes
;
; Revision 1.10  1994/11/07  14:07:55  mikee
; final released version
; recovered by Remy and sent to me
;
; Revision 1.12  1994/11/04  21:28:48  mikee
; modified copyright notification
;
; Revision 1.11  1994/11/04  21:25:21  mikee
; I think it works now!!!
;
; Revision 1.10  1994/11/04  17:00:21  mikee
; still working
;
; Revision 1.9  1994/11/04  14:46:31  mikee
; almost finished
;
; Revision 1.8  1994/11/03  21:24:56  mikee
; seems to work so far
;
; Revision 1.7  1994/11/03  21:22:51  mikee
; trying something
;
; Revision 1.6  1994/11/03  20:11:09  mikee
; tested give-user
; added take and untake
; added and testing steal
;
; Revision 1.5  1994/11/03  18:28:19  mikee
; added a filtered message mode
; added next and previous line/request commands
;
; Revision 1.4  1994/11/03  17:50:32  mikee
; edit buffers working
;
; Revision 1.3  1994/11/03  17:39:56  mikee
; still working
;
; Revision 1.2  1994/11/02  23:41:26  mikee
; still working
;
; Revision 1.1  1994/11/01  23:36:59  mikee
; Initial revision
;

;;;
;;; buffers used by req-mode
;;;
(defvar req-mode-buffer "*Request Info*"
  "Buffer name of req information.")

(defvar req-mode-help-buffer "*Request Help*"
  "Buffer name for the request help information.")

(defvar req-mode-edit-buffer "*Request Edit*"
  "Buffer name for the request edit buffer.")

(defvar req-mode-temp-buffer "*Request Temp*"
  "Buffer name for the request temporary buffer.")

(defvar req-mode-message-buffer "*Request Message*"
  "Buffer name to display the body of a request.")

;;;
;;; environmental variables used by req-mode
;;;
(defvar req-mode-user-variable "LOGNAME"
  "*The system environmental variable that is set the the id
of the current user when that user logs into the system.")

;;;
;;; programs used by req-mode
;;;
(defvar req-mode-program-path "/usr/local/requests/bin"
  "*Path to the req programs.")

(defvar req-mode-query-program "q"
  "*Program to call to generate req information.")

(defvar req-mode-query nil
  "Full path of program to call to generate req information.")

(defvar req-mode-update-program "req"
  "*Program to modify state and content of requests.")

(defvar req-mode-upd nil
  "Full path of program to modify state and content of requests.")

(defvar req-mode-show-program "reqshow"
  "*Program to quickly show content of a request.")

(defvar req-mode-show nil
  "Full path of program to quickly show content of a request.")

(defvar req-mode-mail-alias-file "/etc/mail/aliases"
  "*Name of the local mail aliases file.
This file is used to generate a list of valid user names.")

;;;
;;; lists of regular expressions for modification of a request
;;; prior to that request being shown to a user
;;;
(defvar req-mode-kill-message-lines '("^X-.*: "
                                      "^Status: "
                                      "^From"
                                      "^Received: "
                                      "^[ \t]+id "
                                      "^Return-Path: "
                                      "^Message-Id: "
                                      "^Apparently-To: ")
  "*List of regular expressions of the lines to automatically
remove when showing the body of a request.")

(defvar
  req-mode-change-message-lines '(("X-Request-Acted: " "Date: ")
                                  ("X-Request-Action: " "Action: "))
  "*List of regular expression search and replacement pairs used
to modify the text in the request message buffer.  This list is
processed prior to processing the req-mode-kill-message-lines
list of regular expressions.")

;;;
;;; state toggles that affect the data presentation
;;;
(defvar req-mode-filtered-messages t
  "*If nil then process req-mode-change-message-lines and
req-mode-kill-message-lines lists.")

(defvar req-mode-filter-spec-visible t
  "*Should the filter specification be displayed every
time that the list if requests is regenerated.")

;;;
;;; internal variables
;;;
(defvar req-mode-bogus-lines 3
  "*Number of non-process lines at the top of the display.")

(defvar req-mode-req-array nil
  "Array of process id's. Array index corresponds to line number in
   current req-mode-buffer.")

(defvar req-mode-reqnum-position nil
  "Position of the request number field in the req-mode buffer.")

(defvar req-mode-lines nil
  "Number of lines in the current req-mode buffer.")

(defvar req-mode-filter-spec '()
  "Command line argument to the 'q' program that filters its output.")

(defvar req-mode-user-list '()
  "List of users generated from the /usr/lib/mail/aliases file.")

(defvar req-mode-prio-list '(("low" 1) ("normal" 2) ("high" 3))
  "List of possible priorities.")

;;
;; Issue a req-mode-command to the current buffer, then build the array
;; of process id's and signal numbers.
;;
(defun req-mode-build-request-list ()
  "remove current list of requests and generate a new one"
  (interactive)
  (let* ((buffer-read-only nil)
         (this-line nil)
         (orig-req nil))

    (switch-to-buffer (get-buffer-create req-mode-buffer))
    (setq buffer-read-only nil)
    (setq this-line (count-lines (point-min) (point)))
    (if req-mode-req-array
	(setq orig-req (elt req-mode-req-array this-line))
      (setq orig-req nil))
    (delete-region (point-min) (point-max))
    (insert "\n")

    (apply 'call-process req-mode-query
           nil t nil (delq nil (apply 'append req-mode-filter-spec)))
    (if req-mode-filter-spec-visible
        (progn
          (goto-char (point-min))
          (insert (concat "Filter spec:"
                          (req-mode-list-to-string req-mode-filter-spec)))))

    (goto-char (point-min))

    (if (= (count-lines (point-min) (point-max)) 1)
        (progn
          (goto-char (point-max))
          (insert
           "Req # ! Owner    Age    Told   Status  User       Subject
----- - -------- ------ ------ ------- ---------- ---------------------------
")))

    (goto-char (point-min))
    (let ( bol eol (i 1) )
      (goto-char (point-min))
      (search-forward "Req #")
      (beginning-of-line)
      (setq bol (point))
      (end-of-line)
      (setq eol (point))
      (narrow-to-region bol eol)
      (goto-char bol)
      (setq req-mode-req-position nil)
      (while (or (not (eolp)) (not (numberp req-mode-req-position)))
        (if (looking-at " *Req #")
            (setq req-mode-req-position i))
        (setq i (+ i 1))
        (forward-word 1))
      (widen))
    (setq req-mode-lines (count-lines (point-min) (point-max)))
    (setq req-mode-req-array (make-vector (+ req-mode-lines 1) nil))
    (setq req-mode-signal-array (make-vector (+ req-mode-lines 1) nil))

    (goto-char (point-min))
    (let
        ((i req-mode-bogus-lines)
         (to-skip (- req-mode-req-position 1))
         req-start req-end this-req)
      (forward-line i)
      (while (not (eobp))
        (beginning-of-line)
        (forward-word to-skip)          ; skip to beginning of PID field
        (setq req-start (point))
        (forward-word 1)
        (setq req-end (point))
        (setq this-req
              (string-to-int (buffer-substring req-start req-end)))
        (aset req-mode-req-array i
              (if (> this-req 0) this-req nil))
        (setq i (+ i 1))
        (forward-line 1)))

    (goto-char (point-min))
    (forward-line req-mode-bogus-lines)
    (re-search-forward (concat "^ *" orig-req) nil t)
    (beginning-of-line)
    (delete-other-windows)
    (setq orig-req (count-lines (point-min) (point)))

  (let ((hh (/ (frame-height) 2))
        (wh (+ (count-lines (point-min) (point-max)) 2)))
    (if (< wh hh)
        (split-window (selected-window) wh)
      (split-window (selected-window) hh)))

  (other-window 1)
  (switch-to-buffer (get-buffer-create req-mode-message-buffer))

  (req-mode-show-request (elt req-mode-req-array orig-req)))
  (select-window (get-buffer-window req-mode-buffer))
  (setq buffer-read-only t))

(defun req-mode ()
  "A major-mode for interacting with the 'req' request system.
In req-mode, you manage requests.

The following commands are availabe within req-mode.

C      - Comment  - add a comment to a request
c      - Create   - create a new request
f      - Filter   - limit the display of requests by some critieria
g      - Give     - give a request to someone
K      - Kill     - completely delete a request from the request queue
m      - Merge    - merge two requests into a single request
N      - Notify   - update the time field of the request
P      - Priority - change the priority of the request (see Priority
command below)
r      - Resolve  - resolve a request
R      - reverse  - reverse output
M-s    - Show     - display the current request message
S      - Stall    - change status to 'stalled' - waiting on part, user, etc.
s      - Steal    - acquire ownership of a request from another user
j      - Subject  - supply a more descriptive subject for a request
t      - Take     - acquire ownership of a request that is currently 'unowned'
u      - Untake   - disown a currently owned request - make it 'unowned'
n      - next     - move point to next line displaying next request
p      - previous - move point to next line displaying previous request
C-n    - next     - move point to next line without displaying next request
C-p    - previous - move point to next line without displaying previous request

The filter command (f) has the following sub-commands:

p - Priority - look only at a specific priority
o - Owner    - look only at requests owned by a specific user
U - User     - look only at requests entered by a specific user
O - Open     - look only at open requests
s - Status   - look only at requests with a specific status
u - Unowned  - look only at 'unowned' requests
r - Resolved - look only at resolved requests
M - Merged   - look only at 'merged' requests
m - Modified - look only at 'modified' requests
t - Time     - order requests based on most recently updated
R - Reverse  - reverse the ordering sequence
c - Clear    - clear all filter flags

The Priority command (P) has the following priority levels:

a - All      - \"Overload me with information.\"
l - Low      - \"Lets not forget to do this.\"
n - Normal   - \"I'm having a problem and can't work.\"
h - High     - \"The machine is crashing and you'll soon be fired.\"

The Status command (s) has the following options:

o - Opened   - active issue
s - Stalled  - waiting on a hardware, software, a user, the moon
m - Merged   - multiple, similar requests combined into one
r - Resolved - something already accomplished
"

  (interactive)
  (switch-to-buffer (get-buffer-create req-mode-buffer))
  (kill-all-local-variables)
  (setq major-mode 'req-mode)
  (setq mode-name "Process Mode")

  ;;
  ;; local bindings
  ;;
  (use-local-map (make-sparse-keymap))
  (local-set-key " " 'req-mode-scroll)
  (local-set-key "\177" 'req-mode-unscroll)
  (local-set-key "." 'req-mode-build-request-list)
  (local-set-key "C" 'req-mode-pre-comment)
  (local-set-key "\C-n" 'next-line)
  (local-set-key "\C-p" 'previous-line)
  (local-set-key "n" 'req-mode-next-line)
  (local-set-key "p" 'req-mode-previous-line)
  (local-set-key "c" 'req-mode-pre-create)
  (local-set-key "f" 'req-mode-filter)
  (local-set-key "g" 'req-mode-pre-give)
  (local-set-key "K" 'req-mode-kill)
  (local-set-key "m" 'req-mode-merge)
  (local-set-key "N" 'req-mode-pre-notify)
  (local-set-key "P" 'req-mode-pre-priority)
  (local-set-key "r" 'req-mode-pre-resolve)
  (local-set-key "j" 'req-mode-subject)
  (local-set-key "q" 'req-mode-quit)
  (local-set-key "s" 'req-mode-pre-steal)
  (local-set-key "S" 'req-mode-pre-stall)
  (local-set-key "\M-s" 'req-mode-show)
  (local-set-key "t" 'req-mode-take)
  (local-set-key "u" 'req-mode-untake)
  ;;
  (setq truncate-lines t)
  (set-syntax-table text-mode-syntax-table)
  (setq buffer-read-only t)
  (setq req-mode-query
        (concat req-mode-program-path "/" req-mode-query-program))
  (setq req-mode-upd
        (concat req-mode-program-path "/" req-mode-update-program))
  (setq req-mode-show
        (concat req-mode-program-path "/" req-mode-show-program))
  (req-mode-build-request-list)
  (run-hooks 'req-mode-hook))

;;; move the cursor to the next line and display the next request
(defun req-mode-next-line ()
  "Move the cursor in the request buffer to the next line.
Display the next request in the message buffer."
  (interactive)
  (forward-line)
  (let ((this-line (count-lines (point-min) (point))))
    (req-mode-show-request (elt req-mode-req-array this-line))))

;;; move the cursor to the previous line and display the next request
(defun req-mode-previous-line ()
  "Move the cursor in the request buffer to the next line.
Display the next request in the message buffer."
  (interactive)
  (forward-line -1)
  (let ((this-line (count-lines (point-min) (point))))
    (req-mode-show-request (elt req-mode-req-array this-line))))

;;; display the body of a request in the other window
(defun req-mode-show-request (n)
  "Display the body of a request in the other window."
  (interactive "nRequest Number: ")
  (let ((l req-mode-kill-message-lines)
        (c req-mode-change-message-lines))
    (select-window (get-buffer-window req-mode-message-buffer))
    (setq buffer-read-only nil)
    (make-local-variable 'kill-whole-line)
    (setq kill-whole-line t)
    (delete-region (point-min) (point-max))

    (if (and n (numberp n))
        (progn
          (call-process req-mode-show
                        nil t nil (number-to-string n))
          (if req-mode-filtered-messages
              (progn
                (if (and c (listp c))
                    (while c
                      (goto-char (point-min))
                      (perform-replace (car (car c))
                                       (car (cdr (car c)))
                                       nil t nil)
                      (setq c (cdr c))))
                (if (and l (listp l))
                    (while l
                      (goto-char (point-min))
                      (while (not (eobp))
                        (if (looking-at (car l))
                            (kill-line)
                          (forward-line)))
                      (setq l (cdr l))))))))
    (goto-char (point-min))
    (set-buffer-modified-p nil)
    (setq buffer-read-only t)
    (select-window (get-buffer-window req-mode-buffer))))

;;; scroll forward in the message and onto the next message
;;; at message end
(defun req-mode-scroll ()
  "Scroll forward in the current message.
At the end of the message, go to the next message."
  (interactive)
  (select-window (get-buffer-window req-mode-message-buffer))
  (if (pos-visible-in-window-p (point-max) nil)
      (progn
	(select-window (get-buffer-window req-mode-buffer))
	(req-mode-next-line))
    (scroll-up))
  (select-window (get-buffer-window req-mode-buffer)))

;;; scroll backward in the message and into the previous message
;;; at message beginning
(defun req-mode-unscroll ()
  "Scroll backward in the current message.
At the beginning of the message, go to the previous message."
  (interactive)
  (select-window (get-buffer-window req-mode-message-buffer))
  (if (pos-visible-in-window-p (point-min) nil)
      (progn
	(select-window (get-buffer-window req-mode-buffer))
	(req-mode-previous-line))
    (scroll-down))
  (select-window (get-buffer-window req-mode-buffer)))

;;; quit the mode
(defun req-mode-quit ()
  (interactive)
  (setq req-mode-filter-spec '())
  (if (get-buffer req-mode-buffer)
      (kill-buffer req-mode-buffer))
  (if (get-buffer req-mode-help-buffer)
      (kill-buffer req-mode-help-buffer))
  (if (get-buffer req-mode-edit-buffer)
      (kill-buffer req-mode-edit-buffer))
  (if (get-buffer req-mode-temp-buffer)
      (kill-buffer req-mode-temp-buffer))
  (if (get-buffer req-mode-message-buffer)
      (kill-buffer req-mode-message-buffer))
  (delete-other-windows))

;;; accept an interactive input character for the filter type
(defun req-mode-filter (c)
  (interactive "cFilter (poUOsurMtrcv): ")
  (cond ((char-equal c ?p)
         (req-mode-filter-priority))
        ((char-equal c ?o)
         (let ((ownerid (read-no-blanks-input "Owner: ")))
           (req-mode-add-argument (list "-owner" ownerid))))
        ((char-equal c ?U)
         (let ((userid (read-no-blanks-input "User: ")))
           (req-mode-add-argument (list "-user" userid))))
        ((char-equal c ?O)
         (req-mode-add-argument '("-open" nil)))
        ((char-equal c ?s)
         (req-mode-filter-status))
        ((char-equal c ?u)
         (req-mode-add-argument '("-unowned" nil)))
        ((char-equal c ?r)
         (req-mode-add-argument '("-resolved" nil)))
        ((char-equal c ?M)
         (req-mode-add-argument '("-merged" nil)))
        ((char-equal c ?R)
         (req-mode-add-argument '("-r" nil)))
        ((char-equal c ?t)
         (req-mode-add-argument '("-t" nil)))
        ((char-equal c ?v)
         (setq req-mode-filter-spec-visible
               (not req-mode-filter-spec-visible)))
        ((char-equal c ?\?)
         (let ((cur-win (selected-window)))
           (pop-to-buffer (get-buffer-create req-mode-help-buffer))
           (delete-region (point-min) (point-max))
           (insert "
Possible filter specifications:

p - Priority - look only at a specific priority
o - Owner    - look only at requests owned by a specific user
u - User     - look only at requests entered by a specific user
O - Open     - look only at open requests
s - Status   - look only at requests with a specific status
u - Unowned  - look only at 'unowned' requests
r - Resolved - look only at resolved requests
m - Merged   - look only at 'merged' requests
t - Time     - order requests based on most recently updated
R - Reverse  - reverse the ordering sequence
c - Clear    - clear all filter flags
v - visible  - toggle the filter spec visibility flag

Choose one of the following: p o u O s u r m c

Press C-x 1 to remove this buffer.
")
           (select-window cur-win)))
        ((or (char-equal c ?c) t)
         (setq req-mode-filter-spec '())))
  (req-mode-build-request-list))

;;; accept an interactive input character for the priority level
(defun req-mode-filter-priority ()
  (interactive)
  (let ((p (read-char)))
    (cond ((char-equal p ?l)
           (req-mode-add-argument '("-prio" "low")))
          ((char-equal p ?n)
           (req-mode-add-argument '("-prio" "normal")))
          ((char-equal p ?h)
           (req-mode-add-argument '("-prio" "high")))
          ((char-equal p ?a)
           (req-mode-add-argument '("-prio" "all")))
          ((char-equal p ??)
           (let ((cur-win (selected-window)))
             (pop-to-buffer (get-buffer-create req-mode-help-buffer))
             (delete-region (point-min) (point-max))
             (insert "
Possible priority levels:

a - All      - \"Overwhelm me with everything.\"
l - Low      - \"Lets not forget to do this.\"
n - Normal   - \"I'm having a problem and can't work.\"
h - High     - \"The machine is crashing and you'll soon be fired.\"p

Choose one of the following: l n h

Press C-x 1 to remove this buffer.
")
             (select-window cur-win)))
          (t
           (req-mode-remove-argument '("-prio" nil))))))

;;; accept an interactive input character for the status code
(defun req-mode-filter-status ()
  (interactive)
  (let ((s (read-char)))
    (cond ((char-equal s ?o)
           (req-mode-add-argument '("-status" "open")))
          ((char-equal s ?s)
           (req-mode-add-argument '("-status" "stalled")))
          ((char-equal s ?m)
           (req-mode-add-argument '("-status" "merged")))
          ((char-equal s ?r)
           (req-mode-add-argument '("-status" "resolved")))
          ((char-equal s ??)
           (let ((cur-win (selected-window)))
             (pop-to-buffer (get-buffer-create req-mode-help-buffer))
             (delete-region (point-min) (point-max))
             (insert "
Possible status codes:

o - Opened   - active issue
s - Stalled  - waiting on a hardware, software, a user, the moon
m - Merged   - multiple, similar requests combined into one
r - Resolved - something already accomplished
c - Clear    - remove status filtering

Choose one of the following: o s m r

Press C-x 1 to remove this buffer.
")
             (select-window cur-win)))
          (t
           (req-mode-remove-argument '("-status" nil))))))

; add an argument association
(defun req-mode-add-argument (l)
  "Add an association to an argument list."
  (req-mode-remove-argument l)
  (setq req-mode-filter-spec
        (append req-mode-filter-spec (list l))))

;; remove an argument association
(defun req-mode-remove-argument (l)
  "Remove an association to an argument list."
  (setq req-mode-filter-spec
        (delete
         (assoc (car l) req-mode-filter-spec)
         req-mode-filter-spec)))

;; combine a list into a single string
(defun req-mode-list-to-string (l)
  (let ((y nil))
    (while l
      (if (and (listp l) (listp (car l)))
          (setq y (concat y (req-mode-list-to-string (car l))))
        (setq y (concat y " " (car l))))
      (setq l (cdr l)))
    y))

;; combine two requests into a single request
;; the request at the point is combined into a second request
(defun req-mode-merge (n)
  "Combine two requests into a single request.

The request on the line with the point is combined into a
second request.  The second request number is prompted for when
this function is choosen."

  (interactive "nRequest number: ")
  (let* ((this-line (count-lines (point-min) (point)))
         (req (elt req-mode-req-array this-line)))
    (call-process req-mode-upd nil
                  (get-buffer-create req-mode-temp-buffer)
                  nil "-merge"
                  (number-to-string req)
                  (number-to-string n))))

;; prepare to create a new request
(defun req-mode-pre-create ()
  "Ask a bunch of questions in preparation for
creating a new request."
  (interactive)
  (let ((u nil)
        (s (read-from-minibuffer "Subject: "))
        (p nil)
        (g nil)
        (l nil))
    (if (not req-mode-user-list)
        (req-mode-build-user-list))
    (setq l (append '(("" 0)) req-mode-user-list))
    (setq u (completing-read "Requestor: " req-mode-user-list nil t
                             (getenv req-mode-user-variable)))
    (setq p (completing-read "Priority: " req-mode-prio-list nil t))
    (setq g (completing-read "Give to: " req-mode-user-list nil t))
    (req-mode-explain 'req-mode-create (list s u p g))))

;; now actually create the request
(defun req-mode-create ()
  "Actually create the request."
  (interactive)
  (message "Creating new request.")
  (goto-char (point-min))
  (mapcar (function (lambda (x)
                      (insert (concat x "\n"))))
          req-mode-data)
  (goto-char (point-max))
  (insert "\n.\n")
  (call-process-region (point-min) (point-max) req-mode-upd
                       nil (get-buffer-create req-mode-temp-buffer)
                       nil "-d" "-create"))

;; give a request to an unknown user
(defun req-mode-pre-give ()
  "Interactively query for the user to give a request to."
  (interactive)
  (let ((u nil))
    (if (not req-mode-user-list)
        (req-mode-build-user-list))
    (setq u (completing-read "Userid: " req-mode-user-list nil t))
    (req-mode-explain 'req-mode-give-user u)))

;; give a request to the indicated user
(defun req-mode-give ()
  "Give a request to the indicated user."
  (interactive)
  (message (concat "Giving request to " req-mode-data))
  (call-process-region (point-min) (point-max) req-mode-upd
                       nil (get-buffer-create req-mode-temp-buffer)
                       nil "-give" req-mode-req-num req-mode-data "-"))

;; prepare to steal a request
(defun req-mode-pre-steal ()
  "Prepare to steal a request."
  (interactive)
  (req-mode-explain 'req-mode-steal))

;; steal an already taken request from another user
(defun req-mode-steal ()
  "Steal a request from the indicated user."
  (interactive)
  (message "Stealing request.")
  (call-process-region (point-min) (point-max) req-mode-upd
                       nil (get-buffer-create req-mode-temp-buffer)
                       nil "-steal" req-mode-req-num "-"))

;; steal an already taken request from another user
(defun req-mode-kill ()
  "Steal a request from the indicated user."
  (interactive)
  (let* ((this-line (count-lines (point-min) (point)))
         (req-num (number-to-string (elt req-mode-req-array this-line)))
         (cur-win (selected-window)))
    (message (concat "Killing request " req-num))
    (switch-to-buffer (get-buffer-create req-mode-temp-buffer))
    (delete-region (point-min) (point-max))
    (goto-char (point-min))
    (insert "y\n")
    (call-process-region (point-min) (point-max) req-mode-upd
                         nil (get-buffer-create req-mode-temp-buffer)
                         nil "-kill" req-num)
    (req-mode-build-request-list)))

;; prepare to accept a new comment
(defun req-mode-pre-comment ()
  "Prepare to accept a new comment."
  (interactive)
  (req-mode-explain 'req-mode-comment))

;; add a comment to a request
(defun req-mode-comment ()
  "Add a comment to a existing request."
  (interactive)
  (message "Adding comment to request.")
  (call-process-region (point-min) (point-max) req-mode-upd
                       nil (get-buffer-create req-mode-temp-buffer)
                       nil "-comment" req-mode-req-num "-"))

;; prepare to resolve a request
(defun req-mode-pre-resolve ()
  "Prepare to resolve a request."
  (interactive)
  (req-mode-explain 'req-mode-resolve))

;; resolve a request
(defun req-mode-resolve ()
  "Resolve a current request."
  (interactive)
  (message "Resolving request.")
  (call-process-region (point-min) (point-max) req-mode-upd
                       nil (get-buffer-create req-mode-temp-buffer)
                       nil "-resolve" req-mode-req-num "-"))

;; prepare to notify a user of changes
(defun req-mode-pre-notify ()
  "Prepare to notify a user."
  (interactive)
  (req-mode-explain 'req-mode-notify))

;; enter a notifying entry
(defun req-mode-notify ()
  "A possibly impromptu conversation occurred with the originator
of the request and that conversation is recorded with a notify
action."
  (interactive)
  (message "Adding notifying entry to request.")
  (call-process-region (point-min) (point-max) req-mode-upd
                       nil (get-buffer-create req-mode-temp-buffer)
                       nil "-notify" req-mode-req-num "-"))

;; prepare to stall a request
(defun req-mode-pre-stall ()
  "Prepare to stall a request."
  (interactive)
  (req-mode-explain 'req-mode-stall))

;; stall a request
(defun req-mode-stall ()
  "When a request can not be worked on or completed due to an
external factor such as waiting on a part or on a user, use the
req-mode-stall function to record that fact."
  (interactive)
  (message "Stalling request.")
  (call-process-region (point-min) (point-max) req-mode-upd
                       nil (get-buffer-create req-mode-temp-buffer)
                       nil "-stall" req-mode-req-num "-"))

;; change the subject of a request
(defun req-mode-subject (s)
  "Used to change the subject of a request."
  (interactive "sNew subject: ")
  (message "Changing subject.")
  (let ((this-line (count-lines (point-min) (point))))
    (call-process req-mode-upd
                  nil (get-buffer-create req-mode-temp-buffer)
                  nil "-subject"
                  (number-to-string (elt req-mode-req-array this-line))
                  s)))

;; take request that is currently unowned
(defun req-mode-take ()
  "Take a currently unowned request."
  (interactive)
  (message "Taking request.")
  (let ((this-line (count-lines (point-min) (point))))
    (call-process req-mode-upd
     nil (get-buffer-create req-mode-temp-buffer)
     nil "-take" (number-to-string (elt req-mode-req-array this-line))))
  (req-mode-build-request-list))

;; untake request that is currently unowned
(defun req-mode-untake ()
  "Take a currently unowned request."
  (interactive)
  (message "Untaking request.")
  (let ((this-line (count-lines (point-min) (point))))
    (call-process req-mode-upd
     nil (get-buffer-create req-mode-temp-buffer)
     nil "-untake" (number-to-string (elt req-mode-req-array this-line))))
  (req-mode-build-request-list))

;; display the request at the cursor position
(defun req-mode-show ()
  "Show the request at the cursor position."
  (interactive)
  (let* ((this-line (count-lines (point-min) (point)))
         (req (elt req-mode-req-array this-line)))
    (req-mode-show-request req)))

;; prepare to change the priority of a request
(defun req-mode-pre-priority (p)
  "Prepare to modify the priority of a request."
  (interactive "cPriority (l/n/h): ")
  (req-mode-explain 'req-mode-priority (char-to-string p)))

;; modify the priority of the current request
(defun req-mode-priority ()
  "Modify the priority of the current request."
  (interactive)
  (message (concat "Changing priority to " req-mode-data))
  (call-process-region (point-min) (point-max) req-mode-upd
                       nil (get-buffer-create req-mode-temp-buffer)
                       nil "-prio" req-mode-req-num req-mode-data "-"))


;; this function creates a buffer for editing explanations
(defun req-mode-explain (sym &optional data)
  "Allow the user to enter an explaination."
  (let ((this-line (count-lines (point-min) (point)))
        (hh (/ (frame-height) 2)))
    (select-window (get-buffer-window req-mode-message-buffer))
    (delete-other-windows)
    (split-window (selected-window) hh)
    (other-window 1)
    (switch-to-buffer (get-buffer-create req-mode-edit-buffer))
    (delete-region (point-min) (point-max))
    (kill-all-local-variables)
    (make-local-variable 'req-mode-function)
    (setq req-mode-function sym)
    (make-local-variable 'req-mode-data)
    (setq req-mode-data data)
    (make-local-variable 'req-mode-req-num)
    (if (elt req-mode-req-array this-line)
        (setq req-mode-req-num
              (number-to-string
               (elt req-mode-req-array this-line)))
      (setq req-mode-req-num nil))
    (use-local-map (make-sparse-keymap))
    (local-set-key "\C-c\C-c" 'req-mode-explain-continue)))

;; continue processing the user's explaination
(defun req-mode-explain-continue ()
  "Continue processing the user's action."
  (interactive)
  (funcall req-mode-function)
  (req-mode-build-request-list))

;; The assumption is that users might no be given accounts on all
;; machines, but that the mail aliases file will have all users
;; listed so that mail can be routed properly.
;; Therefore, parse the mail aliases file for user id's.
(defun req-mode-build-user-list ()
  (let ((buf (find-file-noselect req-mode-mail-alias-file))
        (okill-whole-line kill-whole-line)
        (i 0))
    (setq kill-whole-line t)
    (switch-to-buffer buf)
    (setq buffer-read-only nil)
    (auto-save-mode nil)
    (sort-lines nil (point-min) (point-max))

    (goto-char (point-min))
    (while (not (eobp))
      (if (not (looking-at "^[-A-z0-9]+:"))
          (kill-line)
        (forward-line)))

    (goto-char (point-min))
    (perform-replace ":.*$" "" nil t nil)

    (goto-char (point-min))
    (setq req-mode-user-list '())
    (while (not (eobp))
      (setq i (+ i 1))
      (setq req-mode-user-list
            (append req-mode-user-list
                    (list (list (buffer-substring
                                 (prog2
                                   (beginning-of-line) (point))
                                 (prog2
                                   (end-of-line) (point))) i))))
      (forward-line))
    (not-modified)
    (kill-buffer buf)
    (setq kill-whole-line okill-whole-line)))
