;;; fill.el --- fill commands for Emacs

;; Copyright (C) 1985, 1986, 1992, 1994 Free Software Foundation, Inc.

;; Keywords: wp

;; This file is part of GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Commentary:

;; All the commands for filling text.  These are documented in the Emacs
;; manual.

;;; Code:

(defconst fill-individual-varying-indent nil
  "*Controls criterion for a new paragraph in `fill-individual-paragraphs'.
Non-nil means changing indent doesn't end a paragraph.
That mode can handle paragraphs with extra indentation on the first line,
but it requires separator lines between paragraphs.
A value of nil means that any change in indentation starts a new paragraph.")

(defconst sentence-end-double-space t
  "*Non-nil means a single space does not end a sentence.")

(defun set-fill-prefix ()
  "Set the fill prefix to the current line up to point.
Filling expects lines to start with the fill prefix and
reinserts the fill prefix in each resulting line."
  (interactive)
  (setq fill-prefix (buffer-substring
		     (save-excursion (beginning-of-line) (point))
		     (point)))
  (if (equal fill-prefix "")
      (setq fill-prefix nil))
  (if fill-prefix
      (message "fill-prefix: \"%s\"" fill-prefix)
    (message "fill-prefix cancelled")))

(defconst adaptive-fill-mode t
  "*Non-nil means determine a paragraph's fill prefix from its text.")

(defconst adaptive-fill-regexp (if (= regexp-version 19)
				   "[ \t]*\\([>*] +\\)?"
				 ;; for emacs 18 regexp routine
				 "[ \t]+\\|[ \t]*[>*] +")
  "*Regexp to match text at start of line that constitutes indentation.
If Adaptive Fill mode is enabled, whatever text matches this pattern
on the second line of a paragraph is used as the standard indentation
for the paragraph.")

;; use this instead of 
;; (skip-chars-backward "^ \t\n")
;; (skip-chars-backward "^ \n" linebeg)
(defun fill-move-backward-to-break-point (regexp &optional lim)
  (let ((opoint (point)))
    ;; 93.8.23 by kawamoto@ics.es.osaka-u.ac.jp
    ;;     --- $B:G=i$N8l$,(Bfill-column $B$h$jD9$$>l9g(B
    (if (not (re-search-backward regexp lim 'move))
	nil
      ;; we have skipped backward SPC or WAN.  so move to after them.
      (forward-char)
      (if (<= opoint (point))
	  (forward-char -1)))))

;; use instead of
;; (re-search-forward "[ \t]" opoint t)
;; (skip-chars-forward "^ \n")
;; (skip-chars-forward "^ \n")
(defun fill-move-forward-to-break-point (regexp &optional lim)
  (let ((opoint (point)))
    (if (not (re-search-forward regexp lim 'move))
	nil
      (forward-char -1)
      (if (<= (point) opoint)
	  (forward-char))))
  (kinsoku-shori-nobashi))

(defun fill-end-of-sentence-p ()
  (save-excursion
    (skip-chars-backward " ]})\"'")
    (memq (preceding-char) '(?. ?? ?!))))

(defun fill-region-as-paragraph (from to &optional justify-flag)
  "Fill region as one paragraph: break lines to fit `fill-column'.
Prefix arg means justify too.
If `sentence-end-double-space' is non-nil, then period followed by one
space does not end a sentence, so don't break a line there.
From program, pass args FROM, TO and JUSTIFY-FLAG."
  (interactive "r\nP")
  ;; Arrange for undoing the fill to restore point.
  (if (and buffer-undo-list (not (eq buffer-undo-list t)))
      (setq buffer-undo-list (cons (point) buffer-undo-list)))
  ;; Don't let Adaptive Fill mode alter the fill prefix permanently.
  (let ((fill-prefix fill-prefix))
    ;; Figure out how this paragraph is indented, if desired.
    (if (and adaptive-fill-mode
	     (or (null fill-prefix) (string= fill-prefix "")))
	(save-excursion
	  (goto-char (min from to))
	  (if (eolp) (forward-line 1))
	  (forward-line 1)
	  (if (< (point) (max from to))
	      (if (looking-at adaptive-fill-regexp)
		  (setq fill-prefix (buffer-substring (point) (match-end 0))))
	    (goto-char (min from to))
	    (if (eolp) (forward-line 1))
	    ;; If paragraph has only one line, don't assume in general
	    ;; that additional lines would have the same starting
	    ;; decoration.  Assume no indentation.
	    ;; But if left-margin is nonzero, we can assume ordinary
	    ;; lines do have indentation.
	    (if (> left-margin 0)
		(progn
		  (re-search-forward adaptive-fill-regexp)
		  (setq fill-prefix (make-string (current-column) ?\ ))))
	    )))

    (save-restriction
      (narrow-to-region from to)
      (goto-char (point-min))
      (skip-chars-forward "\n")
      (narrow-to-region (point) (point-max))
      (setq from (point))
      (goto-char (point-max))
      (let ((fpre (and fill-prefix (not (equal fill-prefix ""))
		       (regexp-quote fill-prefix))))
	;; Delete the fill prefix from every line except the first.
	;; The first line may not even have a fill prefix.
	(and fpre
	     (progn
	       (if (>= (string-width fill-prefix) fill-column)
		   (error "fill-prefix too long for specified width"))
	       (goto-char (point-min))
	       (forward-line 1)
	       (while (not (eobp))
		 (if (looking-at fpre)
		     (delete-region (point) (match-end 0)))
		 (forward-line 1))
	       (goto-char (point-min))
					; 92.3.26 by M.Kuwada
	       (and (looking-at fpre) (goto-char (match-end 0)))
	       (setq from (point)))))
      ;; from is now before the text to fill,
      ;; but after any fill prefix on the first line.

      ;; Make sure sentences ending at end of line get an extra space.
      ;; loses on split abbrevs ("Mr.\nSmith")
      (goto-char from)
      ;; patch by S.Tomura 88-Jun-30
      ;;$B!cE}9g!d(B
      ;; . + CR             ==> . + SPC + SPC 
      ;; . + SPC + CR +     ==> . + SPC + 
      ;;(while (re-search-forward "[.?!][])}\"']*$" nil t)
      ;;  (insert ? ))
      (while (re-search-forward "[.?!][])}\"']*$" nil t)
	(if (eobp)
	    nil
	  ;; replace CR by two spaces.
	  ;; insert before delete to preserve marker.
	  (insert "  ")
	  ;; delete newline
	  (delete-char 1)))
      ;; end of patch

      ;; Then change all newlines to spaces.
      ;; (subst-char-in-region from (point-max) ?\n ?\ )
      ;;
      ;; patched by S.Tomura 87-Dec-7
      ;; bug fixed by S.Tomura 88-May-25
      ;; modified by  S.Tomura 88-Jun-21
      ;; modified by K.Handa 92-Mar-2
      ;; Spacing is not necessary for charcters of no word-separater.
      ;; The regexp word-across-newline is used for this check.
      (if (not (stringp word-across-newline))
	  (subst-char-in-region from (point-max) ?\n ?\ )
	;;
	;; WAN     +NL+WAN       --> WAN            + WAN
	;; not(WAN)+NL+WAN       --> not(WAN)       + WAN
	;; WAN     +NL+not(WAN)  --> WAN            + not(WAN)
	;; SPC     +NL+not(WAN)  --> SPC            + not(WAN)
	;; not(WAN)+NL+not(WAN)  --> not(WAN) + SPC + not(WAN)
	;;
	(goto-char from)
	(end-of-line)
	(while (not (eobp))
	  ;; 92.8.26 , 92.8.30 by S. Tomura

	  ;; Insert SPC only when point is between nonWAN.  Insert
	  ;; before deleting to preserve marker if possible.
	  (if (or (prog2		; check following char.
		      (forward-char)	; skip newline
		      (or (eobp)
			  (looking-at word-across-newline))
		    (forward-char -1))
		  (prog2		; check previous char.
		      (forward-char -1)
		      (or (eq (following-char) ?\ )
			  (looking-at word-across-newline))
		    (forward-char)))
	      nil
	    (insert ?\ ))
	  (delete-char 1)		; delete newline
	  (end-of-line)))
      ;; After the following processing, there's two spaces at end of sentence
      ;; and single space at end of line within sentence.
      ;; end of patch

      ;; Flush excess spaces, except in the paragraph indentation.
      (goto-char from)
      (skip-chars-forward " \t")
      ;; Nuke tabs while we're at it; they get screwed up in a fill.
      ;; This is quick, but loses when a tab follows the end of a sentence.
      ;; Actually, it is difficult to tell that from "Mr.\tSmith".
      ;; Blame the typist.
      (subst-char-in-region (point) (point-max) ?\t ?\ )
      (while (re-search-forward "   *" nil t)
	(delete-region
	 (+ (match-beginning 0)
	    (if (and sentence-end-double-space
		     (fill-end-of-sentence-p))
		2 1))
	 (match-end 0)))
      (goto-char (point-max))
      (delete-horizontal-space)
      (insert "  ")
      (goto-char (point-min))

      ;; This is the actual filling loop.
      (let ((prefixcol 0) linebeg
	    ;; patch by K.Handa 92-Mar-2
	    (re-break-point (concat "[ \n]\\|" word-across-newline))
	    ;; end of patch
	    )
	(while (not (eobp))
	  (setq linebeg (point))
	  (move-to-column (1+ fill-column))
	  (if (eobp)
	      nil
	    ;; Move back to start of word.
	    ;; (skip-chars-backward "^ \n" linebeg)
	    (fill-move-backward-to-break-point re-break-point linebeg)
	    ;; Don't break after a period followed by just one space.
	    ;; Move back to the previous place to break.
	    ;; The reason is that if a period ends up at the end of a line,
	    ;; further fills will assume it ends a sentence.
	    ;; If we now know it does not end a sentence,
	    ;; avoid putting it at the end of the line.
	    (if sentence-end-double-space
		(while (and (> (point) (+ linebeg 2))
			    (eq (preceding-char) ?\ )
			    (not (eq (following-char) ?\ ))
			    (eq (char-after (- (point) 2)) ?\.))
		  (forward-char -2)
		  (fill-move-backward-to-break-point re-break-point linebeg)))
	    (kinsoku-shori)
	    (if (if (zerop prefixcol)
		    (save-excursion
		      (skip-chars-backward " " linebeg)
		      (bolp))
		  (>= prefixcol (current-column)))
		;; Keep at least one word even if fill prefix exceeds margin.
		;; This handles all but the first line of the paragraph.
		;; Meanwhile, don't stop at a period followed by one space.
		(let ((first t))
		  (move-to-column prefixcol)
		  (while (and (not (eobp))
			      (or first
				  (and (not (bobp))
				       sentence-end-double-space
				       (save-excursion (forward-char -1)
						       (and (looking-at "\\. ")
							    (not (looking-at "\\.  ")))))))
		    (skip-chars-forward " ")
		    ;; (skip-chars-forward "^ \n")
		    (fill-move-forward-to-break-point re-break-point)
		    (setq first nil)))
	      ;; Normally, move back over the single space between the words.
	      (if (eq (preceding-char) ? )
		  (forward-char -1)))
	    (if (and fill-prefix (zerop prefixcol)
		     (< (- (point) (point-min)) (length fill-prefix))
		     (string= (buffer-substring (point-min) (point))
			      (substring fill-prefix 0 (- (point) (point-min)))))
		;; Keep at least one word even if fill prefix exceeds margin.
		;; This handles the first line of the paragraph.
		;; Don't stop at a period followed by just one space.
		(let ((first t))
		  (while (and (not (eobp))
			      (or first
				  (and (not (bobp))
				       sentence-end-double-space
				       (save-excursion (forward-char -1)
						       (and (looking-at "\\. ")
							    (not (looking-at "\\.  ")))))))
		    (skip-chars-forward " ")
		    ;; (skip-chars-forward "^ \n")
		    (fill-move-forward-to-break-point re-break-point)
		    (setq first nil)))))

	  ;; Replace all whitespace here with one newline.
	  ;; Insert before deleting, so we don't forget which side of
	  ;; the whitespace point or markers used to be on.
	  (skip-chars-backward " ")
	  ;; patch by S. Tomura 88-Jun-20
	  ;; 92.4.27 by K.Handa
	  (if mc-flag
	      ;; $B!cJ,3d!d(B  WAN means chars which match word-across-newline.
	      ;; (0)     | SPC + SPC* <EOB>	--> NL
	      ;; (1) WAN | SPC + SPC*		--> WAN + SPC + NL
	      ;; (2)     | SPC + SPC* + WAN	--> SPC + NL  + WAN
	      ;; (3) '.' | SPC + nonSPC		--> '.' + SPC + NL + nonSPC
	      ;; (4) '.' | SPC + SPC		--> '.' + NL
	      ;; (5)     | SPC*			--> NL
	      (let ((start (point))	; 92.6.30 by K.Handa
		    (ch (following-char)))
		(if (and (= ch ? )
			 (progn		; not case (0) -- 92.6.30 by K.Handa
			   (skip-chars-forward " \t")
			   (not (eobp)))
			 (or
			  (progn	; case (1)
			    (goto-char start)
			    (forward-char -1)
			    (looking-at word-across-newline))
			  (progn	; case (2)
			    (goto-char start)
			    (skip-chars-forward " \t")
			    (and (not (eobp))
				 (looking-at word-across-newline)
				 ;; never leave space after the end of sentence
				 (not (fill-end-of-sentence-p))))
			  (progn	; case (3)
			    (goto-char (1+ start))
			    (and (not (eobp))
				 (/= (following-char) ? )
				 (fill-end-of-sentence-p)))))
		    ;; We should keep one SPACE before NEWLINE. (1),(2),(3)
		    (goto-char (1+ start))
		  ;; We should delete all SPACES around break point. (4),(5)
		  (goto-char start))))
	  ;; end of patch
	  (insert ?\n)
	  (delete-horizontal-space)
	  ;; Insert the fill prefix at start of each line.
	  ;; Set prefixcol so whitespace in the prefix won't get lost.
	  (and (not (eobp)) fill-prefix (not (equal fill-prefix ""))
	       (progn
		 (insert fill-prefix)
		 (setq prefixcol (current-column))))
	  ;; Justify the line just ended, if desired.
	  (and justify-flag (not (eobp))
	       (progn
		 (forward-line -1)
		 (justify-current-line)
		 (forward-line 1))))))))

(defun fill-paragraph (arg)
  "Fill paragraph at or after point.  Prefix arg means justify as well.
If `sentence-end-double-space' is non-nil, then period followed by one
space does not end a sentence, so don't break a line there."
  (interactive "P")
  (let ((before (point))
	(bolp (bolp)))
    (save-excursion
      (forward-paragraph)
      (or (bolp) (newline 1))
      (let ((end (point))
	    (beg (progn (backward-paragraph) (point))))
	(goto-char before)
	(fill-region-as-paragraph beg end arg)))
    ;; if around point matches word-across-newline and \n is inserted again.
    (if (and bolp (eolp)) (forward-char 1))))

(defun fill-region (from to &optional justify-flag)
  "Fill each of the paragraphs in the region.
Prefix arg (non-nil third arg, if called from program) means justify as well.
If `sentence-end-double-space' is non-nil, then period followed by one
space does not end a sentence, so don't break a line there."
  (interactive "r\nP")
  (save-restriction
   (narrow-to-region from to)
   (goto-char (point-min))
   (while (not (eobp))
     (let ((initial (point))
	   (end (progn
		 (forward-paragraph 1) (point))))
       (forward-paragraph -1)
       (if (>= (point) initial)
	   (fill-region-as-paragraph (point) end justify-flag)
	 (goto-char end))))))

;;; The following code is valid only for Japanese.

;;; patch by S.Tomura 88-Jun-2, 89-Nov-15
;;; $BF|K\8l$N(Bjustification$B$G$O!"H>3Q6uGr$rF~$l$i$l$k>l=j$O(B
;;; "$B!#!"(B"$B$N8e(B
;;; ""$B$NA0(B
;;; $B1QC18l$HF|K\8l$H$N4V(B
;;; $B$G$"$k!#(B

(defvar ascii-char "[\40-\176]")

(defvar ascii-space "[ \t]")
(defvar ascii-symbols "[\40-\57\72-\100\133-\140\173-\176]")
(defvar ascii-numeric "[\60-\71]")
(defvar ascii-English-Upper "[\101-\132]")
(defvar ascii-English-Lower "[\141-\172]")

(defvar ascii-alphanumeric "[\60-\71\101-\132\141-\172]")

(defvar kanji-char "\\cj")
(defvar kanji-space "$B!!(B")
(defvar kanji-symbols "\\cS")
(defvar kanji-numeric "[$B#0(B-$B#9(B]")
(defvar kanji-English-Upper "[$B#A(B-$B#Z(B]")
(defvar kanji-English-Lower  "[$B#a(B-$B#z(B]")
;;; Bug fixed by Yoshida@CSK on 88-AUG-24
(defvar kanji-hiragana "[$B$!(B-$B$s(B]")
(defvar kanji-katakana "[$B%!(B-$B%v(B]")
;;;
(defvar kanji-Greek-Upper "[$B&!(B-$B&8(B]")
(defvar kanji-Greek-Lower "[$B&A(B-$B&X(B]")
(defvar kanji-Russian-Upper "[$B'!(B-$B'A(B]")
(defvar kanji-Russian-Lower "[$B'Q(B-$B'q(B]")
(defvar kanji-Kanji-1st-Level  "[$B0!(B-$BOS(B]")
(defvar kanji-Kanji-2nd-Level  "[$BP!(B-$Bt$(B]")

(defvar kanji-kanji-char "\\(\\cH\\|\\cK\\|\\cC\\)")

(defvar aletter (concat "\\(" ascii-char "\\|" kanji-char "\\)"))

(defvar kanji-space-insertable (concat 
	   "$B!"(B" aletter                   "\\|"
	   "$B!#(B" aletter                   "\\|"
	   aletter "$B!J(B"                   "\\|"
	   "$B!K(B" aletter                   "\\|"
	   ascii-alphanumeric  kanji-kanji-char "\\|"
	   kanji-kanji-char    ascii-alphanumeric ))

(defvar space-insertable (concat
	  " " aletter                     "\\|"
	  kanji-space-insertable))

(defun find-space-insertable-point ()
  (if (re-search-backward space-insertable nil t)
      (progn (forward-char 1)
	     t)
    nil))
;;; end of patch

(defun justify-current-line ()
  "Add spaces to line point is in, so it ends at `fill-column'."
  (interactive)
  (save-excursion
   (save-restriction
    (let (ncols beg indent end)
      (beginning-of-line)
      (forward-byte (length fill-prefix)) ; 92.3.26 by T.Nakagawa
      (skip-chars-forward " \t")
      (setq indent (current-column))
      (setq beg (point))
      (end-of-line)
      (narrow-to-region beg (point))
      (setq end (point))
      (skip-chars-backward " \t")
      (delete-char (- end (point)))
      (goto-char beg)
      (while (re-search-forward "   *" nil t)
	(delete-region
	 (+ (match-beginning 0)
	    (if (save-excursion
		 (skip-chars-backward " ])\"'")
		 (memq (preceding-char) '(?. ?? ?!)))
		2 1))
	 (match-end 0)))
      (goto-char beg)
      (while (re-search-forward "[.?!][])\"']*\n" nil t)
	(forward-char -1)
	(insert-and-inherit ? ))
      (goto-char (point-max))
      ;; Note that the buffer bounds start after the indentation,
      ;; so the columns counted by INDENT don't appear in (current-column).
      (setq ncols (- fill-column (current-column) indent))
      (if (find-space-insertable-point) ;; (search-backward " " nil t)
	  (while (> ncols 0)
	    (let ((nmove (+ 3 (random 3))))
	      (while (> nmove 0)
		(or (find-space-insertable-point) ;;(search-backward " " nil t)
		    (progn
		     (goto-char (point-max))
		     ;; (search-backward " ")
		     (find-space-insertable-point)))
		(skip-chars-backward " ")
		(setq nmove (1- nmove))))
	    (insert-and-inherit " ")
	    (skip-chars-backward " ")
	    (setq ncols (1- ncols)))))))
  nil)


(defun fill-nonuniform-paragraphs (min max &optional justifyp mailp)
  "Fill paragraphs within the region, allowing varying indentation within each.
This command divides the region into \"paragraphs\",
only at paragraph-separator lines, then fills each paragraph
using as the fill prefix the smallest indentation of any line
in the paragraph.

When calling from a program, pass range to fill as first two arguments.

Optional third and fourth arguments JUSTIFY-FLAG and MAIL-FLAG:
JUSTIFY-FLAG to justify paragraphs (prefix arg),
MAIL-FLAG for a mail message, i. e. don't fill header lines."
  (interactive "r\nP")
  (let ((fill-individual-varying-indent t))
    (fill-individual-paragraphs min max justifyp mailp)))

(defun fill-individual-paragraphs (min max &optional justifyp mailp)
  "Fill paragraphs of uniform indentation within the region.
This command divides the region into \"paragraphs\", 
treating every change in indentation level as a paragraph boundary,
then fills each paragraph using its indentation level as the fill prefix.

When calling from a program, pass range to fill as first two arguments.

Optional third and fourth arguments JUSTIFY-FLAG and MAIL-FLAG:
JUSTIFY-FLAG to justify paragraphs (prefix arg),
MAIL-FLAG for a mail message, i. e. don't fill header lines."
  (interactive "r\nP")
  (save-restriction
    (save-excursion
      (goto-char min)
      (beginning-of-line)
      (if mailp 
	  (while (or (looking-at "[ \t]*[^ \t\n]*:") (looking-at "[ \t]*$"))
	    (if (looking-at "[ \t]*[^ \t\n]*:")
		(search-forward "\n\n" nil 'move)
	      (forward-line 1))))
      (narrow-to-region (point) max)
      ;; Loop over paragraphs.
      (while (progn (skip-chars-forward " \t\n") (not (eobp)))
	(beginning-of-line)
	(let ((start (point))
	      fill-prefix fill-prefix-regexp)
	  ;; Find end of paragraph, and compute the smallest fill-prefix
	  ;; that fits all the lines in this paragraph.
	  (while (progn
		   ;; Update the fill-prefix on the first line
		   ;; and whenever the prefix good so far is too long.
		   (if (not (and fill-prefix
				 (looking-at fill-prefix-regexp)))
		       (setq fill-prefix
			     (buffer-substring (point)
					       (save-excursion (skip-chars-forward " \t") (point)))
			     fill-prefix-regexp
			     (regexp-quote fill-prefix)))
		   (forward-line 1)
		   ;; Now stop the loop if end of paragraph.
		   (and (not (eobp))
			(if fill-individual-varying-indent
			    ;; If this line is a separator line, with or
			    ;; without prefix, end the paragraph.
			    (and 
			(not (looking-at paragraph-separate))
			(save-excursion
			  (not (and (looking-at fill-prefix-regexp)
				    (progn (forward-char (length fill-prefix))
						(looking-at paragraph-separate))))))
			  ;; If this line has more or less indent
			  ;; than the fill prefix wants, end the paragraph.
			  (and (looking-at fill-prefix-regexp)
			       (save-excursion
				 (not (progn (forward-char (length fill-prefix))
					     (or (looking-at paragraph-separate)
						 (looking-at paragraph-start))))))))))
	  ;; Fill this paragraph, but don't add a newline at the end.
	  (let ((had-newline (bolp)))
	    (fill-region-as-paragraph start (point) justifyp)
	    (or had-newline (delete-char -1))))))))

;;; fill.el ends here
