/* ncftpget.c
 *
 * A simple, non-interactive utility to grab files from a remote FTP server.
 * Very useful in shell scripts!
 */

#ifdef HAVE_CONFIG_H
#	include <Config.h>
#endif

#include <unistd.h>
#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>
#include <setjmp.h>

#include <ncftp.h>				/* Library header. */
#include <Strn.h>				/* Library header. */

jmp_buf gJmp;
int gGotSig = 0;
const char *gOperation = "Could not connect to remote host";

extern char *optarg;
extern int optind;

static void
Usage(void)
{
	fprintf(stderr, "Usage: ncftpget [flags] remote-host local-dir remote-path-names...\n");
	fprintf(stderr, "\nFlags:\n\
  -u XX  Use username XX instead of anonymous.\n\
  -p XX  Use password XX with the username.\n\
  -P XX  Use port number XX instead of the default FTP service port (21).\n\
  -d XX  Use the file XX for debug logging.\n\
  -e XX  Use the file XX for error logging.\n\
  -a     Use ASCII transfer type instead of binary.\n\
  -t XX  Timeout after XX seconds.\n");
	fprintf(stderr, "\nExamples:\n\
  ncftpget ftp.wustl.edu . /pub/README /pub/README.too\n\
  ncftpget ftp.wustl.edu . '/pub/README*'\n\
  ncftpget -u gleason -p my.password Cow.probe.net . '/home/mjg/.*rc'\n\
  ncftpget -u gleason Cow.probe.net . /home/mjg/foo.txt  (prompt for password)\n\
  ncftpget -a -d /tmp/debug.log -t 60 ftp.wustl.edu . '/pub/README*'\n");

	fprintf(stderr, "\nLibrary version: %s.\n", gLibNcFTPVersion + 5);
#ifdef UNAME
	fprintf(stderr, "System: %s.\n", UNAME);
#endif
	fprintf(stderr, "\nThis is a freeware program by Mike Gleason (mgleason@probe.net).\n");
	exit(2);
}	/* Usage */



static void
Abort(int sigNum)
{
	static int onceOnly = 0;

	if (onceOnly == 0) {
		++onceOnly;
		gGotSig = sigNum;
		longjmp(gJmp, 1);
	}
}	/* Abort */




static void
Copy(FTPCIPtr cip, char *dstdir, char **files, int binary)
{
	int i;
	int result;
	char *file;

	for (i=0; ; i++) {
		file = files[i];
		if (file == NULL)
			break;
		if (binary != 0)
			result = FTPGetFiles(cip, file, dstdir, kRecursiveNo, kGlobYes);
		else
			result = FTPGetFilesAscii(cip, file, dstdir, kRecursiveNo, kGlobYes);
		if (result != 0)
			fprintf(stderr, "ncftpget: file retrieval error: %s.\n", FTPStrError(result));
	}
}	/* Copy */




void
main(int argc, char **argv)
{
	int result, c;
	volatile int binary = 1;
	FTPLibraryInfo li;
	FTPConnectionInfo fi;
	char *dstdir;
#ifdef HAVE_GETPASS
	char *password;
#endif

	result = FTPInitLibrary(&li);
	if (result < 0) {
		fprintf(stderr, "ncftpget: init library error %d (%s).\n", result, FTPStrError(result));
		exit(3);
	}
	result = FTPInitConnectionInfo(&li, &fi, kDefaultFTPBufSize);
	if (result < 0) {
		fprintf(stderr, "ncftpget: init connection info error %d (%s).\n", result, FTPStrError(result));
		exit(4);
	}

	fi.debugLog = NULL;
	fi.errLog = stderr;
	STRNCPY(fi.user, "anonymous");
	fi.netTimeout = (unsigned int) 180;

	while ((c = getopt(argc, argv, "P:u:p:e:d:t:a")) > 0) switch(c) {
		case 'P':
			fi.port = atoi(optarg);	
			break;
		case 'u':
			STRNCPY(fi.user, optarg);
			break;
		case 'p':
			STRNCPY(fi.pass, optarg);	/* Don't recommend doing this! */
			break;
		case 'e':
			if (strcmp(optarg, "stdout") == 0)
				fi.errLog = stdout;
			else if (optarg[0] == '-')
				fi.errLog = stdout;
			else if (strcmp(optarg, "stderr") == 0)
				fi.errLog = stderr;
			else
				fi.errLog = fopen(optarg, "a");
			break;
		case 'd':
			if (strcmp(optarg, "stdout") == 0)
				fi.debugLog = stdout;
			else if (optarg[0] == '-')
				fi.debugLog = stdout;
			else if (strcmp(optarg, "stderr") == 0)
				fi.debugLog = stderr;
			else
				fi.debugLog = fopen(optarg, "a");
			break;
		case 't':
			fi.netTimeout = (unsigned int) atoi(optarg);	
			break;
		case 'a':
			binary = 0;
			break;
		default:
			Usage();
	}
	if (optind > argc - 3)
		Usage();

#ifdef HAVE_GETPASS
	if (strcmp(fi.user, "anonymous") && strcmp(fi.user, "ftp")) {
		if (fi.pass[0] == '\0') {
			password = getpass("Password: ");		
			if (password != NULL) {
				STRNCPY(fi.pass, password);
				/* Don't leave cleartext password in memory. */
				memset(password, 0, strlen(fi.pass));
			}
		}
	}
#endif

	STRNCPY(fi.host, argv[optind]);
	if (setjmp(gJmp) == 0) {
		signal(SIGINT, Abort);
		signal(SIGTERM, Abort);
		signal(SIGALRM, Abort);
		if ((result = FTPOpenHost(&fi)) < 0) {
			fprintf(stderr, "ncftpget: cannot open %s: %s.\n", fi.host, FTPStrError(result));
			exit(5);
		}
		dstdir = argv[optind + 1];
		gOperation = "could not read file from remote host";
		Copy(&fi, dstdir, argv + optind + 2, binary);
	} else {
		signal(SIGALRM, SIG_IGN);
		signal(SIGINT, SIG_IGN);
		signal(SIGTERM, SIG_IGN);
		if (gGotSig == SIGALRM) {
			fprintf(stderr, "ncftpget: %s: timed-out.\n", gOperation);
		} else {
			fprintf(stderr, "\nncftpget: caught signal, cleaning up...\n");
		}
	}

	if (setjmp(gJmp) == 0) {
		signal(SIGINT, Abort);
		signal(SIGTERM, Abort);
		FTPCloseHost(&fi);
	} else {
		signal(SIGALRM, SIG_IGN);
		signal(SIGINT, SIG_IGN);
		signal(SIGTERM, SIG_IGN);
		if (gGotSig == SIGALRM) {
			fprintf(stderr, "ncftpget: %s: timed-out.\n", gOperation);
		} else {
			fprintf(stderr, "\nncftpget: caught signal, cleaning up...\n");
		}
	}
	exit(0);
}	/* main */
