/* ncftpput.c
 *
 * A simple, non-interactive utility to send files to a remote FTP server.
 * Very useful in shell scripts!
 */

#ifdef HAVE_CONFIG_H
#	include <Config.h>
#endif

#include <unistd.h>
#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>
#include <setjmp.h>

#include <ncftp.h>				/* Library header. */
#include <Strn.h>				/* Library header. */

jmp_buf gJmp;
int gGotSig = 0;
const char *gOperation = "Could not connect to remote host";

extern char *optarg;
extern int optind;

static void
Usage(void)
{
	fprintf(stderr, "Usage: ncftpput [flags] remote-host remote-dir local-files...\n");
	fprintf(stderr, "\nFlags:\n\
  -u XX  Use username XX instead of anonymous.\n\
  -p XX  Use password XX with the username.\n\
  -P XX  Use port number XX instead of the default FTP service port (21).\n\
  -d XX  Use the file XX for debug logging.\n\
  -e XX  Use the file XX for error logging.\n\
  -U XX  Use value XX for the umask.\n\
  -t XX  Timeout after XX seconds.\n\
  -a     Use ASCII transfer type instead of binary.\n\
  -m     Attempt to mkdir the dstdir before copying.\n");
	fprintf(stderr, "\nExamples:\n\
  ncftpput -u gleason -p my.password Elwood.probe.net /home/gleason stuff.txt\n\
  ncftpput -u gleason Elwood.probe.net /home/gleason a.txt (prompt for pass)\n\
  ncftpput -a -u gleason -p my.password -m -U 007 Cow.probe.net /tmp/tmpdir a.txt\n");
	fprintf(stderr, "\nLibrary version: %s.\n", gLibNcFTPVersion + 5);
#ifdef UNAME
	fprintf(stderr, "System: %s.\n", UNAME);
#endif
	fprintf(stderr, "\nThis is a freeware program by Mike Gleason (mgleason@probe.net).\n");
	exit(2);
}	/* Usage */



static void
Abort(int sigNum)
{
	static int onceOnly = 0;

	if (onceOnly == 0) {
		++onceOnly;
		gGotSig = sigNum;
		longjmp(gJmp, 1);
	}
}	/* Abort */




static void
Copy(FTPCIPtr cip, char *dstdir, char **files, int binary)
{
	int i;
	int result;
	char *file;

	for (i=0; ; i++) {
		file = files[i];
		if (file == NULL)
			break;
		if (binary != 0)
			result = FTPPutFiles(cip, file, dstdir, kRecursiveNo, kGlobNo);
		else
			result = FTPPutFilesAscii(cip, file, dstdir, kRecursiveNo, kGlobNo);
		if (result != 0)
			fprintf(stderr, "ncftpput: file send error: %s.\n", FTPStrError(result));
	}
}	/* Copy */



void
main(int argc, char **argv)
{
	int result, c;
	volatile int binary = 1;
	FTPLibraryInfo li;
	FTPConnectionInfo fi;
	volatile int wantMkdir = 0;
	char *volatile Umask = NULL;
	char *dstdir;
#ifdef HAVE_GETPASS
	char *password;
#endif

	result = FTPInitLibrary(&li);
	if (result < 0) {
		fprintf(stderr, "ncftpput: init library error %d (%s).\n", result, FTPStrError(result));
		exit(3);
	}
	result = FTPInitConnectionInfo(&li, &fi, kDefaultFTPBufSize);
	if (result < 0) {
		fprintf(stderr, "ncftpput: init connection info error %d (%s).\n", result, FTPStrError(result));
		exit(4);
	}

	fi.debugLog = NULL;
	fi.errLog = stderr;
	STRNCPY(fi.user, "anonymous");
	fi.netTimeout = (unsigned int) 180;

	while ((c = getopt(argc, argv, "P:u:p:e:d:U:t:ma")) > 0) switch(c) {
		case 'P':
			fi.port = atoi(optarg);	
			break;
		case 'u':
			STRNCPY(fi.user, optarg);
			break;
		case 'p':
			STRNCPY(fi.pass, optarg);	/* Don't recommend doing this! */
			break;
		case 'e':
			if (strcmp(optarg, "stdout") == 0)
				fi.errLog = stdout;
			else if (optarg[0] == '-')
				fi.errLog = stdout;
			else if (strcmp(optarg, "stderr") == 0)
				fi.errLog = stderr;
			else
				fi.errLog = fopen(optarg, "a");
			break;
		case 'd':
			if (strcmp(optarg, "stdout") == 0)
				fi.debugLog = stdout;
			else if (optarg[0] == '-')
				fi.debugLog = stdout;
			else if (strcmp(optarg, "stderr") == 0)
				fi.debugLog = stderr;
			else
				fi.debugLog = fopen(optarg, "a");
			break;
		case 'U':
			Umask = optarg;
			break;
		case 't':
			fi.netTimeout = (unsigned int) atoi(optarg);	
			break;
		case 'm':
			wantMkdir = 1;
			break;
		case 'a':
			binary = 0;	/* Use ascii. */
			break;
		default:
			Usage();
	}
	if (optind > argc - 3)
		Usage();

#ifdef HAVE_GETPASS
	if (strcmp(fi.user, "anonymous") && strcmp(fi.user, "ftp")) {
		if (fi.pass[0] == '\0') {
			password = getpass("Password: ");		
			if (password != NULL) {
				STRNCPY(fi.pass, password);
				/* Don't leave cleartext password in memory. */
				memset(password, 0, strlen(fi.pass));
			}
		}
	}
#endif

	STRNCPY(fi.host, argv[optind]);
	if (setjmp(gJmp) == 0) {
		signal(SIGINT, Abort);
		signal(SIGTERM, Abort);
		signal(SIGALRM, Abort);
		if ((result = FTPOpenHost(&fi)) < 0) {
			fprintf(stderr, "ncftpput: cannot open %s: %s.\n", fi.host, FTPStrError(result));
			exit(5);
		}
		dstdir = argv[optind + 1];
		if (Umask != NULL) {
			gOperation = "could not set umask on remote host";
			result = FTPUmask(&fi, Umask);
			if (result != 0)
				fprintf(stderr, "ncftpput: %s.\n", FTPStrError(result));
		}
		if (wantMkdir != 0) {
			gOperation = "could not mkdir on remote host";
			result = FTPMkdir(&fi, dstdir, kRecursiveYes);
			if (result != 0)
				fprintf(stderr, "ncftpput: %s.\n", FTPStrError(result));
		}
		gOperation = "could not write to file on remote host";
		Copy(&fi, dstdir, argv + optind + 2, binary);
	} else {
		signal(SIGALRM, SIG_IGN);
		signal(SIGINT, SIG_IGN);
		signal(SIGTERM, SIG_IGN);
		if (gGotSig == SIGALRM) {
			fprintf(stderr, "ncftpput: %s: timed-out.\n", gOperation);
		} else {
			fprintf(stderr, "\nncftpput: caught signal, cleaning up...\n");
		}
	}

	if (setjmp(gJmp) == 0) {
		signal(SIGINT, Abort);
		signal(SIGTERM, Abort);
		FTPCloseHost(&fi);
	} else {
		signal(SIGALRM, SIG_IGN);
		signal(SIGINT, SIG_IGN);
		signal(SIGTERM, SIG_IGN);
		if (gGotSig == SIGALRM) {
			fprintf(stderr, "ncftpput: %s: timed-out.\n", gOperation);
		} else {
			fprintf(stderr, "\nncftpput: caught signal, cleaning up...\n");
		}
	}
	exit(0);
}	/* main */
